/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Christopher Blizzard.
 * Portions created by the Initial Developer are Copyright (C) 2001
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Christopher Blizzard <blizzard@mozilla.org>
 *   Chris Lord <chris@linux.intel.com>
 *   Neil Roberts <neil@linux.intel.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "nsCOMPtr.h"
#include "nsIDOMMouseEvent.h"

#include "nsIDOMKeyEvent.h"
#include "nsIDOMUIEvent.h"
#include "nsIDOMEventTarget.h"
#include "nsIDOMHTMLLinkElement.h"

#include "HeadlessEventListener.h"
#include "HeadlessPrivate.h"

HeadlessEventListener::HeadlessEventListener(void)
{
  mOwner = nsnull;
}

HeadlessEventListener::~HeadlessEventListener()
{
}

NS_IMPL_ADDREF(HeadlessEventListener)
NS_IMPL_RELEASE(HeadlessEventListener)
NS_INTERFACE_MAP_BEGIN(HeadlessEventListener)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, nsIDOMKeyListener)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsIDOMEventListener, nsIDOMKeyListener)
  NS_INTERFACE_MAP_ENTRY(nsIDOMKeyListener)
  NS_INTERFACE_MAP_ENTRY(nsIDOMMouseListener)
  NS_INTERFACE_MAP_ENTRY(nsIDOMUIListener)
NS_INTERFACE_MAP_END

nsresult
HeadlessEventListener::Init(HeadlessPrivate *aOwner)
{
  mOwner = aOwner;
  return NS_OK;
}

PRBool
HeadlessEventListener::StringListFind(const nsString &haystack,
                                      const char *needle)
{
  // Utility function to check case-insentively if needle is in the
  // space-separated list of strings in haystack.

  const PRUnichar *hp, *tokend, *hend;

  hp = haystack.BeginReading();
  hend = haystack.EndReading();

  while (true) {
    // Skip leading whitespace
    while (hp < hend && *hp == ' ')
      hp++;
    // If we've reached the end of the string then it wasn't found
    if (hp >= hend)
      return PR_FALSE;
    // Look for the end of the token
    for (tokend = hp + 1; tokend < hend && *tokend != ' '; tokend++);
    // Compare the substrings
    if (Substring(hp, tokend).LowerCaseEqualsASCII(needle))
      return PR_TRUE;
    // Jump to next token
    hp = tokend;
  }
}

NS_IMETHODIMP
HeadlessEventListener::HandleEvent(nsIDOMEvent* aDOMEvent)
{
  nsresult rv;
  nsAutoString type;

  rv = aDOMEvent->GetType(type);
  if (NS_FAILED(rv))
    return rv;

  // Check if the event is link added type
  if (type.Equals(NS_LITERAL_STRING("DOMLinkAdded"))) {
    // Get the link element
    nsCOMPtr<nsIDOMEventTarget> eventTarget;
    rv = aDOMEvent->GetTarget(getter_AddRefs(eventTarget));
    if (NS_FAILED(rv))
      return rv;
    nsCOMPtr<nsIDOMHTMLLinkElement> linkElement
      = do_QueryInterface(eventTarget, &rv);
    if (NS_FAILED(rv))
      return rv;

    // Get the list of relationships for this link
    nsAutoString rel;
    rv = linkElement->GetRel(rel);

    // Check if it is the icon link
    if (StringListFind(rel, "icon")) {
      nsAutoString href;
      rv = linkElement->GetHref(href);
      if (NS_FAILED(rv))
        return rv;
      mOwner->SetIcon(NS_ConvertUTF16toUTF8(href).get());
      g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                    moz_headless_signals[ICON], 0);
    }
  }

  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::KeyDown(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMKeyEvent> keyEvent;
  keyEvent = do_QueryInterface(aDOMEvent);
  if (!keyEvent)
    return NS_OK;
  // Return FALSE to this function to mark the event as not
  // consumed...
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_KEY_DOWN], 0,
                (void *)keyEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::KeyUp(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMKeyEvent> keyEvent;
  keyEvent = do_QueryInterface(aDOMEvent);
  if (!keyEvent)
    return NS_OK;
  // return FALSE to this function to mark this event as not
  // consumed...
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_KEY_UP], 0,
                (void *)keyEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::KeyPress(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMKeyEvent> keyEvent;
  keyEvent = do_QueryInterface(aDOMEvent);
  if (!keyEvent)
    return NS_OK;
  // Return TRUE from your signal handler to mark the event as consumed.
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_KEY_PRESS], 0,
                (void *)keyEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::MouseDown(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr<nsIDOMMouseEvent> mouseEvent;
  mouseEvent = do_QueryInterface(aDOMEvent);
  if (!mouseEvent)
    return NS_OK;
  // Return TRUE from your signal handler to mark the event as consumed.
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_MOUSE_DOWN], 0,
                (void *)mouseEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::MouseUp(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMMouseEvent> mouseEvent;
  mouseEvent = do_QueryInterface(aDOMEvent);
  if (!mouseEvent)
    return NS_OK;
  // Return TRUE from your signal handler to mark the event as consumed.
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_MOUSE_UP], 0,
                (void *)mouseEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::MouseClick(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMMouseEvent> mouseEvent;
  mouseEvent = do_QueryInterface(aDOMEvent);
  if (!mouseEvent)
    return NS_OK;
  // Return TRUE from your signal handler to mark the event as consumed.
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_MOUSE_CLICK], 0,
                (void *)mouseEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::MouseDblClick(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMMouseEvent> mouseEvent;
  mouseEvent = do_QueryInterface(aDOMEvent);
  if (!mouseEvent)
    return NS_OK;
  // Return TRUE from your signal handler to mark the event as consumed.
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_MOUSE_DBL_CLICK], 0,
                (void *)mouseEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::MouseOver(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMMouseEvent> mouseEvent;
  mouseEvent = do_QueryInterface(aDOMEvent);
  if (!mouseEvent)
    return NS_OK;
  // Return TRUE from your signal handler to mark the event as consumed.
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_MOUSE_OVER], 0,
                (void *)mouseEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::MouseOut(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMMouseEvent> mouseEvent;
  mouseEvent = do_QueryInterface(aDOMEvent);
  if (!mouseEvent)
    return NS_OK;
  // Return TRUE from your signal handler to mark the event as consumed.
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_MOUSE_OUT], 0,
                (void *)mouseEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::Activate(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMUIEvent> uiEvent = do_QueryInterface(aDOMEvent);
  if (!uiEvent)
    return NS_OK;
  // Return TRUE from your signal handler to mark the event as consumed.
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_ACTIVATE], 0,
                (void *)uiEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::FocusIn(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMUIEvent> uiEvent = do_QueryInterface(aDOMEvent);
  if (!uiEvent)
    return NS_OK;
  // Return TRUE from your signal handler to mark the event as consumed.
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_FOCUS_IN], 0,
                (void *)uiEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

NS_IMETHODIMP
HeadlessEventListener::FocusOut(nsIDOMEvent* aDOMEvent)
{
  nsCOMPtr <nsIDOMUIEvent> uiEvent = do_QueryInterface(aDOMEvent);
  if (!uiEvent)
    return NS_OK;
  // Return TRUE from your signal handler to mark the event as consumed.
  gint return_val = FALSE;
  g_signal_emit(G_OBJECT(mOwner->mOwningObject),
                moz_headless_signals[DOM_FOCUS_OUT], 0,
                (void *)uiEvent, &return_val);
  if (return_val) {
    aDOMEvent->StopPropagation();
    aDOMEvent->PreventDefault();
  }
  return NS_OK;
}

