/**************************************************************************
 * 
 * Copyright 2007 Tungsten Graphics, Inc., Cedar Park, Texas.
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL TUNGSTEN GRAPHICS AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 **************************************************************************/

/* Authors:  Keith Whitwell <keith@tungstengraphics.com>
 */
#include "imports.h"

#define CLIP_PRIVATE
#include "clip/clip_context.h"

#define CLIP_PIPE_PRIVATE
#include "clip/clip_pipe.h"



struct unfilled_stage {
   struct clip_pipe_stage stage;

   GLuint mode[2];
};


static INLINE struct unfilled_stage *unfilled_stage( struct clip_pipe_stage *stage )
{
   return (struct unfilled_stage *)stage;
}


static void unfilled_begin( struct clip_pipe_stage *stage )
{
   struct unfilled_stage *unfilled = unfilled_stage(stage);

   unfilled->mode[0] = stage->pipe->clip->state.fill_ccw;
   unfilled->mode[1] = stage->pipe->clip->state.fill_cw;

   stage->next->begin( stage->next );
}

static void point( struct clip_pipe_stage *stage,
		   struct vertex_header *v0 )
{
   struct prim_header tmp;
   tmp.v[0] = v0;
   stage->next->point( stage->next, &tmp );
}

static void line( struct clip_pipe_stage *stage,
		  GLboolean reset_line_stipple,
		  struct vertex_header *v0,
		  struct vertex_header *v1 )
{
   struct prim_header tmp;
   tmp.reset_line_stipple = reset_line_stipple;
   tmp.edgeflags = 0;
   tmp.pad = 0;
   tmp.v[0] = v0;
   tmp.v[1] = v1;
   stage->next->line( stage->next, &tmp );
}


static void points( struct clip_pipe_stage *stage,
		    struct prim_header *header )
{
   struct vertex_header *v0 = header->v[0];
   struct vertex_header *v1 = header->v[1];
   struct vertex_header *v2 = header->v[2];

   if (header->edgeflags & 0x1) point( stage, v0 );
   if (header->edgeflags & 0x2) point( stage, v1 );
   if (header->edgeflags & 0x4) point( stage, v2 );
}

static void lines( struct clip_pipe_stage *stage,
		   struct prim_header *header )
{
   struct vertex_header *v0 = header->v[0];
   struct vertex_header *v1 = header->v[1];
   struct vertex_header *v2 = header->v[2];

   if (header->edgeflags & 0x4) line( stage, header->reset_line_stipple, v2, v0 );
   if (header->edgeflags & 0x1) line( stage, 0, v0, v1 );
   if (header->edgeflags & 0x2) line( stage, 0, v1, v2 );
}


/* Unfilled tri:  
 *
 * Note edgeflags in the vertex struct is not sufficient as we will
 * need to manipulate them when decomposing primitives???
 */
static void unfilled_tri( struct clip_pipe_stage *stage,
			  struct prim_header *header )
{
   struct unfilled_stage *unfilled = unfilled_stage(stage);
   GLuint mode = unfilled->mode[header->det < 0];
  
   switch (mode) {
   case FILL_TRI:
      stage->next->tri( stage->next, header );
      break;

   case FILL_LINE:
      lines( stage, header );
      break;

   case FILL_POINT:
      points( stage, header );
      break;
   }   
}

static void unfilled_line( struct clip_pipe_stage *stage,
		       struct prim_header *header )
{
   stage->next->line( stage->next, header );
}


static void unfilled_point( struct clip_pipe_stage *stage,
			struct prim_header *header )
{
   stage->next->point( stage->next, header );
}


static void unfilled_end( struct clip_pipe_stage *stage )
{
   stage->next->end( stage->next );
}

struct clip_pipe_stage *clip_pipe_unfilled( struct clip_pipeline *pipe )
{
   struct unfilled_stage *unfilled = CALLOC_STRUCT(unfilled_stage);

   clip_pipe_alloc_tmps( &unfilled->stage, 0 );

   unfilled->stage.pipe = pipe;
   unfilled->stage.next = NULL;
   unfilled->stage.tmp = NULL;
   unfilled->stage.begin = unfilled_begin;
   unfilled->stage.point = unfilled_point;
   unfilled->stage.line = unfilled_line;
   unfilled->stage.tri = unfilled_tri;
   unfilled->stage.reset_tmps = clip_pipe_reset_tmps;
   unfilled->stage.end = unfilled_end;

   return &unfilled->stage;
}
