/**************************************************************************
 * 
 * Copyright 2007 Tungsten Graphics, Inc., Cedar Park, Texas.
 * All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL TUNGSTEN GRAPHICS AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 **************************************************************************/

 /*
  * Authors:
  *   Keith Whitwell <keith@tungstengraphics.com>
  */

#include "imports.h"

#define CLIP_PRIVATE
#include "clip/clip_context.h"

#define CLIP_PIPE_PRIVATE
#include "clip/clip_pipe.h"


static INLINE struct clip_pipeline *clip_pipeline( struct clip_render *render )
{
   return (struct clip_pipeline *)render;
}


static struct vertex_header *get_vertex( struct clip_pipeline *pipe,
					 GLuint i )
{
   return (struct vertex_header *)(pipe->verts + i * pipe->vertex_size);
}

static struct vertex_header *get_indexed_vertex( struct clip_pipeline *pipe,
						 GLuint i )
{
   return (struct vertex_header *)(pipe->verts + pipe->elts[i] * pipe->vertex_size);
}



static void *pipe_allocate_vertices( struct clip_render *render,
				     GLuint vertex_size,
				     GLuint nr_vertices )
{
   struct clip_pipeline *pipe = clip_pipeline( render );

   pipe->vertex_size = vertex_size;
   pipe->nr_vertices = nr_vertices;
   pipe->verts = MALLOC( nr_vertices * pipe->vertex_size );

   assert(pipe->need_validate == 0);

   pipe->first->begin( pipe->first );

   return pipe->verts;
}

static void pipe_set_prim( struct clip_render *render,
			   GLenum prim )
{
   struct clip_pipeline *pipe = clip_pipeline( render );


   pipe->prim = prim;
}
			  

static const GLuint pipe_prim[GL_POLYGON+1] = {
   PRIM_POINT,
   PRIM_LINE,
   PRIM_LINE,
   PRIM_LINE,
   PRIM_TRI,
   PRIM_TRI,
   PRIM_TRI,
   PRIM_TRI,
   PRIM_TRI,
   PRIM_TRI
};






static void do_point( struct clip_pipeline *draw,
		      unsigned i0 )
{
   struct prim_header prim;
   
   prim.reset_line_stipple = 1;
   prim.edgeflags = 1;
   prim.pad = 0;
   prim.v[0] = draw->get_vertex( draw, i0 );
   draw->first->point( draw->first, &prim );
}


static void do_line( struct clip_pipeline *draw,
		     GLboolean reset_stipple,
		     unsigned i0,
		     unsigned i1 )
{
   struct prim_header prim;
   
   prim.reset_line_stipple = reset_stipple;
   prim.edgeflags = 1;
   prim.pad = 0;
   prim.v[0] = draw->get_vertex( draw, i0 );
   prim.v[1] = draw->get_vertex( draw, i1 );
   draw->first->line( draw->first, &prim );
}

static void do_triangle( struct clip_pipeline *draw,
			 unsigned i0,
			 unsigned i1,
			 unsigned i2 )
{
   struct prim_header prim;
   
   prim.reset_line_stipple = 1;
   prim.edgeflags = ~0;
   prim.pad = 0;
   prim.v[0] = draw->get_vertex( draw, i0 );
   prim.v[1] = draw->get_vertex( draw, i1 );
   prim.v[2] = draw->get_vertex( draw, i2 );
   draw->first->tri( draw->first, &prim );
}
			  

/* A hack -- looks like prim.edgeflags wasn't a good idea, will have
 * to remove.
 */
static void apply_ef( struct prim_header *prim, 
		      unsigned mask )
{
   prim->edgeflags = mask;
   prim->v[0]->edgeflag = (mask >> 0) & 1;
   prim->v[1]->edgeflag = (mask >> 1) & 1;
   prim->v[2]->edgeflag = (mask >> 2) & 1;
}

static void do_ef_triangle( struct clip_pipeline *draw,
			    GLboolean reset_stipple,
			    unsigned ef_mask,
			    unsigned i0,
			    unsigned i1,
			    unsigned i2 )
{
   struct prim_header prim;
   struct vertex_header *v0 = draw->get_vertex( draw, i0 );
   struct vertex_header *v1 = draw->get_vertex( draw, i1 );
   struct vertex_header *v2 = draw->get_vertex( draw, i2 );
   unsigned tmp;

   prim.reset_line_stipple = reset_stipple;

   /* Hmm -- this won't work because in the clipper we just go back to
    * the vertex edgeflags...  Edgeflags in the prim header aren't so
    * useful afterall.
    */
   
   tmp = ((v0->edgeflag << 0) | 
	  (v1->edgeflag << 1) | 
	  (v2->edgeflag << 2));

   

   prim.pad = 0;
   prim.v[0] = v0;
   prim.v[1] = v1;
   prim.v[2] = v2;

   /* Also update the edgeflags in the vertices.  
    */
   apply_ef( &prim, ef_mask & tmp);
   draw->first->tri( draw->first, &prim );
   apply_ef( &prim, tmp);
}


static void do_quad( struct clip_pipeline *draw,
		     unsigned v0,
		     unsigned v1,
		     unsigned v2,
		     unsigned v3 )
{
#if 1
   const unsigned omitEdge2 = ~(1 << 1);
   const unsigned omitEdge3 = ~(1 << 2);
   do_ef_triangle( draw, 1, omitEdge2, v0, v1, v3 );
   do_ef_triangle( draw, 0, omitEdge3, v1, v2, v3 );
#else
   /* This works for unfilled+stipple, but breaks flatshading...
    */
   const unsigned omitEdge1 = ~(1 << 0);
   const unsigned omitEdge3 = ~(1 << 2);
   do_ef_triangle( draw, 1, omitEdge3, v0, v1, v2 );
   do_ef_triangle( draw, 0, omitEdge1, v0, v2, v3 );
#endif
}




/**
 * Main entrypoint to draw some number of points/lines/triangles
 */
static void
draw_prim( struct clip_pipeline *draw, unsigned start, unsigned count )
{
   unsigned i;

//   _mesa_printf("%s (%d) %d/%d\n", __FUNCTION__, draw->prim, start, count );

   switch (draw->prim) {
   case GL_POINTS:
      for (i = 0; i < count; i ++) {
	 do_point( draw,
		   start + i );
      }
      break;

   case GL_LINES:
      for (i = 0; i+1 < count; i += 2) {
	 do_line( draw, 
		  GL_TRUE,
		  start + i + 0,
		  start + i + 1);
      }
      break;

   case GL_LINE_LOOP:  
      if (count >= 2) {
	 for (i = 1; i < count; i++) {
	    do_line( draw, 
		     i == 1, 	/* XXX: only if vb not split */
		     start + i - 1,
		     start + i );
	 }

	 do_line( draw, 
		  0,
		  start + count - 1,
		  start + 0 );
      }
      break;

   case GL_LINE_STRIP:
      if (count >= 2) {
	 for (i = 1; i < count; i++) {
	    do_line( draw,
		     i == 1,
		     start + i - 1,
		     start + i );
	 }
      }
      break;

   case GL_TRIANGLES:
      for (i = 0; i+2 < count; i += 3) {
	 do_ef_triangle( draw,
			 1, 
			 ~0,
			 start + i + 0,
			 start + i + 1,
			 start + i + 2 );
      }
      break;

   case GL_TRIANGLE_STRIP:
      for (i = 0; i+2 < count; i++) {
	 if (i & 1) {
	    do_triangle( draw,
			 start + i + 1,
			 start + i + 0,
			 start + i + 2 );
	 }
	 else {
	    do_triangle( draw,
			 start + i + 0,
			 start + i + 1,
			 start + i + 2 );
	 }
      }
      break;

   case GL_TRIANGLE_FAN:
      if (count >= 3) {
	 for (i = 0; i+2 < count; i++) {
	    do_triangle( draw,
			 start + 0,
			 start + i + 1,
			 start + i + 2 );
	 }
      }
      break;


   case GL_QUADS:
      for (i = 0; i+3 < count; i += 4) {
	 do_quad( draw,
		  start + i + 0,
		  start + i + 1,
		  start + i + 2,
		  start + i + 3);
      }
      break;

   case GL_QUAD_STRIP:
      for (i = 0; i+3 < count; i += 2) {
	 do_quad( draw,
		  start + i + 2,
		  start + i + 0,
		  start + i + 1,
		  start + i + 3);
      }
      break;

   case GL_POLYGON:
      if (count >= 3) {
	 unsigned ef_mask = (1<<2) | (1<<0);

	 for (i = 0; i+2 < count; i++) {

            if (i + 3 >= count)
	       ef_mask |= (1<<1);

	    do_ef_triangle( draw,
			    i == 0,
			    ef_mask,
			    start + i + 1,
			    start + i + 2,
			    start + 0);

	    ef_mask &= ~(1<<2);
	 }
      }
      break;

   default:
      assert(0);
      break;
   }
}




static void pipe_draw_indexed_prim( struct clip_render *render,
				    const GLuint *elts,
				    GLuint count )
{
   struct clip_pipeline *pipe = clip_pipeline( render );

   pipe->get_vertex = get_indexed_vertex;
   pipe->elts = elts;

   draw_prim( pipe, 0, count );
}

static void pipe_draw_prim( struct clip_render *render,
			    GLuint start,
			    GLuint count )
{
   struct clip_pipeline *pipe = clip_pipeline( render );

   pipe->get_vertex = get_vertex;
   pipe->elts = NULL;

   draw_prim( pipe, start, count );
}


static void pipe_release_vertices( struct clip_render *render,
				   void *vertices,
				   GLuint vertex_size,
				   GLuint vertices_used )
{
   struct clip_pipeline *pipe = clip_pipeline( render );

   pipe->first->end( pipe->first );

   FREE(pipe->verts);
   pipe->verts = NULL;
}

static void pipe_destroy( struct clip_render *render )
{
   struct clip_pipeline *pipe = clip_pipeline( render );
//   _mesa_printf("%s\n", __FUNCTION__);

   _mesa_free(pipe);
}

struct clip_render *clip_create_prim_render( struct clip_context *draw )
{
   struct clip_pipeline *pipe = CALLOC_STRUCT(clip_pipeline);

   /* XXX: Add casts here to avoid the compiler messages:
    */
   pipe->render.start_render = NULL;
   pipe->render.allocate_vertices = pipe_allocate_vertices;
   pipe->render.set_prim = pipe_set_prim;
   pipe->render.draw_prim = pipe_draw_prim;
   pipe->render.draw_indexed_prim = pipe_draw_indexed_prim;
   pipe->render.release_vertices = pipe_release_vertices;
   pipe->render.flush = NULL;
   pipe->render.destroy = pipe_destroy;

   pipe->clip = draw;
   pipe->prim = 0;

   pipe->emit      = clip_pipe_emit( pipe );
   pipe->unfilled  = clip_pipe_unfilled( pipe );
   pipe->twoside   = clip_pipe_twoside( pipe );
   pipe->offset    = clip_pipe_offset( pipe );
   pipe->clipper   = clip_pipe_clip( pipe );
   pipe->flatshade = clip_pipe_flatshade( pipe );
   pipe->cull      = clip_pipe_cull( pipe );
   pipe->wide      = clip_pipe_wide( pipe );
   pipe->stipple   = clip_pipe_stipple( pipe );

   return &pipe->render;
}



GLboolean clip_pipe_validate_state( struct clip_render *render )
{
   struct clip_pipeline *pipe = clip_pipeline( render );
   
   /* Dependent on driver state and primitive:
    */
   struct clip_pipe_stage *next = pipe->emit;
   GLboolean install = GL_FALSE;
   GLboolean need_flat = GL_FALSE;
   const GLbitfield active_prims = clip_get_active_prims(pipe->clip);

   pipe->need_validate = 0;

   if (pipe->clip->state.force_prim_pipe)
      install = GL_TRUE;

   /* Some primitive state needs to be revalidated according to new
    * unfilled modes.
    */
   pipe_set_prim( &pipe->render, pipe->prim );

   /* Note: always draw lines with quads to be conformant */
   if ((active_prims & (1 << GL_LINES)) ||
       ((active_prims & (1 << GL_POINTS)) && 
	(pipe->clip->state.point_size != 1.0 ||
	 pipe->clip->state.use_vertex_pointsize))) 
   {
      pipe->wide->next = next;
      next = pipe->wide;
      install = GL_TRUE;
   }


   if ((active_prims & (1 << GL_LINES)) && 
       pipe->clip->state.line_stipple_pattern != 0xffff) 
   {
      pipe->stipple->next = next;
      next = pipe->stipple;
      install = GL_TRUE;
      need_flat = GL_TRUE;
   }
   


   if (pipe->clip->vb_prims & (1 << GL_TRIANGLES)) 
   {   
      if (pipe->clip->state.fill_cw != FILL_TRI ||
	  pipe->clip->state.fill_ccw != FILL_TRI) {

	 pipe->unfilled->next = next;
	 next = pipe->unfilled;
	 install = GL_TRUE;
      }
	 
      if (install && 
	  (pipe->clip->state.offset_cw ||
	   pipe->clip->state.offset_ccw)) {
	 pipe->offset->next = next;
	 next = pipe->offset;
      }

      if (pipe->clip->state.light_twoside) {
	 pipe->twoside->next = next;
	 next = pipe->twoside;
	 install = GL_TRUE;
      }

      /* All the above require the determinant which is calculated
       * below.  Can't cull before clipping as we don't have ndc
       * coords for clipped vertices?
       */
      if (install)
      {
	 pipe->cull->next = next;
	 next = pipe->cull;
      }
   }


   if (pipe->clip->vb_state.clipped_prims) {
      pipe->clipper->next = next;
      next = pipe->clipper;
      install = GL_TRUE;
      need_flat = GL_TRUE;
   }

   if (need_flat) {
      if (pipe->clip->state.flatshade) {
	 pipe->flatshade->next = next;
	 next = pipe->flatshade;
      }
   }
   

//   install = GL_TRUE;

   pipe->first = next;
   return install;
}


void clip_pipe_set_hw_render( struct clip_render *render,
			       struct clip_render *hw )
{
}


void clip_pipe_set_clip_state( struct clip_render *render,
				struct clip_state *state )
{
   struct clip_pipeline *pipe = clip_pipeline( render ); 

   pipe->clip->revalidate = 1;
   pipe->need_validate = 1;
}

void clip_pipe_invalidate( struct clip_render *render )
{
   struct clip_pipeline *pipe = clip_pipeline( render ); 

   pipe->clip->revalidate = 1;
   pipe->need_validate = 1;
}


void clip_pipe_reset_vertex_indices( struct clip_pipeline *pipe )
{
   GLuint i;

   for (i = 0; i < pipe->nr_vertices; i++) {
      struct vertex_header *v0 = get_vertex( pipe, i );
      v0->index = ~0;
   }

   pipe->first->reset_tmps( pipe->first );
}


#define MAX_VERTEX_SIZE ((2 + FRAG_ATTRIB_MAX) * 4 * sizeof(GLfloat))

void clip_pipe_alloc_tmps( struct clip_pipe_stage *stage, GLuint nr )
{
   stage->nr_tmps = nr;

   if (nr) {
      GLubyte *store = MALLOC(MAX_VERTEX_SIZE * nr);
      GLuint i;

      stage->tmp = MALLOC(sizeof(struct vertex_header *) * nr);
      
      for (i = 0; i < nr; i++)
	 stage->tmp[i] = (struct vertex_header *)(store + i * MAX_VERTEX_SIZE);
   }
}

void clip_pipe_free_tmps( struct clip_pipe_stage *stage )
{
   if (stage->tmp) {
      FREE(stage->tmp[0]);
      FREE(stage->tmp);
   }
}


void clip_pipe_reset_tmps( struct clip_pipe_stage *stage )
{
   GLuint i;

   for (i = 0; i < stage->nr_tmps; i++)
      stage->tmp[i]->index = ~0;

   stage->next->reset_tmps( stage->next );
}


/* Helpers:
 */
void clip_passthrough_tri( struct clip_pipe_stage *stage,
			   struct prim_header *header )
{
   stage->next->tri( stage->next, header );
}

void clip_passthrough_line( struct clip_pipe_stage *stage,
			    struct prim_header *header )
{
   stage->next->line( stage->next, header );
}

void clip_passthrough_point( struct clip_pipe_stage *stage,
			     struct prim_header *header )
{
   stage->next->point( stage->next, header );
}
