/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "task-list.h"

#include <libwnck/libwnck.h>

G_DEFINE_TYPE (TaskList, task_list, GTK_TYPE_HBOX);

#define TASK_LIST_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  TASK_TYPE_LIST, \
  TaskListPrivate))

struct _TaskListPrivate
{
  WnckScreen *screen;
  GHashTable *win_table;
  guint timer;
  guint counter;

  gboolean show_all_windows;
};

enum
{
  PROP_0,
  
  PROP_SHOW_ALL_WINDOWS
};

/* D&D stuff */
static const GtkTargetEntry drop_types[] =
{
  { "STRING", 0, 0 },
  { "text/plain", 0, 0},
  { "text/uri-list", 0, 0}
};
static const gint n_drop_types = G_N_ELEMENTS(drop_types);

static void
update_hints (WnckWindow *window, GtkWidget *button)
{
  GtkWidget *parent;
  gint x, y, x1, y1;

  /* Skip problems */
  if (!WNCK_IS_WINDOW (window)) return;
  if (!GTK_IS_WIDGET (button)) return;

  /* Skip invisible windows */
  if (!GTK_WIDGET_VISIBLE (button)) return;

  x = y = 0;

  /* Recursively compute the button's coordinates */
  for (parent = button; parent; parent = parent->parent)
  {
    if (parent->parent)
    {
      x += parent->allocation.x;
      y += parent->allocation.y;
    }
    else
    {
      x1 = y1 = 0;
      if (GDK_IS_WINDOW (parent->window))
        gdk_window_get_origin (parent->window, &x1, &y1);
      x += x1; y += y1;
      break;
    }
  }
    
  /* Set the minimize hint for the window */
  wnck_window_set_icon_geometry (window, x, y,
                                 button->allocation.width,
                                 button->allocation.height);
}

static void
set_minimize_hints (TaskList *list)
{
  TaskListPrivate *priv;
    
  g_return_if_fail (TASK_IS_LIST (list));
  priv = list->priv;

  g_hash_table_foreach (priv->win_table,
                        (GHFunc)update_hints,
                        NULL);

}

/*
 * Show all windows code 
 */
static gboolean
ensure_list (TaskList *list)
{
  TaskListPrivate *priv = list->priv;
  WnckWorkspace *current;
  GList *children, *c;

  current = wnck_screen_get_active_workspace (priv->screen);

  if (!WNCK_IS_WORKSPACE (current))
    return FALSE;

  children = gtk_container_get_children (GTK_CONTAINER (list));
  for (c = children; c; c = c->next)
  { 
    GtkWidget *widget = c->data;
    WnckWindow *window = g_object_get_data (G_OBJECT (widget), "WnckWindow");

    if (priv->show_all_windows 
        || (wnck_window_is_on_workspace (window, current)
            && wnck_window_is_in_viewport (window, current)))
    {
      if (!wnck_window_is_skip_tasklist (window))
        gtk_widget_show (widget);
    }
    else
    {
      gtk_widget_hide (widget);
    }
  }
  g_list_free (children);

  /* Re-set minimize hints */
  set_minimize_hints (list);

  return FALSE;
}

static void
task_list_set_show_all_windows (TaskList *list, gboolean show_all_windows)
{
  TaskListPrivate *priv = list->priv;

  priv->show_all_windows = show_all_windows;

  ensure_list (list);

  g_debug ("Show all windows: %s", show_all_windows ? "true" : "false");
}

/* Callback */
static gboolean
on_leave_notify_event (GtkWidget        *button,
                       GdkEventCrossing *event,
                       TaskList         *list)
{
  g_return_val_if_fail (TASK_IS_LIST (list), FALSE);

  gtk_widget_queue_draw (GTK_WIDGET (list));

  return FALSE;
}

static gboolean
on_button_released (GtkWidget      *button,
                    GdkEventButton *event,
                    WnckWindow     *window)
{
  g_return_val_if_fail (WNCK_IS_WINDOW (window), FALSE);

  if (event->button == 1)
  {
    WnckWorkspace *space;

    if (wnck_window_is_active (window))
    {
      wnck_window_minimize (window);
    }
    else
    {
      space = wnck_window_get_workspace (window);
      wnck_workspace_activate (space, event->time);
      wnck_window_activate (window, event->time);
    }
    
    return TRUE;
  }

  return FALSE;
}

static gboolean
on_button_pressed (GtkWidget      *button,
                   GdkEventButton *event,
                   WnckWindow     *window)
{
  g_return_val_if_fail (WNCK_IS_WINDOW (window), FALSE);

  if (event->button == 3)
  {
    GtkWidget *menu = wnck_action_menu_new (window);
    gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
                    event->button, event->time);
    return TRUE;
  }

  return FALSE;
}

static gboolean
on_query_tooltip (GtkWidget *button,
                  gint x, gint y, 
                  gboolean keyboard_mode, 
                  GtkTooltip *tooltip,
                  WnckWindow *window)
{
  g_return_val_if_fail (WNCK_IS_WINDOW (window), FALSE);

  gtk_tooltip_set_text (tooltip, wnck_window_get_name(window));
  gtk_tooltip_set_icon (tooltip, wnck_window_get_icon (window));

  return TRUE;
}

static gboolean
activate_window (GtkWidget *button)
{
  gint active;

  g_return_val_if_fail (GTK_IS_WIDGET (button), FALSE);

  active = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "drag-true"));

  if (active)
  { 
    WnckWindow *window;

    window = (WnckWindow*)g_object_get_data (G_OBJECT (button), "WnckWindow");
    if (WNCK_IS_WINDOW (window))
      wnck_window_activate (window, time (NULL));
  }
  
  g_object_set_data (G_OBJECT (button), "drag-true", GINT_TO_POINTER (0));
  
  return FALSE;
}

static void
on_drag_leave (GtkWidget      *button,
               GdkDragContext *context,
               guint           time)
{
  g_object_set_data (G_OBJECT (button), "drag-true", GINT_TO_POINTER (0));
}

static gboolean
on_drag_motion (GtkWidget      *button,
                GdkDragContext *context,
                gint            x, 
                gint            y,
                guint           t)
{
  gint active;

  active = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (button), "drag-true"));
  
  if (!active)
  {
    g_object_set_data (G_OBJECT (button), "drag-true", GINT_TO_POINTER (1));

    g_timeout_add (1000, (GSourceFunc)activate_window, button);
  }

  return FALSE;
}

static void
on_icon_changed (WnckWindow *window, GtkButton *button)
{
  if (WNCK_IS_WINDOW (window) && GTK_IS_BUTTON (button))
  {
    gtk_image_set_from_pixbuf (GTK_IMAGE (gtk_button_get_image (button)), 
                               wnck_window_get_mini_icon (window));
    gtk_widget_queue_draw (GTK_WIDGET (button));
  }
}

static void
set_state (WnckWindow *window, GtkButton *button, gboolean *found)
{
  TaskListPrivate *priv = TASK_LIST (task_list_get_default ())->priv;

  /* Skip problems */
  if (!WNCK_IS_WINDOW (window)) 
    return;
  if (!GTK_IS_WIDGET (button)) 
    return;

  /* Skip invisible windows */
  if (!GTK_WIDGET_VISIBLE (button)) 
    return;

  /* Skip windows that don't need attention */
  if (!wnck_window_or_transient_needs_attention (window)) 
    return;

  /* Blink the button */
  gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), 
                               priv->counter & 1);

  /* Don't destroy the timeout */
  *found = TRUE;
}

static gboolean
on_blink (TaskList *list)
{
  TaskListPrivate *priv;
  gboolean found = FALSE;

  g_return_val_if_fail (TASK_IS_LIST (list), FALSE);
  priv = list->priv;

  /* Increment the blink counter */
  priv->counter ++;

  g_hash_table_foreach (priv->win_table, 
                        (GHFunc)set_state,
                        &found);

  /* Continue the periodic timeout if needed */
  if (!found) priv->timer = 0;
  return found;
}

static void
on_state_changed (WnckWindow      *window, 
                  WnckWindowState  changed_mask,
                  WnckWindowState  new_state,
                  TaskList        *list)
{
  TaskListPrivate *priv;
  GtkWidget *button;
  WnckWorkspace *current;

  g_return_if_fail (WNCK_IS_WINDOW (window));
  g_return_if_fail (TASK_IS_LIST (list));
  priv = list->priv;

  /* Get the button */
  button = g_hash_table_lookup (priv->win_table, window);
  g_return_if_fail (GTK_IS_WIDGET (button));

  /* Get the workspace */
  current = wnck_screen_get_active_workspace (priv->screen);

  /* Check if the button should be hidden */
  if (wnck_window_is_skip_tasklist (window)
      || (current
      && !wnck_window_is_on_workspace (window, current)
      && !priv->show_all_windows))
    gtk_widget_hide (button);
  else
  {
    gtk_widget_show (button);
    
    /* If the window needs attention and there is no timer... */
    if (!priv->timer &&
        wnck_window_or_transient_needs_attention (window) )
    {
      /* Start the timer */
      priv->timer = g_timeout_add (500, (GSourceFunc)on_blink, list);
      priv->counter = 0;
    }
  }

  /* Update the button's state */
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button),
    wnck_screen_get_active_window (priv->screen) == window);

  /* Re-set minimize hints */
  ensure_list (list);
  set_minimize_hints (list);
}

static void
on_workspace_changed (WnckWindow *window, TaskList *list)
{
  g_return_if_fail (TASK_IS_LIST (list));

  ensure_list (list);
}

static void
on_window_opened (WnckScreen *screen, 
                  WnckWindow *window,
                  TaskList   *list)
{
  TaskListPrivate *priv;
  WnckWindowType type;
  GtkWidget *button;

  g_return_if_fail (TASK_IS_LIST (list));
  priv = list->priv;

  type = wnck_window_get_window_type (window);

  if (type == WNCK_WINDOW_DESKTOP
      || type == WNCK_WINDOW_DOCK
      || type == WNCK_WINDOW_SPLASHSCREEN
      || type == WNCK_WINDOW_MENU)
    return;

  button = g_object_new (GTK_TYPE_TOGGLE_BUTTON,
                         "has-tooltip", TRUE,
                         "image", gtk_image_new_from_pixbuf (
                                    wnck_window_get_mini_icon (window)),
                         "name", "tasklist-button2", 
                         "relief", GTK_RELIEF_NONE,
                         NULL);
  /* D&D */
	gtk_widget_add_events (GTK_WIDGET (button),GDK_ALL_EVENTS_MASK);
	gtk_drag_dest_set (GTK_WIDGET (button),
                           GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
                           drop_types, n_drop_types,
                           GDK_ACTION_COPY);
	gtk_drag_dest_add_uri_targets (GTK_WIDGET (button));
	gtk_drag_dest_add_text_targets (GTK_WIDGET (button));
  g_signal_connect (button, "drag-motion", 
                    G_CALLBACK (on_drag_motion), NULL);
  g_signal_connect (button, "drag-leave", 
                    G_CALLBACK (on_drag_leave), NULL);
 
  gtk_box_pack_start (GTK_BOX (list), button, FALSE, FALSE, 0);
  
  g_signal_connect (button, "button-release-event",
                    G_CALLBACK (on_button_released), window);
  g_signal_connect (button, "button-press-event",
                    G_CALLBACK (on_button_pressed), window);
  g_signal_connect (button, "leave-notify-event",
                    G_CALLBACK (on_leave_notify_event), list);
  g_signal_connect (button, "query-tooltip", 
                    G_CALLBACK (on_query_tooltip), window);
  g_signal_connect (window, "icon-changed",
                    G_CALLBACK (on_icon_changed), button);
  g_signal_connect (window, "state-changed",
                    G_CALLBACK (on_state_changed), list);
  g_signal_connect (window, "workspace-changed",
                    G_CALLBACK (on_workspace_changed), list);
  
  g_hash_table_insert (priv->win_table, window, button);
  g_object_set_data (G_OBJECT (button), "WnckWindow", window);

  /* Do an initial state change to avoid code duplication here */
  on_state_changed (window, 0xFFFFFFFF, wnck_window_get_state (window), list);

  /* Show the window if it's button is visible (why??) */
  if (GTK_WIDGET_VISIBLE (button))
    wnck_window_activate (window, GDK_CURRENT_TIME);
}

static void
on_window_closed (WnckScreen *screen, 
                  WnckWindow *window,
                  TaskList   *list)
{
  TaskListPrivate *priv;
  GtkWidget *button;

  g_return_if_fail (TASK_IS_LIST (list));
  priv = list->priv;

  button = g_hash_table_lookup (priv->win_table, window);

  if (GTK_IS_WIDGET (button))
    gtk_widget_destroy (button);

  g_hash_table_remove (priv->win_table, window);

  /* Set minimize hints */
  set_minimize_hints (list);
}

static void
on_active_window_changed (WnckScreen *screen, 
                          WnckWindow *old_window,
                          TaskList   *list)
{
  TaskListPrivate *priv;
  WnckWindow *act_window;
  GtkWidget *old_button;
  GtkWidget *act_button;

  g_return_if_fail (TASK_IS_LIST (list));
  priv = list->priv;

  old_button = g_hash_table_lookup (priv->win_table, old_window);
  if (GTK_IS_WIDGET (old_button))
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (old_button), FALSE);
  }

  act_window = wnck_screen_get_active_window (priv->screen);
  act_button = g_hash_table_lookup (priv->win_table, act_window);
  if (GTK_IS_WIDGET (act_button))
  {
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (act_button), TRUE);
  }

  ensure_list (list);
}

static void
on_active_workspace_changed (WnckScreen    *screen,
                             WnckWorkspace *old_workspace,
                             TaskList      *list)
{
  g_return_if_fail (TASK_IS_LIST (list));

  ensure_list (list);
}

static void
on_active_viewport_changed (WnckScreen    *screen,
                            TaskList      *list)
{
  g_return_if_fail (TASK_IS_LIST (list));

  ensure_list (list);
}
/* GObject stuff */
static void
task_list_finalize (GObject *object)
{
  TaskListPrivate *priv;

  priv = TASK_LIST_GET_PRIVATE (object);

  /* Remove the blink timer */
  if (priv->timer) g_source_remove (priv->timer);

  G_OBJECT_CLASS (task_list_parent_class)->finalize (object);
}

static void
task_list_get_property (GObject    *object,
                        guint       prop_id,
                        GValue     *value,
                        GParamSpec *pspec)
{
  TaskList *list = TASK_LIST (object);
  TaskListPrivate *priv;

  g_return_if_fail (TASK_IS_LIST (list));
  priv = list->priv;

  switch (prop_id)
  {
    case PROP_SHOW_ALL_WINDOWS:
      g_value_set_boolean (value, priv->show_all_windows);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}

static void
task_list_set_property (GObject      *object,
                        guint         prop_id,
                        const GValue *value,
                        GParamSpec   *pspec)
{
  TaskList *list = TASK_LIST (object);
  TaskListPrivate *priv;

  g_return_if_fail (TASK_IS_LIST (list));
  priv = list->priv;

  switch (prop_id)
  {
    case PROP_SHOW_ALL_WINDOWS:
      task_list_set_show_all_windows (list, g_value_get_boolean (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}

static void
task_list_class_init (TaskListClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = task_list_finalize;
  obj_class->set_property = task_list_set_property;
  obj_class->get_property = task_list_get_property;

  g_object_class_install_property (obj_class,
    PROP_SHOW_ALL_WINDOWS,
    g_param_spec_boolean ("show_all_windows",
                          "Show All Windows",
                          "Show windows from all workspaces",
                          TRUE,
                          G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  g_type_class_add_private (obj_class, sizeof (TaskListPrivate));
}

static void
task_list_init (TaskList *list)
{
  TaskListPrivate *priv;
 	
  priv = list->priv = TASK_LIST_GET_PRIVATE (list);

  priv->screen = wnck_screen_get_default ();

  priv->win_table = g_hash_table_new (NULL, NULL);

  /* No blink timer */
  priv->timer = 0;

  gtk_container_set_border_width (GTK_CONTAINER (list), 0);

  g_signal_connect (priv->screen, "window-opened",
                    G_CALLBACK (on_window_opened), list);
  g_signal_connect (priv->screen, "window-closed",
                    G_CALLBACK (on_window_closed), list);
  g_signal_connect (priv->screen, "active-workspace-changed",
                    G_CALLBACK (on_active_workspace_changed), list);
  g_signal_connect (priv->screen, "viewports-changed",
                    G_CALLBACK (on_active_viewport_changed), list);
  g_signal_connect (priv->screen, "active-window-changed",
                    G_CALLBACK (on_active_window_changed), list);
}

GtkWidget *
task_list_new (void)

{
  GtkWidget *list = NULL;

  list = g_object_new (TASK_TYPE_LIST, 
                       "homogeneous", FALSE, 
                       "spacing", 2, 
                       NULL);
  
  g_idle_add ((GSourceFunc)ensure_list, list);
  
  return list;
}

GtkWidget *
task_list_get_default (void)
{
  static GtkWidget *list = NULL;

  if (!list)
    list = task_list_new ();

  return list;
}

gboolean    
task_list_get_desktop_visible (TaskList *list)
{
  GList *children, *c;
  gboolean all_minimised = TRUE;

  g_return_val_if_fail (TASK_IS_LIST (list), TRUE);

  children = gtk_container_get_children (GTK_CONTAINER (list));
  for (c = children; c; c = c->next)
  {
    WnckWindow *window;
    
    window = g_object_get_data (G_OBJECT (c->data), "WnckWindow");

    if (WNCK_IS_WINDOW (window) && !wnck_window_is_minimized (window))
      all_minimised = FALSE;
  }
  g_list_free (children);

  return all_minimised;
}

