/**
 * @file backend/xcb/geis_gestures.h
 * @brief internal GEIS XCB back end class
 *
 * Copyright 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */
#ifndef GEIS_XCB_GRAIL_GESTURES_H_
#define GEIS_XCB_GRAIL_GESTURES_H_

#include <string.h>


/*
 * Clears all the bits in a bitmask.
 */
static inline void
_grail_mask_clear(uint16_t mask_len, uint32_t* mask)
{
  memset(mask, 0, mask_len * sizeof(uint32_t));
}

/*
 * Combines the bits in one bitmask with the bits in another.
 */
static inline void
_grail_mask_or(uint16_t  mask_len, uint32_t* mask_lhs, uint32_t* mask_rhs)
{
  uint16_t i;
  for (i = 0; i < mask_len; ++i)
    mask_lhs[i] |= mask_rhs[i];
}

GeisStatus geis_xcb_backend_add_drag_attrs(GeisGestureClass gesture_class);

GeisStatus geis_xcb_backend_add_pinch_attrs(GeisGestureClass gesture_class);

GeisStatus geis_xcb_backend_add_rotate_attrs(GeisGestureClass gesture_class);

GeisStatus geis_xcb_backend_add_tap_attrs(GeisGestureClass gesture_class);

GeisInteger geis_xcb_backend_primitive_class(GeisInteger grail_gesture_type);

GeisSize geis_xcb_backend_map_grail_attrs(GeisInteger grail_type,
                                          int         grail_version,
                                          GeisSize    num_properties,
                                          float      *properties,
                                          GeisFrame   frame);

/**
 * Initializes the XCB gesture bitmask.
 *
 * @param[out] mask_len  The number of 32-bit flags in the bitmask.
 * @param[out] mask      The bitmask.
 *
 * Marks all gesture bits as set in the mask, depending on the settings of the
 * subscription flags.
 */
void geis_xcb_backend_gesture_bitmask_init(uint16_t              *mask_len,
                                           uint32_t              *mask);

/**
 * Clears the XCB gesture bits for gestures not matching a class.
 *
 * @param[in] class_name The name of the gestire class.
 * @param[in] mask_len   The number of 32-bit flags in the bitmask.
 * @param[in] mask       The bitmask.
 *
 * The XCB gesture botmask bits for any gestired not belonging to the named
 * class are cleared.
 */
void geis_xcb_backend_gesture_bitmask_filter_class(GeisString  class_name,
                                                   uint16_t   *mask_len,
                                                   uint32_t   *mask);

/**
 * Clears the XCB gesture bits for gestures with touches not in a range.
 *
 * @param[in] touches   The number of touches to allow.
 * @param[in] op        The maximum number of touches to allow.
 * @param[in] mask_len  The number of 32-bit flags in the grail bitmask.
 * @param[in] mask      The grail bitmask.
 *
 * The (min, max) pair represents an open interval.  The XCB gesture bitmask bits
 * for any gesture with touches outside of that interval are cleared.
 */
void geis_xcb_backend_gesture_bitmask_filter_touches(GeisInteger          touches,
                                                     GeisFilterOperation  op,
                                                     uint16_t            *mask_len,
                                                     uint32_t            *mask);

/**
 * Clears the XCB gesture bits for gestures that are/are not continuations.
 *
 * @param[in] is_system  Indicates whether the gestires should be system
 *                       gestures or not.
 * @param[in] mask_len   The number of 32-bit flags in the bitmask.
 * @param[in] mask       The bitmask.
 *
 * The XCB gesture botmask bits for any gestired not belonging to the named
 * class are cleared.
 */
void geis_xcb_backend_gesture_bitmask_filter_system(GeisBoolean is_system,
                                                    uint16_t   *mask_len,
                                                    uint32_t   *mask);

void geis_xcb_backend_gesture_bitmask_filter_grabs(GeisBoolean is_grab,
                                                   uint16_t   *mask_len,
                                                   uint32_t   *mask);


#endif /* GEIS_XCB_GRAIL_GESTURES_H_ */
