/**
 * geis_instance_table.h
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.	 See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51
 * Franklin St, Fifth Floor, Boston, MA  02110-1301 US
 */
#include "geis_instance_table.h"

#include "geis_instance.h"
#include <stdlib.h>

#define GROWTH_FACTOR 2


typedef struct _GeisInstanceTableEntry
{
  uint32_t      id;
  GeisInstance  type;
} *GeisInstanceTableEntry;

struct _GeisInstanceTable
{
  size_t                 capacity;
  size_t                 size;
  GeisInstanceTableEntry data;
};


GeisInstanceTable
geis_instance_table_new(int size_hint)
{
  GeisInstanceTable table = malloc(sizeof(struct _GeisInstanceTable));
  if (!table)
  {
    /* TODO: report error */
  }

  table->capacity = size_hint;
  table->size = 0;
  table->data = calloc(table->capacity,
                       size_hint * sizeof(struct _GeisInstanceTableEntry));
  if (!table->data)
  {
    /* TODO: report error */
  }

  return table;
}


/**
 * Adds (or replaces) a gesture type in the table.
 */
void
geis_instance_table_add(GeisInstanceTable table,
                        GeisInstance      instance)
{
  size_t i;
  uint32_t window_id = geis_instance_get_window_id(instance);

  /* If the ID is already present, just replace the gesture type */
  for (i = 0; i < table->size; ++i)
  {
    if (table->data[i].id == window_id)
    {
      table->data[i].type = instance;
      return;
    }
  }

  /* If there is no space to insert, grow the table. */
  if (table->size >= table->capacity)
  {
    size_t new_capacity = table->capacity * GROWTH_FACTOR;
    GeisInstanceTableEntry new_data = realloc(table->data,
                               new_capacity * sizeof(struct _GeisInstanceTableEntry));
    if (!new_data)
    {
      /* TODO: report error */
    }

    table->capacity = new_capacity;
    table->data     = new_data;
  }

  table->data[table->size].id   = window_id;
  table->data[table->size].type = instance;
  ++table->size;
}


GeisInstance
geis_instance_table_get(GeisInstanceTable table,
                        uint32_t          window_id)
{
  size_t i;
  for (i = 0; i < table->size; ++i)
  {
    if (table->data[i].id == window_id)
    {
      return table->data[i].type;
    }
  }
  return NULL;
}


void
geis_instance_table_free(GeisInstanceTable table)
{
  free(table->data);
  free(table);
}

