/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */
using Unity;
using Unity.Place;
using Gee;
 
namespace Unity.ApplicationsPlace.Utils
{ 
  
  public AppInfo? get_app_info_for_actor (string actor)
  {
    string desktop_id = get_desktop_id_for_actor (actor);
    return AppInfoManager.get_instance().lookup (desktop_id);
  }
  
  public bool check_actor_has_category (string actor, Set<string>? categories)
  {
    if (categories == null)
      return false;
  
    AppInfoManager appman = AppInfoManager.get_instance ();
    string desktop_id = get_desktop_id_for_actor (actor);
    AppInfo? appinfo =  appman.lookup (desktop_id);
    
    if (appinfo == null)
      return false;
    
    Gee.List<string>? actor_cats = appman.get_categories (desktop_id);
    if (actor_cats == null)
      return false;
    
    foreach (var cat in actor_cats)
    {
      if (cat in categories)
        return true;
    }
    
    return false;
  }
  
  public string get_desktop_id_for_actor (string actor)
  {
    if (actor.has_prefix("application://"))
      return actor.substring(14).dup();
    else if (actor.has_prefix("app://"))
      return actor.substring(6).dup();
    else if (actor.has_prefix("/"))
      {
        var f = File.new_for_path (actor);
        return f.get_basename ();
      }
    
    /* At this point we assume that @actor is already the basename of the
     * .desktop file */
    return actor;
  }
  
  /* Sloppy and null-safe equality checking. null == "" and strings
   * are stripped of whitespace before comparison */
  private bool search_has_really_changed (Search? old_search,
                                          Search? new_search)
  {
    if (old_search == null && new_search == null)
      return false;
    
    string s1, s2;
   
    if (old_search == null)
      {
        s1 = new_search.get_search_string ();
        if (s1 == null || s1.strip() == "")
          return false;
        else
          return true;
      }
    else if (new_search == null)
      {
        s2 = old_search.get_search_string ();
        if (s2 == null || s2.strip() == "")
          return false;
        else
          return true;
      }
    else
      {
        s1 = new_search.get_search_string ();
        s2 = old_search.get_search_string ();
        if (s1 == null)
          {
            if (s2 == null || s2.strip() == "")
              return false;
            else
              return true;
          }
        else if (s2 == null)
          {
            if (s1 == null || s1.strip() == "")
              return false;
            else
              return true;
          }
        else
          return s1.strip () != s2.strip ();
      }
  }
  
  /* Run through @filter_set and remove any row in which URI is not
   * in @valid_uris */
  public void apply_uri_filter (Set<string> valid_uris,
                                Dee.ResultSet filter_set)
  {
    var model = filter_set.get_model ();
    uint n_removed = 0;
    
    /* Anything in filter_set that is not in the event_uris set, is removed
     * from the results_model */
    foreach (var row in filter_set)
    {
      if (!(model.get_string (row, ResultsColumn.URI) in valid_uris))
        {
          //debug ("Removed: %s", model.get_string (row, ResultsColumn.URI));
          model.remove (row);
          n_removed++;
        }
    }
    
    debug ("Removed %u rows from result set", n_removed);
  }
}
