/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */

using GLib;

namespace Unity.MusicLens {

  public class MusicStoreCollection : Object
  {

    private const string MUSICSTORE_BASE_URI = "http://musicsearch.ubuntu.com/v1/";

    public MusicStoreCollection ()
    {
    }

    public async void search (LensSearch search, SearchType search_type,
			owned List<FilterParser> filters, int max_results = -1, Cancellable cancellable) throws IOError
    {
      string? uri = build_search_uri (search.search_string, filters);

      if (uri == null)
        return;	  

      var results_model = search.results_model;
      File file = File.new_for_uri (uri);

      yield read_musicstore_search_result_async (file, results_model, cancellable);
    }

    private async void read_musicstore_search_result_async (File file, Dee.Model model, Cancellable cancellable)
    {
      var timer = new Timer ();
      debug ("Searching %s", file.get_uri ());

      try {
        var stream = yield file.read_async (Priority.DEFAULT, cancellable);
	var dis = new DataInputStream (stream);
	var parser = new Json.Parser ();
	yield parser.load_from_stream_async (dis, cancellable);
	var root_object = parser.get_root ().get_object ();
	
	foreach (var r in root_object.get_array_member ("results").get_elements ()) {
	  var result_obj = r.get_object ();
	  
	  string type = result_obj.get_string_member ("type");
	  if (type == "track") {
	    Track track = new Track ();
	    track.title = result_obj.get_string_member ("title");
	    track.artist = result_obj.get_string_member ("artist");
	    track.uri = result_obj.get_string_member ("purchase_url");
	    track.artwork_path = result_obj.get_string_member ("image");
	    track.mime_type = "audio-x-generic";

	    // FIXME drag n drop uri needs to be the u1ms:// link
	    model.append (track.uri, track.artwork_path, 2, track.mime_type, track.title, track.artist, track.uri);
	  
	  } else if (type == "album") {
	    Album album = new Album ();
	    album.title = result_obj.get_string_member ("title");
	    album.artist = result_obj.get_string_member ("artist");
	    album.uri = result_obj.get_string_member ("purchase_url");
	    album.artwork_path = result_obj.get_string_member ("image");
	    
	    model.append (album.uri, album.artwork_path, 2, "audio-x-generic", album.title, album.artist, album.uri);
	  }
        }

	debug ("Retrieved '%s' in %fms", file.get_uri (), timer.elapsed()*1000);
	debug ("model has %u rows after search", model.get_n_rows ());

      } catch (Error e) {
	warning ("Error reading URL '%s': %s", file.get_uri (), e.message);
      }
    }

    private string? build_search_uri (string query, List<FilterParser> filters)
    {
      if (query.strip() == "")
        return null;
    
      MusicStoreFilterParser store_parser;
      string musicstore_base_uri = MUSICSTORE_BASE_URI;
      if (GLib.Environment.get_variable("MUSICSTORE_URI") != null)
        musicstore_base_uri = GLib.Environment.get_variable("MUSICSTORE_URI");
      debug ("Using base URI of '%s'", musicstore_base_uri);
      StringBuilder uri = new StringBuilder (musicstore_base_uri);
      uri.append ("search?q=");
      uri.append (Uri.escape_string (query, "", false));

      foreach (FilterParser parser in filters) {
	  if (parser is GenreFilterParser)
	    store_parser = new MusicStoreGenreFilterParser (parser as GenreFilterParser);
	  else if (parser is DecadeFilterParser)
	    store_parser = new MusicStoreDecadeFilterParser (parser as DecadeFilterParser);
	  else
	    continue;

	  uri.append (store_parser.parse ());
      }
      
      uri.append ("&pagesize=10");

      return uri.str;
    }
  }
}
