/*
** ucview_dbus_server.c
** 
*/

/*
  Copyright (C) 2007  Arne Caspari <arne@unicap-imaging.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "ucview_dbus_server.h"
#include <glib.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <unicap.h>
#include <string.h>
#include <gtk/gtk.h>
#include <unicap.h>
#include <unicapgtk.h>


#include "ucview.h"
#include "user_config.h"
#include "device_dialog.h"

enum
{
   PROP_0 = 0,
};



static void server_object_set_property( GObject *object, 
					guint property_id, 
					const GValue *value, 
					GParamSpec *pspec );
static void server_object_get_property( GObject *object, 
					guint property_id, 
					GValue *value, 
					GParamSpec *pspec );


static gboolean server_open_device( ServerObject *obj, gchar *devid, GError **error );
static gboolean server_show_device_dialog( ServerObject *object, GError **error );
static gboolean server_new_window( ServerObject *obj, GError **error );
static gboolean server_list_device_paths( ServerObject *server, gchar ***paths, GError **error );
static void ucview_dbus_destroy_event_cb( GtkWidget *ucv, ServerObject *server );

gboolean server_echo_string( ServerObject *server, gchar *original, gchar **echo, GError **error)
{
   *echo = g_strdup (original);
   
   g_message( "echo_string\n" );

   return TRUE;
}



#include "server-bindings.h"

G_DEFINE_TYPE( ServerObject, server_object, G_TYPE_OBJECT );


static void server_object_class_init(ServerObjectClass *klass)
{
   GObjectClass *object_class;
   GError *error = NULL;

   object_class = G_OBJECT_CLASS( klass );
   
   object_class->set_property = server_object_set_property;
   object_class->get_property = server_object_get_property;

   /* Init the DBus connection, per-klass */
   klass->connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
   if (klass->connection == NULL)
   {
      g_warning("Unable to connect to dbus: %s", error->message);
      g_error_free (error);
   }

}

static void server_object_init(ServerObject *server)
{
   ServerObjectClass *klass = SERVER_OBJECT_GET_CLASS( server );
   DBusGProxy *driver_proxy;
   unsigned int request_ret;
   GError *error = NULL;
   
   server->dbus_paths = 0;
   server->instance_count = 0;

   dbus_g_object_type_install_info( SERVER_OBJECT_TYPE, &dbus_glib_server_object_object_info );

   /* Register DBUS path */
   dbus_g_connection_register_g_object (klass->connection,
					"/org/unicapimaging/UCView",
					G_OBJECT (server));
   
   /*    Register the service name, the constant here are defined in dbus-glib-bindings.h */
   driver_proxy = dbus_g_proxy_new_for_name (klass->connection,
					     DBUS_SERVICE_DBUS,
					     DBUS_PATH_DBUS,
					     DBUS_INTERFACE_DBUS);
	 
   if(!org_freedesktop_DBus_request_name (driver_proxy,
					  "org.unicapimaging.UCView",
					  0, &request_ret,    /* See tutorial for more infos about these */
					  &error))
   {
      g_warning("Unable to register service: %s", error->message);
      g_error_free (error);
   }
	 
   g_object_unref (driver_proxy);

}

static void server_object_set_property( GObject *object, 
					guint property_id, 
					const GValue *value, 
					GParamSpec *pspec )
{
   switch( property_id )
   {

      default:
	 g_warning( "Unknown property\n" );
	 break;
   }
}

static void server_object_get_property( GObject *object, 
					guint property_id, 
					GValue *value, 
					GParamSpec *pspec )
{
   switch( property_id )
   {
      default:
	 g_warning( "Unknown property\n" );
	 break;
   }
}

static void ucview_dbus_destroy_event_cb( GtkWidget *ucv, ServerObject *server )
{
   const gchar *path;
   GList *e;
   gboolean keep_alive;
   
   
   g_object_get( ucv, "keep-dbus-alive", &keep_alive, NULL );

   path = ucview_window_get_dbus_path( UCVIEW_WINDOW( ucv ) );
   
   e = g_list_find( server->dbus_paths, path );
   if( e )
   {
      server->dbus_paths = g_list_remove_link( server->dbus_paths, e );
      g_list_free( e );
   } else
   {
      g_warning( "Expected path in dbus_paths\n" );
   }

   server->instance_count--;
   
   if( !keep_alive )
   {
      if( server->instance_count == 0 )
      {
	 gtk_main_quit();
      }
   }
}


static gboolean server_open_device( ServerObject *server, gchar *devid, GError **error )
{
   unicap_device_t devspec, device;
   unicap_handle_t handle;
   GtkWidget *ucv;
   
   unicap_void_device( &devspec );
   strcpy( devspec.identifier, devid );
   
   if( !SUCCESS( unicap_enumerate_devices( &devspec, &device, 0 ) ) )
   {
      g_set_error( error, g_quark_from_static_string( "open" ), 0x100, "No such device!" );
      return FALSE;
   }

   if( unicap_is_stream_locked( &device ) )
   {
      g_set_error( error, g_quark_from_static_string( "open" ), 0x101, "Device already in use!" );
      return FALSE;
   }

   if( !SUCCESS( unicap_open( &handle, &device ) ) )
   {
      g_set_error( error, g_quark_from_static_string( "open" ), 0x102, "Failed to open device!" );
      return FALSE;
   }
   
   gdk_threads_enter();


   ucv = ucview_window_new( handle );
   user_config_restore_device( handle );
   server->instance_count++;
   g_signal_connect( G_OBJECT( ucv ), "destroy", G_CALLBACK( ucview_dbus_destroy_event_cb ), server );
   gtk_widget_show_all( ucv );
   
   if( ucview_window_get_dbus_path( UCVIEW_WINDOW( ucv ) ) )
   {
      server->dbus_paths = g_list_append( server->dbus_paths, (gpointer)ucview_window_get_dbus_path( UCVIEW_WINDOW( ucv ) ) );
   }
   
   gdk_threads_leave();

   return TRUE;
}

static gboolean server_new_window( ServerObject *server, GError **error )
{
   GtkWidget *ucv;
   GtkWidget *device_dialog;
   unicap_handle_t handle;
   unicap_device_t device;
   
   gdk_threads_enter();

   if( !SUCCESS( unicap_enumerate_devices( NULL, &device, 0 ) ) )
   {
      GtkWidget *dlg;
      dlg = gtk_message_dialog_new( NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
				    "No supported video capture device found!" );
      gtk_dialog_run( GTK_DIALOG( dlg ) );
      gdk_threads_leave();
      gtk_widget_destroy( dlg );
/*       g_set_error( error, 0, 100, "No supported video capture device found!" ); */
      return TRUE;
   }

   device_dialog = g_object_new( DEVICE_DIALOG_TYPE, "restore-device", TRUE, NULL );
   gtk_widget_show_all( device_dialog );
   handle = device_dialog_run( DEVICE_DIALOG( device_dialog ) );
   gtk_widget_destroy( device_dialog );

/*    if( handle ) */
/*    { */
/*       unicap_device_t device; */
      
/*       unicap_get_device( handle, &device ); */
      
/*       if( unicap_is_stream_locked( &device ) ) */
/*       { */
/* 	 unicap_close( handle ); */
/* 	 handle = NULL; */
/*       } */
/*    } */

   if( !handle )
   {
      GtkWidget *dlg;
      dlg = gtk_message_dialog_new( NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
				    "No device!" );
      gtk_dialog_run( GTK_DIALOG( dlg ) );
      gdk_threads_leave();
      gtk_widget_destroy( dlg );
      if( server->instance_count == 0 )
      {
	 gtk_main_quit();
      }
/*       g_set_error( error, 0, 101, "No device selected" ); */
      return TRUE;
   }

   ucv = ucview_window_new( NULL );
   server->instance_count++;
   g_signal_connect( G_OBJECT( ucv ), "destroy", G_CALLBACK( ucview_dbus_destroy_event_cb ), server );
   gtk_widget_show( ucv );
   g_object_set( ucv, "unicap-handle", handle, NULL );
   
   if( ucview_window_get_dbus_path( UCVIEW_WINDOW( ucv ) ) )
   {
      server->dbus_paths = g_list_append( server->dbus_paths, (gpointer)ucview_window_get_dbus_path( UCVIEW_WINDOW( ucv ) ) );
   }
   
   gdk_threads_leave();

   return TRUE;
}

static gboolean server_list_device_paths( ServerObject *server, gchar ***paths, GError **error )
{
   int num_paths;
   GList *e;
   int i;
   gchar **strs;
   
   num_paths = g_list_length( server->dbus_paths );
   strs = g_new( gchar *, num_paths + 1 );
   
   for( e = g_list_first( server->dbus_paths ), i = 0; e; e = g_list_next( e ), i++ )
   {
      gchar *path = (gchar*)e->data;
      
      strs[i] = g_strdup( path );
   }
   strs[i] = NULL;
   *paths = strs;
   
   return TRUE;
}


static gboolean server_show_device_dialog( ServerObject *server, GError **error )
{
   GtkWidget *ucv;
   GtkWidget *device_dialog;
   unicap_handle_t handle;
   GtkWidget *window;
   GtkWidget *vbox;
   GtkWidget *label;
   
   gdk_threads_enter();

   window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
   vbox = gtk_vbox_new( FALSE, 12 );
   gtk_container_add( GTK_CONTAINER( window ), vbox );
   label = gtk_label_new( _("Looking for video capture devices" ) );
   gtk_box_pack_start( GTK_BOX( vbox ), label, FALSE, FALSE, 12 );
   gtk_widget_show_all( window );

   device_dialog = g_object_new( DEVICE_DIALOG_TYPE, "restore-device", FALSE, NULL );
   gtk_widget_show_all( device_dialog );
   gtk_widget_destroy( window );
   handle = device_dialog_run( DEVICE_DIALOG( device_dialog ) );
   gtk_widget_destroy( device_dialog );

   if( handle )
   {
      unicap_device_t device;
      
      unicap_get_device( handle, &device );
      
      if( unicap_is_stream_locked( &device ) )
      {
	 unicap_close( handle );
	 handle = NULL;
      }
   }

   if( !handle )
   {
      GtkWidget *dlg;
      dlg = gtk_message_dialog_new( NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
				    "No device selected." );
      gtk_dialog_run( GTK_DIALOG( dlg ) );
      gdk_threads_leave();
      gtk_widget_destroy( dlg );

      if( server->instance_count == 0 )
      {
	 gtk_main_quit();
      }
      
/*       g_set_error( error, 0, 101, "No device selected" ); */
      return TRUE;
   }

   ucv = ucview_window_new( handle );
   
   user_config_restore_device( handle );
   server->instance_count++;
   g_signal_connect( G_OBJECT( ucv ), "destroy", G_CALLBACK( ucview_dbus_destroy_event_cb ), server );
   gtk_widget_show_all( ucv );
   if( ucview_window_get_dbus_path( UCVIEW_WINDOW( ucv ) ) )
   {
      server->dbus_paths = g_list_append( server->dbus_paths, (gpointer)ucview_window_get_dbus_path( UCVIEW_WINDOW( ucv ) ) );
   }
   
   gdk_threads_leave();

   return TRUE;
}
