# ubuntuone.storageprotocol.client - the storage protocol client
#
# Author: Lucio Torre <lucio.torre@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License version 3,
# as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
"""
The storage protocol client.

"""
from twisted.internet.protocol import ClientFactory
from twisted.internet import reactor, defer
from twisted.python import log

import uuid
from functools import partial
from itertools import chain
import logging
from oauth import oauth
from ubuntuone.storageprotocol import protocol_pb2
from ubuntuone.storageprotocol import request
from ubuntuone.storageprotocol import sharersp


log_debug = partial(log.msg, loglevel=logging.DEBUG)

class StorageClient(request.RequestHandler):
    """ A Basic Storage Protocol client."""

    # we are a client, we do odd requests
    REQUEST_ID_START = 1

    def __init__(self):
        """Create the client. done by the factory."""
        request.RequestHandler.__init__(self)
        self.root_id = None
        self.root_id_defers = []
        self._node_state_callback = None
        self._share_change_callback = None
        self._share_answer_callback = None
        self._free_space_callback = None
        self._account_info_callback = None
        self.line_mode = True

    def protocol_version(self):
        """ask for the protocol version

        will return a deferred that will get called with
        the request object when completed
        """
        p = ProtocolVersion(self)
        p.start()
        return p.deferred

    def dataReceived(self, data):
        """extends dataReceived.

        this method first reads the protocol hello line then switches back to
        len+data."""
        if self.line_mode:
            # first read the hello line, then back to binary.
            try:
                pos = data.index("\r\n")
            except ValueError:
                return
            self.line_mode = False
            data = data[pos+2:]

        request.RequestHandler.dataReceived(self, data)

    def dummy_authenticate(self, credentials):
        """Authenticate to a server using the 'dummy auth' provider.

        Returns a deferred that will get called with the request
        object when completed.
        """
        p = Authenticate(self, {'dummy_token': credentials})
        p.start()
        return p.deferred

    def oauth_authenticate(self, consumer, token):
        """Authenticate to a server using the OAuth provider.

        @param consumer: the OAuth consumer to authenticate with.
        @type consumer: `oauth.OAuthConsumer`
        @param token: a previously acquired OAuth access token.
        @type consumer: `oauth.OAuthToken`

        Returns a deferred that will get called with the request
        object when completed.
        """
        request = oauth.OAuthRequest.from_consumer_and_token(
            oauth_consumer=consumer,
            token=token,
            http_method="GET",
            http_url="storage://server")
        request.sign_request(
            oauth.OAuthSignatureMethod_PLAINTEXT(), consumer, token)

        # Make sure all parameter values are strings.
        auth_parameters = dict(
            (key, str(value)) for key, value in request.parameters.iteritems())
        p = Authenticate(self, auth_parameters)
        p.start()
        return p.deferred

    def handle_ROOT(self, message):
        """handle incoming ROOT message

        will notify if someone is waiting for this information
        """
        self.root_id = message.root.node
        if self.root_id_defers:
            for d in self.root_id_defers:
                d.callback(self.root_id)
            self.root_id_defers = []

    def handle_NODE_STATE(self, message):
        """handle incoming NODE_STATE"""
        self.notify_node_state(message.node_state)

    def handle_NOTIFY_SHARE(self, message):
        """handle incoming NOTIFY_SHARE"""
        self.notify_share_change(message)

    def handle_SHARE_DELETED(self, message):
        """handle incoming SHARE_DELETED"""
        self.notify_share_change(message)

    def handle_SHARE_ACCEPTED(self, message):
        """handle incoming SHARE_ACCEPTED"""
        self.notify_share_answer(message.share_accepted)

    def handle_BEGIN_CONTENT(self, message):
        """Accepts and discards a misplaced BEGIN_CONTENT.

        It can happen that while the server receives a PUT_CONTENT request
        and that it tells us to BEGIN_CONTENT, we cancelled the request,
        received the OK, and this side's request is gone, so receive this
        message here.
        """

    def handle_FREE_SPACE_INFO(self, message):
        """handle unsolicited FREE_SPACE_INFO"""
        self.notify_free_space(message.free_space_info)

    def handle_ACCOUNT_INFO(self, message):
        """handle unsolicited ACCOUNT_INFO"""
        self.notify_account_info(message.account_info)

    def get_root(self):
        """get the root id through a deferred"""
        if self.root_id is not None:
            return defer.succeed(self.root_id)
        else:
            d = defer.Deferred()
            self.root_id_defers.append(d)
            return d

    def make_dir(self, share, parent, name):
        """ create a directory named name on the node parent

        the new node id will be on request.new_dir_id
        """
        p = MakeDir(self, share, parent, name)
        p.start()
        return p.deferred

    def make_file(self, share, parent, name):
        """ create a file named name on the node parent

        the new node id will be on request.new_file_id
        """
        p = MakeFile(self, share, parent, name)
        p.start()
        return p.deferred

    def move(self, share, node, new_parent, new_name):
        """ move a node
        """
        p = Move(self, share, node, new_parent, new_name)
        p.start()
        return p.deferred

    def unlink(self, share, node):
        """ unlink a node
        """
        p = Unlink(self, share, node)
        p.start()
        return p.deferred

    def get_content(self, share, node, hash, offset=0,
                    callback=None, node_attr_callback=None):
        """ get the content of node with hash

        the content will be on request.content
        or callback will be called for every piece that arrives
        """
        req = self.get_content_request(share, node, hash, offset,
                                       callback, node_attr_callback)
        return req.deferred

    def get_content_request(self, share, node, hash, offset=0,
                            callback=None, node_attr_callback=None):
        """ get the content of node with hash, returns the request

        the content will be on request.content
        or callback will be called for every piece that arrives
        """
        p = GetContent(self, share, node, hash, offset,
                                        callback, node_attr_callback)
        p.start()
        return p

    def put_content(self, share, node, previous_hash, hash,
                    crc32, size, deflated_size, fd):
        """ put the content of fd into file node
        """
        req = self.put_content_request(share, node, previous_hash,
                                       hash, crc32, size, deflated_size, fd)
        return req.deferred

    def put_content_request(self, share, node, previous_hash, hash,
                            crc32, size, deflated_size, fd):
        """ put the content of fd into file node, return the request
        """
        p = PutContent(self, share, node, previous_hash, hash, crc32, size,
                                                            deflated_size, fd)
        p.start()
        return p

    def query(self, items):
        """ get the current hash for items if changed

        items is a list of (node, hash) tuples
        """
        r = MultiQuery(self, items)
        r.start()
        return r.deferred

    def get_free_space(self, share_id):
        """Get quota information for the given share (or the user's own
        space)."""
        r = FreeSpaceInquiry(self, share_id)
        r.start()
        return r.deferred

    def get_account_info(self):
        """Get account information (like purchased space etc.)."""
        r = AccountInquiry(self)
        r.start()
        return r.deferred

    def set_node_state_callback(self, callback):
        """define the function to be called when a node_state message arrives

        the function will be called with the message as argument
        """
        self._node_state_callback = callback

    def notify_node_state(self, node_state):
        """Call the current node state callback, if any, with the share, node,
        and hash given in the message.

        @param: node_state - a (raw) NodeState message

        """
        if self._node_state_callback:
            self._node_state_callback(node_state.share, node_state.node,
                                      node_state.hash)

    def set_free_space_callback(self, callback):
        """Sets the quota notification callback, which will be called
        with the share id and free bytes.

        """
        self._free_space_callback = callback

    def notify_free_space(self, free_space_info):
        """Call the current quota info callback, if any, with the share
        and available bytes.

        @param: free_space_info - a (raw) FreeSpaceInfo message

        """
        if self._free_space_callback:
            self._free_space_callback(free_space_info.share_id,
                                      free_space_info.free_bytes)

    def set_account_info_callback(self, callback):
        """Sets the account info notification callback; the callback
        will be passed a raw AccountInfo structure when it is called."""
        self._account_info_callback = callback

    def notify_account_info(self, account_info):
        """Call the current account info callback, if any."""
        if self._account_info_callback:
            self._account_info_callback(account_info)

    def set_share_change_callback(self, callback):
        """Define the function to be called when something changed regarding
        a share.
        """
        self._share_change_callback = callback

    def notify_share_change(self, msg):
        """Call the current changed share callback, if any, with the
        notify info.

        @param msg: - a (raw) NotifyShare message
        """
        if not self._share_change_callback:
            return

        if msg.type == protocol_pb2.Message.NOTIFY_SHARE:
            info = sharersp.NotifyShareHolder.load_from_msg(msg.notify_share)
            self._share_change_callback("changed", info)
        elif msg.type == protocol_pb2.Message.SHARE_DELETED:
            self._share_change_callback("deleted",
                                        uuid.UUID(msg.share_deleted.share_id))
        else:
            raise ValueError("Received a callback with wrong message type!")

    def set_share_answer_callback(self, callback):
        """Define the function to be called when a share answer is received."""
        self._share_answer_callback = callback

    def notify_share_answer(self, msg):
        """Call the current share answer callback, if any, with the info.

        @param msg: - a (raw) ShareAccepted message
        """
        if self._share_answer_callback:
            if msg.answer == protocol_pb2.ShareAccepted.YES:
                answer = "Yes"
            elif msg.answer == protocol_pb2.ShareAccepted.NO:
                answer = "No"
            else:
                raise ValueError("Not supported ShareAccepted answer")
            self._share_answer_callback(uuid.UUID(msg.share_id), answer)

    def create_share(self, node, share_to, name, access_level):
        """Creates a share to other user.

        node: which node to share.
        share_to: the id of the receiving user.
        name: the name of the share
        access_level: the permissions on the share

        There's no need to indicate where the node lives, as it only can be
        in own root (there's no re-sharing).
        """
        r = CreateShare(self, node, share_to, name, access_level)
        r.start()
        return r.deferred

    def delete_share(self, share_id):
        """Deletes a share we have offered.

        @param share_id: the id of the share to delete
        """
        r = DeleteShare(self, share_id)
        r.start()
        return r.deferred

    def accept_share(self, share_id, answer):
        """Accepts (or not) a share from other user.

        share_id: the share id
        answer: if it was accepted ("Yes") or not ("No")
        """
        r = AcceptShare(self, share_id, answer)
        r.start()
        return r.deferred

    def list_shares(self):
        """List all the shares the user is involved.

        This includes the shares the user created, and those that were
        shared to her.
        """
        p = ListShares(self)
        p.start()
        return p.deferred

    def query_caps(self, caps):
        """Query the server to discover its capabilitis.

        The server should answer if it supports or not all the given
        caps.
        """
        r = QuerySetCaps(self, caps)
        r.start()
        return r.deferred

    def set_caps(self, caps):
        """Set the server to this capabilities."""
        r = QuerySetCaps(self, caps, set_mode=True)
        r.start()
        return r.deferred


class GetContent(request.Request):
    """A Request to get the content of a node id

    @ivar data: the content of the node (available upon success)
    """

    def __init__(self, protocol, share, node_id, hash,
                 offset=0, callback=None, node_attr_callback=None):
        """ request the content of node with hash

        @param protocol: the request handler
        @param share: the share node or root
        @param node_id: the node id of the node we want to read
        @param hash: the hash of the content of the version we have
        @param offset: offset for reading
        @param callback: function to call when data arrives

        """
        request.Request.__init__(self, protocol)
        self.share = share
        self.node_id = node_id
        self.hash = hash
        self.offset = offset
        self.callback = callback
        self.node_attr_callback = node_attr_callback
        self.parts = []

    def _start(self):
        "send GET_CONTENT"
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.GET_CONTENT
        message.get_content.node = str(self.node_id)
        message.get_content.hash = str(self.hash)
        message.get_content.share = self.share
        message.get_content.offset = self.offset
        self.sendMessage(message)

    def processMessage(self, message):
        """process messages
        """
        # pylint: disable-msg=W0201
        if message.type == protocol_pb2.Message.NODE_ATTR:
            if self.node_attr_callback is not None:
                self.node_attr_callback(
                    deflated_size=message.node_attr.deflated_size,
                    size=message.node_attr.size,
                    hash=message.node_attr.hash,
                    crc32=message.node_attr.crc32)
        elif message.type == protocol_pb2.Message.BYTES:
            if self.cancelled:
                # don't care about more bytes if already cancelled
                return
            if self.callback is not None:
                self.callback(message.bytes.bytes)
            else:
                self.parts.append(message.bytes.bytes)
        elif message.type == protocol_pb2.Message.EOF:
            if self.cancelled:
                # eof means that the cancel request arrived late. this is the
                # end.
                self.done()
                return
            if self.callback is None:
                self.data = "".join(self.parts)
            self.done()
        elif message.type == protocol_pb2.Message.ERROR:
            self.error(request.StorageRequestError(self, message))
        elif message.type == protocol_pb2.Message.OK:
            self.done()
        elif message.type == protocol_pb2.Message.CANCELLED:
            self.error(request.RequestCancelledError("CANCELLED"))
        else:
            self.error(request.StorageProtocolProtocolError(message))

    def _cancel(self):
        """Cancels the current download."""
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.CANCEL_REQUEST
        self.sendMessage(message)

class ListShares(request.Request):
    """List all the shares the user is involved.

    This includes the shares the user created, and those that were
    shared to her.
    """

    def _start(self):
        '''Sends the LIST_SHARES message to the server.'''
        # pylint: disable-msg=W0201
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.LIST_SHARES
        self.sendMessage(message)
        self.shares = []

    def processMessage(self, message):
        '''Process the answer from the server.'''
        if message.type == protocol_pb2.Message.SHARES_INFO:
            share = sharersp.ShareResponse.load_from_msg(message.shares)
            self.shares.append(share)
        elif message.type == protocol_pb2.Message.SHARES_END:
            self.done()
        else:
            self.error(request.StorageRequestError(self, message))

class CreateShare(request.Request):
    """Creates a share."""

    # these are the valid access levels and their translation to the
    # protocol message
    _valid_access_levels = {
        "View": protocol_pb2.CreateShare.VIEW,
        "Modify": protocol_pb2.CreateShare.MODIFY,
    }

    def __init__(self, protocol, node_id, share_to, name, access_level):
        """Creates a share.

        @param node_id: which node will be root to share.
        @param share_to: the id of the receiving user.
        @param name: the name of the share
        @param access_level: the permissions on the share

        """
        request.Request.__init__(self, protocol)
        self.node = node_id
        self.share_to = share_to
        self.name = name
        self.access_level = access_level
        self.share_id = None

    def _start(self):
        '''Sends the CREATE_SHARE message to the server.'''
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.CREATE_SHARE
        message.create_share.node = self.node
        message.create_share.share_to = self.share_to
        message.create_share.name = self.name

        # we make this testing here and not in __init__ because it should
        # be done when creating the message (to support that the access_level
        # is changed between instantiating and message creation)
        try:
            message_access_level = self._valid_access_levels[self.access_level]
        except KeyError:
            raise ValueError("Invalid access level! (%r)" % self.access_level)
        message.create_share.access_level = message_access_level

        self.sendMessage(message)

    def processMessage(self, message):
        '''Process the answer from the server.'''
        if message.type == protocol_pb2.Message.SHARE_CREATED:
            self.share_id = message.share_created.share_id
            self.done()
        elif message.type == protocol_pb2.Message.OK:
            # XXX: this is for PROTOCOL_VERSION=1 backward compatibility
            self.done()
        else:
            self.error(request.StorageRequestError(self, message))


class AcceptShare(request.Request):
    """Acceptss a share (or not)."""

    # these are the valid answers and their translation to the
    # protocol message
    _valid_answer = {
        "Yes": protocol_pb2.ShareAccepted.YES,
        "No": protocol_pb2.ShareAccepted.NO,
    }

    def __init__(self, protocol, share_id, answer):
        """Accepts (or not) a share from other user.

        @param share_id: the share id
        @param answer: if it was accepted ("Yes") or not ("No")
        """
        request.Request.__init__(self, protocol)
        self.share_id = share_id
        self.answer = answer

    def _start(self):
        '''Sends the SHARE_ACCEPTED message to the server.'''
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.SHARE_ACCEPTED
        message.share_accepted.share_id = str(self.share_id)

        # we make this testing here and not in __init__ because it should
        # be done when creating the message (to support that the answer
        # is changed between instantiating and message creation)
        try:
            message.share_accepted.answer = self._valid_answer[self.answer]
        except KeyError:
            raise ValueError("Invalid answer! (%r)" % self.answer)

        self.sendMessage(message)

    def processMessage(self, message):
        '''Process the answer from the server.'''
        if message.type == protocol_pb2.Message.OK:
            self.done()
        else:
            self.error(request.StorageRequestError(self, message))


class DeleteShare(request.Request):
    """Deletes a share."""

    def __init__(self, protocol, share_id):
        """Deletes a share we had offered to someone else.

        @param share_id: the share id
        """
        request.Request.__init__(self, protocol)
        self.share_id = share_id

    def _start(self):
        """Sends the DELETE_SHARE message to the server."""
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.DELETE_SHARE
        message.delete_share.share_id = str(self.share_id)

        self.sendMessage(message)

    def processMessage(self, message):
        """Process the answer from the server."""
        if message.type == protocol_pb2.Message.OK:
            self.done()
        else:
            self.error(request.StorageRequestError(self, message))


class Unlink(request.Request):
    """ unlink a node
    """
    # pylint: disable-msg=C0111

    def __init__(self, protocol, share, node_id):
        """ request that node_id be unlinked

        @param protocol: the request handler
        @param share: the share node or root
        @param node_id: the node id of the node we want to unlink

        """
        request.Request.__init__(self, protocol)
        self.share = share
        self.node = node_id

    def _start(self):
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.UNLINK
        message.unlink.share = self.share
        message.unlink.node = self.node

        self.sendMessage(message)

    def processMessage(self, message):
        if message.type == protocol_pb2.Message.OK:
            self.done()
        else:
            self.error(request.StorageRequestError(self, message))


class Move(request.Request):
    """ move a node
    """
    def __init__(self, protocol, share, node_id, new_parent_id, new_name):
        """create the move request

        @param protocol: the request handler
        @param share: the share node or root
        @param node_id: the node id of the node we want to move
        @param new_parent_id: the id of the new parent
        @param new_name: the new name for this node
        @param callback: function to call when data arrives

        """
        request.Request.__init__(self, protocol)
        self.share = share
        self.node_id = node_id
        self.new_parent_id = new_parent_id
        self.new_name = new_name

    def _start(self):
        """send MOVE"""
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.MOVE
        message.move.share = self.share
        message.move.node = self.node_id
        message.move.new_parent_node = str(self.new_parent_id)
        message.move.new_name = self.new_name

        self.sendMessage(message)

    def processMessage(self, message):
        """process messages"""
        if message.type == protocol_pb2.Message.OK:
            self.done()
        else:
            # XXX:
            # lucio.torre
            # handle more error messages
            self.error(request.StorageRequestError(self, message))

class MultiQuery(object):
    """Creates a Request-like object that encapsulates many Query requests

    we may need to split this request into many Query rests if the list of items
    to query is to big to fit in one message
    """
    def __init__(self, protocol, items):
        """ create a multiquery

        @param protocol: the request handler
        @param items: a list of (node, hash) tuples
        """
        items = iter(items)
        defers = []
        self.queries = []

        while True:
            r = Query(protocol, items)
            self.queries.append(r)
            defers.append(r.deferred)
            if r.overflow:
                items = chain([r.overflow], items)
            else:
                break

        self.deferred = defer.DeferredList(defers, consumeErrors=True)

    def start(self):
        """start the queries"""
        for q in self.queries:
            q.start()

class Query(request.Request):
    """query about the hash of a node_id

    @ivar remains: the items that could not fit in the query
    @ivar response: the node state messages that were received
    """

    def __init__(self, protocol, items):
        """ generates a query message to send to the server

        puts as much items as it can inside the message
        whats left is left in self.remainder

        @param protocol: the request handler
        @param items: a list of (node, hash, share) tuples
        """

        request.Request.__init__(self, protocol)
        self.query_message = qm = protocol_pb2.Message()
        qm.id = 0 # just to have something in the field when calculating size
        qm.type = protocol_pb2.Message.QUERY
        self.response = []
        self.overflow = None
        items_that_fit = []

        def add_items(msg, *args):
            """add items to query"""
            for share, node, hash in args:
                qi = msg.query.add()
                qi.share = share
                qi.node = str(node)
                qi.hash = hash

        for item in items:
            add_items(qm, item)
            if qm.ByteSize() > request.MAX_MESSAGE_SIZE:
                self.overflow = item
                break
            items_that_fit.append(item)

        if self.overflow is not None:
            qm.ClearField("query")
            add_items(qm, *items_that_fit)

    def _start(self):
        """send QUERY"""
        self.sendMessage(self.query_message)

    def processMessage(self, message):
        """handle messages"""
        if message.type == protocol_pb2.Message.NODE_STATE:
            self.response.append(message.node_state)
            self.protocol.notify_node_state(message.node_state)
        elif message.type == protocol_pb2.Message.QUERY_END:
            self.done()
        else:
            # XXX:
            # lucio.torre
            # handle more error messages
            self.error(message)


class BytesMessageProducer(object):
    """Produce BYTES messages from a file."""

    def __init__(self, request, fh, offset):
        """Create a BytesMessageProducer."""
        self.request = request
        self.producing = False
        self.fh = fh
        self.offset = offset
        self.finished = False

    def resumeProducing(self):
        """IPushProducer interface."""
        self.producing = True
        self.go()

    def stopProducing(self):
        """IPushProducer interface."""
        self.producing = False

    def pauseProducing(self):
        """IPushProducer interface."""
        self.producing = False

    def go(self):
        """While producing, generates data.

        Reads a little from the file, generates a BYTES message, and pass the
        control to the reactor.  If no more data, it finishes with EOF.
        """
        if not self.producing or self.request.cancelled or self.finished:
            return

        if self.offset:
            self.fh.seek(self.offset)
        data = self.fh.read(request.MAX_PAYLOAD_SIZE)
        if data:
            if self.offset:
                self.offset += len(data)
            response = protocol_pb2.Message()
            response.type = protocol_pb2.Message.BYTES
            response.bytes.bytes = data
            self.request.sendMessage(response)
            reactor.callLater(0, self.go)
        else:
            message = protocol_pb2.Message()
            message.type = protocol_pb2.Message.EOF
            self.request.sendMessage(message)
            self.producing = False
            self.finished = True


class PutContent(request.Request):
    """PUT content request"""
    def __init__(self, protocol, share, node_id,
                 previous_hash, hash, crc32, size,
                 deflated_size, fd):
        """ put content into a node

        @param protocol: the request handler
        @param share: the share node or root
        @param node_id: the node to receive the content
        @param previous_hash: the hash the node has (for conflict checking)
        @param hash: the hash hint for the new content
        @param crc32: the crc32 hint for the new content
        @param size: the size hint for the new content
        @param fd: a file-like object to read data from

        """
        request.Request.__init__(self, protocol)
        self.share = share
        self.node_id = node_id
        self.previous_hash = previous_hash
        self.hash = hash
        self.crc32 = crc32
        self.size = size
        self.deflated_size = deflated_size
        self.fd = fd

    def _start(self):
        """send PUT_CONTENT"""
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.PUT_CONTENT
        message.put_content.share = self.share
        message.put_content.node = str(self.node_id)
        message.put_content.previous_hash = self.previous_hash
        message.put_content.hash = self.hash
        message.put_content.crc32 = self.crc32
        message.put_content.size = self.size
        message.put_content.deflated_size = self.deflated_size
        self.sendMessage(message)

    def processMessage(self, message):
        """handle messages"""
        if message.type == protocol_pb2.Message.BEGIN_CONTENT:
            message_producer = BytesMessageProducer(
                self, self.fd, message.begin_content.offset)
            self.registerProducer(message_producer, streaming=True)
        elif message.type == protocol_pb2.Message.OK:
            self.done()
        elif message.type == protocol_pb2.Message.CANCELLED:
            self.error(request.RequestCancelledError("CANCELLED"))
        else:
            self.error(request.StorageRequestError(self, message))

    def _cancel(self):
        """Cancels the current upload."""
        if self.producer is not None:
            self.producer.stopProducing()
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.CANCEL_REQUEST
        self.sendMessage(message)


class MakeObject(request.Request):
    """ Handles the creation of new objects

    on completion it will have the attribute 'new_id' with the
    node id of the created object

    @cvar create_message: must be overridden with the correct creation message
        to send
    @cvar response_message: must be overridden with the correct creation success
        message that will be received

    @ivar new_id: the id of the node that was created (available upon success)
    @ivar new_parent_id: the parent id the node now exists under
    @ivar new_name: the name the node now exists under
    """

    def __init__(self, protocol, share, parent_id, name):
        """ create a node

        @param protocol: the request handler
        @param share: the share node or root
        @param parent_id: the desired parent id
        @param name: the desired name

        """
        request.Request.__init__(self, protocol)
        self.share = share
        self.parent_id = parent_id
        self.name = name

    def _start(self):
        """send $MAKE message"""
        message = protocol_pb2.Message()
        message.type = self.create_message

        message.make.share = self.share
        message.make.parent_node = str(self.parent_id)
        message.make.name = self.name
        self.sendMessage(message)

    def processMessage(self, message):
        """handle messages"""
        # pylint: disable-msg=W0201
        if message.type == self.create_response:
            self.new_id = message.new.node
            self.new_parent_id = message.new.parent_node
            self.new_name = message.new.name
            self.done()
        else:
            self.error(request.StorageRequestError(self, message))

class MakeDir(MakeObject):
    """Extends MakeObject to make directories"""
    create_message = protocol_pb2.Message.MAKE_DIR
    create_response = protocol_pb2.Message.NEW_DIR


class MakeFile(MakeObject):
    """Extends MakeObject to make files"""
    create_message = protocol_pb2.Message.MAKE_FILE
    create_response = protocol_pb2.Message.NEW_FILE


class ProtocolVersion(request.Request):
    """ Handles the protocol version query

    when completed will contain the servers protocol version
    on `other_protocol_version`

    @ivar other_protocol_version: the other peer's protocol version (available
        upon success)
    """

    def _start(self):
        """send PROTOCOL_VERSION"""
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.PROTOCOL_VERSION
        message.protocol.version = self.protocol.PROTOCOL_VERSION
        self.sendMessage(message)

    def processMessage(self, message):
        """handle messages"""
        # pylint: disable-msg=W0201
        if message.type == protocol_pb2.Message.PROTOCOL_VERSION:
            self.other_protocol_version = message.protocol.version
            self.done()
        else:
            self.error(request.StorageRequestError(self, message))

class Authenticate(request.Request):
    """ Request to authenticate the user
    """

    def __init__(self, protocol, auth_parameters):
        """create an authentication request

        @param protocol: the request handler
        @param auth_parameters: a dictionary of authentication parameters.
        """
        request.Request.__init__(self, protocol)
        self.auth_parameters = auth_parameters

    def _start(self):
        """send AUTH_REQUEST"""
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.AUTH_REQUEST
        for key, value in self.auth_parameters.items():
            param = message.auth_parameters.add()
            param.name = key
            param.value = value
        self.sendMessage(message)

    def processMessage(self, message):
        """handle messages"""
        if message.type == protocol_pb2.Message.AUTH_AUTHENTICATED:
            self.done()
        elif message.type == protocol_pb2.Message.ERROR:
            # as the error travels with the exception, we send all here
            self.error(request.StorageRequestError(self, message))
        else:
            self.error(request.StorageProtocolError(
                        "Authentication Error:"+str(message)))


class QuerySetCaps(request.Request):
    """Queries or Sets the server to use capabilities."""

    def __init__(self, protocol, caps, set_mode=False):
        """Generates a query_caps or set_caps message to send to the server.

        @param protocol: the request handler
        @param caps: a list of capabilities to ask for or to set
        """
        request.Request.__init__(self, protocol)
        self.caps = caps
        self.accepted = None
        self.redirect_hostname = None
        self.redirect_port = None
        self.redirect_srvrecord = None
        self.set_mode = set_mode

    def _start(self):
        """Send QUERY_CAPS or SET_CAPS"""
        message = protocol_pb2.Message()
        if self.set_mode:
            message.type = protocol_pb2.Message.SET_CAPS
            for cap in self.caps:
                qc = message.set_caps.add()
                qc.capability = cap
        else:
            message.type = protocol_pb2.Message.QUERY_CAPS
            for cap in self.caps:
                qc = message.query_caps.add()
                qc.capability = cap

        self.sendMessage(message)

    def processMessage(self, message):
        """Handle the message"""
        if message.type == protocol_pb2.Message.ACCEPT_CAPS:
            self.accepted = message.accept_caps.accepted
            self.redirect_hostname = message.accept_caps.redirect_hostname
            self.redirect_port = message.accept_caps.redirect_port
            self.redirect_srvrecord = message.accept_caps.redirect_srvrecord
            self.done()
        else:
            self.error(request.StorageRequestError(self, message))


class FreeSpaceInquiry(request.Request):
    """Query available space."""

    def __init__(self, protocol, share_id):
        '''Initializes the request.'''
        request.Request.__init__(self, protocol)
        self.share_id = share_id

    def _start(self):
        '''Sends the FREE_SPACE_INQUIRY message to the server.'''
        # pylint: disable-msg=W0201
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.FREE_SPACE_INQUIRY
        message.free_space_inquiry.share_id = self.share_id
        self.sendMessage(message)
        self.free_bytes = None

    def processMessage(self, message):
        '''Process the answer from the server.'''
        if message.type == protocol_pb2.Message.FREE_SPACE_INFO:
            self.free_bytes = message.free_space_info.free_bytes
            self.done()
        else:
            self.error(request.StorageRequestError(self, message))


class AccountInquiry(request.Request):
    """Query account information."""

    def _start(self):
        '''Sends the FREE_SPACE_INQUIRY message to the server.'''
        # pylint: disable-msg=W0201
        message = protocol_pb2.Message()
        message.type = protocol_pb2.Message.ACCOUNT_INQUIRY
        self.sendMessage(message)
        self.purchased_bytes = None

    def processMessage(self, message):
        '''Process the answer from the server.'''
        if message.type == protocol_pb2.Message.ACCOUNT_INFO:
            self.purchased_bytes = message.account_info.purchased_bytes
            self.done()
        else:
            self.error(request.StorageRequestError(self, message))


class ThrottlingStorageClient(StorageClient):
    """
    This is the throttling version of the StorageClient protocol.
    """
    factory = None

    def connectionMade(self):
        """Handle connection Made."""
        if self.factory.client is None:
            self.factory.client = self
        StorageClient.connectionMade(self)

    def connectionLost(self, reason=None):
        """Handle connection lost."""
        if self.factory.client is self:
            self.factory.unregisterProtocol(self)
        StorageClient.connectionLost(self, reason=reason)

    def write(self, data):
        """Transport API to capture bytes written."""
        if self.factory.client is self:
            self.factory.registerWritten(len(data))
        StorageClient.write(self, data)

    def writeSequence(self, seq):
        """Transport API to capture bytes written in a sequence."""
        if self.factory.client is self:
            self.factory.registerWritten(sum(len(x) for x in seq))
        StorageClient.writeSequence(self, seq)

    def dataReceived(self, data):
        """Override transport default to capture bytes read."""
        if self.factory.client is self:
            self.factory.registerRead(len(data))
        StorageClient.dataReceived(self, data)

    def throttleReads(self):
        """Pause self.transport."""
        self.transport.pauseProducing()

    def unthrottleReads(self):
        """Resume self.transport."""
        self.transport.resumeProducing()

    def throttleWrites(self):
        """Pause producing."""
        self.pauseProducing()

    def unthrottleWrites(self):
        """Resume producing."""
        self.resumeProducing()


class StorageClientFactory(ClientFactory):
    """StorageClient factory"""
    # pylint: disable-msg=W0232
    protocol = StorageClient


class ThrottlingStorageClientFactory(StorageClientFactory):
    """
    This is the throttling version of StorageClientFactory.
    """

    protocol = ThrottlingStorageClient

    def __init__(self, read_limit=None, write_limit=None):
        """creates the instance"""
        self.readLimit = read_limit # max bytes we should read per second
        self.writeLimit = write_limit # max bytes we should write per second
        self.readThisSecond = 0
        self.writtenThisSecond = 0
        self.unthrottleReadsID = None
        self.resetReadThisSecondID = None
        self.unthrottleWritesID = None
        self.resetWriteThisSecondID = None
        self.stopped = True

    def callLater(self, period, func, *args, **kwargs):
        """
        Wrapper around L{reactor.callLater} for test purpose.
        """
        return reactor.callLater(period, func, *args, **kwargs)

    def maybeCallLater(self, id, period, func):
        """do a callLater only if we don't have a DelayedCall with the
        specified id already running.
        """
        delayed_call = getattr(self, id)
        # check if we already have a DelayedCall running
        if delayed_call is None or (not delayed_call.active() \
           and delayed_call.cancelled):
            return self.callLater(period, func)
        return delayed_call

    def registerWritten(self, length):
        """
        Called by protocol to tell us more bytes were written.
        """
        self.writtenThisSecond += length
        self.checkWriteBandwidth()

    def registerRead(self, length):
        """
        Called by protocol to tell us more bytes were read.
        """
        self.readThisSecond += length
        self.checkReadBandwidth()

    def checkReadBandwidth(self):
        """Checks if we've passed bandwidth limits."""
        if self.readLimit is not None and \
           self.readThisSecond > self.readLimit and \
           self.unthrottleReadsID is None:
            self.throttleReads()
            throttle_time = (float(self.readThisSecond) / self.readLimit) - 1.0
            log_debug("pause reads for: %s", str(throttle_time))
            self.unthrottleReadsID = self.maybeCallLater('unthrottleReadsID',
                                                         throttle_time,
                                                         self.unthrottleReads)

    def checkWriteBandwidth(self):
        """Checks if we've passed bandwidth limits."""
        if self.writeLimit is not None and \
           self.writtenThisSecond > self.writeLimit and \
           self.unthrottleWritesID is None:
            self.throttleWrites()
            throttle_time = (float(self.writtenThisSecond) / self.writeLimit) \
                    - 1.0
            log_debug("pause writes for: %s", str(throttle_time))
            self.unthrottleWritesID = self.maybeCallLater('unthrottleWritesID',
                                                      throttle_time,
                                                      self.unthrottleWrites)

    def _resetReadThisSecond(self):
        """reset the counter named with 'name' every 1 second"""
        # check the bandwidth limits
        self.readThisSecond = 0
        self.resetReadThisSecondID = self.callLater(1,
                                                   self._resetReadThisSecond)

    def _resetWrittenThisSecond (self):
        """reset the counter named with 'name' every 1 second"""
        self.writtenThisSecond = 0
        self.resetWriteThisSecondID = self.callLater(1,
                                                self._resetWrittenThisSecond)

    def throttleReads(self):
        """Throttle reads on all protocols."""
        if self.client is not None:
            self.client.throttleReads()

    def unthrottleReads(self):
        """Stop throttling reads on all protocols."""
        self.unthrottleReadsID = None
        if self.client is not None:
            self.client.unthrottleReads()

    def throttleWrites(self):
        """Throttle writes on all protocols."""
        if self.client is not None:
            self.client.throttleWrites()

    def unthrottleWrites(self):
        """Stop throttling writes on all protocols."""
        self.unthrottleWritesID = None
        if self.client is not None:
            self.client.unthrottleWrites()

    def buildProtocol(self, addr):
        """build the protocol and start the counters reset loops"""
        if self.readLimit is not None:
            self._resetReadThisSecond()
        if self.writeLimit is not None:
            self._resetWrittenThisSecond()
        self.stopped = False
        return StorageClientFactory.buildProtocol(self, addr)

    def unregisterProtocol(self, protocol):
        """Stop all DelayedCall we have around"""
        for delayed in [self.unthrottleReadsID, self.resetReadThisSecondID,
                        self.unthrottleWritesID, self.resetWriteThisSecondID]:
            self._cancel_delayed_call(delayed)


    def _cancel_delayed_call(self, delayed):
        """Safely cancel a DelayedCall."""
        if delayed is not None and not delayed.cancelled \
           and delayed.active():
            try:
                delayed.cancel()
            except defer.AlreadyCalledError:
                # ignore AlreadyCalledError
                pass


if __name__ == "__main__":
    # these 3 lines show the different ways of connecting a client to the
    # server

    # using tcp
    reactor.connectTCP('localhost', 20100, StorageClientFactory())

    # using ssl
    #reactor.connectSSL('localhost', 20101, StorageClientFactory(),
    #           ssl.ClientContextFactory())

    # using ssl over a proxy
    #from ubuntuone.storageprotocol import proxy_tunnel
    #proxy_tunnel.connectHTTPS('localhost', 3128,
    #        'localhost', 20101, StorageClientFactory(),
    #        user="test", passwd="test")

    reactor.run()
