# -*- coding: utf-8 -*-

#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Test the linux utils functions."""

import codecs
import os

from twisted.internet import defer

from ubuntuone.controlpanel import utils
from ubuntuone.controlpanel.tests import TestCase, USER_HOME

REAL_CONTENT = u"""
XDG_DESKTOP_DIR="$HOME/Desktop"
XDG_DOWNLOAD_DIR="$HOME/Downloads"
XDG_PUBLICSHARE_DIR="$HOME/Public"
XDG_DOCUMENTS_DIR="$HOME/Documents"
XDG_MUSIC_DIR="$HOME/Music"
XDG_PICTURES_DIR="$HOME/Pictures"
XDG_VIDEOS_DIR="$HOME/Videos"
"""

NON_ASCII = u"""
XDG_DESKTOP_DIR="$HOME/桌面"
XDG_DOWNLOAD_DIR="$HOME/下载"
XDG_DOCUMENTS_DIR="$HOME/文档"
"""

LATIN_1 = u"""
XDG_DESKTOP_DIR="$HOME/Súbeme"
XDG_DOWNLOAD_DIR="$HOME/Bájame"
XDG_DOCUMENTS_DIR="$HOME/Mis ñoños Documentos"
"""


class AutoupdaterTestCase(TestCase):
    """Test the code that is used for the auto update process."""

    @defer.inlineCallbacks
    def test_are_updates_present(self):
        """We should never have updates."""
        are_present = yield utils.are_updates_present(None)
        self.assertFalse(are_present)

    def test_perform_update(self):
        """Test the method that performs the update."""
        self.assertEqual(None, utils.perform_update())


class DefaultFoldersTestCase(TestCase):
    """Test suite for the default_folders retriever."""

    def _create_faked_file(self, content='', encoding=None):
        """Create a faked file to be parsed and clean it up afterwards."""
        filename = './dirs-parser-test.dirs'
        if os.path.exists(filename):
            os.remove(filename)

        if encoding is None:
            encoding = 'utf-8'

        with codecs.open(filename, 'w', encoding=encoding) as f:
            f.write(content)

        self.addCleanup(os.remove, filename)

        return filename

    def test_default_folders_not_file(self):
        """Default folders are empty if the file does not exist."""
        path = 'foo/bar/baz/yadda/yo'
        assert not os.path.exists(path)
        actual = utils.default_folders(dirs_path=path)

        self.assertEqual(actual, [])

    def test_default_folders_empty_file(self):
        """Default folders are empty if the file empty."""
        filename = self._create_faked_file(content='')
        actual = utils.default_folders(dirs_path=filename)

        self.assertEqual(actual, [])

    def test_default_folders_only_comments(self):
        """Default folders are empty if only comments in the file."""
        content = u"# yadda yadda\n# foo bar baz\n#puipuipui"
        filename = self._create_faked_file(content=content)
        actual = utils.default_folders(dirs_path=filename)

        self.assertEqual(actual, [])

    def test_default_folders_syntax_error(self):
        """Default folders do not explode on syntax error."""
        content = u"foo ~ bar\nYADDA=DOO"
        filename = self._create_faked_file(content=content)
        actual = utils.default_folders(dirs_path=filename)

        self.assertEqual(actual, [u'DOO'])

    def test_default_folders_parsed(self):
        """Default folders parses the content of the file."""
        filename = self._create_faked_file(content=REAL_CONTENT)
        actual = utils.default_folders(user_home=USER_HOME, dirs_path=filename)

        expected = (i.split(u'=')[1].strip(u'"') for i in REAL_CONTENT.split())
        expected = [i.replace(u'$HOME', USER_HOME) for i in expected]
        self.assertEqual(actual, expected)

    def test_default_folders_non_ascii(self):
        """Default folders parses the content if it's non-ascii."""
        filename = self._create_faked_file(content=NON_ASCII)
        actual = utils.default_folders(user_home=USER_HOME, dirs_path=filename)

        expected = (i.split(u'=')[1].strip(u'"') for i in NON_ASCII.split())
        expected = [i.replace(u'$HOME', USER_HOME) for i in expected]
        self.assertEqual(actual, expected)

    def test_default_folders_bad_encoding(self):
        """Default folders parses the content if it's non-ascii."""
        filename = self._create_faked_file(content=LATIN_1, encoding='latin-1')
        actual = utils.default_folders(user_home=USER_HOME, dirs_path=filename)

        self.assertEqual(actual, [])
