# test_config - tests for ubuntuone.oauthdesktop.config
#
# Author: Stuart Langridge <stuart.langridge@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""Tests for the OAuth client code for StorageFS."""

import os, StringIO
from contrib.mocker import MockerTestCase
from ubuntuone.oauthdesktop import config
from ubuntuone.oauthdesktop.config import get_config

class ConfigFileTests(MockerTestCase):
    """Test the config file finder"""

    def test_get_file_from_tmp(self):
        """Does the tmp file get chosen?"""
        
        # mock that tmp config file exists
        tmp_config_file = os.path.realpath(os.path.join(
          os.path.split(config.__file__)[0], "../../data/oauth_urls"
        ))
        osp = self.mocker.replace("os.path")
        osp.isfile(tmp_config_file)
        self.mocker.result(True)
        self.mocker.replay()
        
        conf = get_config()
        self.assertEqual(conf.FILENAME, tmp_config_file)

    def test_get_file_from_home(self):
        """Does our home config file get chosen?"""
        
        # mock that home config file exists
        home_config_file = os.path.expanduser("~/.config/ubuntuone/oauth_urls")
        osp = self.mocker.replace("os.path")
        osp.exists(home_config_file)
        self.mocker.result(True)
        self.mocker.replay()
        
        conf = get_config(use_tmpconfig=False) # pretend not in source tree
        self.assertEqual(conf.FILENAME, home_config_file)

    def test_get_file_from_etc(self):
        """Does our /etc config file get chosen?"""
        
        # mock that etc config file exists
        etc_config_file = os.path.expanduser("/etc/xdg/ubuntuone/oauth_urls")
        osp = self.mocker.replace("os.path")
        osp.exists(etc_config_file)
        self.mocker.result(True)
        self.mocker.replay()
        
        conf = get_config(use_tmpconfig=False) # pretend not in source tree
        self.assertEqual(conf.FILENAME, etc_config_file)

    def test_tmp_file_parses(self):
        """Does the tmp file get chosen and parse correctly?"""
        
        # mock that tmp config file exists
        tmp_config_file = os.path.realpath(os.path.join(
          os.path.split(config.__file__)[0], "../../data/oauth_urls"
        ))
        osp = self.mocker.replace("os.path")
        osp.isfile(tmp_config_file)
        self.mocker.result(True)
        
        sio = StringIO.StringIO("[default]\n")
        mock_open = self.mocker.replace(open)
        mock_open(tmp_config_file)
        self.mocker.result(sio)
        
        self.mocker.replay()
        
        conf = get_config()
        self.assertTrue(conf.has_section("default"))


