# -*- coding: utf-8 -*-
#
# Copyright 2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Tests for the Setup Account page Qt UI."""

from PyQt4 import QtCore

from ubuntu_sso.qt import forgotten_password_page as gui
from ubuntu_sso.qt.tests import (
    PageBaseTestCase,
    FakePageUiStyle,
)

# is ok to access private method/attrs in tests
# pylint: disable=W0212, E1101


class ForgottenPasswordTestCase(PageBaseTestCase):
    """Test the SetupAccountPage code."""

    ui_class = gui.ForgottenPasswordPage
    ui_siganls = ('passwordResetTokenSent',)
    ui_backend_siganls = ('PasswordResetTokenSent', 'PasswordResetError')

    def test_request_new_password(self):
        """Test the request_new_password function."""
        email = 'email@example.com'
        self.ui.ui.email_line_edit.setText(email)
        self.ui.request_new_password()
        self.assertEqual(self._overlay_show_counter, 1)
        self.assertFalse(self.ui.isEnabled())
        self.assert_backend_called('request_password_reset_token',
            self.app_name, email)

    def test_email_address(self):
        """Test the email_address property."""
        value = self.ui.email_address
        self.assertEqual(value, unicode(self.ui.ui.email_line_edit.text()))
        self.assertTrue(isinstance(value, unicode))

    def test_send_button_clicked(self):
        """Test the send_button property."""
        self.patch(self.ui, "hide_error", self._set_called)
        self.ui.ui.send_button.clicked.emit(True)
        self.assert_backend_called('request_password_reset_token',
            self.app_name, '')
        expected = ((), {})
        self.assertEqual(expected, self._called)

    def test_initialize_page_with_email(self):
        """Test the initialization method."""
        self.ui.ui.email_line_edit.setText('mail@example')
        button = FakePageUiStyle()
        self.patch(self.ui.ui, "send_button", button)
        self.ui.initializePage()
        self.assertTrue(button.properties['default'])
        self.assertTrue(button.isEnabled())

    def test_set_translated_strings(self):
        """Test the translated string method."""
        self.assert_title_correct(gui.FORGOTTEN_PASSWORD_TITLE)
        subtitle = gui.FORGOTTEN_PASSWORD_SUBTITLE
        self.assert_subtitle_correct(subtitle.format(app_name=self.app_name))

        self.assertEqual(unicode(self.ui.ui.email_address_label.text()),
            gui.EMAIL_LABEL)
        self.assertEqual(unicode(self.ui.ui.send_button.text()),
            gui.RESET_PASSWORD)

    def test_connect_ui(self):
        """Test the connect ui method."""
        self.assertEqual(self.ui.ui.email_line_edit.receivers(
            QtCore.SIGNAL("textChanged(const QString&)")), 2)
        self.assertEqual(self.ui.ui.send_button.receivers(
            QtCore.SIGNAL("clicked()")), 1)

    def test_validate_not_valid(self):
        """Test validate method."""
        self.ui.ui.email_line_edit.setText('')
        button = FakePageUiStyle()
        self.patch(self.ui.ui, "send_button", button)
        self.ui._validate()
        self.assertFalse(button.isEnabled())

    def test_validate_valid(self):
        """Test validate method."""
        self.ui.ui.email_line_edit.setText('mail@example')
        button = FakePageUiStyle()
        self.patch(self.ui.ui, "send_button", button)
        self.ui._validate()
        self.assertTrue(button.isEnabled())

    def test_on_password_reset_token_sent(self):
        """Test on_password_reset_token_sent method."""
        email = 'foo@ubuntu.com'
        self.assert_signal_emitted(self.ui.passwordResetTokenSent, (email,),
            self.ui.on_password_reset_token_sent, self.app_name, email)

    def test_on_password_reset_error(self):
        """Test on_password_reset_error method."""
        self.patch(self.ui, "show_error", self._set_called)
        error = {'errtype': 'FooBarBaz'}

        self.ui.on_password_reset_error(self.app_name, error)

        expected = ((self.ui, gui.REQUEST_PASSWORD_TOKEN_WRONG_EMAIL), {})
        self.assertTrue(expected, self._called)
