# telepathy-butterfly - an MSN connection manager for Telepathy
#
# Copyright (C) 2007 Johann Prieur <johann.prieur@gmail.com>
# Copyright (C) 2011-2012 Canonical, Ltd.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import logging
import dbus

import telepathy

from tpufa.android.androidcontacts import AndroidContacts

__all__ = ['UfAAvatars']

logger = logging.getLogger('UfAAvatars')

SUPPORTED_AVATAR_MIME_TYPES = dbus.Array(["image/png", "image/jpeg",
    "image/gif"], signature='s')
MINIMUM_AVATAR_PIXELS = dbus.UInt32(96)
RECOMMENDED_AVATAR_PIXELS = dbus.UInt32(96)
MAXIMUM_AVATAR_PIXELS = dbus.UInt32(192)
MAXIMUM_AVATAR_BYTES = dbus.UInt32(500 * 1024)

class UfAAvatars(\
        telepathy.server.ConnectionInterfaceAvatars,
        AndroidContacts):

    def __init__(self):
        self._avatar_known = False
        telepathy.server.ConnectionInterfaceAvatars.__init__(self)
        AndroidContacts.__init__(self)

        dbus_interface = telepathy.CONNECTION_INTERFACE_AVATARS
        self._implement_property_get(dbus_interface, {
            'SupportedAvatarMIMETypes':
                lambda: SUPPORTED_AVATAR_MIME_TYPES,
            'MinimumAvatarHeight': lambda: MINIMUM_AVATAR_PIXELS,
            'MinimumAvatarWidth': lambda: MINIMUM_AVATAR_PIXELS,
            'RecommendedAvatarHeight': lambda: RECOMMENDED_AVATAR_PIXELS,
            'RecommendedAvatarWidth': lambda: RECOMMENDED_AVATAR_PIXELS,
            'MaximumAvatarHeight': lambda: MAXIMUM_AVATAR_PIXELS,
            'MaximumAvatarWidth': lambda: MAXIMUM_AVATAR_PIXELS,
            'MaximumAvatarBytes': lambda: MAXIMUM_AVATAR_BYTES,
            })

    def GetAvatarRequirements(self):
        return (SUPPORTED_AVATAR_MIME_TYPES,
                MINIMUM_AVATAR_PIXELS, MINIMUM_AVATAR_PIXELS,
                MAXIMUM_AVATAR_PIXELS, MAXIMUM_AVATAR_PIXELS,
                MAXIMUM_AVATAR_BYTES)

    def GetKnownAvatarTokens(self, contacts):
        result = {}
        contact = None
        for handle_id in contacts:
            if handle_id == self._self_handle.id:
                result[self._self_handle] = ""
                continue
            self.fill_contact_info(handle_id)
            if handle_id in self.contactInfo:
                contact = self.contactInfo[handle_id]
            if contact and 'avatar' in contact:
                handle = self.handle(telepathy.HANDLE_TYPE_CONTACT, handle_id)
                # if we have avatar 'key', then it is safe to read the index 0 directly
                result[handle] = contact['avatar'][0]
        return result

    def RequestAvatars(self, contacts):
        for handle_id in contacts:
            self.fill_contact_info(handle_id)
            if not handle_id in self.contactInfo:
                return
            contact = self.contactInfo[handle_id]
            handle = self.handle(telepathy.HANDLE_TYPE_CONTACT, handle_id)
            token = contact['avatar'][0]
            avatar = contact['avatar'][1]
            imgtype = contact['avatar'][2]
            self.AvatarRetrieved(handle, token, avatar, 'image/' + imgtype)

    def SetAvatar(self, avatar, mime_type):
        # not supported
        return None

    def ClearAvatar(self):
        # not supported
        pass
