/*
 * bootinfo.c - boot information
 *
 *  inforead  -o offset {-1|-2|-3|-4 -l length}
 *  infowrite -o offset {-1|-2|-3|-4 -l length} data
 *
 * modification information
 * ------------------------
 * 2009/06/29 : first release
 */

/* Include Files */
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <mtd/mtd-user.h>

#define MTDNAME     ("/dev/mtd2")
#define INFO_LENGTH (0x10000)

unsigned char infodata [INFO_LENGTH];
unsigned char chkdata [INFO_LENGTH];

/*
 * eraseBlock - MTD erase block
 *  in        : fd     - MTD descriptor
 *              length - erase length
 *  return    : OK/ERROR
 */
static int eraseBlock (int fd, int length)
{
	int	     rc;
	mtd_info_t   meminfo;
	erase_info_t erase;

	rc = ioctl(fd, MEMGETINFO, &meminfo);
	if (rc) {
		perror("eraseBlock - MEMGETINFO");
		return -1;
	}
	erase.length = meminfo.erasesize;
	for ( erase.start = 0 ; erase.start < length ; erase.start += meminfo.erasesize ){
		rc = ioctl(fd, MEMERASE, &erase);
		if (rc) {
			perror("eraseBlock - MEMERASE");
			return -1;
		}
	}
	return 0;
}

/*
 * writeBlock - write Block
 *  in        : buf    - write data
 *              length - write length
 *  return    : OK/ERROR
 */
static int writeBlock (unsigned char *buf, int length)
{
	int  rc, ret, fd;

	fd = open (MTDNAME, O_RDWR);
	if (fd < 0) {
		fprintf (stderr, "%s: %s\n", MTDNAME, strerror(errno));
		return -1;
	}

	rc = eraseBlock (fd, length);
	if (rc != 0) {
		goto exit;
	}

	ret = write (fd, buf, length);
	if (ret != length) {
		perror ("writeBlock");
		rc = -1;
	}
exit:
	close (fd);
	return rc;
}

/*
 * readBlock - read Block
 *  in        : buf    - read data
 *              length - read length
 *  return    : OK/ERROR
 */
static int readBlock (unsigned char *buf, int length)
{
	int    rc = 0, ret, fd;

	fd = open (MTDNAME, O_RDWR);
	if (fd < 0) {
		fprintf (stderr, "%s: %s\n", MTDNAME, strerror(errno));
		return -1;
	}

	ret = read (fd, buf, length);
	if (ret != length) {
		perror ("readBlock");
		rc = -1;
	}

	close (fd);
	return rc;
}

/*
 * inforead - boot information read
 */
static int inforead (int argc, char *argv[])
{
	int            usage, offset, length, i, rc;
	unsigned char *p;

	usage  = 0;
	offset = -1;
	length = -1;
	for ( i = 1 ; i < argc ; i++ ) {
		if (argv [i][0] == '-') {
			if (argv [i][1] == '1') {
				length = 1;
			} else if (argv [i][1] == '2') {
				length = 2;
			} else if (argv [i][1] == '3') {
				length = 3;
			} else if (argv [i][1] == '4') {
				length = 4;
			} else if (argv [i][1] == 'l') {
				if ((i + 1) < argc) {
					if ((argv [i+1][0] == '0') &&
					    ((argv [i+1][1] == 'x') ||
					     (argv [i+1][1] == 'X'))) {
						sscanf (argv [i+1], "%x", &length);
					} else {
						sscanf (argv [i+1], "%d", &length);
					}
				} else {
					usage++;
				}
				i++;
			} else if (argv [i][1] == 'o') {
				if ((i + 1) < argc) {
					if ((argv [i+1][0] == '0') &&
					    ((argv [i+1][1] == 'x') ||
					     (argv [i+1][1] == 'X'))) {
						sscanf (argv [i+1], "%x", &offset);
					} else {
						sscanf (argv [i+1], "%d", &offset);
					}
				} else {
					usage++;
				}
				i++;
			} else {
				usage++;
			}
		} else {
			usage++;
		}
	}
	if ((0 < usage) || (offset == -1)) {
		fprintf (stderr, "Usage: -o offset {-1|-2|-3|-4 -l length}\n");
		return 1;
	}

	/*
	 * default length
	 */
	if (length == -1) {
		length = 4;
	}

	/*
	 * check length
	 */
	if (sizeof(infodata) < (offset + length)) {
		fprintf (stderr, "over offset+length %d\n", sizeof(infodata));
		return 1;
	}

	/*
	 * read boot info
	 */
	rc = readBlock (infodata, sizeof(infodata));
	if (rc != 0) {
		return 1;
	}

	p = &(infodata [offset]);
	if (length == 1) {
		fprintf (stdout, "offset:%X value:%02X\n", offset, p[0]);
	} else if (length == 2) {
		fprintf (stdout, "offset:%X value:%02X%02X\n", offset, p[1], p[0]);
	} else if (length == 3) {
		fprintf (stdout, "offset:%X value:%02X%02X%02X\n",
				 offset, p[2], p[1], p[0]);
	} else if (length == 4) {
		fprintf (stdout, "offset:%X value:%02X%02X%02X%02X\n",
				 offset, p[3], p[2], p[1], p[0]);
	} else {
		if (length <= 32) {
			fprintf (stdout, "offset:%X %dbytes  ", offset, length);
			for ( i = 0 ; i < length ; i++ ) {
				unsigned char c = *p++;
				if (isprint (c)) {
					fprintf (stdout, "%c", c);
				} else {
					fprintf (stdout, ".");
				}
			}
			fprintf (stdout, "\n");
		} else {
			fprintf (stdout, "offset:%X %dbytes\n", offset, length);
			for ( i = 0 ; i < length ; i++ ) {
				unsigned char c = *p++;
				if (isprint (c)) {
					fprintf (stdout, "%c", c);
				} else {
					fprintf (stdout, ".");
				}
				if ((i % 64) == 63) {
					fprintf (stdout, "\n");
				}
			}
			if ( i % 64 ) {
				fprintf (stdout, "\n");
			}
		}
	}

	return 0;
}

/*
 * infowrite - boot information write
 */
static int infowrite (int argc, char *argv[])
{
	int            usage, offset, length, i, rc, value;
	char          *pvalue;
	unsigned char *p, *pn;

	usage  = 0;
	offset = -1;
	length = -1;
	pvalue = 0;
	for ( i = 1 ; i < argc ; i++ ) {
		if (argv [i][0] == '-') {
			if (argv [i][1] == '1') {
				length = 1;
			} else if (argv [i][1] == '2') {
				length = 2;
			} else if (argv [i][1] == '3') {
				length = 3;
			} else if (argv [i][1] == '4') {
				length = 4;
			} else if (argv [i][1] == 'l') {
				if ((i + 1) < argc) {
					if ((argv [i+1][0] == '0') &&
					    ((argv [i+1][1] == 'x') ||
					     (argv [i+1][1] == 'X'))) {
						sscanf (argv [i+1], "%x", &length);
					} else {
						sscanf (argv [i+1], "%d", &length);
					}
				} else {
					usage++;
				}
				i++;
			} else if (argv [i][1] == 'o') {
				if ((i + 1) < argc) {
					if ((argv [i+1][0] == '0') &&
					    ((argv [i+1][1] == 'x') ||
					     (argv [i+1][1] == 'X'))) {
						sscanf (argv [i+1], "%x", &offset);
					} else {
						sscanf (argv [i+1], "%d", &offset);
					}
				} else {
					usage++;
				}
				i++;
			} else {
				usage++;
			}
		} else if (pvalue == 0) {
			pvalue = argv [i];
		} else {
			usage++;
		}
	}
	if ((0 < usage) || (offset == -1) || (pvalue == 0)) {
		fprintf (stderr, "Usage: -o offset {-1|-2|-3|-4 -l length} data\n");
		return 1;
	}

	/*
	 * default length
	 */
	if (length == -1) {
		length = 4;
	}

	/*
	 * check length
	 */
	if (sizeof(infodata) < (offset + length)) {
		fprintf (stderr, "over offset+length %d\n", sizeof(infodata));
		return 1;
	}

	/*
	 * read boot info
	 */
	rc = readBlock (infodata, sizeof(infodata));
	if (rc != 0) {
		return 1;
	}

	/*
	 * update data
	 */
	p = &(infodata [offset]);
	if ((length == 1) || (length == 2) || (length == 3) || (length == 4)) {
		if ((pvalue [0] == '0') &&
		    ((pvalue [1] == 'x') || (pvalue [1] == 'X'))) {
			sscanf (pvalue, "%x", &value);
		} else {
			sscanf (pvalue, "%d", &value);
		}
		if (length == 1) {
			p [0] = value;
		} else if (length == 2) {
			p [0] = value;
			p [1] = value >> 8;
		} else if (length == 3) {
			p [0] = value;
			p [1] = value >> 8;
			p [2] = value >> 16;
		} else {
			p [0] = value;
			p [1] = value >> 8;
			p [2] = value >> 16;
			p [3] = value >> 24;
		}
	} else {
		memcpy (p, pvalue, length);
	}

	/*
	 * write boot info
	 */
	rc = writeBlock (infodata, sizeof(infodata));
	if (rc != 0) {
		return 1;
	}

	/*
	 * read check
	 */
	rc = readBlock (chkdata, sizeof(chkdata));
	if (rc != 0) {
		return 1;
	}
	p  = &(infodata [offset]);
	pn = &(chkdata [offset]);
	for ( i = 0 ; i < length ; i++ , p++ , pn++ ) {
		if (*p != *pn) {
			fprintf (stdout, "Verify error offset:%x %02X - %02X\n",
				 i, *p, *pn);
			rc = 2;
			break;
		}
	}

	return rc;
}

/*
 * main - boot information
 */
int main(int argc, char *argv[])
{
	int   rc;
	char *p;

	/*
	 * program name
	 */
	p = rindex (argv [0], '/');
	if (p == 0) {
		p = argv [0];
	} else {
		p++;	/* skip "/" */
	}
	if (strcmp (p, "inforead") == 0) {
		rc = inforead (argc, argv);
	} else if (strcmp (p, "infowrite") == 0) {
		rc = infowrite (argc, argv);
	} else {
		rc = 1;
	}
	exit ( rc );
}
