/*
 * Copyright (c) 2015 Intel Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "PSHService.h"
#include "SensorEnumerator.h"
#include "PSHSensor.h"
#include <dirent.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include "Log.h"
#include "PSHAccelerometer.h"
#include "PSHAmbientTemperature.h"
#include "PSHCustomerSensor.h"
#include "PSHGameRotationVector.h"
#include "PSHGeomagneticRotationVector.h"
#include "PSHGlanceGesture.h"
#include "PSHGravity.h"
#include "PSHGyroscope.h"
#include "PSHGyroscopeUncalibrated.h"
#include "PSHHeartRate.h"
#include "PSHLight.h"
#include "PSHLinearAcceleration.h"
#include "PSHMagneticField.h"
#include "PSHMagneticFieldUncalibrated.h"
#include "PSHOrientation.h"
#include "PSHPickUpGesture.h"
#include "PSHPressure.h"
#include "PSHProximity.h"
#include "PSHRelativeHumidity.h"
#include "PSHRotationVector.h"
#include "PSHSignificantMotion.h"
#include "PSHStepCounter.h"
#include "PSHStepDetector.h"
#include "PSHTemperature.h"
#include "PSHTiltDetector.h"
#include "PSHWakeGesture.h"
#include "PSHFaceDetection.h"
#include "PSHMeter.h"

// Current PSH driver only support 4096 bytes in maximum
#define RESP_BUF_MAX_SIZE 4096

PSHService::PSHService()
        :ctrlFd(-1), dataFd(-1), dataSizeFd(-1)
{
        memset(&metaEvent, 0, sizeof(metaEvent));
        metaEvent.version = META_DATA_VERSION;
        metaEvent.type = SENSOR_TYPE_META_DATA;
        metaEvent.meta_data.what = META_DATA_FLUSH_COMPLETE;

        if (!initializeInterfaces())
                return;

        dataBuf = new char[128 * 1024];
        respBuf = new unsigned char[RESP_BUF_MAX_SIZE];
        uint8_t resetCmd[4] = { 0, 1, 0, 0};
        size_t resp_size;

        // setup DDR
        sendControlCommand(resetCmd, 4, NULL, &resp_size);
        // Todo: handle the response if needed

        int error = pthread_mutex_init(&mFlushQueueLock, NULL);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_init error: %s", __FUNCTION__, __LINE__, strerror(error));
        }

        if (!getStatus())
                return;
}

PSHService::~PSHService()
{
        int error = pthread_mutex_destroy(&mFlushQueueLock);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_destroy error: %s", __FUNCTION__, __LINE__, strerror(error));
        }
}

#define MAX_STRING_SIZE 64
bool PSHService::initializeInterfaces()
{
        /* open control node */
        ctrlFd = open("/sys/class/pci_bus/0000:00/device/0000:00:0e.0/ctrl", O_RDWR);
        if (ctrlFd == -1) {
                ALOGE("%s line: %d open %s failed: %s", __FUNCTION__, __LINE__, "/sys/class/pci_bus/0000:00/device/0000:00:0e.0/ctrl", strerror(errno));
                return false;
        }

        /* open data node */
        dataFd = open("/sys/class/pci_bus/0000:00/device/0000:00:0e.0/data", O_RDONLY);
        if (dataFd == -1) {
                ALOGE("%s line: %d open %s failed: %s", __FUNCTION__, __LINE__, "/sys/class/pci_bus/0000:00/device/0000:00:0e.0/data", strerror(errno));
                return false;
        }

        /* open data_size node */
        dataSizeFd = open("/sys/class/pci_bus/0000:00/device/0000:00:0e.0/data_size", O_RDONLY);
        if (dataSizeFd == -1) {
                ALOGE("%s line: %d open %s failed: %s", __FUNCTION__, __LINE__, "/sys/class/pci_bus/0000:00/device/0000:00:0e.0/data_size", strerror(errno));
                return false;
        }

        return true;
}

#define CMD_GET_STATUS                11
bool PSHService::getStatus()
{
#define LINK_AS_CLIENT                (0)
#define LINK_AS_MONITOR                (1)
#define LINK_AS_REPORTER        (2)
        struct link_info {
                unsigned char id;
                unsigned char ltype;
                unsigned short slide;
        } __attribute__ ((packed));

        struct sensor_info {
                unsigned char id;
                unsigned char status;
                unsigned short freq;
                unsigned short data_cnt;
                unsigned short slide;
                unsigned short priv;
#define SENSOR_ATTRIB_HAVE_CALIB        ((unsigned short) 0x1 << 12)
                unsigned short attri;

                short freq_max;        // -1, means no fixed data rate
                char name[SNR_NAME_MAX_LEN + 1];

                unsigned char health;
                unsigned char link_num;
                struct link_info linfo[0];
        } __attribute__ ((packed));

        struct psh_cmd_resp *resp;
        struct sensor_info *info;
        unsigned char getStatusCmd[12] = { 0, CMD_GET_STATUS, 0, 0,
                                           0xff, 0xff, 0xff, 0xff,
                                           0xff, 0xff, 0xff, 0xff };
        size_t resp_size = 0;

        if (!sendControlCommand(getStatusCmd, 12, NULL, &resp_size)) {
                ALOGE("%s line: %d send control command failed!", __FUNCTION__, __LINE__);
                return false;
        }

        int left = resp_size;
        while (left > 0) {
                resp = reinterpret_cast<struct psh_cmd_resp *>(respBuf + (resp_size -left));
                info = reinterpret_cast<struct sensor_info *>(resp->buf);
                left -= sizeof(struct psh_cmd_resp) + resp->data_len;
                if (left < 0) {
                        ALOGW("%s line: %d data_len: %u left: %d", __FUNCTION__, __LINE__, resp->data_len, left);
                        break;
                }
                if (resp->type != CMD_GET_STATUS) {
                        ALOGW("%s line: %d Unknown response type: %u", __FUNCTION__, __LINE__, resp->type);
                        continue;
                }
                if (resp->data_len == 0) {
                        ALOGD("%s line: %d status response end.", __FUNCTION__, __LINE__);
                        break;
                }
                ALOGV_IF(VERBOSE, "%s line: %d status response: (%u, \"%s\", %u)", __FUNCTION__, __LINE__, info->id, info->name, resp->data_len);
                psh_sensor_information_t information;
                information.id = info->id;
                information.freq_max = info->freq_max;
                information.support_calibration = (info->attri & SENSOR_ATTRIB_HAVE_CALIB) == 0 ? false : true;
                sensorsInfo[info->name] = information;
        }

        return true;
}

bool PSHService::sendControlCommand(const void *buf, size_t count, void **resp_buf, size_t *resp_size)
{
        bool result = true;
        ssize_t ret;
        const struct psh_cmd *cmd = reinterpret_cast<const struct psh_cmd *>(buf);

        *resp_size = 0;
        if (resp_buf != NULL)
                *resp_buf = respBuf;

        if (count < sizeof(struct psh_cmd) || count > sizeof(struct psh_cmd) + CMD_PARAM_MAX_SIZE) {
                ALOGE("%s line: %d Invalid command size: %d", __FUNCTION__, __LINE__, count);
                return false;
        }

        ALOGV_IF(VERBOSE, "%s line: %d command tran_id: %u, cmd_id: %u, sensor_id: %u", __FUNCTION__, __LINE__, cmd->tran_id, cmd->cmd_id, cmd->sensor_id);
        ret = write(ctrlFd, buf, count);
        if (ret < 0) {
                ALOGE("%s line: %d write to control node error: %s", __FUNCTION__, __LINE__, strerror(errno));
                result = false;
        }

        do {
                ret = read(ctrlFd, respBuf + *resp_size, RESP_BUF_MAX_SIZE);
                if (ret < 0) {
                        ALOGE("%s line: %d read from control node error: %s", __FUNCTION__, __LINE__, strerror(errno));
                        return false;
                }
                *resp_size += ret;
        } while (ret > 0);
        ALOGV_IF(VERBOSE, "%s line: %d returns: %d, %d", __FUNCTION__, __LINE__, ret, *resp_size);

        return result;
}

void PSHService::getDataFds(std::queue<int> &dataFdQueue)
{
        dataFdQueue.push(dataSizeFd);
}

struct cal_param_info {
        char size;
        char data[0];
} __attribute__ ((packed));

struct resp_calibration {
        unsigned char result;
        struct cal_param_info info;
} __attribute__ ((packed));

bool PSHService::handleEvents(int fd)
{
        char dataSizeBuf[8] = { 0 };
        int dataSize, left;
        int ret = pread(fd, dataSizeBuf, 8, 0);
        if (ret < 0) {
                ALOGE("%s line: %d Read data size error: %s", __FUNCTION__, __LINE__, strerror(errno));
                return false;
        }
        if (ret == 0) {
                ALOGE("%s line: %d Read data return length 0.", __FUNCTION__, __LINE__);
                return false;
        }

        dataSizeBuf[7] = 0;
        sscanf(dataSizeBuf, "%d", &dataSize);
        if ((dataSize <= 0) || (dataSize > 128 * 1024)) {
                ALOGE("%s line: %d Invalid data size: %d", __FUNCTION__, __LINE__, dataSize);
                return false;
        }

        left = dataSize;

        /* sysfs has limitation that max 4K data can be returned at once */
        while (left > 0) {
                ret = read(dataFd, dataBuf + dataSize - left, left);
                if (ret < 0) {
                        ALOGE("%s line: %d Read data error: %s", __FUNCTION__, __LINE__, strerror(errno));
                        return false;
                }
                if (ret == 0) {
                        ALOGE("%s line: %d Read data return length 0.", __FUNCTION__, __LINE__);
                        return false;
                }
                left = left - ret;
        }
        ret = dataSize;

        struct psh_cmd_resp *resp = reinterpret_cast<struct psh_cmd_resp *>(dataBuf);
        char *p = dataBuf;
        while (ret > 0) {
                if (resp->type == RESP_STREAMING) {
                        if (sensorsTable.count(resp->sensor_id)) {
                                Sensor *sensor = sensorsTable[resp->sensor_id];
                                sensor->handleEvents(resp->buf, resp->data_len);
                        } else {
                                ALOGE("%s line: %d Cannot find sensor with id: %d", __FUNCTION__, __LINE__, resp->sensor_id);
                        }
                } else if (resp->type == RESP_PSH_EVT) {
#define PSH_EVT_ID_FLUSH_DONE ((unsigned char)1)
                        struct resp_psh_evt {
                                unsigned char evt_id;
                                unsigned char evt_buf_len;
                                char evt_buf[0];
                        } __attribute__ ((packed));
                        struct resp_psh_evt *evt = reinterpret_cast<struct resp_psh_evt *>(resp->buf);

                        if (resp->data_len != sizeof(*evt) + evt->evt_buf_len)
                                ALOGE("Get a resp_psh_evt with wrong data_len: %d\n", resp->data_len);

                        if (evt->evt_id == PSH_EVT_ID_FLUSH_DONE) {
                                lockFlushQueue();
                                while (!flushQueue.empty()) {
                                        struct flushParams &params = flushQueue.front();
                                        metaEvent.meta_data.sensor = params.handle;
                                        params.session->dispatchEvents(sizeof(metaEvent), &metaEvent);
                                        flushQueue.pop();
                                }
                                unlockFlushQueue();
                        }
                } else if (resp->type == RESP_CAL_RESULT) {
                        struct resp_calibration *calibration = reinterpret_cast<struct resp_calibration *>(resp->buf);
                        // Todo: handle/store calibration result if necessary
                        ALOGD("%s line: %d calibration result: %u sensor id: %u", __FUNCTION__, __LINE__, calibration->result, resp->sensor_id);
                } else {
                        // Todo: handle useful response
                        ALOGW("%s line: %d Unexcepted response: %u", __FUNCTION__, __LINE__, resp->type);
                }

                ret = ret - (sizeof(struct psh_cmd_resp) + resp->data_len);
                p = p + sizeof(struct psh_cmd_resp) + resp->data_len;
                resp = reinterpret_cast<struct psh_cmd_resp *>(p);

        }
        return false;
}

void PSHService::addToFlushQueue(int handle, Session *session)
{
        struct flushParams params = { handle, session };
        flushQueue.push(params);
}

void PSHService::removeFromFlushQueue()
{
        flushQueue.pop();
}

#define DEVICE_INITIALIZED   (1 << 0)
#define ADDINFO_INITIALIZED  (1 << 1)
#define PRIVDATA_INITIALIZED (1 << 2)
int PSHService::enumerateSensors(SensorEnumerator * enumerator, const xmlNodePtr ishRoot, std::vector<Sensor *> &sensors)
{
        struct sensor_device_t device;
        sensor_additional_information_t information;
        psh_private_data_t data;
        int index = 0;
        Sensor * sensor = NULL;

        memset(&device, 0, sizeof(device));
        memset(&data, 0, sizeof(data));

        for (xmlNodePtr node = ishRoot->xmlChildrenNode; node != NULL; node = node->next) {
                xmlNodePtr p = node->xmlChildrenNode;
                const xmlChar * typeStr = xmlGetProp(node, reinterpret_cast<const xmlChar*>("type"));
                unsigned int result = 0;
                while (p != NULL) {
                        if (!strcmp(reinterpret_cast<const char *>(p->name), "device")) {
                                bool ret = enumerator->initializeSensorDevice(p, device, reinterpret_cast<const char *>(typeStr));
                                p = p->next;
                                if (ret)
                                        result |= DEVICE_INITIALIZED;
                        } else if (!strcmp(reinterpret_cast<const char *>(p->name), "additional_information")) {
                                enumerator->initializeSensorAdditionalInformation(p, information);
                                p = p->next;
                                result |= ADDINFO_INITIALIZED;
                        } else if (!strcmp(reinterpret_cast<const char *>(p->name), "private_data")) {
                                bool ret = initializePrivateData(p, data, device);
                                p = p->next;
                                if (ret)
                                        result |= PRIVDATA_INITIALIZED;
                        }
                }
                if (result != (DEVICE_INITIALIZED | ADDINFO_INITIALIZED | PRIVDATA_INITIALIZED))
                        continue;
                sensor = constructSensorByType(device, information, data);
                if (sensor == NULL) {
                        ALOGW("%s line: %d Cannot construct sensor: type: %d  name: %s", __FUNCTION__, __LINE__, device.type, device.name);
                        continue;
                }

                sensors.push_back(sensor);

                index++;
                sensorsTable[data.id] = sensor;
        }

        return index;
}

bool PSHService::initializePrivateData(xmlNodePtr node, psh_private_data_t &data, struct sensor_device_t &device)
{
        xmlNodePtr p = node->xmlChildrenNode;
        xmlChar *str = NULL;
        std::string name;

        while (p != NULL) {
                if (!strcmp(reinterpret_cast<const char *>(p->name), "name")) {
                        str = xmlNodeGetContent(p);
                        if (str != NULL) {
                                name = reinterpret_cast<char *>(str);
                                xmlFree(str);
                        }
                }
                p = p->next;
        }

        if (name.length() == 0)
                name = typeTable.getPSHTypeName(device.type);

        if (!sensorsInfo.count(name)) {
                ALOGE("%s line: %d cannot find sensor with identify name: %s", __FUNCTION__, __LINE__, name.c_str());
                return false;
        }
        if (name.length() > SNR_NAME_MAX_LEN) {
                ALOGE("%s line: %d invalid name: %s, maximum length: %d actual length: %d", __FUNCTION__, __LINE__, name.c_str(), SNR_NAME_MAX_LEN, name.length());
                return false;
        }

        strncpy(data.name, name.c_str(), SNR_NAME_MAX_LEN);

        psh_sensor_information_t info = sensorsInfo[name];
        data.id = info.id;
        data.support_calibration = info.support_calibration;

        int32_t minDelay = info.freq_max <= 0 ? 0 : 1000000 / info.freq_max;
        if (device.minDelay != minDelay && device.minDelay > 0) {
                ALOGW("%s line: %d sensor %s minDelay does not align: config: %dus real: %dus", __FUNCTION__, __LINE__, name.c_str(), device.minDelay, minDelay);
        }

        return true;
}

Sensor * PSHService::constructSensorByType(const struct sensor_device_t &device, const sensor_additional_information_t &information, const psh_private_data_t &data)
{
        Sensor * sensor = NULL;

        switch (device.type) {
        case SENSOR_TYPE_ACCELEROMETER:
                sensor = new PSHAccelerometer(device, information, data, this);
                break;
        case SENSOR_TYPE_MAGNETIC_FIELD:
                sensor = new PSHMagneticField(device, information, data, this);
                break;
        case SENSOR_TYPE_ORIENTATION:
                sensor = new PSHOrientation(device, information, data, this);
                break;
        case SENSOR_TYPE_GYROSCOPE:
                sensor = new PSHGyroscope(device, information, data, this);
                break;
        case SENSOR_TYPE_LIGHT:
                sensor = new PSHLight(device, information, data, this);
                break;
        case SENSOR_TYPE_PRESSURE:
                sensor = new PSHPressure(device, information, data, this);
                break;
        case SENSOR_TYPE_TEMPERATURE:
                sensor = new PSHTemperature(device, information, data, this);
                break;
        case SENSOR_TYPE_PROXIMITY:
        case SENSOR_TYPE_CONTINUOUS_PROXIMITY:
                sensor = new PSHProximity(device, information, data, this);
                break;
        case SENSOR_TYPE_GRAVITY:
                sensor = new PSHGravity(device, information, data, this);
                break;
        case SENSOR_TYPE_LINEAR_ACCELERATION:
                sensor = new PSHLinearAcceleration(device, information, data, this);
                break;
        case SENSOR_TYPE_ROTATION_VECTOR:
                sensor = new PSHRotationVector(device, information, data, this);
                break;
        case SENSOR_TYPE_RELATIVE_HUMIDITY:
                sensor = new PSHRelativeHumidity(device, information, data, this);
                break;
        case SENSOR_TYPE_AMBIENT_TEMPERATURE:
                sensor = new PSHAmbientTemperature(device, information, data, this);
                break;
        case SENSOR_TYPE_MAGNETIC_FIELD_UNCALIBRATED:
                sensor = new PSHMagneticFieldUncalibrated(device, information, data, this);
                break;
        case SENSOR_TYPE_GAME_ROTATION_VECTOR:
                sensor = new PSHGameRotationVector(device, information, data, this);
                break;
        case SENSOR_TYPE_GYROSCOPE_UNCALIBRATED:
                sensor = new PSHGyroscopeUncalibrated(device, information, data, this);
                break;
        case SENSOR_TYPE_SIGNIFICANT_MOTION:
                sensor = new PSHSignificantMotion(device, information, data, this);
                break;
        case SENSOR_TYPE_STEP_DETECTOR:
                sensor = new PSHStepDetector(device, information, data, this);
                break;
        case SENSOR_TYPE_STEP_COUNTER:
                sensor = new PSHStepCounter(device, information, data, this);
                break;
        case SENSOR_TYPE_GEOMAGNETIC_ROTATION_VECTOR:
                sensor = new PSHGeomagneticRotationVector(device, information, data, this);
                break;
        case SENSOR_TYPE_HEART_RATE:
                sensor = new PSHHeartRate(device, information, data, this);
                break;
        case SENSOR_TYPE_TILT_DETECTOR:
                sensor = new PSHTiltDetector(device, information, data, this);
                break;
        case SENSOR_TYPE_WAKE_GESTURE:
                sensor = new PSHWakeGesture(device, information, data, this);
                break;
        case SENSOR_TYPE_GLANCE_GESTURE:
                sensor = new PSHGlanceGesture(device, information, data, this);
                break;
        case SENSOR_TYPE_PICK_UP_GESTURE:
                sensor = new PSHPickUpGesture(device, information, data, this);
                break;
        case SENSOR_TYPE_FACE_DETECTION:
                sensor = new PSHFaceDetection(device, information, data, this);
                break;
        case SENSOR_TYPE_METER:
                sensor = new PSHMeter(device, information, data, this);
                break;
        default:
                sensor = new PSHCustomerSensor(device, information, data, this);
                break;
        }

        return sensor;
}

void PSHService::lockFlushQueue()
{
        int error = pthread_mutex_lock(&mFlushQueueLock);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_lock error: %s", __FUNCTION__, __LINE__, strerror(error));
        }
}

void PSHService::unlockFlushQueue()
{
        int error = pthread_mutex_unlock(&mFlushQueueLock);
        if (error != 0) {
                ALOGE("%s line: %d pthread_mutex_unlock error: %s", __FUNCTION__, __LINE__, strerror(error));
        }
}
