#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/ioctl.h>
#include <Log.h>
#include "smhi_client.h"

#define IOCTL_HECI_CONNECT_CLIENT _IOWR('H' , 0x01, heci_guid_t)
#define HECI_CLIENT_SMHI_GUID { 0xbb579a2e, 0xcc54, 0x4450, { 0xb1, 0xd0, 0x5e, 0x75, 0x20, 0xdc, 0xad, 0x25 } }

typedef enum
{
        // SMHI Commands - defined in ISH FAS
        SMHI_GET_VERSION                = 1,
        SMHI_ISH_RESET                  = 2,
        SMHI_SELF_TEST                  = 3,
        SMHI_GET_SENSOR_CONF            = 4,
        SMHI_SET_SENSOR_CONF            = 5,
        SMHI_RESTORE_SENSOR_CONFIG      = 6,
        SMHI_GET_EXCEPTIONS             = 7,
        SMHI_GET_TIME                   = 8,
        SMHI_GET_SENSOR_PROP_VALUE       = 9,
        SMHI_SET_SENSOR_PROP_VALUE       = 0xA,
        SMHI_PWR_MEASURE_DEBUG       = 0xB,
        SMHI_PNR_SESSION_CONTROL        = 0xC,
        SMHI_PNR_PB_SET_BUF             = 0xD,
        SMHI_PNR_PB_GET_BUF             = 0xE,
        SMHI_PNR_REC_PUBLISH_BUF        = 0xF,
        SMHI_SET_I2C_INFO=0x10,

        // debug SMHI commands
        SMHI_GET_STAT                   = 0x70,
        SMHI_RESET_STAT                 = 0x71,
        SMHI_PM_SET_CONFIG              = 0x72,
        SMHI_PM_GET_CUR_CONFIG          = 0x73,
        // add extra debug commands here
#ifdef SMHI_DEBUG
        SMHI_SET_START_TIME_DEBUG       = 0x79,
        SMHI_SENSOR_CORE_DUMP           = 0x7A,
        SMHI_GET_HID_STATISTICS         = 0x7B,
#endif
        SMHI_PERFSYS_RESET              = 0x7C,
        SMHI_PERFSYS_GET                = 0x7D,
        SMHI_PERFMON_CTRL               = 0x7E,
        SMHI_PERFMON_GET_FIXED          = 0x7F,

        SMHI_COMMAND_LAST
} smhi_command_id;

typedef enum
{
        SMHI_SUCCESS         = 0,
        SMHI_MSG_UNSUPPORTED = 1,
        SMHI_FAILURE         = 2,
        SMHI_WIP             = 3
} smhi_host_status;

#pragma pack(1)

typedef struct {
        unsigned int  data1;
        unsigned short data2;
        unsigned short data3;
        unsigned char  data4[8];
} heci_guid_t;

typedef struct
{
        uint8_t command : 7;
        uint8_t is_response : 1;
        uint8_t reserved[2];
        uint8_t status;
} smhi_msg_header_t;

/********* SMHI_GET_TIME *********/
typedef struct
{
        uint64_t    ms_since_ish_reset;
} smhi_get_time_res;

/********* SMHI_GET_VERSION *********/
typedef struct
{
        ish_version_t           version;
} smhi_get_version_res;

/********* SMHI_SET_SENSOR_CONF *********/
typedef struct
{
        smhi_luid_t             sensor_luid;
        smhi_sensor_config_t    config_type     : 8; // 1 byte
        uint8_t                 reserved[3];         // align to dword
        uint32_t                data_size;
        uint8_t                 config_data[0];
} smhi_set_sensor_conf_req;

typedef struct _sc_sdt_sensor_data {
        sc_sensor_luid luid;/**< LUID of sensor */
        uint16_t  data_length;/**< length of sdt_info_entry data in bytes */
        uint8_t   reserved;/**< reserved */
        uint8_t   num_entries;/**< Number of entries */
        sc_sdt_info_entry   sdt_info_entry[];/**< SDT info */
} sc_sdt_sensor_data;

#pragma pack()

int smhi_client_connect()
{
        int handle;
        int result;
        heci_guid_t smhi_client_guid = HECI_CLIENT_SMHI_GUID;

        handle = open("/dev/ish", O_RDWR);
        if (handle < 0) {
                ALOGE("%s line: %d open /dev/ish error: %s", __FUNCTION__, __LINE__, strerror(errno));
                return -1;
        }

        result = ioctl(handle, IOCTL_HECI_CONNECT_CLIENT, &smhi_client_guid);
        if (result < 0) {
                ALOGE("%s line: %d ioctl /dev/ish error: %s", __FUNCTION__, __LINE__, strerror(errno));
                close(handle);
                return -1;
        }

        return handle;
}

int smhi_client_disconnect(int handle)
{
        int ret;

        ret = close(handle);
        if (ret < 0) {
                ALOGE("%s line: %d close error: %s", __FUNCTION__, __LINE__, strerror(errno));
                return -1;
        }

        return ret;
}

#define SMHI_RES_DEFAULT_BUF_SIZE (128)
static size_t smhi_client_get(int handle, smhi_command_id cmd, char **res, long unsigned int res_size)
{
        static size_t smhi_buf_size = sizeof(smhi_msg_header_t) + SMHI_RES_DEFAULT_BUF_SIZE;
        static char *buf = NULL;

        if ((buf == NULL) || (res_size + sizeof(smhi_msg_header_t) > smhi_buf_size)) {
                if (buf != NULL)
                        free(buf);
                if (res_size + sizeof(smhi_msg_header_t) > smhi_buf_size)
                        smhi_buf_size = res_size + sizeof(smhi_msg_header_t);
                buf = (char*)malloc(smhi_buf_size);
                if (buf == NULL) {
                        ALOGE("%s line: %d malloc error!", __FUNCTION__, __LINE__);
                        return 0;
                }
        }

        memset(buf, 0, smhi_buf_size);
        smhi_msg_header_t *header = (smhi_msg_header_t *)buf;
        *res = buf + sizeof(smhi_msg_header_t);

        header->command = cmd;
        int ret = write(handle, (void *)buf, sizeof(smhi_msg_header_t));
        if (ret < 0) {
                ALOGE("%s line: %d write to driver error: %s", __FUNCTION__, __LINE__, strerror(errno));
                goto error;
        } else if (ret != sizeof(smhi_msg_header_t)) {
                ALOGE("%s line: %d write to driver size error: expect: %lu actual: %d", __FUNCTION__, __LINE__, sizeof(smhi_msg_header_t), ret);
                goto error;
        }

        ret = read(handle, (void *)buf, sizeof(smhi_msg_header_t) + res_size);
        if (ret < 0) {
                ALOGE("%s line: %d read from driver error: %s", __FUNCTION__, __LINE__, strerror(errno));
                goto error;
        } else if (ret != sizeof(smhi_msg_header_t) + res_size) {
                ALOGE("%s line: %d read from driver size error: expect: %lu actual: %d", __FUNCTION__, __LINE__, sizeof(smhi_msg_header_t) + res_size, ret);
                goto error;
        } else if (header->status != SMHI_SUCCESS) {
                ALOGE("%s line: %d smhi response status error! error status: %d", __FUNCTION__, __LINE__, header->status);
                goto error;
        }

        return res_size;

error:
        return 0;
}

uint64_t smhi_client_get_time(int handle)
{
        smhi_get_time_res *time;
        long unsigned int res_size = sizeof(smhi_get_time_res);

        res_size = smhi_client_get(handle, SMHI_GET_TIME, (char **)(&time), res_size);

        if (res_size != sizeof(smhi_get_time_res)) {
                ALOGE("%s line: %d error: %lu", __FUNCTION__, __LINE__, res_size);
                return 0;
        }
        return time->ms_since_ish_reset;
}

ish_version_t smhi_client_get_version(int handle)
{
        ish_version_t version = { 0 };
        smhi_get_version_res *res;
        long unsigned int res_size = sizeof(smhi_get_version_res);

        res_size = smhi_client_get(handle, SMHI_GET_VERSION, (char **)(&res), res_size);

        if (res_size != sizeof(smhi_get_version_res)) {
                ALOGE("%s line: %d error: %lu", __FUNCTION__, __LINE__, res_size);
        } else {
                version = res->version;
        }

        return  version;
}

int smhi_client_set_sensor_conf(int handle, smhi_luid_t luid, smhi_sensor_config_t config, int num_entries, sc_sdt_info_entry *entries, size_t data_size)
{
        size_t total_size = sizeof(smhi_msg_header_t) + sizeof(smhi_set_sensor_conf_req) + sizeof(sc_sdt_sensor_data) + num_entries * (sizeof(sc_sdt_info_entry) + data_size);
        char *buf = (char*)malloc(total_size);
        if (buf == NULL) {
                ALOGE("%s line: %d malloc error!", __FUNCTION__, __LINE__);
                return -1;
        }

        memset(buf, 0, total_size);
        smhi_msg_header_t *header = (smhi_msg_header_t *)buf;
        smhi_set_sensor_conf_req *req = (smhi_set_sensor_conf_req *)(buf + sizeof(smhi_msg_header_t));
        sc_sdt_sensor_data *sensor_data = (sc_sdt_sensor_data *)(buf + sizeof(smhi_msg_header_t) + sizeof(smhi_set_sensor_conf_req));

        header->command = SMHI_SET_SENSOR_CONF;
        req->sensor_luid = luid;
        req->config_type = config;
        req->data_size = sizeof(sc_sdt_sensor_data) + num_entries * (sizeof(sc_sdt_info_entry) + data_size);
        sensor_data->luid = luid;
        sensor_data->data_length = num_entries * (sizeof(sc_sdt_info_entry) + data_size);
        sensor_data->num_entries = num_entries;
        memcpy(sensor_data->sdt_info_entry, entries, num_entries * (sizeof(sc_sdt_info_entry) + data_size));

        int ret = write(handle, (void *)buf, total_size);
        if (ret < 0) {
                ALOGE("%s line: %d write to driver error: %s", __FUNCTION__, __LINE__, strerror(errno));
                goto error_set_conf;
        } else if (ret != total_size) {
                ALOGE("%s line: %d write to driver size error: except: %d actual: %d", __FUNCTION__, __LINE__, total_size, ret);
                goto error_set_conf;
        }

        ret = read(handle, (void *)buf, sizeof(smhi_msg_header_t));
        if (ret < 0) {
                ALOGE("%s line: %d read from driver error: %s", __FUNCTION__, __LINE__, strerror(errno));
                goto error_set_conf;
        } else if (ret != sizeof(smhi_msg_header_t)) {
                ALOGE("%s line: %d read from driver size error: except: %d actual: %d", __FUNCTION__, __LINE__, sizeof(smhi_msg_header_t), ret);
                goto error_set_conf;
        } else if (header->status != SMHI_SUCCESS) {
                ALOGE("%s line: %d smhi response status error! error status: %d", __FUNCTION__, __LINE__, header->status);
                goto error_set_conf;
        }

        free(buf);
        return 0;

error_set_conf:
        free(buf);
        return -1;
}
