#!/usr/bin/env python
#
# This file is part of Python Download Manager
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Author: Kenneth Christiansen <kenneth.christiansen@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#

from downloadmanager.client import DownloadManager
from downloadmanager.constants import DownloadState

import sys
import os
import ecore
import dbus
import urllib
import urlparse
import locale
from optparse import OptionParser

try:
    from e_dbus import DBusEcoreMainLoop
    DBusEcoreMainLoop(set_as_default=True)
except:
    import dbus.ecore

locale.setlocale(locale.LC_ALL, "")

remaining = []

def report_item(item):
    bytes_written, size = item.get_progress()
    if not size:
        percent = "Undef."
    else:
        p = bytes_written * 100.0 / size
        if 0 <= p <= 100:
            percent = "% 3.1f%%" % p
        else:
            percent = "???.?%"

    if bytes_written == size:
        space = " " * len("%9d" % size)
        print "%s, size = %9d             %s  (%s)" % \
              (percent, size, space, item.http_uri)
    else:
        print "%s, bytes written = %9d of %9d (%s)" % \
              (percent, bytes_written, size, item.http_uri)

def report():
    if not remaining:
        print "no downloads"
    else:
        print "---- BEGIN ----"
        for item in remaining:
            report_item(item)
        print "---- END ----"
    return True

def state_changed_cb(item, state, status=-2):
    if state == DownloadState.IN_PROGRESS:
        print "Download has started..: %r -> %r" % \
              (item.http_uri, item.local_path)
        remaining.append(item)
    elif state == DownloadState.COMPLETED:
        print "Download has finished.: %r" % (item.http_uri,)
        _dispose_download(item)
    elif state == DownloadState.CANCELLED:
        print "Download was cancelled: %r" % (item.http_uri,)
        _dispose_download(item)
    elif state == DownloadState.EXCEPTION:
        exc = item.get_exception_info()
        type = int(exc[0])
        code = int(exc[1])
        msg = str(exc[2])
        print "Downloading error.....: %r" % (item.http_uri,)
        print "                      : %d-%d %r" % (type, code, msg)
        _dispose_download(item)

def _dispose_download(download_item):
    try:
        remaining.remove(download_item)
    except ValueError:
        pass
    if not remaining:
        ecore.main_loop_quit()

def start_download(mger, source, destination, resume):
    item = mger.add(source, destination)
    item.on_state_changed_add(state_changed_cb)
    item.start(resume)
    return item

def get_free_outfile(url):
    path = urlparse.urlparse(url)[2]
    filename = os.path.basename(urllib.url2pathname(path))
    if not os.path.exists(filename):
        return filename
    i = 1
    while 1:
        f = "%s.%d" % (filename, i)
        if not os.path.exists(f):
            return f
        i += 1

if __name__ == "__main__":

    usage = "Usage: %prog [options] <url>"
    parser = OptionParser(usage=usage)
    parser.add_option("-c", "--continue", action="store_true",
                      dest="resume", default=False,
                      help="resume getting a partially-downloaded file.")
    parser.add_option("-O", "--output-document", action="store", dest="outfile",
                      metavar="FILE", help="write documents to FILE.")

    options, args = parser.parse_args()

    if not args:
        parser.error("Missing url")

    mger = DownloadManager()
    for url in args:
        if options.outfile:
            destination = options.outfile
        else:
            destination = get_free_outfile(url)

        start_download(mger, url, destination, options.resume)

    ecore.timer_add(1.0, report)
    ecore.main_loop_begin()
