/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment viewport projection example
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#include <pgm/pgm.h>

/* Text regeneration timeout tag */
static guint regenerate_tag = 0;

/* Text regeneration handler */
static gboolean
regenerate_cb (gpointer data)
{
  PgmDrawable *text = PGM_DRAWABLE (data);

  pgm_drawable_regenerate (text);
  regenerate_tag = 0;

  return FALSE;
}

/* Configure event handler */
static void
configure_event_cb (PgmViewport *viewport,
                    PgmEventConfigure *event,
                    gpointer data)
{
  /* Remove current regeneration timeout */
  if (regenerate_tag)
    g_source_remove (regenerate_tag);

  /* Then add the new regeneration timeout */
  regenerate_tag = g_timeout_add (250, regenerate_cb, data);
}

/* Key press event handler */
static void
keypress_event_cb (PgmViewport *viewport,
                   PgmEventKey *event,
                   gpointer data)
{
  static gboolean fullscreen = FALSE;

  if (event->type == PGM_KEY_PRESS)
    {
      switch (event->keyval)
        {
          /* Fullscreen */
        case PGM_f:
          fullscreen = !fullscreen;
          pgm_viewport_set_fullscreen (viewport, fullscreen);
          break;

        case PGM_1:
        case PGM_KP_1:
          pgm_viewport_set_canvas_rotation
            (viewport, PGM_VIEWPORT_ROTATION_NONE);
          break;

        case PGM_2:
        case PGM_KP_2:
          pgm_viewport_set_canvas_rotation
            (viewport, PGM_VIEWPORT_ROTATION_90);
          break;

        case PGM_3:
        case PGM_KP_3:
          pgm_viewport_set_canvas_rotation
            (viewport, PGM_VIEWPORT_ROTATION_180);
          break;

        case PGM_4:
        case PGM_KP_4:
          pgm_viewport_set_canvas_rotation
            (viewport, PGM_VIEWPORT_ROTATION_270);
          break;

        case PGM_5:
        case PGM_KP_5:
          pgm_viewport_set_canvas_reflection
            (viewport, PGM_VIEWPORT_REFLECTION_NONE);
          break;

        case PGM_6:
        case PGM_KP_6:
          pgm_viewport_set_canvas_reflection
            (viewport, PGM_VIEWPORT_REFLECTION_HORIZONTAL_FLIP);
          break;

        case PGM_7:
        case PGM_KP_7:
          pgm_viewport_set_canvas_reflection
            (viewport, PGM_VIEWPORT_REFLECTION_VERTICAL_FLIP);
          break;

          /* Quit */
        case PGM_q:
        case PGM_Escape:
          pgm_main_quit ();
          break;

        default:
          break;
        }
    }
}

/* Entry point */
int
main (int argc,
      char *argv[])
{
  PgmViewport *viewport = NULL;
  PgmDrawable *pattern, *text;
  PgmCanvas *canvas;

  /* Init */
  pgm_init (&argc, &argv);

  /* OpenGL viewport creation */
  pgm_viewport_factory_make ("opengl", &viewport);
  if (!viewport)
    {
      g_print ("Cannot create the 'opengl' viewport\n");
      return -1;
    }
  pgm_viewport_set_title (viewport, "Viewport projection");
  pgm_viewport_set_size (viewport, 640, 480);

  /* Pattern creation */
  pattern = pgm_image_new_from_file ("examples/pictures/pattern.png", 0);
  pgm_image_set_layout (PGM_IMAGE (pattern), PGM_IMAGE_ZOOMED);
  pgm_image_set_alignment (PGM_IMAGE (pattern), PGM_IMAGE_TOP_LEFT);
  pgm_drawable_set_size (pattern, 640.0f, 480.0f);
  pgm_drawable_set_position (pattern, 0.0f, 0.0f, 0.0f);
  pgm_drawable_set_bg_color (pattern, 255, 255, 255, 0);
  pgm_drawable_set_fg_color (pattern, 20, 20, 20, 255);
  pgm_drawable_show (pattern);

  /* Text creation */
  text = pgm_text_new ("<span face='MgOpen Modata'>"
                       "<u><b>Rotation</b></u>\n"
                       "'1': do not rotate\n"
                       "'2': 90° counter-clockwise rotation\n"
                       "'3': 180° counter-clockwise rotation\n"
                       "'4': 270° counter-clockwise rotation\n"
                       "\n"
                       "<u><b>Reflection</b></u>\n"
                       "'5': do not flip\n"
                       "'6': horizontal flip\n"
                       "'7': vertical flip\n"
                       "</span>");
  pgm_text_set_font_height (PGM_TEXT (text), 0.085f);
  pgm_drawable_set_size (text, 600.0f, 400.0f);
  pgm_drawable_set_position (text, 10.0f, 10.0f, 0.0f);
  pgm_drawable_set_bg_color (text, 255, 255, 255, 0);
  pgm_drawable_set_fg_color (text, 225, 225, 255, 185);
  pgm_drawable_show (text);

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_canvas_set_size (canvas, 640.0f, 480.0f);
  pgm_viewport_set_canvas (viewport, canvas);
  pgm_canvas_add_many (canvas, PGM_DRAWABLE_MIDDLE, pattern, text, NULL);

  /* Main loop */
  g_signal_connect (G_OBJECT (viewport), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);
  g_signal_connect (G_OBJECT (viewport), "configure-event",
                    G_CALLBACK (configure_event_cb), text);
  g_signal_connect (G_OBJECT (viewport), "key-press-event",
                    G_CALLBACK (keypress_event_cb), NULL);
  pgm_viewport_show (viewport);
  pgm_main ();

  /* Deinit */
  gst_object_unref (canvas);
  gst_object_unref (viewport);
  pgm_deinit ();

  return 0;
}
