
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk_fb.c 1157 2006-09-17 15:11:58Z mschwerin $
 *
 */
#include "config.h"

#include <errno.h>
#include <fcntl.h>
#include <semaphore.h>
#include <stdio.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <xine.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_LINUX_KD_H
#include <linux/kd.h>
#endif

#include "event.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "odk_plugin.h"
#include "otk.h"

#define K_Delete        ('\033' | ('\133'<<8) | ('\063'<<16) | ('\176'<<24))
#define K_Page_Up       ('\033' | ('\133'<<8) | ('\065'<<16) | ('\176'<<24))
#define K_Page_Down     ('\033' | ('\133'<<8) | ('\066'<<16) | ('\176'<<24))
#define K_Up            ('\033' | ('\133'<<8) | ('\101'<<16))
#define K_Down          ('\033' | ('\133'<<8) | ('\102'<<16))
#define K_Right         ('\033' | ('\133'<<8) | ('\103'<<16))
#define K_Left          ('\033' | ('\133'<<8) | ('\104'<<16))
#define K_Home          ('\033' | ('\117'<<8) | ('\110'<<16))
#define K_End           ('\033' | ('\117'<<8) | ('\106'<<16))
#define K_Return        ('\015')
#define K_Space         ('\040')
#define K_Escape        ('\033')
#define K_Tab           ('\011')
#define K_BackSpace     ('\177')

#define K_0             '0'
#define K_1             '1'
#define K_2             '2'
#define K_3             '3'
#define K_4             '4'
#define K_5             '5'
#define K_6             '6'
#define K_7             '7'
#define K_8             '8'
#define K_9             '9'

typedef struct {
    odk_window_t window_class;
    fb_visual_t vis;

    int width;
    int height;

    int keep_going;

    int tty_fd;
    struct termios ti_cur;
    struct termios ti_save;

    /* The event handler. */
    odk_cb event_handler;
    void *event_handler_data;
} fb_window_t;


static void
frame_output_cb (void *fb_win_p,
                 int video_width, int video_height,
                 double video_pixel_aspect,
                 int *dest_x, int *dest_y,
                 int *dest_width, int *dest_height,
                 double *dest_pixel_aspect, int *win_x, int *win_y)
{
    fb_window_t *fb_win = (fb_window_t *) fb_win_p;

    static int width = 0;
    static int height = 0;

    if (fb_win == NULL)
        return;

    if ((width != video_width) || (height != video_height)) {
        oxine_event_t ev;

        width = video_width;
        height = video_height;

        ev.type = OXINE_EVENT_FRAME_FORMAT_CHANGED;
        ev.data.pos.x = width;
        ev.data.pos.y = height;

        fb_win->event_handler (fb_win->event_handler_data, &ev);
    }
    *win_x = 0;
    *win_y = 0;

    *dest_x = fb_win->window_class.output_xpos;
    *dest_y = fb_win->window_class.output_ypos;

    *dest_width = fb_win->window_class.output_width;
    *dest_height = fb_win->window_class.output_height;
}


static int
show (odk_window_t * this)
{
    return 1;
}


static int
hide (odk_window_t * this)
{
    return 1;
}


static int
fullscreen (odk_window_t * this, int mode)
{
    return 1;
}


static int
displace (odk_window_t * this, int x, int y, int w, int h)
{
    fb_window_t *win = (fb_window_t *) this;
    win->width = w;
    win->height = h;
    return 1;
}


static void
exit_keyboard (fb_window_t * win)
{
    tcsetattr (win->tty_fd, TCSANOW, &win->ti_save);

#ifdef HAVE_LINUX_KD_H
    if (ioctl (win->tty_fd, KDSETMODE, KD_TEXT) == -1)
        warn ("Failed to set /dev/tty to text mode: %s", strerror (errno));
#endif

    close (win->tty_fd);
}


static int
next_event (fb_window_t * win)
{
    unsigned char c[4];
    int n;
    int i;
    int k;

    n = read (win->tty_fd, c, 4);
#ifdef DEBUG_INPUT_EVENTS
    debug ("Got key: 0 %03o", c[0]);
#endif
    for (k = 0, i = 0; i < n; i++) {
        k |= c[i] << (i << 3);
#ifdef DEBUG_INPUT_EVENTS
        if (i > 0)
            debug ("         %d %03o", i, c[i]);
#endif
    }

    return k;
}



static int
init_keyboard (fb_window_t * win)
{
    win->tty_fd = open ("/dev/tty", O_RDWR);
    if (win->tty_fd == -1) {
        error ("Failed to open /dev/tty: %s", strerror (errno));
        return 0;
    }
#ifdef HAVE_LINUX_KD_H
    if (ioctl (win->tty_fd, KDSETMODE, KD_GRAPHICS) == -1)
        warn ("Failed to set /dev/tty to graphics mode: %s",
              strerror (errno));
#endif

    tcgetattr (win->tty_fd, &win->ti_save);

    win->ti_cur = win->ti_save;
    win->ti_cur.c_cc[VMIN] = 1;
    win->ti_cur.c_cc[VTIME] = 0;
    win->ti_cur.c_iflag &= ~(IGNBRK | BRKINT | PARMRK | ISTRIP |
                             INLCR | IGNCR | ICRNL | IXON);
    win->ti_cur.c_lflag &= ~(ECHO | ECHONL | ISIG | ICANON);

    if (tcsetattr (win->tty_fd, TCSAFLUSH, &win->ti_cur) == -1) {
        error ("Failed to change terminal attributes: %s", strerror (errno));
        close (win->tty_fd);
        return 0;
    }

    return 1;
}


static void
handle_event (fb_window_t * win, int key)
{
    if (!win->event_handler)
        return;

    oxine_event_t ev;
    ev.type = OXINE_EVENT_KEY;
    ev.source.key = OXINE_KEY_NULL;

    switch (key) {
    case K_0:
        ev.source.key = OXINE_KEY_0;
        break;
    case K_1:
        ev.source.key = OXINE_KEY_1;
        break;
    case K_2:
        ev.source.key = OXINE_KEY_2;
        break;
    case K_3:
        ev.source.key = OXINE_KEY_3;
        break;
    case K_4:
        ev.source.key = OXINE_KEY_4;
        break;
    case K_5:
        ev.source.key = OXINE_KEY_5;
        break;
    case K_6:
        ev.source.key = OXINE_KEY_6;
        break;
    case K_7:
        ev.source.key = OXINE_KEY_7;
        break;
    case K_8:
        ev.source.key = OXINE_KEY_8;
        break;
    case K_9:
        ev.source.key = OXINE_KEY_9;
        break;
    case 'x':
        ev.source.key = OXINE_KEY_AUDIO_CHANNEL;
        ev.data.how = +1;
        break;
    case 'X':
        ev.source.key = OXINE_KEY_AUDIO_CHANNEL;
        ev.data.how = -1;
        break;
    case 'c':
        ev.source.key = OXINE_KEY_AUDIO_OFFSET;
        ev.data.how = +9;
        break;
    case 'C':
        ev.source.key = OXINE_KEY_AUDIO_OFFSET;
        ev.data.how = -9;
        break;
    case 'r':
        ev.source.key = OXINE_KEY_SPU_CHANNEL;
        ev.data.how = +1;
        break;
    case 'R':
        ev.source.key = OXINE_KEY_SPU_CHANNEL;
        ev.data.how = -1;
        break;
    case 't':
        ev.source.key = OXINE_KEY_SPU_OFFSET;
        ev.data.how = +9;
        break;
    case 'T':
        ev.source.key = OXINE_KEY_SPU_OFFSET;
        ev.data.how = -9;
        break;
    case 'v':
    case '+':
        ev.source.key = OXINE_KEY_VOLUME;
        ev.data.how = +5;
        break;
    case 'V':
    case '-':
        ev.source.key = OXINE_KEY_VOLUME;
        ev.data.how = -5;
        break;
    case 'H':
    case 'h':
    case '?':
        ev.source.key = OXINE_KEY_HELP;
        break;
    case 'S':
    case 's':
        ev.source.key = OXINE_KEY_STOP;
        break;
    case 'Q':
        ev.source.key = OXINE_KEY_QUIT;
        break;
    case 'q':
        ev.source.key = OXINE_KEY_SHUTDOWN;
        break;
    case 'O':
    case 'o':
        ev.source.key = OXINE_KEY_MENU_OSD;
        break;
    case 'E':
    case 'e':
        ev.source.key = OXINE_KEY_EJECT;
        break;
    case 'A':
    case 'a':
        ev.source.key = OXINE_KEY_ASPECT_RATIO;
        break;
    case 'I':
    case 'i':
        ev.source.key = OXINE_KEY_DEINTERLACE;
        break;
    case 'F':
    case 'f':
        ev.source.key = OXINE_KEY_FULLSCREEN;
        break;
    case 'm':
        ev.source.key = OXINE_KEY_VOLMUTE;
        break;
    case 'M':
        ev.source.key = OXINE_KEY_PLAYMODE;
        break;
    case K_Space:
        ev.source.key = OXINE_KEY_PPLAY;
        break;
    case K_Page_Up:
    case 'p':
        ev.source.key = OXINE_KEY_PREV;
        break;
    case K_Page_Down:
    case 'n':
        ev.source.key = OXINE_KEY_NEXT;
        break;
    case K_Escape:
        ev.source.key = OXINE_KEY_MENU_MAIN;
        break;
    case K_Delete:
        ev.source.key = OXINE_KEY_REMOVE;
        break;
    case K_Up:
        ev.source.key = OXINE_KEY_SPEED;
        ev.data.how = +1;
        break;
    case K_Down:
        ev.source.key = OXINE_KEY_SPEED;
        ev.data.how = -1;
        break;
    case K_Right:
        ev.source.key = OXINE_KEY_SKIP;
        ev.data.how = +30;
        break;
    case K_Left:
        ev.source.key = OXINE_KEY_SKIP;
        ev.data.how = -30;
        break;
    case K_Home:
        ev.source.key = OXINE_KEY_FIRST;
        break;
    case K_End:
        ev.source.key = OXINE_KEY_LAST;
        break;
    case K_BackSpace:
        ev.source.key = OXINE_KEY_BACK;
        break;
        /*
           case XK_KP_8:
           case XK_KP_Up:
           ev.source.key = OXINE_KEY_UP;
           break;
           case XK_KP_2:
           case XK_KP_Down:
           ev.source.key = OXINE_KEY_DOWN;
           break;
           case XK_KP_4:
           case XK_KP_Left:
           ev.source.key = OXINE_KEY_LEFT;
           break;
           case XK_KP_6:
           case XK_KP_Right:
           ev.source.key = OXINE_KEY_RIGHT;
           break;
           case XK_KP_0:
           case XK_KP_Insert:
           ev.source.key = OXINE_KEY_SELECT;
           break;
         */
    case K_Return:
        ev.source.key = OXINE_KEY_ACTIVATE;
        break;
    case K_Tab:
        ev.source.key = OXINE_KEY_NEXT_WIDGET;
        break;
    default:
        break;
    }

    if (ev.source.key != OXINE_KEY_NULL)
        win->event_handler (win->event_handler_data, &ev);
}


static void
event_loop (odk_window_t * this)
{
    fb_window_t *win = (fb_window_t *) this;

    win->keep_going = 1;
    while (win->keep_going) {
        int key = next_event (win);
        handle_event (win, key);
    }
    win->keep_going = 0;
}


static void
set_event_handler (odk_window_t * this,
                   void (*cb) (void *data, oxine_event_t * ev), void *data)
{
    fb_window_t *win = (fb_window_t *) this;
    win->event_handler = cb;
    win->event_handler_data = data;
}


static void
stop_event_loop (odk_window_t * this)
{
    fb_window_t *win = (fb_window_t *) this;
    win->keep_going = 0;
}


static int
dispose (odk_window_t * this)
{
    fb_window_t *win = (fb_window_t *) this;

    stop_event_loop (this);
    xine_close_video_driver (this->xine, this->video_port);
    exit_keyboard (win);
    this->video_port = NULL;

    return 1;
}


odk_window_t *fb_construct (xine_t * xine, const char *driver);

/*
 * Constructor
 */
odk_window_t *
fb_construct (xine_t * xine, const char *driver)
{
    fb_window_t *win = ho_new (fb_window_t);

    win->vis.frame_output_cb = frame_output_cb;
    win->vis.user_data = win;

    win->window_class.xine = xine;
    win->window_class.video_port =
        xine_open_video_driver (xine, driver, XINE_VISUAL_TYPE_FB,
                                (void *) &win->vis);
    if (!win->window_class.video_port) {
        warn ("Failed to open video port '%s', falling back to auto.",
              driver);
        win->window_class.video_port =
            xine_open_video_driver (xine, NULL, XINE_VISUAL_TYPE_FB,
                                    (void *) &(win->vis));
    }

    if (!win->window_class.video_port) {
        error ("Failed to open video port!");
        ho_free (win);
        return NULL;
    }

    if (!init_keyboard (win)) {
        error (_("Failed to initialize keyboard!"));
        xine_close_video_driver (win->window_class.xine,
                                 win->window_class.video_port);
        win->window_class.video_port = NULL;
        ho_free (win);
        return NULL;
    }

    win->window_class.show = show;
    win->window_class.hide = hide;
    win->window_class.displace = displace;
    win->window_class.dispose = dispose;
    win->window_class.fullscreen = fullscreen;
    win->window_class.event_loop = event_loop;
    win->window_class.stop_event_loop = stop_event_loop;
    win->window_class.set_event_handler = set_event_handler;

    return (odk_window_t *) win;
}
