/* Copyright 2011 Canonical, Ltd. This software is licensed under the GNU
 * Lesser General Public License version 3 or later (see the file COPYING).
 */

#ifndef OIF_QML_GESTURE_H_
#define OIF_QML_GESTURE_H_

#include <QDeclarativeItem>

#include <geis/geis.h>

class GestureArea;

/**
 * A property type used for configuring gesture begin and end conditions based
 * on the number of active touches.
 */

class TouchLimits : public QObject {
  Q_OBJECT
  /**
   * This property holds the number of touches required to begin a gesture.
   *
   * The default value is two touches.
   */
  Q_PROPERTY(int start READ start WRITE set_start NOTIFY startChanged)

 public:
  explicit TouchLimits(QObject* parent = 0);

  int start() const { return start_; }
  void set_start(int new_value);

 signals:
  void startChanged();

 private:
  int start_;

  Q_DISABLE_COPY(TouchLimits)
};

/**
 * An item describing a gesture subscription.
 */

class Gesture : public QDeclarativeItem {
  Q_OBJECT
  Q_ENUMS(DeviceType)

  /**
   * This property holds the gesture begin and end conditions based on the
   * number of active touches.
   */
  Q_PROPERTY(TouchLimits* touches READ touches)

  /**
   * This property holds the device type.
   *
   * Only devices matching the device type will generate events. The default
   * value is All.
   */
  Q_PROPERTY(
    DeviceType devices READ devices WRITE set_devices NOTIFY
    devicesChanged)

  /**
   * \internal
   *
   * This property holds whether the gesture is a global system gesture.
   *
   * This property is only intended to be used by Unity. The existence and
   * implementation of this property may change without notice. A fully
   * supported public alternative may be provided in the future.
   */
  Q_PROPERTY(
    bool _UNITY_globalGesture READ _UNITY_globalGesture WRITE
    set_UNITY_globalGesture NOTIFY _UNITY_globalGestureChanged)

 public:
  explicit Gesture(QDeclarativeItem* parent = 0);
  ~Gesture();

  /**
   * An enumeration of classes of multitouch input devices for subscriptions.
   *
   * See Device::DeviceType for descriptions of the device classes.
   */
  enum DeviceType {
    /** All Device::TouchScreen devices */
    TouchScreens,
    /** All Device::TouchPad devices */
    TouchPads,
    /** All Device::Independent devices */
    Independents,
    /** All devices */
    All
  };

  TouchLimits* touches() { return &touches_; }
  DeviceType devices() const { return devices_; }
  void set_devices(DeviceType new_value);
  bool _UNITY_globalGesture() const { return _UNITY_globalGesture_; }
  void set_UNITY_globalGesture(bool new_value);

  GeisSubscription subscription() const { return subscription_; }
  void set_subscription(GeisSubscription subscription);
  bool EnableSubscription();
  bool DisableSubscription();
  void DeleteSubscription();

 signals:
  void typeChanged();
  void devicesChanged();
  void _UNITY_globalGestureChanged();

 private:
  TouchLimits touches_;
  DeviceType devices_;
  bool _UNITY_globalGesture_;
  GeisSubscription subscription_;

  Q_DISABLE_COPY(Gesture)
};

QML_DECLARE_TYPE(Gesture)
QML_DECLARE_TYPE(TouchLimits)

#endif  // OIF_QML_GESTURE_H_
