#include <linux/interrupt.h>
#include "iwl-em-intr.h"
#include "iwl-trans.h"
#include "pcie/internal.h"
#include "trans_slave/idi_internal.h"
#include "iwl-op-mode.h"
#include "iwl-drv.h"
#include "iwl-config.h"
#include "iwl-prph.h"
#include "iwl-debug.h"
#include "iwl-amfh.h"

#ifdef IWL_IDI_LLS_INTA_TEST
static struct timer_list data_timer;
#endif
/*
 * IWL Emulation Logger
 */
#define INTA_LOG_LEVEL_WARN		KERN_ERR
#define INTA_LOG_PREFIX			"[INTA]"

#ifdef __IWL_INTA_LOG_ENABLED__
#define INTA_TRACE_ENTER \
	IWL_EM_TRACE_ENTER(__INTA_LOG_LEVEL_TRACE__, INTA_LOG_PREFIX)
#define INTA_TRACE_EXIT \
	IWL_EM_TRACE_EXIT(__INTA_LOG_LEVEL_TRACE__, INTA_LOG_PREFIX)
#define INTA_TRACE_EXIT_RET_STR(_ret) \
	IWL_EM_LOG(__INTA_LOG_LEVEL_TRACE__, INTA_LOG_PREFIX, \
							"<<< "_ret)
#define INTA_TRACE_EXIT_RET(_ret) \
	IWL_EM_TRACE_EXIT_RET(__INTA_LOG_LEVEL_TRACE__, \
				INTA_LOG_PREFIX, _ret)
#define IWL_INTA_LOG(fmt, args ...) \
	IWL_EM_LOG(__INTA_LOG_LEVEL_DEBUG__, \
		INTA_LOG_PREFIX, fmt, ## args)
#define IWL_INTA_LOG_HEX_DUMP(msg, p, len) \
	IWL_EM_LOG_HEX_DUMP(msg, p, len)
#else

#define INTA_TRACE_ENTER
#define INTA_TRACE_EXIT
#define INTA_TRACE_EXIT_RET_STR(_ret)
#define INTA_TRACE_EXIT_RET(_ret)
#define IWL_INTA_LOG(fmt, args ...)
#define IWL_INTA_LOG_HEX_DUMP(msg, p, len)

#endif

/* Always print error messages */
#define IWL_INTA_LOG_ERR(fmt, args ...) \
	IWL_EM_LOG_ERR(fmt, ## args)

/**
 * holds the data of the irq's
 */
struct iwl_inta_em_t {

	/* iwl_trans_idi who uses this intr */
	struct iwl_trans *idi_trans;

	/* handle idi irq - simulates irq context */
	irqreturn_t (*idi_irq_handler)(unsigned long);

	/* handle idi irq - simulates thread context */
	irqreturn_t (*idi_irq_thread)(unsigned long);

	/* simulates the hw register holding the pending irqs */
	atomic_t idi_irq_mask;

};

static struct iwl_inta_em_t iwl_idi_inta_em;
static struct iwl_inta_em_t *iwl_inta_em_data = &iwl_idi_inta_em;

/*****************************************************
 * INIT AND STOP
 ****************************************************/

/* should be called after start_hw and before actual interupts can be sent */
void iwl_inta_em_init(struct iwl_trans *trans)
{
	INTA_TRACE_ENTER;
	iwl_inta_em_data->idi_trans = trans;
	atomic_set(&iwl_inta_em_data->idi_irq_mask, 0);
	IWL_INTA_LOG("inta handler initialized");
	INTA_TRACE_EXIT;
}

void iwl_inta_em_start(struct iwl_trans *trans,
		       irqreturn_t (*idi_irq_handler)(unsigned long),
		       irqreturn_t (*idi_irq_thread)(unsigned long))
{
	INTA_TRACE_ENTER;
	iwl_inta_em_data->idi_irq_handler = idi_irq_handler;
	iwl_inta_em_data->idi_irq_thread = idi_irq_thread;

#ifdef IWL_IDI_LLS_INTA_TEST
	iwl_idi_test_lls();
#endif

	IWL_INTA_LOG("inta handler started");
	INTA_TRACE_EXIT;
}

/* free relevant memory */
int iwl_inta_em_stop(void)
{
	INTA_TRACE_ENTER;

#ifdef IWL_IDI_LLS_INTA_TEST
	del_timer_sync(&data_timer);
#endif
	IWL_INTA_LOG("inta handler stopped");
	INTA_TRACE_EXIT;
	return 0;
}

void iwl_inta_em_free(void)
{
	INTA_TRACE_ENTER;
	IWL_INTA_LOG("inta handler freed");
	INTA_TRACE_EXIT;
}

/************************************************
 * SET CALLED IRQ FROM OUTSIDE
 ************************************************/

static void atomic_bitwise_and(int i, atomic_t *v)
{
	int old;
	int new;

	do {
		old = atomic_read(v);
		new = old & i;
	} while (atomic_cmpxchg(v, old, new) != old);
}

/* updates the called_irq's according to "Called" - replace with called mask */
void iwl_inta_em_set_irq_called(u32 called, struct iwl_trans *trans)
{
	INTA_TRACE_ENTER;
	atomic_bitwise_and(called, &iwl_inta_em_data->idi_irq_mask);
	INTA_TRACE_EXIT;
}

/* updates the called_irq's according to "Called" - adding called mask */
void iwl_inta_em_add_irq_called(u32 called)
{
	INTA_TRACE_ENTER;
	atomic_or(called, &iwl_inta_em_data->idi_irq_mask);
	INTA_TRACE_EXIT;
}

/* updates the called_irq's according to "Called" - adding called mask */
void iwl_inta_em_clean_irq_called(struct iwl_trans *trans)
{
	INTA_TRACE_ENTER;
	atomic_set(&iwl_inta_em_data->idi_irq_mask, 0);
	INTA_TRACE_EXIT;
}

/************************************************
 * REACH IRQ DATA FROM OUTSIDE
 ************************************************/

/* returns a u32 specifing which idi irq are pending */
u32 iwl_inta_em_read_called(struct iwl_trans *trans)
{
	u32 called;
	INTA_TRACE_ENTER;
	called = atomic_read(&iwl_inta_em_data->idi_irq_mask);
	INTA_TRACE_EXIT_RET(called);
	return called;
}

/* returns the iwl_trans_idi pointed by the data */
struct iwl_trans *iwl_inta_em_get_trans(void)
{
	return iwl_inta_em_data->idi_trans;
}

/************************************************
 * HELPERS
 ************************************************/

/* handles irq's */
irqreturn_t iwl_pcie_irq_handler_idi(int irq, void *dev_id)
{
	struct iwl_trans *trans = dev_id;
	struct iwl_trans *em_idi_trans = iwl_inta_em_get_trans();
	u32 idi_irq;
	irqreturn_t (*idi_irq_handler)(unsigned long) =
		iwl_inta_em_data->idi_irq_handler;
	irqreturn_t (*idi_irq_thread)(unsigned long) =
		iwl_inta_em_data->idi_irq_thread;
	irqreturn_t irq_ret = IRQ_HANDLED;
	INTA_TRACE_ENTER;

	/* handle the pcie dependent irq and detect pending idi irq */
	idi_irq = iwl_inta_em_handle_pcie(trans);

	/* update the idi mask */
	iwl_inta_em_add_irq_called(idi_irq);

	/* if there are pending idi-irq, handle them */
	while (iwl_inta_em_read_called(iwl_inta_em_data->idi_trans)) {
		irq_ret = idi_irq_handler((unsigned long)em_idi_trans);
		if (irq_ret == IRQ_WAKE_THREAD)
			irq_ret = idi_irq_thread((unsigned long)em_idi_trans);
	}
	INTA_TRACE_EXIT_RET(irq_ret);
	return irq_ret;
}

/**
 * tasklet for iwlagn interrupt - detects and handles pci-dependent irq.
 *detect and return mask for idi dependent irq
 */
u32 iwl_inta_em_handle_pcie(struct iwl_trans *trans)
{
	struct iwl_trans_pcie *trans_pcie =
		IWL_TRANS_GET_PCIE_TRANS(trans);
	struct isr_statistics *isr_stats = &trans_pcie->isr_stats;
	u32 inta = 0;
	u32 handled = 0;
	unsigned long flags;
	u32 idi_irq = 0;
	u32 i;
#ifdef CPTCFG_IWLWIFI_DEBUG
	u32 inta_mask;
#endif

	INTA_TRACE_ENTER;

	spin_lock_irqsave(&trans_pcie->irq_lock, flags);

	/* Ack/clear/reset pending uCode interrupts.
	 * Note:  Some bits in CSR_INT are "OR" of bits in CSR_FH_INT_STATUS,
	 */
	/* There is a hardware bug in the interrupt mask function that some
	 * interrupts (i.e. CSR_INT_BIT_SCD) can still be generated even if
	 * they are disabled in the CSR_INT_MASK register. Furthermore the
	 * ICT interrupt handling mechanism has another bug that might cause
	 * these unmasked interrupts fail to be detected. We workaround the
	 * hardware bugs here by ACKing all the possible interrupts so that
	 * interrupt coalescing can still be achieved.
	 */
	iwl_write32(trans, CSR_INT,
		    trans_pcie->inta | ~trans_pcie->inta_mask);

	inta = trans_pcie->inta;

#ifdef CPTCFG_IWLWIFI_DEBUG
	if (iwl_have_debug_level(IWL_DL_ISR)) {
		/* just for debug */
		inta_mask = iwl_read32(trans, CSR_INT_MASK);
		IWL_DEBUG_ISR(trans, "inta 0x%08x, enabled 0x%08x\n",
			      inta, inta_mask);
	}
#endif

	/* saved interrupt in inta variable now we can reset trans_pcie->inta */
	trans_pcie->inta = 0;

	spin_unlock_irqrestore(&trans_pcie->irq_lock, flags);
	/* Now service all interrupt bits discovered above. */
	if (inta & CSR_INT_BIT_HW_ERR) {

		handled |= CSR_INT_BIT_HW_ERR;
		idi_irq |= IWL_IDI_HW_ERR_INT;
		return idi_irq;
	}

#ifdef CPTCFG_IWLWIFI_DEBUG
	if (iwl_have_debug_level(IWL_DL_ISR)) {
		/* NIC fires this, but we don't use it, redundant with WAKEUP */
		if (inta & CSR_INT_BIT_SCD) {
			IWL_DEBUG_ISR(trans,
				      "Scheduler finished to transmit the frame/frames.\n");
			isr_stats->sch++;
		}

		/* Alive notification via Rx interrupt will do the real work */
		if (inta & CSR_INT_BIT_ALIVE) {
			IWL_DEBUG_ISR(trans, "Alive interrupt\n");
			isr_stats->alive++;
		}
	}
#endif
	/* Safely ignore these bits for debug checks below */
	inta &= ~(CSR_INT_BIT_SCD | CSR_INT_BIT_ALIVE);

	/* HW RF KILL switch toggled */
	if (inta & CSR_INT_BIT_RF_KILL) {
		bool hw_rfkill;

		hw_rfkill = iwl_is_rfkill_set(trans);
		IWL_WARN(trans, "RF_KILL bit toggled to %s.\n",
			 hw_rfkill ? "disable radio" : "enable radio");

		isr_stats->rfkill++;

		iwl_op_mode_hw_rf_kill(trans->op_mode, hw_rfkill);

		handled |= CSR_INT_BIT_RF_KILL;
	}

	/* Chip got too hot and stopped itself */
	if (inta & CSR_INT_BIT_CT_KILL) {
		IWL_ERR(trans, "Microcode CT kill error detected.\n");
		isr_stats->ctkill++;
		handled |= CSR_INT_BIT_CT_KILL;
	}

	/* Error detected by uCode */
	if (inta & CSR_INT_BIT_SW_ERR) {
		handled |= CSR_INT_BIT_SW_ERR;
		idi_irq |= IWL_IDI_SW_ERR_INT;
	}

	/* uCode wakes up after power-down sleep */
	if (inta & CSR_INT_BIT_WAKEUP) {
		IWL_DEBUG_ISR(trans, "Wakeup interrupt\n");
		iwl_pcie_rxq_inc_wr_ptr(trans, &trans_pcie->rxq);
		for (i = 0; i < trans->cfg->base_params->num_of_queues; i++)
			iwl_pcie_txq_inc_wr_ptr(trans, &trans_pcie->txq[i]);
		isr_stats->wakeup++;
		handled |= CSR_INT_BIT_WAKEUP;
	}

	/* All uCode command responses, including Tx command responses,
	 * Rx "responses" (frame-received notification), and other
	 * notifications from uCode come through here*/
	if (inta & (CSR_INT_BIT_FH_RX | CSR_INT_BIT_SW_RX |
		    CSR_INT_BIT_RX_PERIODIC)) {
		IWL_DEBUG_ISR(trans, "Rx interrupt\n");
		if (inta & (CSR_INT_BIT_FH_RX | CSR_INT_BIT_SW_RX)) {
			handled |= (CSR_INT_BIT_FH_RX | CSR_INT_BIT_SW_RX);
			iwl_write32(trans, CSR_FH_INT_STATUS,
				    CSR_FH_INT_RX_MASK);
		}
		if (inta & CSR_INT_BIT_RX_PERIODIC) {
			handled |= CSR_INT_BIT_RX_PERIODIC;
			iwl_write32(trans,
				    CSR_INT, CSR_INT_BIT_RX_PERIODIC);
		}
		/* Sending RX interrupt require many steps to be done in the
		 * the device:
		 * 1- write interrupt to current index in ICT table.
		 * 2- dma RX frame.
		 * 3- update RX shared data to indicate last write index.
		 * 4- send interrupt.
		 * This could lead to RX race, driver could receive RX interrupt
		 * but the shared data changes does not reflect this;
		 * periodic interrupt will detect any dangling Rx activity.
		 */

		/* Disable periodic interrupt; we use it as just a one-shot. */
		iwl_write8(trans, CSR_INT_PERIODIC_REG,
			   CSR_INT_PERIODIC_DIS);

		iwl_amfh_rx_handler();

		/* Enable periodic interrupt in 8 msec only if we received
		 * real RX interrupt (instead of just periodic int), to catch
		 * any dangling Rx interrupt.  If it was just the periodic
		 * interrupt, there was no dangling Rx activity, and no need
		 * to extend the periodic interrupt; one-shot is enough.
		 */
		if (inta & (CSR_INT_BIT_FH_RX | CSR_INT_BIT_SW_RX)) {
			iwl_write8(trans, CSR_INT_PERIODIC_REG,
				   CSR_INT_PERIODIC_ENA);
				isr_stats->rx++;
			}
	}

	/* This "Tx" DMA channel is used only for loading uCode */
	if (inta & CSR_INT_BIT_FH_TX) {
		iwl_write32(trans, CSR_FH_INT_STATUS, CSR_FH_INT_TX_MASK);
		IWL_DEBUG_ISR(trans, "uCode load interrupt\n");
		isr_stats->tx++;
		handled |= CSR_INT_BIT_FH_TX;
		idi_irq |= IWL_IDI_FH_TX_INT;
	}

	if (inta & ~handled) {
		IWL_ERR(trans, "Unhandled INTA bits 0x%08x\n", inta & ~handled);
		isr_stats->unhandled++;
	}

	if (inta & ~(trans_pcie->inta_mask))
		IWL_WARN(trans, "Disabled INTA bits 0x%08x were pending\n",
			 inta & ~trans_pcie->inta_mask);

	/* Re-enable all interrupts */
	/* only Re-enable if disabled by irq */
	if (test_bit(STATUS_INT_ENABLED, &trans_pcie->status))
		iwl_enable_interrupts(trans);
	/* Re-enable RF_KILL if it occurred */
	else if (handled & CSR_INT_BIT_RF_KILL)
		iwl_enable_rfkill_int(trans);

	INTA_TRACE_EXIT_RET(idi_irq);
	return idi_irq;
}

#ifdef IWL_IDI_LLS_INTA_TEST
/************************************************
 * TESTING
 ************************************************/

/**
 *the function called by the timer - set lls irq as
 * pending and calls self again
 */
static void iwl_idi_test_lls_func(unsigned long data)
{
	int mili = 5000;
	iwl_inta_em_add_irq_called(IWL_IDI_LLS_INT);
	data_timer.expires += msecs_to_jiffies(mili);
	add_timer(&data_timer);
}

/* set timer to periodicly append lls irq */
void iwl_idi_test_lls(void)
{
	INTA_TRACE_ENTER;
	data_timer.expires = jiffies + msecs_to_jiffies(2000);
	data_timer.data = 0;
	data_timer.function = &iwl_idi_test_lls_func;
	init_timer(&data_timer);
	add_timer(&data_timer);
	INTA_TRACE_EXIT;
}
#endif
