/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#include "iwl-devtrace.h"
#include "shared.h"
#include "iwl-op-mode.h"
/* FIXME: need to abstract out TX command (once we know what it looks like) */
#include "dvm/commands.h"

/* FIXME: change values to be unique for each bus? */
#define IWL_SLV_TX_Q_HIGH_THLD 320
#define IWL_SLV_TX_Q_LOW_THLD 256

#define IWL_SLV_TXQ_GET_ENTRY(txq_entry, outer_ptr)\
	((outer_ptr) = container_of((txq_entry),\
				    typeof(*(outer_ptr)), txq_entry))

/* AL memory pool manager, must be called from a locked context. */

/**
 * iwl_slv_al_mem_pool_init - initialize pool manager
 * @pm - corresponding pool manager
 * @num_elems - number of the elements
 * Returns 0 upon success, negative error otherwise
 */
int iwl_slv_al_mem_pool_init(struct iwl_slv_al_mem_pool *pm, u8 num_elems)
{
	if (WARN_ON(pm == NULL ||
		    num_elems > sizeof(pm->used) * BITS_PER_BYTE))
		return -EINVAL;

	pm->pool_size = num_elems;
	pm->free_count = num_elems;
	pm->next_idx = 0;
	bitmap_zero(pm->used, sizeof(pm->used) * BITS_PER_BYTE);

	/* check that free_count and pool_size are large enough */
	BUILD_BUG_ON(BIT(sizeof(pm->pool_size) * BITS_PER_BYTE) <
		(sizeof(pm->used) * BITS_PER_BYTE));
	BUILD_BUG_ON(sizeof(pm->pool_size) != sizeof(pm->free_count));
	BUILD_BUG_ON(sizeof(pm->pool_size) != sizeof(pm->next_idx));

	return 0;
}

/**
 * iwl_slv_pool_mgr_alloc - allocate item from the pool and return its index.
 * @trans_slv - the transport
 * @pm - corresponding pool manager
 * Returns index upon success, negative error otherwise
 */
int iwl_slv_al_mem_pool_alloc(struct iwl_trans_slv_tx *slv_tx,
			      struct iwl_slv_al_mem_pool *pm)
{
	u8 i;

	lockdep_assert_held(&slv_tx->mem_rsrc_lock);

	i = find_first_zero_bit(pm->used, pm->pool_size);

	if (WARN_ON(i >= pm->pool_size))
		return -EINVAL;

	pm->free_count--;
	set_bit(i, pm->used);

	return i;
}

int iwl_slv_al_mem_pool_alloc_cb(struct iwl_trans_slv_tx *slv_tx,
				 struct iwl_slv_al_mem_pool *pm)
{
	u8 i;

	lockdep_assert_held(&slv_tx->mem_rsrc_lock);

	i = find_next_zero_bit(pm->used, pm->pool_size, pm->next_idx);

	if (i >= pm->pool_size)
		return -EINVAL;

	pm->next_idx = (i + 1) % pm->pool_size;
	pm->free_count--;
	set_bit(i, pm->used);

	return i;
}

/**
 * iwl_slv_al_pool_mgr_free - free item according to index
 * @trans_slv - the transport
 * @pm - corresponding pool manager
 * @idx - the index of the item to free
 * Returns 0 upon success, negative error otherwise
 */
int iwl_slv_al_mem_pool_free(struct iwl_trans_slv_tx *slv_tx,
			     struct iwl_slv_al_mem_pool *pm, u8 idx)
{
	lockdep_assert_held(&slv_tx->mem_rsrc_lock);

	/* Check that the index is legal and was allocated */
	if (WARN_ON(idx >= pm->pool_size || !test_bit(idx, pm->used)))
		return -EINVAL;

	__clear_bit(idx, pm->used);
	pm->free_count++;

	return 0;
}

struct iwl_slv_txq_entry *iwl_slv_txq_peek_next(struct iwl_trans_slv *trans_slv,
						u8 txq_id)
{
	struct iwl_slv_tx_queue *txq = &trans_slv->txqs[txq_id];
	struct iwl_slv_txq_entry *txq_entry = NULL;
	struct list_head *element;

	spin_lock_bh(&trans_slv->txq_lock);

	if (list_empty(&txq->waiting))
		goto exit;

	element = txq->waiting.next;
	txq_entry = list_entry(element, struct iwl_slv_txq_entry, list);

exit:
	spin_unlock_bh(&trans_slv->txq_lock);
	return txq_entry;
}

void iwl_slv_txq_move_to_sent(struct iwl_trans_slv *trans_slv, u8 txq_id,
			      struct iwl_slv_txq_entry *txq_entry)
{
	struct iwl_slv_tx_queue *txq = &trans_slv->txqs[txq_id];
	struct list_head *element;

	spin_lock_bh(&trans_slv->txq_lock);

	element = txq->waiting.next;
	list_del(element);
	list_add_tail(&txq_entry->list, &txq->sent);
	txq->waiting_count--;

	spin_unlock_bh(&trans_slv->txq_lock);
}

/**
 * iwl_slv_tx_stop - stop tx, don't free resources yet.
 */
void iwl_slv_tx_stop(struct iwl_trans *trans)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	if (WARN(!trans_slv->policy_wq,
		 "trans_slv->policy_wq is NULL. stop before init is done?")) {
		return;
	}
	cancel_work_sync(&trans_slv->policy_trigger);
	flush_workqueue(trans_slv->policy_wq);
}

static inline void iwl_slv_tx_destroy_caches(struct iwl_trans_slv *trans_slv)
{
	if (trans_slv->cmd_entry_pool) {
		kmem_cache_destroy(trans_slv->cmd_entry_pool);
		trans_slv->cmd_entry_pool = NULL;
	}

	if (trans_slv->data_entry_pool) {
		kmem_cache_destroy(trans_slv->data_entry_pool);
		trans_slv->data_entry_pool = NULL;
	}
}

static int iwl_slv_tx_alloc_caches(struct iwl_trans_slv *trans_slv)
{
	int hcmd_entry_size;

	/* cmd_entry has a bus specific field which contains the
	 * dev_cmd and also headroom, if needed by bus */
	hcmd_entry_size = sizeof(struct iwl_slv_tx_cmd_entry) +
			  sizeof(struct iwl_device_cmd) +
			  trans_slv->config.hcmd_headroom;

	trans_slv->cmd_entry_pool =
		kmem_cache_create("iwl_slv_cmd_entry",
				  hcmd_entry_size, sizeof(void *), 0, NULL);

	if (unlikely(!trans_slv->cmd_entry_pool))
		goto error;

	trans_slv->data_entry_pool =
		kmem_cache_create("iwl_slv_data_entry",
				  sizeof(struct iwl_slv_tx_data_entry),
				  sizeof(void *), 0, NULL);

	if (unlikely(!trans_slv->data_entry_pool))
		goto error;

	return 0;

error:
	iwl_slv_tx_destroy_caches(trans_slv);
	return -ENOMEM;
}

static inline int iwl_slv_tx_alloc_queues(struct iwl_trans_slv *trans_slv)
{
	int size, i, ret;

	size = (trans_slv->config.max_queues_num) *
		sizeof(struct iwl_slv_tx_queue);
	trans_slv->txqs = kzalloc(size, GFP_KERNEL);
	if (!trans_slv->txqs)
		return -ENOMEM;

	ret = iwl_slv_tx_alloc_caches(trans_slv);
	if (ret)
		goto error_free;

	for (i = 0; i < trans_slv->config.max_queues_num; i++) {
		INIT_LIST_HEAD(&trans_slv->txqs[i].waiting);
		INIT_LIST_HEAD(&trans_slv->txqs[i].sent);
	}

	return 0;

error_free:
	kfree(trans_slv->txqs);
	trans_slv->txqs = NULL;

	return ret;
}

void iwl_slv_free_data_queue(struct iwl_trans *trans, int txq_id)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_slv_tx_queue *txq = &trans_slv->txqs[txq_id];
	struct iwl_slv_tx_data_entry *data_entry;
	struct iwl_slv_txq_entry *txq_entry, *tmp;

	/* waiting queue - no need to handle DTU memory */
	list_for_each_entry_safe(txq_entry, tmp, &txq->waiting, list) {
		list_del(&txq_entry->list);
		IWL_SLV_TXQ_GET_ENTRY(txq_entry, data_entry);
		iwl_op_mode_free_skb(trans->op_mode, data_entry->skb);
		kmem_cache_free(trans_slv->data_entry_pool, data_entry);
	}

	/* sent queue - need to free DTU memory */
	list_for_each_entry_safe(txq_entry, tmp, &txq->sent, list) {
		list_del(&txq_entry->list);
		IWL_SLV_TXQ_GET_ENTRY(txq_entry, data_entry);
		trans_slv->config.free_dtu_mem(trans, &txq_entry->reclaim_info);
		iwl_op_mode_free_skb(trans->op_mode, data_entry->skb);
		kmem_cache_free(trans_slv->data_entry_pool, data_entry);
	}
}

static inline
void iwl_slv_free_cmd_entry(struct iwl_trans *trans,
			    struct iwl_slv_tx_cmd_entry *cmd_entry)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	trans_slv->config.clean_dtu(trans, cmd_entry->txq_entry.reclaim_info);

	kfree(cmd_entry->hcmd_meta.dup_buf);
	cmd_entry->hcmd_meta.dup_buf = NULL;
	kmem_cache_free(trans_slv->cmd_entry_pool, cmd_entry);
}

/**
* iwl_slv_free_queues - free data in all queues.
*
* Locking - this function should be called after AL is stopped, so no DMA
* interrupts can occur at this stage; op_mode also will not send new
* data/command at this stage. All workers should be stopped before this call.
* Thus, no locking is required here.
*/
static void iwl_slv_free_queues(struct iwl_trans *trans)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_slv_tx_queue *txq;
	struct iwl_slv_tx_cmd_entry *cmd_entry;
	struct iwl_slv_txq_entry *txq_entry, *tmp;
	int i;

	if (WARN_ON(trans_slv == NULL || trans_slv->txqs == NULL))
		return;

	txq = &trans_slv->txqs[trans_slv->cmd_queue];

	/* waiting command queue, not mapped addresses */
	list_for_each_entry_safe(txq_entry, tmp, &txq->waiting, list) {
		list_del(&txq_entry->list);
		IWL_SLV_TXQ_GET_ENTRY(txq_entry, cmd_entry);
		iwl_slv_free_cmd_entry(trans, cmd_entry);
	}

	/* sent command queue, need to check if mapped and then unmap */
	list_for_each_entry_safe(txq_entry, tmp, &txq->sent, list) {
		list_del(&txq_entry->list);
		trans_slv->config.free_dtu_mem(trans, &txq_entry->reclaim_info);
		IWL_SLV_TXQ_GET_ENTRY(txq_entry, cmd_entry);
		iwl_slv_free_cmd_entry(trans, cmd_entry);
	}

	/* data queues */
	for (i = 0; i < trans_slv->config.max_queues_num; i++)
		if (i != trans_slv->cmd_queue)
			iwl_slv_free_data_queue(trans, i);

	iwl_slv_tx_destroy_caches(trans_slv);

	kfree(trans_slv->txqs);
	trans_slv->txqs = NULL;
}

/**
* iwl_slv_tx_free - free all the resources, assumes tx is stopped.
*/
void iwl_slv_tx_free(struct iwl_trans *trans)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	if (WARN_ON_ONCE(!trans_slv))
		return;

	iwl_slv_free_queues(trans);

	if (trans_slv->policy_wq) {
		destroy_workqueue(trans_slv->policy_wq);
		trans_slv->policy_wq = NULL;
	}
}

int iwl_slv_init(struct iwl_trans *trans)
{
	struct iwl_trans_slv *trans_slv;
	int ret;

	BUG_ON(!trans);
	trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	BUG_ON(!trans_slv);

	/* verify that the configure ops was called before init */
	if (WARN_ON_ONCE(!trans_slv->config.policy_trigger)) {
		ret = EINVAL;
		goto error;
	}

	spin_lock_init(&trans_slv->txq_lock);

	ret = iwl_slv_tx_alloc_queues(trans_slv);
	if (ret)
		goto error;

	/* Initialize the wait queue for commands */
	init_waitqueue_head(&trans_slv->wait_command_queue);

	/* initialize policy data */
	trans_slv->policy_wq = alloc_workqueue("slv_policy_wq",
					IWL_SLV_POLICY_WQ_FLAGS, 1);
	INIT_WORK(&trans_slv->policy_trigger, trans_slv->config.policy_trigger);

	return 0;
error:
	IWL_ERR(trans, "%s failed, ret %d\n", __func__, ret);
	return ret;
}

/* iwl_slv_tx_get_cmd_entry - get requested cmd entry */
void iwl_slv_tx_get_cmd_entry(struct iwl_trans *trans,
			      struct iwl_rx_packet *pkt,
			      struct iwl_slv_tx_cmd_entry **cmd_entry)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_slv_tx_queue *txq = &trans_slv->txqs[trans_slv->cmd_queue];
	u16 sequence = le16_to_cpu(pkt->hdr.sequence);
	int txq_id = SEQ_TO_QUEUE(sequence);
	struct iwl_slv_txq_entry *txq_entry, *tmp;
	struct iwl_device_cmd *dev_cmd;
	int cmd_list_idx;

	if (WARN(txq_id != trans_slv->cmd_queue,
		 "wrong command queue %d (should be %d), sequence 0x%X\n",
		 txq_id, trans_slv->cmd_queue, sequence)) {
		iwl_print_hex_error(priv, pkt, 32);
		return;
	}

	/* FIXME - when not found? */
	spin_lock_bh(&trans_slv->txq_lock);

	if (WARN(list_empty(&txq->sent), "empty sent queue.\n")) {
		spin_unlock_bh(&trans_slv->txq_lock);
		return;
	}

	cmd_list_idx = 0;
	list_for_each_entry_safe(txq_entry, tmp, &txq->sent, list) {
		IWL_SLV_TXQ_GET_ENTRY(txq_entry, *cmd_entry);
		dev_cmd = iwl_cmd_entry_get_dev_cmd(trans_slv, *cmd_entry);

		if (dev_cmd->hdr.sequence != pkt->hdr.sequence) {
			cmd_list_idx++;
			continue;
		}

		if (cmd_list_idx)
			IWL_WARN(trans, "%s not first entry, idx %d\n",
				 __func__, cmd_list_idx);

		list_del(&txq_entry->list);

		/* get out of the loop when the required command is found */
		break;
	}

	spin_unlock_bh(&trans_slv->txq_lock);
	return;
}

/**
 * iwl_slv_tx_cmd_complete - handler for processing command response;
 * called from the Rx flow. DTU resources are freed in
 * iwl_slv_tx_get_cmd_entry.
 */
void iwl_slv_tx_cmd_complete(struct iwl_trans *trans,
			     struct iwl_rx_cmd_buffer *rxcb,
			     struct iwl_slv_tx_cmd_entry *cmd_entry,
			     int handler_status)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_device_cmd *dev_cmd;

	if (WARN_ON(!cmd_entry)) {
		IWL_WARN(trans, "Invalid cmd entry");
		return;
	}

	if (cmd_entry->hcmd_meta.flags & CMD_WANT_SKB) {
		struct page *p;

		cmd_entry->hcmd_meta.source->resp_pkt = rxb_addr(rxcb);
		p = rxb_steal_page(rxcb);

		cmd_entry->hcmd_meta.source->_rx_page_addr =
					(unsigned long)page_address(p);
		cmd_entry->hcmd_meta.source->_rx_page_order =
					trans_slv->rx_page_order;
		cmd_entry->hcmd_meta.source->handler_status =
							handler_status;
	}

	dev_cmd = iwl_cmd_entry_get_dev_cmd(trans_slv, cmd_entry);
	if (!(cmd_entry->hcmd_meta.flags & CMD_ASYNC)) {
		if (!test_bit(STATUS_HCMD_ACTIVE, &trans_slv->status)) {
			IWL_WARN(trans,
				 "HCMD_ACTIVE already clear for command %s\n",
				 trans_slv_get_cmd_string(trans_slv,
					dev_cmd->hdr.cmd));
		}
		clear_bit(STATUS_HCMD_ACTIVE, &trans_slv->status);
		IWL_DEBUG_INFO(trans, "Clearing HCMD_ACTIVE for command %s\n",
			       trans_slv_get_cmd_string(trans_slv,
					dev_cmd->hdr.cmd));
		wake_up_interruptible(&trans_slv->wait_command_queue);
	}

	iwl_slv_free_cmd_entry(trans, cmd_entry);
}

/**
 * iwl_trans_slv_tx_data_reclaim - free until ssn, not inclusive. Assuming that ssn
 * is always ahead of the sequence in sent queue.
 */
void iwl_trans_slv_tx_data_reclaim(struct iwl_trans *trans, int txq_id,
				   int ssn, struct sk_buff_head *skbs)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_slv_tx_queue *txq = &trans_slv->txqs[txq_id];
	struct iwl_slv_tx_data_entry *data_entry;
	struct iwl_slv_txq_entry *txq_entry, *tmp;
	int idx, tfd_num = ssn & (trans_slv->config.tfds_num - 1);
	u16 seq_ctrl;
	int freed = 0;

	IWL_DEBUG_TX_REPLY(trans, "reclaim: q %d, ssn %d, tfd %d\n",
			   txq_id, ssn, tfd_num);

	spin_lock_bh(&trans_slv->txq_lock);

	list_for_each_entry_safe(txq_entry, tmp, &txq->sent, list) {
		IWL_SLV_TXQ_GET_ENTRY(txq_entry, data_entry);

		BUG_ON(data_entry->cmd == NULL);

		seq_ctrl = le16_to_cpu(data_entry->cmd->hdr.sequence);
		idx = SEQ_TO_INDEX(seq_ctrl) &
			(trans_slv->config.tfds_num - 1);
		/* freeing up to ssn - not inclusive*/
		if (idx == tfd_num)
			break;

		__skb_queue_tail(skbs, data_entry->skb);

		list_del(&txq_entry->list);
		trans_slv->config.clean_dtu(trans, txq_entry->reclaim_info);
		kmem_cache_free(trans_slv->data_entry_pool, data_entry);
		freed++;
	}

	/* FIXME: need to handle AMPDU */
	if ((txq->waiting_count < IWL_SLV_TX_Q_LOW_THLD) &&
	    test_and_clear_bit(txq_id, trans_slv->queue_stopped_map)) {
		iwl_op_mode_queue_not_full(trans->op_mode, txq_id);
		IWL_DEBUG_TX(trans, "wake %d", txq_id);
	}

	spin_unlock_bh(&trans_slv->txq_lock);

	queue_work(trans_slv->policy_wq, &trans_slv->policy_trigger);

	IWL_DEBUG_TX_REPLY(trans, "reclaim: freed %d\n", freed);
}

static void iwl_slv_tx_copy_hcmd(struct iwl_device_cmd *out_cmd,
				 struct iwl_host_cmd *cmd)
{
	int i;
	u8 *cmd_dest = out_cmd->payload;

	for (i = 0; i < IWL_MAX_CMD_TBS_PER_TFD; i++) {
		if (!cmd->len[i])
			continue;
		if (cmd->dataflags[i] & (IWL_HCMD_DFL_NOCOPY |
					 IWL_HCMD_DFL_DUP))
			break;

		memcpy(cmd_dest, cmd->data[i], cmd->len[i]);
		cmd_dest += cmd->len[i];
	}
}

static int
iwl_slv_tx_set_meta_for_hcmd(struct iwl_trans *trans,
			     struct iwl_host_cmd *cmd,
			     struct iwl_slv_tx_cmd_entry *cmd_entry)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_slv_tx_hcmd_meta *hcmd_meta;
	struct iwl_slv_txq_entry *txq_entry = &cmd_entry->txq_entry;
	struct iwl_slv_tx_dtu_meta *dtu_meta = &txq_entry->dtu_meta;
	struct iwl_slv_tx_chunk_info *const chunk_info = dtu_meta->chunk_info;
	int i, chunk_idx, ret;

	bool had_nocopy = false;
	bool had_dup = false;

	hcmd_meta = &cmd_entry->hcmd_meta;
	hcmd_meta->dup_buf = NULL;
	hcmd_meta->copy_size = sizeof(struct iwl_cmd_header);
	hcmd_meta->hcmd_size = sizeof(struct iwl_cmd_header);
	ret = 0;

	/* the first block is always the copied part of dev cmd */
	chunk_idx = 1;
	dtu_meta->chunks_num = 1;
	dtu_meta->total_desc_num = 0;

	for (i = 0; i < IWL_MAX_CMD_TBS_PER_TFD; i++) {
		if (!cmd->len[i])
			continue;

		if (cmd->dataflags[i] & IWL_HCMD_DFL_NOCOPY) {
			if (WARN_ON(cmd->dataflags[i] & IWL_HCMD_DFL_DUP)) {
				ret = -EINVAL;
				goto free_dup_buf;
			}
			had_nocopy = true;
			chunk_info[chunk_idx].addr = (u8 *)cmd->data[i];
		} else if (cmd->dataflags[i] & IWL_HCMD_DFL_DUP) {
			/* only allowed once */
			if (WARN_ON(had_dup)) {
				ret = -EINVAL;
				goto free_dup_buf;
			}
			had_dup = true;
			hcmd_meta->dup_buf = kmemdup(cmd->data[i],
						     cmd->len[i],
						     GFP_ATOMIC);
			if (!hcmd_meta->dup_buf)
				return -ENOMEM;
			chunk_info[chunk_idx].addr = hcmd_meta->dup_buf;
		} else {
			/* NOCOPY and DUP must not be followed by normal! */
			if (WARN_ON(had_nocopy || had_dup)) {
				ret = -EINVAL;
				goto free_dup_buf;
			}
			hcmd_meta->copy_size += cmd->len[i];
		}
		if (had_nocopy || had_dup) {
			chunk_info[chunk_idx].len = cmd->len[i];

			/* TXCs num for the current chunk */
			chunk_info[chunk_idx].desc_num =
				DIV_ROUND_UP(cmd->len[i],
					     trans_slv->config.tb_size);

			dtu_meta->total_len +=
					chunk_info[chunk_idx].len;
			dtu_meta->total_desc_num +=
					chunk_info[chunk_idx].desc_num;

			dtu_meta->chunks_num++;
			chunk_idx++;
		}
		hcmd_meta->hcmd_size += cmd->len[i];
	}

	/* set up the first chunk with the copied part */
	chunk_info[0].addr = (u8 *)iwl_cmd_entry_get_dev_cmd(trans_slv,
							     cmd_entry);
	chunk_info[0].len = hcmd_meta->copy_size;
	chunk_info[0].desc_num = DIV_ROUND_UP(hcmd_meta->copy_size,
					      trans_slv->config.tb_size);
	dtu_meta->total_desc_num += chunk_info[0].desc_num;
	dtu_meta->total_len += chunk_info[0].len;

	if (WARN_ON(dtu_meta->total_desc_num >
		    trans_slv->config.max_desc_count)) {
		IWL_ERR(trans, "%s failed, wrong desc count %d\n",
			__func__, dtu_meta->total_desc_num);
		ret = -IWL_SLV_TX_GEN_ERR;
		goto free_dup_buf;
	}

	/* If any of the command structures end up being larger than
	 * the TFD_MAX_PAYLOAD_SIZE and they aren't dynamically
	 * allocated into separate TFDs, then we will need to
	 * increase the size of the buffers.
	 */
	if (WARN_ON(hcmd_meta->copy_size > TFD_MAX_PAYLOAD_SIZE)) {
		ret = -EINVAL;
		goto free_dup_buf;
	}

	if (WARN(had_nocopy && (cmd->flags & CMD_ASYNC),
		 "Bad flags: 0x%x", cmd->flags)) {
		ret = -EINVAL;
		goto free_dup_buf;
	}

	hcmd_meta->flags = cmd->flags;
	if (cmd->flags & CMD_WANT_SKB) {
		WARN_ON(cmd->flags & CMD_ASYNC);
		hcmd_meta->source = cmd;
	}

	if (had_nocopy)
		hcmd_meta->source = cmd;

free_dup_buf:
	if (ret < 0)
		kfree(hcmd_meta->dup_buf);

	return ret;
}

/**
* iwl_slv_tx_enqueue_hcmd - add host command to the queue and
* trigger policy mechanism.
*/
static int iwl_slv_tx_enqueue_hcmd(struct iwl_trans *trans,
				   struct iwl_host_cmd *cmd)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_slv_tx_queue *txq;
	struct iwl_slv_tx_cmd_entry *cmd_entry;
	struct iwl_device_cmd *dev_cmd;
	u16 seq_num;
	int ret;

	BUG_ON(!trans_slv->txqs);
	BUG_ON(!cmd);

	IWL_DEBUG_HC(trans, "Enqueue cmd 0x%x, flags 0x%x\n",
		     cmd->id, cmd->flags);

	txq = &trans_slv->txqs[trans_slv->cmd_queue];
	cmd_entry = kmem_cache_alloc(trans_slv->cmd_entry_pool, GFP_ATOMIC);
	if (unlikely(!cmd_entry))
		return -ENOMEM;

	memset(cmd_entry, 0, sizeof(*cmd_entry));

	ret = iwl_slv_tx_set_meta_for_hcmd(trans, cmd, cmd_entry);
	if (ret) {
		IWL_ERR(trans, "%s (%d): get_hcmd_size failed\n",
			__func__, __LINE__);
		goto error_free;
	}
	dev_cmd = iwl_cmd_entry_get_dev_cmd(trans_slv, cmd_entry);
	iwl_slv_tx_copy_hcmd(dev_cmd, cmd);
	dev_cmd->hdr.cmd = cmd->id;
	dev_cmd->hdr.flags = 0;

	spin_lock_bh(&trans_slv->txq_lock);

	seq_num = txq->waiting_last_idx & (trans_slv->config.tfds_num - 1);
	dev_cmd->hdr.sequence =
		cpu_to_le16((u16)QUEUE_TO_SEQ(trans_slv->cmd_queue) |
			    INDEX_TO_SEQ(seq_num));

	ret = txq->waiting_last_idx;

	/* FIXME: wrapping */
	txq->waiting_count++;
	txq->waiting_last_idx++;

	trace_iwlwifi_dev_hcmd(trans->dev, cmd, cmd_entry->hcmd_meta.hcmd_size,
			       &dev_cmd->hdr);

	list_add_tail(&cmd_entry->txq_entry.list, &txq->waiting);

	spin_unlock_bh(&trans_slv->txq_lock);
	queue_work(trans_slv->policy_wq, &trans_slv->policy_trigger);

	return ret;

error_free:
	kmem_cache_free(trans_slv->cmd_entry_pool, cmd_entry);
	return ret;
}

static int
iwl_slv_tx_set_meta_for_data(struct iwl_trans *trans,
			     struct iwl_slv_tx_data_entry *data_entry,
			     struct iwl_slv_tx_dtu_meta *dtu_meta)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_tx_cmd *tx_cmd;
	struct ieee80211_hdr *hdr;
	u16 len, firstlen, secondlen, hdr_len;

	hdr = (struct ieee80211_hdr *)data_entry->skb->data;
	hdr_len = ieee80211_hdrlen(hdr->frame_control);
	tx_cmd = (struct iwl_tx_cmd *)data_entry->cmd->payload;

	/* compute overall lengths */
	len = sizeof(struct iwl_tx_cmd) +
		sizeof(struct iwl_cmd_header) + hdr_len;
	firstlen = ALIGN(len, 4);
	secondlen = data_entry->skb->len - hdr_len;

	/* Tell NIC about any 2-byte padding after MAC header */
	if (firstlen != len)
		tx_cmd->tx_flags |= TX_CMD_FLG_MH_PAD_MSK;

	/* tx data contains at most two memory chunks */
	dtu_meta->chunks_num = 1;

	dtu_meta->chunk_info[0].addr = (u8 *)data_entry->cmd;
	dtu_meta->chunk_info[0].len = firstlen;
	dtu_meta->chunk_info[0].desc_num =
		DIV_ROUND_UP(firstlen, trans_slv->config.tb_size);

	if (secondlen) {
		dtu_meta->chunks_num++;
		dtu_meta->chunk_info[1].addr =
			data_entry->skb->data + hdr_len;
		dtu_meta->chunk_info[1].len = secondlen;
		dtu_meta->chunk_info[1].desc_num =
			DIV_ROUND_UP(secondlen, trans_slv->config.tb_size);
	}

	dtu_meta->total_desc_num =
		dtu_meta->chunk_info[0].desc_num +
		dtu_meta->chunk_info[1].desc_num;

	dtu_meta->total_len =
		dtu_meta->chunk_info[0].len +
		dtu_meta->chunk_info[1].len;

	if (WARN_ON(dtu_meta->total_len > 0xFFF)) {
		IWL_ERR(trans, "%s: length too big (%d)",
			__func__, dtu_meta->total_len);
		return -IWL_SLV_TX_GEN_ERR;
	}

	dtu_meta->total_len = (dtu_meta->total_len & 0xFFF) |
			       (tx_cmd->sta_id << 12);

	if (WARN_ON(dtu_meta->total_desc_num >
		    trans_slv->config.max_desc_count)) {
		IWL_ERR(trans, "%s failed, wrong desc count %d\n",
			__func__, dtu_meta->total_desc_num);
		return -IWL_SLV_TX_GEN_ERR;
	}

	return 0;
}

void iwl_trans_slv_tx_set_ssn(struct iwl_trans *trans, int txq_id, int ssn)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	spin_lock_bh(&trans_slv->txq_lock);

	trans_slv->txqs[txq_id].waiting_last_idx = ssn;

	spin_unlock_bh(&trans_slv->txq_lock);
}

/**
* iwl_trans_slv_tx_data_send - process header and add packet to the waiting queue.
*/
int iwl_trans_slv_tx_data_send(struct iwl_trans *trans, struct sk_buff *skb,
			       struct iwl_device_cmd *dev_cmd, int txq_id)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct ieee80211_hdr *hdr = (struct ieee80211_hdr *)skb->data;
	struct iwl_slv_tx_queue *txq;
	struct iwl_slv_tx_data_entry *data_entry;
	u16 seq_num;

	BUG_ON(!trans_slv->txqs);
	BUG_ON(!skb);

	/* verify that this skb doesn't belong to any other queue */
	if (WARN_ON(skb->next != NULL || skb->prev != NULL)) {
		IWL_ERR(trans, "Inavlid skb\n");
		return -EINVAL;
	}

	if (WARN_ON(txq_id >= trans_slv->config.max_queues_num)) {
		IWL_ERR(trans, "Inavlid txq id (%d)\n", txq_id);
		return -EINVAL;
	}

	txq = &trans_slv->txqs[txq_id];
	data_entry = kmem_cache_alloc(trans_slv->data_entry_pool, GFP_ATOMIC);
	if (unlikely(!data_entry))
		return -ENOMEM;

	memset(data_entry, 0, sizeof(*data_entry));

	data_entry->skb = skb;
	data_entry->cmd = dev_cmd;

	iwl_slv_tx_set_meta_for_data(trans, data_entry,
				     &data_entry->txq_entry.dtu_meta);

	spin_lock_bh(&trans_slv->txq_lock);

	dev_cmd->hdr.cmd = REPLY_TX;
	seq_num = txq->waiting_last_idx & (trans_slv->config.tfds_num - 1);
	dev_cmd->hdr.sequence = cpu_to_le16((u16)(QUEUE_TO_SEQ(txq_id) |
					    INDEX_TO_SEQ(seq_num)));

	IWL_DEBUG_TX(trans, "txq_id %d, seq %d, last_idx %d, sequence 0x%X\n",
		     txq_id, seq_num, txq->waiting_last_idx,
		     dev_cmd->hdr.sequence);

	list_add_tail(&data_entry->txq_entry.list, &txq->waiting);

	txq->waiting_count++;
	txq->waiting_last_idx++;

	/* FIXME: compute AC for agg */
	if (!ieee80211_has_morefrags(hdr->frame_control) &&
	    (txq->waiting_count > IWL_SLV_TX_Q_HIGH_THLD))
		if (!test_and_set_bit(txq_id, trans_slv->queue_stopped_map)) {
			iwl_op_mode_queue_full(trans->op_mode, txq_id);
			IWL_DEBUG_TX(trans, "stop %d", txq_id);
		}

	spin_unlock_bh(&trans_slv->txq_lock);

	queue_work(trans_slv->policy_wq, &trans_slv->policy_trigger);
	return 0;
}

static int iwl_slv_send_cmd_async(struct iwl_trans *trans,
				struct iwl_host_cmd *cmd)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	int ret;

	/* An asynchronous command can not expect an SKB to be set. */
	if (WARN_ON(cmd->flags & CMD_WANT_SKB))
		return -EINVAL;

	ret = iwl_slv_tx_enqueue_hcmd(trans, cmd);
	if (ret < 0) {
		IWL_ERR(trans,
			"Error sending %s: enqueue_hcmd failed: %d\n",
			trans_slv_get_cmd_string(trans_slv, cmd->id), ret);
		return ret;
	}
	return 0;
}

/**
* iwl_slv_tx_cancel_cmd - cancel command, but don't delete it yet in order
* not to confuse the command complete.
*/
static void iwl_slv_tx_cancel_cmd(struct iwl_trans *trans, int cmd_idx)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_slv_tx_queue *txq = &trans_slv->txqs[trans_slv->cmd_queue];
	struct iwl_slv_tx_cmd_entry *cmd_entry;
	struct iwl_device_cmd *dev_cmd;
	struct iwl_slv_txq_entry *txq_entry;
	__le16 sequence;

	sequence = cpu_to_le16(QUEUE_TO_SEQ(trans_slv->cmd_queue) |
			       INDEX_TO_SEQ(cmd_idx));

	spin_lock_bh(&trans_slv->txq_lock);
	list_for_each_entry(txq_entry, &txq->sent, list) {
		IWL_SLV_TXQ_GET_ENTRY(txq_entry, cmd_entry);
		dev_cmd = iwl_cmd_entry_get_dev_cmd(trans_slv, cmd_entry);
		if (dev_cmd->hdr.sequence == sequence) {
			if (cmd_entry->hcmd_meta.flags & CMD_WANT_SKB)
				cmd_entry->hcmd_meta.flags &= ~CMD_WANT_SKB;
			break;
		}
	}
	spin_unlock_bh(&trans_slv->txq_lock);
}

static int iwl_slv_send_cmd_sync(struct iwl_trans *trans,
				struct iwl_host_cmd *cmd)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	int cmd_idx;
	int ret;

	IWL_DEBUG_INFO(trans, "Attempting to send sync command %s\n",
		       trans_slv_get_cmd_string(trans_slv, cmd->id));

	if (WARN_ON(test_and_set_bit(STATUS_HCMD_ACTIVE,
				     &trans_slv->status))) {
		IWL_ERR(trans, "Command %s: a command is already active!\n",
			trans_slv_get_cmd_string(trans_slv, cmd->id));
		return -EIO;
	}

	IWL_DEBUG_INFO(trans, "Setting HCMD_ACTIVE for command %s\n",
		       trans_slv_get_cmd_string(trans_slv, cmd->id));

	cmd_idx = iwl_slv_tx_enqueue_hcmd(trans, cmd);
	if (cmd_idx < 0) {
		ret = cmd_idx;
		clear_bit(STATUS_HCMD_ACTIVE, &trans_slv->status);
		IWL_ERR(trans,
			"Error sending %s: enqueue_hcmd failed: %d\n",
			trans_slv_get_cmd_string(trans_slv, cmd->id), ret);
		return ret;
	}

	ret = wait_event_interruptible_timeout(
			trans_slv->wait_command_queue,
			!test_bit(STATUS_HCMD_ACTIVE, &trans_slv->status),
			HOST_COMPLETE_TIMEOUT);
	if (!ret) {
		if (test_bit(STATUS_HCMD_ACTIVE, &trans_slv->status)) {
			IWL_ERR(trans,
				"Error sending %s: time out after %dms.\n",
				trans_slv_get_cmd_string(trans_slv, cmd->id),
				jiffies_to_msecs(HOST_COMPLETE_TIMEOUT));

			clear_bit(STATUS_HCMD_ACTIVE, &trans_slv->status);
			IWL_DEBUG_INFO(trans,
				       "Clearing HCMD_ACTIVE for command %s\n",
				       trans_slv_get_cmd_string(trans_slv,
								cmd->id));
			ret = -ETIMEDOUT;
			goto cancel;
		}
	}

	if ((cmd->flags & CMD_WANT_SKB) && !cmd->resp_pkt) {
		IWL_ERR(trans, "Error: Response NULL in '%s'\n",
			trans_slv_get_cmd_string(trans_slv, cmd->id));
		ret = -EIO;
		goto cancel;
	}

	return 0;

cancel:
	if (cmd->flags & CMD_WANT_SKB) {
		/*
		 * Cancel the CMD_WANT_SKB flag for the cmd in the
		 * TX cmd queue. Otherwise in case the cmd comes
		 * in later, it will possibly set an invalid
		 * address (cmd->meta.source).
		 */
		iwl_slv_tx_cancel_cmd(trans, cmd_idx);
	}

	if (cmd->resp_pkt) {
		iwl_free_resp(cmd);
		cmd->resp_pkt = NULL;
	}

	return ret;
}

int iwl_trans_slv_send_cmd(struct iwl_trans *trans, struct iwl_host_cmd *cmd)
{
	if (cmd->flags & CMD_ASYNC)
		return iwl_slv_send_cmd_async(trans, cmd);

	return iwl_slv_send_cmd_sync(trans, cmd);
}

void iwl_slv_rx_handle_dispatch(struct iwl_trans *trans,
				struct iwl_rx_cmd_buffer *rxcb)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	int reclaim, ret;
	struct iwl_rx_packet *pkt = rxb_addr(rxcb);
	struct iwl_slv_tx_cmd_entry *cmd_entry = NULL;

	reclaim = !(pkt->hdr.sequence & SEQ_RX_FRAME);
	if (reclaim) {
		int i;

		for (i = 0; i < trans_slv->n_no_reclaim_cmds; i++) {
			if (trans_slv->no_reclaim_cmds[i] == pkt->hdr.cmd) {
				reclaim = false;
				break;
			}
		}
	}

	if (reclaim) {
		iwl_slv_tx_get_cmd_entry(trans, pkt, &cmd_entry);
		if (cmd_entry == NULL)
			return;

		ret = iwl_op_mode_rx(trans->op_mode, rxcb,
				     iwl_cmd_entry_get_dev_cmd(trans_slv,
							       cmd_entry));
		if (!rxcb->_page_stolen)
			iwl_slv_tx_cmd_complete(trans, rxcb, cmd_entry, ret);
		else
			IWL_WARN(trans, "Claim null rxb?\n");
	} else {
		iwl_op_mode_rx(trans->op_mode, rxcb, NULL);
	}
	return;
}
