/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#ifndef __IWL_LLS_H__
#define __IWL_LLS_H__

#include "iwl-trans.h"

/* TXBU BITs */
#define IWL_LLS_TXBU_SIGNATURE	0xdea1
#define IWL_LLS_EX_MODE_MASK	BIT(7)
#define IWL_LLS_INTEN_MASK	BIT(6)
#define IWL_LLS_TXC_SC_MASK	BIT(5)
#define IWL_LLS_TXC_PB_MASK	(BIT(0) | BIT(1))
#define IWL_LLS_TXC_PA_MASK	(BIT(0) | BIT(1))
#define IWL_LLS_TXC_DST_MASK	0xFFFFF

/**
 * struct iwl_idi_txbu_header - IDI Tx Burst header
 *
 * @signature[0:15]: Used for stream synchronization purposes
 * @seq_num[15:31]: Used as a uniqe identifier for eacg of the TXBUs
 * @txc_count:	bits[0:4]: The number of TXCs in this burst
 *		bits[5-7]: Reserved.
 * @queue:	bits[0:4]: Extended TXBU mode only.
 *		Select which queue to update in the SCHD block; used to
 *		calculate the entry in the LUT and BC tables.
 *		bits[5-7]: Reserved.
 * @tfd_index[16:23]:	Extended TXBU mode only.
 *		Calculate the next TFD ubdex value that is written to the
 *		SCHD_WR_PTR.
 * @flags:	bits[0:4]: Reserved.
 *		bit[5]: txsc - TXC size control. enable/disable TXC size
 *		checking.
 *		bit[6]: inten - Interrupt control. post TXBU Interrupt control
 *		at the end of the TXBU processing.
 *		bit[7]: ex - extended: control the TXBU type (basic\extended).
 * @lut_value:	[0:7] Extended TXBU mode only. The byte count value that will
 *		be will written to the SFDB_LUT[queue][tfd_index]
 *		[8-15] Reserved
 * @byte_cnt_value: Extended TXBU mode only. The byte count value that will
 *		   be will written to the SFDB_BC[queue][tfd_index]
 */
struct iwl_lls_txbu {
	__le16 signature;
	__le16 seq_num;
	u8 txc_count;
	u8 queue;
	u8 tfd_index;
	u8 flags;
	u8 lut_value;
	u8 reserved;
	__le16 byte_cnt_value;
} __packed;

/*.
*			[2:7]: Reserved.
* @padd_after:		[0:1]: Pad after data.
*			[2:7]: Reserved.
* @len:			[16:31]: TXC data length.
* @dest_and_flags:	[0:19]: Destination address.
*			[20:29]: Reserved.
*			[30]: TFDI - Compressed TFD info (debug).
*			[31]: REGA - Refuster Access (debug).
*/
struct iwl_lls_txc {
	u8 pb;
	u8 pa;
	__le16 len;
	__le32 dest_and_flags;
} __packed;

/*
 * Callback for the IDI DBB DMA interrupt. The callback must always be called in
 * an async. manner.
 * In the context of the emulation, need to consider the correct context for
 * calling this function.
 * @chan: the channel that the interrupt was called from.
 * @data: A void pointer to data that was registered in the set irq handler.
*/
typedef void (*iwl_idi_lls_processing_irq) (__le32 chan, void *data);

/*
 * Initialize the LLS TX component.
 * Should be called as part of the global AL intialization flow.
 *
 *@trans - transport to the pci.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_lls_em_init(struct iwl_trans *trans);

/**
 * Free the LLS TX emulation block.
 * Synchronously returns when the LLS work for current job is finished.
 */
void iwl_lls_em_free(void);

/**
 * Start the LLS TX Emulation.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_lls_em_start(void);

/**
 * Stop the LLS TX Emulation.
 * Used to stop all parsing jobs.
 * Synchronious function - returns only after all parsing operations are
 * stopped.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_lls_em_stop(void);

/**
 * Registers an interrupt handler for LLs interrupts to the IDI.
 *
 * @param irq_handler - interrupt handler to register.
 * @param channel - the channel number on which the interrupt will be called.
 * @param data  void pointer which will be given to the interrupt handler.
 */
void iwl_lls_em_register_inter(iwl_idi_lls_processing_irq irq_handler,
			      u8 channel, void *data);

/*
 * Push data to the LLS pipe for the target channel.
 *
 * @param buffer - the target address to push from tp the pipe.
 * @param size - The amount of data in bytes to push to the pipe.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_lls_em_push_to_pipe(u32 channel, void *buffer, u32 size);

#endif	/*__IWL_LLS_H__*/
