/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#include <linux/pci.h>
#include <linux/kthread.h>

#include "iwl-io.h"
#include "pcie/internal.h"

#include "iwl-idi.h"
#include "iwl-emulation.h"
#include "iwl-amfh.h"

/*
 * IWL AMFH Logger
 */
#define AMFH_LOG_PREFIX			"[IWL_AMFH]"

#ifdef __IWL_AMFH_LOG_ENABLED__
#define AMFH_TRACE_ENTER \
	IWL_EM_TRACE_ENTER(__AMFH_LOG_LEVEL_TRACE__, AMFH_LOG_PREFIX)
#define AMFH_TRACE_EXIT \
	IWL_EM_TRACE_EXIT(__AMFH_LOG_LEVEL_TRACE__, AMFH_LOG_PREFIX)
#define AMFH_TRACE_EXIT_RET_STR(_ret) \
	IWL_EM_LOG(__AMFH_LOG_LEVEL_TRACE__, AMFH_LOG_PREFIX, \
							"<<< "_ret)
#define AMFH_TRACE_EXIT_RET(_ret) \
	IWL_EM_TRACE_EXIT_RET(__AMFH_LOG_LEVEL_TRACE__, \
				AMFH_LOG_PREFIX, _ret)
#define IWL_AMFH_LOG(fmt, args ...) \
	IWL_EM_LOG(__AMFH_LOG_LEVEL_DEBUG__, \
		AMFH_LOG_PREFIX, fmt, ## args)

#define IWL_AMFH_LOG_HEX_DUMP(msg, p, len) \
	IWL_EM_LOG_HEX_DUMP(msg, p, len)
#else

#define AMFH_TRACE_ENTER
#define AMFH_TRACE_EXIT
#define AMFH_TRACE_EXIT_RET_STR(_ret)
#define AMFH_TRACE_EXIT_RET(_ret)
#define IWL_AMFH_LOG(fmt, args ...)
#define IWL_AMFH_LOG_HEX_DUMP(msg, p, len)

#endif
/* Always print error messages */
#define IWL_AMFH_LOG_ERR(fmt, args ...) \
	IWL_EM_LOG_ERR(fmt, ## args)


/*
 * Timeout when waiting to IDI stream to have more space avaialble.
 * Waiting more than that for the IDI to pull data from the stream means
 * it was probably stuck and nobody told us.
 */
#define AMFH_PAUSING_TIMEOUT 2000

/* Signatures and sizes which are pushed to the IDI stream */
static const u32 AMFH_KICK0_SIGNATURE = 0xDEADBEAF;
static const u32 AMFH_CMPL1_SIGNATURE = 0x0;
static const u32 AMFH_CMPL1_SIZE = sizeof(AMFH_CMPL1_SIGNATURE);
static const u32 AMFH_CMPL2_PART1_SIGNATURE = 0x0;
static const u32 AMFH_CMPL2_PART2_SIGNATURE = 0x3;
static const u32 AMFH_CMPL2_SIZE = sizeof(AMFH_CMPL2_PART1_SIGNATURE) +
				   sizeof(AMFH_CMPL2_PART2_SIGNATURE);
static const u32 AMFH_CMPL2_DUMMY = 0xFFFFFFFF;

/**
 * struct iwl_amfh_rx_queue:
 * @bd: driver's pointer to buffer of receive buffer descriptors (rbd)
 * @bd_dma: bus address of buffer of receive buffer descriptors (rbd)
 * @pages: list of allocated pages, each referenced by the corresponding rbd
 * @pages_dma: dma addresses of the pages in the same index
 * @read: Shared index to newest available Rx buffer
 * @write: Shared index to oldest written Rx packet
 * @write_actual: the write pointer as last written to the HW
 * @rb_stts: driver's pointer to receive buffer status
 * @rb_stts_dma: bus address of receive buffer status
 */
struct iwl_amfh_rx_queue {
	__le32 *bd;
	dma_addr_t bd_dma;
	struct page *pages[RX_QUEUE_SIZE];
	dma_addr_t pages_dma[RX_QUEUE_SIZE];
	u32 read;
	u32 write;
	u32 write_actual;
	struct iwl_rb_status *rb_stts;
	dma_addr_t rb_stts_dma;
};

struct iwl_amfh_t {

	/* Configuration supplied on init */
	struct iwl_al_amfh_config *init_cfg;

	/* Configuration supplied on start */
	struct iwl_amfh_config *amfh_config;

	/*
	 * This indicates the number of RBDs waiting in the RX ring.
	 * When the RX handle is called it updates this value, and whenever
	 * a burst ends it is updated again. If it's still positive (more RX,
	 * or the burst hasn't exhausted all frames) - start another RX work.
	 */
	u32 rx_waiting;

	/*
	 * Indicates that the IDI stream was read and AMFH should try to push
	 * again, in case it was paused due to lack of space on IDI stream.
	 */
	bool resume;

	/* Indicates an RX interrupt that AMFH should handle. */
	atomic_t rx_interrupt;

	/* Number of frames already transferred in this burst. */
	u32 curr_burst_frames;

	/* RX Queue of the PCI ring */
	struct iwl_amfh_rx_queue rxq;

	/* transport which have pointers to all other structs */
	struct iwl_trans *trans;

	/* for prints only */
	struct device *dev;

	/* AMFH main handler thread and waitqueue */
	struct task_struct *workloop_thread;
	wait_queue_head_t handler_queue;
};


/* Global struct declaration + pointer which will be externed
   + short handler to this C file alone. */
static struct iwl_amfh_t iwl_amfh_em_global;
struct iwl_amfh_t *iwl_amfh_em = &iwl_amfh_em_global;
static struct iwl_amfh_t *amfh = &iwl_amfh_em_global;


/***************************
 * START OF RXQ FUNCTIONS
 **************************/

/**
 * RXQ mode of operation
 * This RX queue is a simplification of the PCIe RX queue.
 * The queue uses two pointers (indices), read and write pointer,
 * to manage its operations. The device updates the read pointer, which points
 * the the last index of valid data which was written the queue.
 * The driver (AMFH emulation) updates the write pointer, telling the device
 * where is the last index which can be written into, which is always the
 * one *before* the write pointer.
 * The queue is initialized with dma-mapped pages for each slot in the queue,
 * so that these 4 variables point to the same location in memory:
 * pages[i] -- the virtual address (struct page *) of the allocated page
 * pages_dma[i] -- the bus address of the allocated page, as mapped by us
 * bd[i] -- memory descriptor which contains the dma address of the page,
 * using the format the FH expects (little-endian shifted 32 bit address)
 * bd_dma[i] -- the physical address of the bd, which is really what the device
 * reads.
 * All the pages are allocated on init, and when the device writes new data,
 * it advances the read pointer and interupts the host.
 * The host reads up to the new read index (inclusive) from the page.
 * When a specific descriptor is read, it can be written again by the device,
 * so the host increments *both* read and write pointer simultaneuosly with
 * each buffer being read - so the write pointer always trails the host's
 * read pointer by one. When the write pointer hits a multiple of 8, the host
 * also writes the updated pointer to the device.
 *
 * Note: both updating the host's read/write pointers and updating the device
 * write pointer (calling iwl_amfh_rxq_update_write_pointer) should be called
 * by the user of the ring - these operations don't happen automatically.
 **/


/**
 * iwl_amfh_rxq_update_write_ptr - Update the write pointer for the RX queue.
 * The function updates the write pointer only when it is a multiple of 8
 * because it is preferred by the FH
 * (may round down non-aligned write pointers).
 * This function also assumes shadow registers are enabled.
 */
static void iwl_amfh_rxq_update_write_ptr(void)
{
	struct iwl_amfh_rx_queue *rxq = &(amfh->rxq);

	/* Device expects a multiple of 8 */
	if (rxq->write_actual == (rxq->write & ~0x7))
		return;
	else
		rxq->write_actual = rxq->write & ~0x7;

	iwl_write32(amfh->trans, FH_RSCSR_CHNL0_WPTR, rxq->write_actual);
}


/**
 * iwl_amfh_rxq_free_pages - Frees all the pages allocated for the queue
 */
static void iwl_amfh_rxq_free_pages(int until_index)
{
	struct iwl_amfh_rx_queue *rxq = &amfh->rxq;
	u32 rx_page_order = get_order(amfh->init_cfg->rb_size);
	int i;

	for (i = 0; i < until_index; i++) {
		rxq->bd[i] = 0;
		dma_unmap_page(amfh->trans->dev, rxq->pages_dma[i],
			       PAGE_SIZE << rx_page_order,
			       DMA_FROM_DEVICE);
		__free_pages(rxq->pages[i], rx_page_order);
	}
}


/**
 * iwl_amfh_rxq_alloc_pages - allocates all the pages for the ring with
 * with the mask given.
 */
static int iwl_amfh_rxq_alloc_pages(gfp_t gfp_mask)
{
	struct iwl_amfh_rx_queue *rxq = &(amfh->rxq);
	int i;
	u32 rx_page_order = get_order(amfh->init_cfg->rb_size);


	if (rx_page_order > 0)
		gfp_mask |= __GFP_COMP;

	for (i = 0; i < RX_QUEUE_SIZE; i++) {

		rxq->pages[i] = alloc_pages(gfp_mask, rx_page_order);
		if (!rxq->pages[i]) {
			IWL_AMFH_LOG_ERR("Failed to allocate pages for RX");
			goto alloc_error;
		}

		/* Mapping the pages to a DMA address, and cheking that
		the address is no more than 36 bits, and 256 bit
		aligned (8 LSB are zero). */
		rxq->pages_dma[i] = dma_map_page(amfh->trans->dev,
			rxq->pages[i], 0, PAGE_SIZE << rx_page_order,
			DMA_FROM_DEVICE);
		BUG_ON(rxq->pages_dma[i] & ~DMA_BIT_MASK(36));
		BUG_ON(rxq->pages_dma[i] & DMA_BIT_MASK(8));

		/* Pointing the i'th RBD to that page */
		rxq->bd[i] = cpu_to_le32((u32)(rxq->pages_dma[i] >> 8));
	}

	return 0;

alloc_error:
	/* Freeing already-allocated pages, up until the failing index. */
	iwl_amfh_rxq_free_pages(i);
	return -ENOMEM;
}


/**
 * iwl_amfh_rxq_alloc_queue - allocates RBDs, pages, and status variables
 **/
static int iwl_amfh_rxq_alloc_queue(struct device *dev)
{
	struct iwl_amfh_rx_queue *rxq = &amfh->rxq;

	memset(&amfh->rxq, 0, sizeof(amfh->rxq));

	if (WARN_ON(rxq->bd || rxq->rb_stts))
		return -EINVAL;

	/* Allocate the circular buffer of Read Buffer Descriptors (RBDs) */
	rxq->bd = dma_alloc_coherent(dev, sizeof(__le32) *
					amfh->init_cfg->num_rbds,
				    &rxq->bd_dma, GFP_KERNEL);
	if (!rxq->bd)
		goto err_bd;
	memset(rxq->bd, 0, sizeof(__le32) * amfh->init_cfg->num_rbds);

	/*Allocate the driver's pointer to receive buffer status */
	rxq->rb_stts = dma_alloc_coherent(dev, sizeof(*rxq->rb_stts),
					  &rxq->rb_stts_dma, GFP_KERNEL);
	if (!rxq->rb_stts)
		goto err_rb_stts;
	memset(rxq->rb_stts, 0, sizeof(*rxq->rb_stts));

	/* Allocates the DMA pages of the ring, error handling is inside */
	if (iwl_amfh_rxq_alloc_pages(GFP_KERNEL))
		goto err_rb_stts;

	return 0;

err_rb_stts:
	dma_free_coherent(dev, sizeof(__le32) * amfh->init_cfg->num_rbds,
			  rxq->bd, rxq->bd_dma);
	memset(&rxq->bd_dma, 0, sizeof(rxq->bd_dma));
	rxq->bd = NULL;
err_bd:
	return -ENOMEM;

}


/**
 * iwl_amfh_rx_hw_init - initializes the PCIe HW
 **/
static void iwl_amfh_rx_hw_init(void)
{
	struct iwl_trans *trans = amfh->trans;
	struct iwl_amfh_rx_queue *rxq = &(amfh->rxq);

	/* Stop Rx DMA */
	iwl_write_direct32(trans, FH_MEM_RCSR_CHNL0_CONFIG_REG, 0);

	/* Reset driver's Rx queue write index */
	iwl_write_direct32(trans, FH_RSCSR_CHNL0_RBDCB_WPTR_REG, 0);

	/* Tell device where to find RBD circular buffer in DRAM */
	iwl_write_direct32(trans, FH_RSCSR_CHNL0_RBDCB_BASE_REG,
			   (u32)(rxq->bd_dma >> 8));

	/* Tell device where in DRAM to update its Rx status */
	iwl_write_direct32(trans, FH_RSCSR_CHNL0_STTS_WPTR_REG,
			   rxq->rb_stts_dma >> 4);

	/* Set interrupt coalescing timer to default (2048 usecs) */
	iwl_write8(trans, CSR_INT_COALESCING, IWL_HOST_INT_TIMEOUT_DEF);
}

/**
 * iwl_amfh_rxq_init - allocates and inits all RXQ and HW related
 **/
static int iwl_amfh_rxq_init(void)
{
	struct iwl_amfh_rx_queue *rxq = &amfh->rxq;

	int err;

	if (!rxq->bd) {
		err = iwl_amfh_rxq_alloc_queue(amfh->trans->dev);
		if (err)
			return err;
	}


	/* Set us so that we have processed and used all buffers, but have
	 * not restocked the Rx queue with fresh buffers */
	rxq->read = 0;
	rxq->write = RX_QUEUE_SIZE - 1;
	rxq->write_actual = 0;

	iwl_amfh_rx_hw_init();

	iwl_amfh_rxq_update_write_ptr();

	return 0;
}


/**
 * iwl_amfh_rxq_start - start DMA on the rx queue
 **/
static void iwl_amfh_rxq_start(void)
{
	const u32 rfdnlog = RX_QUEUE_SIZE_LOG; /* 256 RBDs */
	/* FIXME: RX_RB_TIMEOUT for all devices? */
	u32 rb_timeout = RX_RB_TIMEOUT;
	u32 rb_size_flag = amfh->init_cfg->rb_size == (8 * 1024) ?
		FH_RCSR_RX_CONFIG_REG_VAL_RB_SIZE_8K :
		FH_RCSR_RX_CONFIG_REG_VAL_RB_SIZE_4K;


	/* Enable Rx DMA
	 * FH_RCSR_CHNL0_RX_IGNORE_RXF_EMPTY is set because of HW bug in
	 *      the credit mechanism in 5000 HW RX FIFO
	 * Direct rx interrupts to hosts
	 * Rx buffer size 4 or 8k
	 * RB timeout 0x10
	 * 256 RBDs
	 */
	iwl_write_direct32(amfh->trans, FH_MEM_RCSR_CHNL0_CONFIG_REG,
			   FH_RCSR_RX_CONFIG_CHNL_EN_ENABLE_VAL |
			   FH_RCSR_CHNL0_RX_IGNORE_RXF_EMPTY |
			   FH_RCSR_CHNL0_RX_CONFIG_IRQ_DEST_INT_HOST_VAL |
			   FH_RCSR_CHNL0_RX_CONFIG_SINGLE_FRAME_MSK |
			   rb_size_flag |
			   (rb_timeout << FH_RCSR_RX_CONFIG_REG_IRQ_RBTH_POS)|
			   (rfdnlog << FH_RCSR_RX_CONFIG_RBDCB_SIZE_POS));
}


/**
 * iwl_amfh_rxq_stop - stops DMA operation
 **/
static int iwl_amfh_rxq_stop(void)
{

	/* stop Rx DMA */
	iwl_write_direct32(amfh->trans, FH_MEM_RCSR_CHNL0_CONFIG_REG, 0);
	return iwl_poll_direct_bit(amfh->trans, FH_MEM_RSSR_RX_STATUS_REG,
			    FH_RSSR_CHNL0_RX_STATUS_CHNL_IDLE, 1000);
}

/**
 * iwl_amfh_rxq_free - frees the rxq context
 **/
static void iwl_amfh_rxq_free(void)
{
	struct iwl_amfh_rx_queue *rxq = &amfh->rxq;

	/*if rxq->bd is NULL, it means that nothing has been allocated,
	 * exit now */
	if (!rxq->bd) {
		IWL_DEBUG_INFO(amfh, "Free NULL rx context\n");
		return;
	}

	iwl_amfh_rxq_free_pages(RX_QUEUE_SIZE);

	dma_free_coherent(amfh->trans->dev, sizeof(__le32) *
					amfh->init_cfg->num_rbds,
					rxq->bd, rxq->bd_dma);
	memset(&rxq->bd_dma, 0, sizeof(rxq->bd_dma));
	rxq->bd = NULL;

	if (rxq->rb_stts)
		dma_free_coherent(amfh->trans->dev,
				  sizeof(struct iwl_rb_status),
				  rxq->rb_stts, rxq->rb_stts_dma);
	else
		IWL_DEBUG_INFO(amfh, "Free rxq->rb_stts which is NULL\n");
	memset(&rxq->rb_stts_dma, 0, sizeof(rxq->rb_stts_dma));
	rxq->rb_stts = NULL;
}


/*************************
 * END OF RXQ FUNCTIONS
 ************************/

/*
 * AMFH mode of operation:
 * After init and start, AMFH operation is through a single thread whose
 * function is prv_amfh_idle. This function in general waits on hadnler_queue
 * for an RX inidcation from the HW (a call to iwl_amfh_rx_handler).
 * When such indication is received, prv_amfh_handle_request is called to
 * start transfer the RX packets to the IDI stream.
 *
 * The IDI stream works in bursts on several packets at once, a notion which
 * PCIe doesn't have. In order to create such bursts over PCIe ring, we take
 * the number of packets waiting on the ring when prv_andh_handle_request was
 * called as the burst size.
 *
 * Each burst is structed in the following format:
 * 1. KICK0 signature of 4 bytes.
 * 2. Length of the following packet, 4 bytes.
 * 3. The payload of the RX packet itself, according to the length.
 * 4. length of the CMPL signature (4 for CMPL1, 8 for CMPL2)
 * 5(a). CMPL1 signature of 4 bytes.
 * 5(b). CMPL2 signature of 8 bytes, and a dummy write of 4 bytes.
 *
 * If a burst contains one packet, it only has 1+2+3+4+5b.
 * If a burst contains multiple packets, it has 1+(2+3+4+5a){1,}+(2+3+4+5b)
 * This means that for each non-last packet we write the length and payload,
 * then adding CMPL1 signature, and for the last one we use CMPL2 signature.
 *
 * Each push to the IDI stream uses the calls in iwl_amfH_config to check for
 * available space, and if there is enough it pushes the data. If there is
 * not enough space, we wait for it to be available (on the waitqueue) and
 * when the IDI reads some data it must resume us (it does so in any case).
 *
 * When a burst ends, it can either be because there is no more RX, or because
 * we reached the maximal burst size. In both cases we return to prv_amfh_idle
 * and check for more RX, and if there is none, go to sleep until an interrupt
 * wakes us again (or a call to iwl_amfh_stop).
 */


static int prv_amfh_push_data(const void *src, u32 len)
{
	int ret = 0;
	long time_left;

	AMFH_TRACE_ENTER;

	if (!amfh->amfh_config->available(len)) {
		time_left =
			wait_event_interruptible_timeout(
					amfh->handler_queue,
					(amfh->resume &&
					 amfh->amfh_config->available(len)) ||
					kthread_should_stop(),
				msecs_to_jiffies(AMFH_PAUSING_TIMEOUT));

		if (kthread_should_stop()) {
			IWL_AMFH_LOG("AMFH thread was called for a stop");
			ret = -ERESTARTSYS;
			goto exit;
		}

		/* If the timeout was reached, it means IDI thread is stuck. */
		if (WARN_ON(time_left == 0)) {
			IWL_AMFH_LOG_ERR("AMFH stuck due to IDI thread idle");
			ret = -ERESTARTSYS;
			goto exit;
		}

		amfh->resume = false;
	}

	IWL_AMFH_LOG("Pushing %d bytes from 0x%p to IDI stream.", len, src);
	amfh->amfh_config->push(src, len);
exit:
	AMFH_TRACE_EXIT_RET(ret);
	return ret;
}

/* Utility functions which helps use prv_amfh_push_data */
#define amfh_push_data(src, len) do {			\
	ret = prv_amfh_push_data(src, len);		\
	if (ret)					\
		goto exit;				\
} while (0)

#define amfh_push_int(x) amfh_push_data(&(x), sizeof(x))

/**
 * prv_update_unhandled_rx - updates the # of RBDs in the ring.
 *
 * This should be called when receiveing an RX interrupt and whenever we
 * have finished a burst.
 * The number of RBDs found when starting the burst (RX interrupt)
 * will define the upper limit for the burst (under the limit of max
 * frames per burst) - any other RBDs will be read in the next burst.
 **/
static void prv_update_unhandled_rx(void)
{
	u32 sw_read_index, hw_read_index;
	int value; /* This is int because the result can be negative */

	AMFH_TRACE_ENTER;
	hw_read_index = le16_to_cpu(amfh->rxq.rb_stts->closed_rb_num) &  0x0FFF;
	sw_read_index = amfh->rxq.read;
	value = hw_read_index - sw_read_index;
	if (value < 0)
		value += amfh->init_cfg->num_rbds;
	amfh->rx_waiting = (u32)value;
	IWL_AMFH_LOG("Unhandled RBDs: %d (SW: %u HW: %u)",
		     value, sw_read_index, hw_read_index);
	AMFH_TRACE_EXIT;
}


/**
 * prv_amfh_copy_from_ring - copies RX packet payload and len to the IDI.
 *
 * It is a simplified version of iwl_rx_handler, and extracts one RB from
 * the ring, passes it forward to the driver, and updates the ring.
 *
 * Returns 0, or -ERESTARTSYS in case push failed.
 **/
static int prv_amfh_copy_from_ring(void)
{
	struct iwl_amfh_rx_queue *rxq = &amfh->rxq;
	struct iwl_rx_packet *pkt;
	int len, ret;

	AMFH_TRACE_ENTER;

	pkt = page_address(rxq->pages[rxq->read]);
	IWL_AMFH_LOG("Got pkt cmd = 0x%X", pkt->hdr.cmd);

	len = le32_to_cpu(pkt->len_n_flags) & FH_RSCSR_FRAME_SIZE_MSK;
	len += sizeof(u32); /* account for status word */

	IWL_AMFH_LOG("Received packet of size %d", len);
	amfh_push_int(len);
	amfh_push_data(pkt, len);

	rxq->read = (rxq->read + 1) & RX_QUEUE_MASK;
	rxq->write = (rxq->read - 1) & RX_QUEUE_MASK;
	iwl_amfh_rxq_update_write_ptr();

exit:
	AMFH_TRACE_EXIT_RET(ret);
	return ret;
}


/**
 * prv_amfh_handle_request - pushes RX burst to IDI stream.
 *
 * For details, see "AMFH mode of operation" above.
 * Doesn't return anything because its return value is irrelevant,
 * but still have a notion of return value for debugging purposes.
 **/
static void prv_amfh_handle_request(void)
{
	bool more_rx;
	int ret;

	AMFH_TRACE_ENTER;

	atomic_set(&amfh->rx_interrupt, 0);
	prv_update_unhandled_rx();
	amfh->curr_burst_frames = 0;

	if (amfh->rx_waiting == 0) {
		IWL_AMFH_LOG(
			"RX interrupt received but no RBDs waiting in ring");
		ret = -EIO;
		goto exit;
	}

	amfh_push_int(AMFH_KICK0_SIGNATURE);

	do {
		ret = prv_amfh_copy_from_ring();
		if (ret)
			goto exit;
		amfh->rx_waiting--;
		amfh->curr_burst_frames++;

		/*
		 * We close the burst if there's no more data or maximal size
		 * of a burst is reached.
		 */
		more_rx = ((amfh->rx_waiting > 0) &&
			(amfh->curr_burst_frames <
			amfh->init_cfg->max_burst_size));
		if (more_rx) {
			amfh_push_int(AMFH_CMPL1_SIZE);
			amfh_push_int(AMFH_CMPL1_SIGNATURE);
		}
	} while (more_rx);

	amfh_push_int(AMFH_CMPL2_SIZE);
	amfh_push_int(AMFH_CMPL2_PART1_SIGNATURE);
	amfh_push_int(AMFH_CMPL2_PART2_SIGNATURE);
	amfh_push_int(AMFH_CMPL2_DUMMY);

exit:
	IWL_AMFH_LOG("Exiting with status %d", ret);
	AMFH_TRACE_EXIT;
}

/**
 * prv_amfh_idle - main AMFH thread.
 *
 * For details, see "AMFH mode of operation" above.
 **/
static int prv_amfh_idle(void *unused)
{
	AMFH_TRACE_ENTER;
	while (!kthread_should_stop()) {
		prv_amfh_handle_request();
		/* rx_waiting > 0 ==> no wait, loop again to handler */
		prv_update_unhandled_rx();
		wait_event_interruptible(amfh->handler_queue,
					 atomic_read(&amfh->rx_interrupt) ||
					 amfh->rx_waiting ||
					 kthread_should_stop());
	}
	AMFH_TRACE_EXIT;
	return 0;
}

int iwl_amfh_init(struct iwl_al_amfh_config *al_config,
	struct iwl_trans *trans)
{
	AMFH_TRACE_ENTER;
	BUG_ON(!al_config);
	BUG_ON(!trans);
	amfh->init_cfg = al_config;
	amfh->trans = trans;
	amfh->dev = trans->dev;
	init_waitqueue_head(&amfh->handler_queue);

	AMFH_TRACE_EXIT;
	return 0;
}

void iwl_amfh_free(void)
{
	AMFH_TRACE_ENTER;
	AMFH_TRACE_EXIT;
}

int iwl_amfh_start(struct iwl_amfh_config *amfh_config)
{
	int ret;
	AMFH_TRACE_ENTER;

	BUG_ON(amfh_config == NULL);
	amfh->amfh_config = amfh_config;

	amfh->resume = 0;
	amfh->rx_waiting = 0;
	amfh->curr_burst_frames = 0;
	atomic_set(&amfh->rx_interrupt, 0);

	/* RX ring initialization */
	ret = iwl_amfh_rxq_init();
	if (ret) {
		IWL_AMFH_LOG_ERR("Failed to initialize RXQ");
		goto exit;
	}

	amfh->workloop_thread = kthread_run(prv_amfh_idle, NULL,
		"AMFH workloop");

	if (IS_ERR(amfh->workloop_thread)) {
		IWL_AMFH_LOG_ERR("Could not create AMFH workloop thread task");
		ret = -EFAULT;
		goto err_kthread;
	}

	iwl_amfh_rxq_start();
	goto exit;

err_kthread:
	iwl_amfh_rxq_free();
exit:
	AMFH_TRACE_EXIT_RET(ret);
	return ret;
}

int iwl_amfh_stop(void)
{
	int ret = 0;
	AMFH_TRACE_ENTER;

	BUG_ON(task_is_dead(amfh->workloop_thread));

	ret = kthread_stop(amfh->workloop_thread);

	iwl_amfh_rxq_stop();
	iwl_amfh_rxq_free();

	AMFH_TRACE_EXIT_RET(ret);
	return ret;
}

int iwl_amfh_resume(void)
{
	amfh->resume = true;
	IWL_AMFH_LOG("Waking up AMFH thread");
	wake_up_interruptible(&amfh->handler_queue);
	return 0;
}

u32 iwl_amfh_rx_handler(void)
{
	IWL_AMFH_LOG("Requesting RX processing");
	atomic_set(&amfh->rx_interrupt, 1);
	wake_up_interruptible(&amfh->handler_queue);
	return 0;
}
