////////////////////////////////////////////////////////////////////////////////
//3456789 123456789 123456789 123456789 123456789 123456789 123456789 123456789
//      10        20        30        40        50        60        70        80
//
// test-em-px-cm-inch
//
// test-em-px-cm-inch.c - test em/px/cm/inch conversions
//
// Copyright 2012 Canonical Ltd.
//
// Authors:
//    Mirco "MacSlow" Mueller <mirco.mueller@canonical.com>
//
// This program is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License version 3, as published
// by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranties of
// MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
// PURPOSE.  See the GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License along
// with this program.  If not, see <http://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <math.h>
#include <cairo/cairo.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#define GNOME_DESKTOP_SCHEMA "org.gnome.desktop.interface"
#define GSETTINGS_FONT_KEY   "font-name"

// Canonical colours
// 119  41  83	 Canonical Aubergine
// 255 255 255	 White
//  94  39  80	 Canonical Dark Aubergine
// 119  33 111	 Canonical Light Aubergine
// 174 167 159	 Warm Grey
//  51  51  51	 Cool Grey

// Ubuntu colours
// 221  72  20	 Ubuntu Orange
//  44   0  30	 Server Dark Aubergine
// 174 167 159	 Warm Grey
//  51  51  51	 Cool Grey
//   0   0   0	 Black
// 255 255 255	 White

gdouble pixels_per_em   = 0.0;
gdouble pixels_per_inch = 0.0;
gdouble pixels_per_cm   = 0.0;

// conversion prototypes
gint    em2px(gdouble em);
gdouble em2cm(gdouble em);
gdouble em2inch(gdouble em);
gdouble px2em(gint px);
gdouble px2cm(gint px);
gdouble px2inch(gint px);
gdouble cm2em(gdouble cm);
gint    cm2px(gdouble cm);
gdouble inch2em(gdouble inch);
gint    inch2px(gdouble inch);

// utility-functions ///////////////////////////////////////////////////////////

gdouble
get_font_size()
{
	GSettings*            gnome_settings = NULL;
	gchar*                font_name      = NULL;
	PangoFontDescription* desc           = NULL;
	gdouble               points         = 0.0f;

	gnome_settings = g_settings_new(GNOME_DESKTOP_SCHEMA);
	font_name = g_settings_get_string(gnome_settings, GSETTINGS_FONT_KEY);
	desc = pango_font_description_from_string(font_name);
	points = (gdouble) pango_font_description_get_size(desc) /
		 (gdouble) PANGO_SCALE;
	pango_font_description_free(desc);
	g_free((gpointer) font_name);
	g_object_unref(gnome_settings);

	return points;
}

gdouble
align(gdouble val, gboolean odd)
{
  gdouble fract = val - (gint) val;

  if (odd)
  {
    // for strokes with an odd line-width
    if (fract != 0.5)
      return (gdouble) ((gint) val + 0.5);
    else
      return val;
  }
  else
  {
    // for strokes with an even line-width
    if (fract != 0.0)
      return (gdouble) ((gint) val);
    else
      return val;
  }
}

void
draw_text(cairo_t* cr, GtkWidget* widget, gint x, gint y, gint width, gint height, char* text)
{
	PangoLayout*          layout = NULL;
	PangoFontDescription* desc   = NULL;

	layout = pango_cairo_create_layout(cr);
	desc = pango_font_description_from_string("Ubuntu 10.0");
	pango_layout_set_font_description(layout, desc);
	pango_font_description_free(desc);

	pango_layout_set_width(layout, width);
	pango_layout_set_height(layout, height);
	pango_layout_set_markup(layout, text, -1);

	pango_cairo_context_set_font_options(pango_layout_get_context(layout),
					     gdk_screen_get_font_options(gtk_widget_get_screen(widget)));

	cairo_move_to(cr, x, y);
	pango_cairo_show_layout(cr, layout);
	g_object_unref(layout);
}

void
draw_rulers(cairo_t* cr, GtkWidget* widget, gint x, gint y, gint h, gint gap)
{
	// cm-ruler background
	cairo_rectangle(cr,
			align(x, FALSE),
			align(y, FALSE),
			cm2px(6.0),
			h);

	// inch-ruler background
	cairo_rectangle(cr,
			align(x, FALSE),
			align(y + h + gap, FALSE),
			inch2px(2.0),
			h);
	cairo_set_source_rgb(cr, 255.0, 255.0, 255.0);
	cairo_fill(cr);

	// cm-ruler label
	cairo_set_source_rgb(cr, 174.0/255.0, 167.0/255.0, 159.0/255.0);
	draw_text(cr, widget, x + cm2px(0.2), y + cm2px(0.1), cm2px(3.0), -1, "<big><b>6cm</b></big>\0");

	// inch-ruler label
	draw_text(cr, widget, x + cm2px(0.2), y + h + gap + cm2px(0.1), cm2px(3.0), -1, "<big><b>2inch</b></big>\0");

	// cm-ruler tick-marks
	cairo_set_line_width(cr, 1.0);
	gdouble x_tick = 0.0;
	gdouble y_tick = 0.0;
	gdouble y_step = 0.0;
	gdouble step   = 0.0;
	gint    offset = 0;
	for(offset = 0; offset <= 60; offset++)
	{
		if(offset % 10 == 0)
			step = cm2px(0.2);
		else if (offset % 5)
			step = cm2px(0.05);
		else
			step = cm2px(0.1);

		x_tick = align(x + cm2px(offset * 0.1), TRUE);
		y_tick = align(y + h - step, TRUE);
		y_step = align(y + h, TRUE);
		cairo_move_to(cr, x_tick, y_tick);
		cairo_line_to(cr, x_tick, y_step);
	}
	cairo_set_source_rgb(cr, 0.0, 0.0, 0.0);
	cairo_stroke(cr);

	// inch-ruler tick-marks
	x_tick = 0.0;
	y_tick = 0.0;
	y_step = 0.0;
	step   = 0.0;
	offset = 0;
	for(offset = 0; offset <= 20; offset++)
	{
		if(offset % 10 == 0)
			step = cm2px(0.2);
		else if (offset % 5)
			step = cm2px(0.05);
		else
			step = cm2px(0.1);

		x_tick = align(x + inch2px(offset * 0.1), TRUE);
		y_tick = align(y + 2 * h + gap - step, TRUE);
		y_step = align(y + 2 * h + gap, TRUE);
		cairo_move_to(cr, x_tick, y_tick);
		cairo_line_to(cr, x_tick, y_step);
	}
	cairo_set_source_rgb(cr, 0.0, 0.0, 0.0);
	cairo_stroke(cr);
}

void
draw_em_rects(cairo_t* cr, gint x, gint y, gint gap)
{
	cairo_set_line_width(cr, 1.0);
	cairo_rectangle(cr,
			align(x, TRUE),
			align(y, TRUE),
			em2px(1.0),
			em2px(1.0));
	cairo_rectangle(cr,
			align(x, TRUE),
			align(y + em2px(1.0) + gap, TRUE),
			em2px(2.0),
			em2px(2.0));
	cairo_rectangle(cr,
			align(x, TRUE),
			align(y + em2px(3.0) + 2 * gap, TRUE),
			em2px(3.0),
			em2px(3.0));
	cairo_set_source_rgb(cr, 51.0/255.0, 51.0/255.0, 51.0/255.0);
	cairo_stroke(cr);
}

void
draw_info_text(cairo_t* cr, GtkWidget* widget, gint x, gint y, gint width, gint height)
{
	PangoLayout*          layout = NULL;
	PangoFontDescription* desc   = NULL;

	layout = pango_cairo_create_layout(cr);
	//text_font = 
	desc = pango_font_description_from_string("Ubuntu 10.0");
	//g_free((gpointer) text_font_face);
	pango_layout_set_font_description(layout, desc);
	pango_font_description_free(desc);

	pango_layout_set_width(layout, width);
	pango_layout_set_height(layout, height);
	pango_layout_set_markup(layout, "<b>Font:</b> Ubuntu 10\n<b>DPI:</b> 145\0", -1);

	pango_cairo_context_set_font_options(pango_layout_get_context(layout),
					     gdk_screen_get_font_options(gtk_widget_get_screen(widget)));

	cairo_set_source_rgb(cr, 1.0, 1.0, 1.0);
	cairo_move_to(cr, x, y);
	pango_cairo_show_layout(cr, layout);
	g_object_unref(layout);
}

// unit-conversion /////////////////////////////////////////////////////////////

gint
em2px(gdouble em)
{
	return (gint) roundl(em * pixels_per_em);
}

gdouble
em2cm(gdouble em)
{
	return (em2px(em) / pixels_per_cm);
}

gdouble
em2inch(gdouble em)
{
	return (em2px(em) / pixels_per_inch);
}

gdouble
px2em(gint px)
{
	return ((gdouble) px / pixels_per_em);
}

gdouble
px2cm(gint px)
{
	return ((gdouble) px / pixels_per_cm);
}

gdouble
px2inch(gint px)
{
	return ((gdouble) px / pixels_per_inch);
}

gdouble
cm2em(gdouble cm)
{
	return (px2em(cm2px(cm)));
}

gint
cm2px(gdouble cm)
{
	return (gint) roundl(cm * pixels_per_cm);
}

gdouble
inch2em(gdouble inch)
{
	return (px2em(inch2px(inch)));
}

gint
inch2px(gdouble inch)
{
	return (gint) roundl(inch * pixels_per_inch);
}

// signal-callbacks ////////////////////////////////////////////////////////////

gboolean
draw_handler(GtkWidget* widget,
             cairo_t*   cr,
             gpointer   data)
{
	cairo_pattern_t* pattern = NULL;
	gint             width   = gtk_widget_get_allocated_width(widget);
	gint             height  = gtk_widget_get_allocated_height(widget);

	cairo_scale(cr, 1.0, 1.0);
	cairo_set_operator(cr, CAIRO_OPERATOR_CLEAR);
	cairo_paint(cr);
	cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
	cairo_rectangle(cr, 0.0, 0.0, width, height);
	pattern = cairo_pattern_create_linear(0.0,
					      0.0,
					      0.0,
					      (gdouble) height);
	if (pattern)
	{

		cairo_pattern_add_color_stop_rgb(pattern,
						 0.0,
						 221.0 / 255.0,
						 72.0 / 255.0,
						 20.0 / 255.0);
		cairo_pattern_add_color_stop_rgb(pattern,
						 1.0,
						 174.0 / 255.0,
						 167.0 / 255.0,
						 159.0 / 255.0);
		cairo_set_source(cr, pattern);
		cairo_fill(cr);
		cairo_pattern_destroy(pattern);
	}
	else
	{
		cairo_set_source_rgba(cr, 1.0, 0.5, 0.25, 1.0);
		cairo_fill(cr);
	}

	gdouble margin_cm = 0.5;
	gdouble gap_cm    = 0.5;
	gdouble height_cm = 0.5;

	draw_rulers(cr,
		    widget,
		    cm2px(margin_cm),
		    height - cm2px(margin_cm) - cm2px(gap_cm) - 2 * cm2px(height_cm),
		    cm2px(height_cm),
		    cm2px(gap_cm));
	draw_em_rects(cr,
		      cm2px(margin_cm),
		      cm2px(margin_cm),
		      cm2px(gap_cm));
	draw_info_text(cr,
		       widget,
		       width / 2,
		       cm2px(margin_cm),
		       width / 2,
		       -2);

	return TRUE;
}

void
destroy_handler(GtkWidget* object,
                gpointer   data)
{
	gtk_main_quit ();
}
 
////////////////////////////////////////////////////////////////////////////////

int
main(int    argc,
     char** argv)
{
	g_type_init ();
	gtk_init (&argc, &argv);

	GdkScreen*   screen        = gdk_screen_get_default(); // not ref'ed
	gint         max_monitors  = gdk_screen_get_n_monitors (screen);
	gint         monitor       = 0;
	GdkRectangle geo           = {0, 0, 0, 0};
	gint         width         = 0;
	gint         height        = 0;
	gdouble      dpi           = 0.0;
	gdouble      points        = 0.0;
	gdouble      dpi_vert      = 0.0;
	gdouble      dpi_horiz     = 0.0;
	GtkWidget*   window        = NULL;

	points = get_font_size();
	g_print("font-size: %3.2f\n", points);

	for (monitor = 0; monitor < max_monitors; monitor++)
	{
		width  = gdk_screen_get_monitor_width_mm (screen, monitor);
		height = gdk_screen_get_monitor_height_mm (screen, monitor);
		if (width == -1 || height == -1)
		{
			GdkDisplay* display     = gdk_screen_get_display(screen); // not ref'ed
			Display*    display_x11 = gdk_x11_display_get_xdisplay(display); // not ref'ed

			g_print("Could not determine geometry in mm of each monitor. "
				"You might happen to run a TwinView-setup.\n");
			g_print("screen 0: %dmm * %dmm, %dpx * %dpx\n",
				DisplayWidthMM(display_x11, 0),
				DisplayHeightMM(display_x11, 0),
				DisplayWidth(display_x11, 0),
				DisplayHeight(display_x11, 0));

			// assume a default DPI of 96.0
			dpi = 96.0;
			g_print("Assuming a screen-DPI of 96.0\n");

			// since we can't get the physical size only iterate once
			monitor = max_monitors;
		}
		else
		{
			gdk_screen_get_monitor_geometry (screen, monitor, &geo);
			g_print("monitor %d: %dmm * %dmm, %dpx * %dpx\n",
				monitor,
				width,
				height,
				geo.width,
				geo.height);

			// determine current system DPI
			// remember: 1 inch == 2.54 cm == 25.4 mm
			dpi_horiz  = (double) geo.width / ((double) width / 25.4);
			dpi_vert = (double) geo.height / ((double) height / 25.4);
			if(dpi_horiz != dpi_vert)
				dpi = (dpi_horiz + dpi_vert) / 2.0;
			else
				dpi = dpi_horiz;
		}

		g_print("determined screen-DPI: %3.2f\n\n", dpi);
		pixels_per_em = points / 72.0f * dpi;
		pixels_per_inch = dpi;
		pixels_per_cm = dpi / 2.54;

		g_print("On this monitor these measurements apply:\n\n");
		g_print("\t1em   = %dpx\n",     em2px(1.0));
		g_print("\t1em   = %4.3fcm\n",  em2cm(1.0));
		g_print("\t1em   = %4.3f\"\n",  em2inch(1.0));
		g_print("\t100px = %4.3fem\n",  px2em(100));
		g_print("\t100px = %4.3fcm\n",  px2cm(100));
		g_print("\t100px = %4.3f\"\n",  px2inch(100));
		g_print("\t1cm   = %4.3fem\n",  cm2em(1.0));
		g_print("\t1cm   = %dpx\n",     cm2px(1.0));
		g_print("\t1\"    = %4.3fem\n", inch2em(1.0));
		g_print("\t1\"    = %dpx\n",    inch2px(1.0));

		g_print("\nSmallest possible errors:\n\n");
		g_print("\t1px = %4.3fem (%3.2f %%)\n", px2em(1), px2em(1)/1.0);
		g_print("\t1px = %4.3fcm (%3.2f %%)\n", px2cm(1), px2cm(1)/1.0);
		g_print("\t1px = %4.3f\" (%3.2f %%)\n", px2inch(1), px2inch(1)/1.0);
	}

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), "EM/pixel/cm/inch Test");
	gtk_window_move(GTK_WINDOW(window),
			(gint) roundl(cm2px(2.0)),
			(gint) roundl(cm2px(2.0)));
	gtk_widget_set_size_request(window,
				    (gint) roundl(cm2px(8.0)),
				    (gint) roundl(cm2px(6.0)));
	g_signal_connect (G_OBJECT (window),
			  "draw",
			  G_CALLBACK (draw_handler),
			  NULL);
	g_signal_connect (G_OBJECT (window),
			  "destroy",
			  G_CALLBACK (destroy_handler),
			  NULL);
	gtk_widget_show_all(window);

	gtk_main();
	
	return 0;
}

