
#include "ENMApp.h"

#if defined(__WXMSW__)

#include "Win32WlanScanner.h"
#include "DeviceController.h"
#include "ENMUtil.h"
#include "wlan.h"
#include "wpa/sha1.h"
#include <wx/msw/registry.h>
#include <fstream>
using namespace std;

// Max value of APs researched once
static const int MAX_AP_NUMBER = 32;
static const int NUMBER_BSSIDS = 100;
static const int BSSID_LIST_LENGTH = sizeof(NDIS_802_11_BSSID_LIST_EX) * NUMBER_BSSIDS;

#if defined(__WXMSW__)
static wxString logFile(_T("C:\\WlanScanner.log"));
#else
static wxString logFile(_T("/tmp/enm/WlanScanner.log"));
#endif

Win32WlanScanner::Win32WlanScanner() : WlanScanner()
{
    memset(m_openProfile, 0, PROFILE_MAX_SIZE);
    memset(m_wepProfile, 0, PROFILE_MAX_SIZE);
    memset(m_wpaProfile, 0, PROFILE_MAX_SIZE);
    m_pBssidList = new char[BSSID_LIST_LENGTH];
}

Win32WlanScanner::~Win32WlanScanner()
{
    CloseWlanHandle();
    delete[] m_pBssidList;
}

bool Win32WlanScanner::OpenWlanHandle()
{
    DWORD dwNegotiatedVersion = 0;
	return m_isHandleOpen = WlanOpenHandle(1, NULL, &dwNegotiatedVersion, &m_hClientHandle) == ERROR_SUCCESS;
}

bool Win32WlanScanner::CloseWlanHandle()
{
    if (!m_isHandleOpen)
        return false;

    if (WlanCloseHandle(m_hClientHandle, NULL) == ERROR_SUCCESS)
    {
        m_isHandleOpen = false;
        return true;
    }

    return false;
}

bool Win32WlanScanner::IsHandleOpen()
{
    return m_isHandleOpen;
}

WlanResult Win32WlanScanner::StartWlanScan()
{
    if (!m_isHandleOpen)
        return WLAN_INVALID_HANDLE;

    PWLAN_INTERFACE_INFO_LIST pInterfaceList = NULL;
	if (WlanEnumInterfaces(m_hClientHandle, NULL, &pInterfaceList) != ERROR_SUCCESS)
	{
		return WLAN_ERROR;
	}
	if (pInterfaceList->dwNumberOfItems == 0)
	{
	    // no WLAN interfaces
		WlanFreeMemory(pInterfaceList);
		return WLAN_NO_INTERFACE;
	}
	PWLAN_INTERFACE_INFO pInterface = &pInterfaceList->InterfaceInfo[0];

	DWORD result = WlanScan(m_hClientHandle, &pInterface->InterfaceGuid, NULL, NULL, NULL);
	WlanFreeMemory(pInterfaceList);
	if (result == ERROR_SUCCESS)
        return WLAN_SUCCESS;
    else
        return WLAN_ERROR;
}

bool Win32WlanScanner::IsWlanInterfaceAvailable()
{
    if (!m_isHandleOpen)
        return false;

    PWLAN_INTERFACE_INFO_LIST pInterfaceList = NULL;
	if (WlanEnumInterfaces(m_hClientHandle, NULL, &pInterfaceList) != ERROR_SUCCESS)
	{
		return false;
	}
	bool result = pInterfaceList->dwNumberOfItems > 0;
	WlanFreeMemory(pInterfaceList);
	return result;
}

wxString Win32WlanScanner::StringToHex(const char* str, size_t length)
{
    wxString result;
    if ( str != NULL )
    {
        for (size_t i = 0; i < length; i ++)
        {
            result += wxString::Format(_T("%02X"), str[i]);
        }
    }
    return result;
}

bool Win32WlanScanner::UpdateCurrentAP(APStatus& apStatus)
{
    ENMUtil::Log(logFile, _T("Enter Win32WlanScanner::UpdateCurrentAP"));
    bool result = GetCurrentConnectedNetwork(&apStatus);
    ENMUtil::Log(logFile, wxString::Format(_T("Return Win32WlanScanner::UpdateCurrentAP: %d\n"), result));
    return result;
}

WlanResult Win32WlanScanner::RegisterWlanNotification(void* pData, bool enable)
{
    if (!m_isHandleOpen)
        return WLAN_INVALID_HANDLE;

    DWORD source = WLAN_NOTIFICATION_SOURCE_ALL;
    if (!enable)
        source = WLAN_NOTIFICATION_SOURCE_NONE;
    DWORD result = WlanRegisterNotification(m_hClientHandle, source, TRUE, OnWlanNotify, pData, NULL, NULL);
    if (result == ERROR_SUCCESS)
        return WLAN_SUCCESS;
    else
        return WLAN_REGISTER_ERROR;
}

bool Win32WlanScanner::IsWlanAdapterGatewayAvailable()
{
    if (!m_isHandleOpen)
        return false;

    PWLAN_INTERFACE_INFO_LIST pInterfaceList = NULL;
	if (WlanEnumInterfaces(m_hClientHandle, NULL, &pInterfaceList) != ERROR_SUCCESS)
	{
		return false;
	}
	if (pInterfaceList->dwNumberOfItems == 0)
	{
		WlanFreeMemory(pInterfaceList);
		return false;
	}

	PWLAN_INTERFACE_INFO pInterface = &pInterfaceList->InterfaceInfo[0];
    wxString strGuid = GuidToString(pInterface->InterfaceGuid);
    WlanFreeMemory(pInterfaceList);
    wxRegKey keyConn(_T("HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Services\\") + strGuid + _T("\\Parameters\\Tcpip"));
    if ( !keyConn.Exists() )
    {
        return false;
    }

    wxString strGateway;
    wxString strDhcpGateway;
    if ( keyConn.HasValue(_T("DhcpDefaultGateway")) )
    {
        keyConn.QueryValue(_T("DhcpDefaultGateway"), strDhcpGateway);
    }
    if ( keyConn.HasValue(_T("DefaultGateway")) )
    {
        keyConn.QueryValue(_T("DefaultGateway"), strGateway);
    }
    return strGateway.Length() > 0 || strDhcpGateway.Length() > 0;
}

void Win32WlanScanner::LoadProfiles()
{
    wxString pfOpen(::wxGetApp().strAppPath + _T("profile\\open.xml"));
    wxString pfWep(::wxGetApp().strAppPath + _T("profile\\wep-128.xml"));
    wxString pfWpa(::wxGetApp().strAppPath + _T("profile\\wpa-psk.xml"));

    ifstream open(pfOpen.c_str());
    open.read(m_openProfile, PROFILE_MAX_SIZE);
    open.close();
    m_open = wxString::FromAscii(m_openProfile);

    ifstream wep(pfWep.c_str());
    wep.read(m_wepProfile, PROFILE_MAX_SIZE);
    wep.close();
    m_wep = wxString::FromAscii(m_wepProfile);

    ifstream wpa(pfWpa.c_str());
    wpa.read(m_wpaProfile, PROFILE_MAX_SIZE);
    wpa.close();
    m_wpa = wxString::FromAscii(m_wpaProfile);
}

bool Win32WlanScanner::GetBssidList(HANDLE handle, NDIS_802_11_BSSID_LIST_EX* pBssidList)
{
    if ( handle == INVALID_HANDLE_VALUE || pBssidList == NULL )
    {
        return false;
    }

    DWORD oidcode = OID_802_11_BSSID_LIST;
    ULONG bytesreturned;
    DWORD dicRet = DeviceIoControl(handle, IOCTL_NDIS_QUERY_GLOBAL_STATS, &oidcode, sizeof(oidcode), (ULONG *) pBssidList, sizeof(NDIS_802_11_BSSID_LIST_EX) * NUMBER_BSSIDS, &bytesreturned, NULL);
    DWORD dicError = GetLastError();

    return dicRet != 0;
}

#if FALSE // deprecated!
NDIS_802_11_BSSID_LIST_EX* Win32WlanScanner::GetBssidList(HANDLE handle)
{
    if ( handle == INVALID_HANDLE_VALUE )
    {
        return NULL;
    }

    ULONG oidcode;
    ULONG bytesreturned;
    NDIS_802_11_BSSID_LIST_EX* pBssidList = (NDIS_802_11_BSSID_LIST_EX *) VirtualAlloc(NULL, sizeof(NDIS_802_11_BSSID_LIST_EX) * NUMBER_BSSIDS, MEM_RESERVE | MEM_COMMIT, PAGE_READWRITE);
    if( pBssidList == NULL )
    {
        return NULL;
    }

    memset(pBssidList, 0, sizeof(NDIS_802_11_BSSID_LIST_EX) * NUMBER_BSSIDS);
    oidcode = OID_802_11_BSSID_LIST;
    DWORD dicRet = DeviceIoControl(handle, IOCTL_NDIS_QUERY_GLOBAL_STATS, &oidcode, sizeof(oidcode), (ULONG *) pBssidList, sizeof(NDIS_802_11_BSSID_LIST_EX) * NUMBER_BSSIDS, &bytesreturned, NULL);
    DWORD dicError = GetLastError();
//    ofstream fout;
//    fout.open("C:\\diclog.txt", ios::app);
//    fout << "DeviceIoControl return: " << dicRet << endl;
    if( dicRet == 0 )
    {
//        fout << "DeviceIoControl error: " << dicError << endl;
        VirtualFree(pBssidList, sizeof(NDIS_802_11_BSSID_LIST_EX) * NUMBER_BSSIDS, MEM_RELEASE);
        pBssidList = NULL;
    }
//    fout.close();
    return pBssidList;
}
#endif

WlanResult Win32WlanScanner::ScanWlanNetwork(AccessPointList* pList)
{
    if (pList == NULL)
        return WLAN_INVALID_PARAMETER;

    bool complete = true;
    AccessPointList listFromWifi;
    memset(&listFromWifi, 0, sizeof(AccessPointList));
    WlanResult resultFromWifi = GetNetworkFromWifi(&listFromWifi);
    if (resultFromWifi != WLAN_SUCCESS)
    {
        return resultFromWifi;
    }

    AccessPointList listFromNdis;
    memset(&listFromNdis, 0, sizeof(AccessPointList));
    WlanResult resultFromNdis = GetNetworkFromNdis(&listFromNdis);
    complete = resultFromNdis == WLAN_SUCCESS;

    if ( !complete || listFromNdis.nCount == 0 )
    {
        CopyAccessPointList(pList, &listFromWifi);
        return WLAN_DATA_LOST;
    }

	for ( int i = 0; i < listFromNdis.nCount; i++ )
	{
		for ( int j = 0; j < listFromWifi.nCount; j++ )
		{
		    if ( listFromNdis.apList[i].ssid == listFromWifi.apList[j].ssid )
		    {
		        wmemcpy(&listFromNdis.apList[i].name[0], &listFromWifi.apList[j].name[0], AP_SSID_MAX_LENGTH);
#if GETTYPE_THROUGH_WLANAPI
		        listFromNdis.apList[i].type = listFromWifi.apList[j].type;
#endif
		        listFromNdis.apList[i].bssType = listFromWifi.apList[j].bssType;
		        listFromNdis.apList[i].auth = listFromWifi.apList[j].auth;
		        listFromNdis.apList[i].signalQuality = listFromWifi.apList[j].signalQuality;
		    }
		}
	}
	CopyAccessPointList(pList, &listFromNdis);
    return WLAN_SUCCESS;
}

bool Win32WlanScanner::CopyAccessPointList(AccessPointList* pNewList, AccessPointList* pSrcList)
{
    if (pSrcList == NULL)
        return false;

    if (pNewList == pSrcList)
        return false;

    pNewList->nCount = 0;
    if (pSrcList->nCount == 0)
        return true;

    memcpy(pNewList, pSrcList, sizeof(AccessPointList));
//
//    size_t count = 0;
//    char* copied = new char[pSrcList->nCount];
//    memset(copied, 0, pSrcList->nCount);
//    for ( int i = 0 ; i < pSrcList->nCount ; i++ )
//    {
//        if ( copied[i] == 1 ) continue;
//        pNewList->apList[count] = pSrcList->apList[i];
//        count ++;
//        for ( int j = i + 1 ; j < pSrcList->nCount ; j++ )
//        {
//            if (pSrcList->apList[i].ssid == pSrcList->apList[j].ssid)
//            {
//                copied[j] = 1;
//            }
//        }
//    }
//    delete[] copied;
//    pNewList->nCount = count;
    return true;
}

WlanResult Win32WlanScanner::ConnectWlanNetwork(const APStatus* ap, const wxString& key, int keyIndex)
{
    DWORD result = 0;

    if (ap == NULL)
        return WLAN_INVALID_PARAMETER;

    if (!m_isHandleOpen)
        return WLAN_INVALID_HANDLE;

    PWLAN_INTERFACE_INFO_LIST pInterfaceList = NULL;
	if (WlanEnumInterfaces(m_hClientHandle, NULL, &pInterfaceList) != ERROR_SUCCESS)
	{
		return WLAN_ERROR;
	}
	if (pInterfaceList->dwNumberOfItems == 0)
	{
		WlanFreeMemory(pInterfaceList);
		return WLAN_NO_INTERFACE;
	}
	PWLAN_INTERFACE_INFO pInterface = &pInterfaceList->InterfaceInfo[0];

    char buf[PROFILE_MAX_SIZE];
    memset(buf, 0, PROFILE_MAX_SIZE);
    wxString ap_name = wxString::FromUTF8((const char*)&ap->ssid.ucSsid, ap->ssid.length);
    wxString hex = StringToHex((const char*)ap->ssid.ucSsid, ap->ssid.length);

    wxString profile;
    if (ap->auth == WLAN_SECURITY_WEP)
    {
        profile = wxString::Format(m_wep, ap_name, hex, ap_name, key, keyIndex - 1);
    }
    else if (ap->auth == WLAN_SECURITY_NONE)
    {
        profile = wxString::Format(m_open, ap_name, hex, ap_name);
    }
    else if (ap->auth == WLAN_SECURITY_WPA_PSK)
    {
        string s = ENMUtil::WxStringToString(key);
        const char* passphrase = s.c_str();
        unsigned char psk[32] = { 0 };
        char encoded[65] = { 0 };
        pbkdf2_sha1(passphrase, (const char*)ap->ssid.ucSsid, ap->ssid.length, 4096, psk, 32);
        char* p = &encoded[0];
        for ( size_t i = 0; i < 32; i++ )
        {
            sprintf(p, "%02X", psk[i]);
            p += 2;
        }
        profile = wxString::Format(m_wpa, ap_name, hex, ap_name, wxString::FromUTF8(encoded));
    }
    else
    {
        WlanFreeMemory(pInterfaceList);
        return WLAN_UNKNOWN_AUTH;
    }

    DWORD reasonCode = 0;
	result = WlanSetProfile(m_hClientHandle, &pInterface->InterfaceGuid, 0, profile.fn_str(), NULL, TRUE, NULL, &reasonCode);
    if (result != ERROR_SUCCESS)
    {
        if (reasonCode != 0)
        {
            wchar_t reason[1024];
            wmemset(reason, 0, 1024);
            WlanReasonCodeToString(reasonCode, 1024, reason, NULL);
        }
        return WLAN_PROFILE_ERROR;
    }

    WLAN_CONNECTION_PARAMETERS param;
    param.wlanConnectionMode = wlan_connection_mode_profile;
    param.strProfile = ap_name.fn_str();
    param.pDot11Ssid = (PDOT11_SSID)malloc(sizeof(DOT11_SSID));
    param.pDot11Ssid->uSSIDLength = ap->ssid.length;
    memcpy(param.pDot11Ssid->ucSSID, ap->ssid.ucSsid, AP_SSID_MAX_LENGTH);
    param.pDesiredBssidList = NULL;
    memcpy(&param.dot11BssType, &ap->bssType, sizeof(DOT11_BSS_TYPE));
    param.dwFlags = 0;
    result = WlanConnect(m_hClientHandle, &pInterface->InterfaceGuid, &param, NULL);
    WlanFreeMemory(pInterfaceList);
    if (result == ERROR_SUCCESS)
        return WLAN_SUCCESS;
    else
        return WLAN_ERROR;
}

WlanResult Win32WlanScanner::GetNetworkFromWifi(AccessPointList* pList)
{
    if (pList == NULL)
        return WLAN_INVALID_PARAMETER;

    if (!m_isHandleOpen)
        return WLAN_INVALID_HANDLE;

    PWLAN_INTERFACE_INFO_LIST pInterfaceList = NULL;
	if ( WlanEnumInterfaces(m_hClientHandle, NULL, &pInterfaceList) != ERROR_SUCCESS )
	{
		return WLAN_ERROR;
	}
	if ( pInterfaceList->dwNumberOfItems == 0 )
	{
	    // no WLAN interfaces
		WlanFreeMemory(pInterfaceList);
		return WLAN_NO_INTERFACE;
	}
	PWLAN_INTERFACE_INFO pInterface = &pInterfaceList->InterfaceInfo[0];

	PWLAN_AVAILABLE_NETWORK_LIST pNetworkList = NULL;
	if ( WlanGetAvailableNetworkList(m_hClientHandle, &pInterface->InterfaceGuid, WLAN_AVAILABLE_NETWORK_INCLUDE_ALL_MANUAL_HIDDEN_PROFILES, NULL, &pNetworkList) != ERROR_SUCCESS )
	{
		if (pNetworkList != NULL)
		{
			WlanFreeMemory(pNetworkList);
		}
        WlanFreeMemory(pInterfaceList);
		return WLAN_SCAN_ERROR;
	}
	if (pNetworkList->dwNumberOfItems == 0)
	{
	    WlanFreeMemory(pInterfaceList);
		WlanFreeMemory(pNetworkList);
		return WLAN_SUCCESS;
	}

    int detected = 0;
    memset(pList->apList, 0, sizeof(APStatus) * 32);
    for (size_t j = 0; j < pNetworkList->dwNumberOfItems; j ++)
    {
        APStatus ap;
        memset(&ap, 0, sizeof(APStatus));
        ENMWlanGetAPStatus(&pNetworkList->Network[j], &ap);
        if ( ap.ssid.length > 0 || ap.signalQuality > 0 || ap.channel > 0 )
        {
            pList->apList[detected++] = ap;
        }
        if ( detected >= MAX_AP_NUMBER )
        {
            break;
        }
    }
    pList->nCount = detected;

    WlanFreeMemory(pInterfaceList);
    WlanFreeMemory(pNetworkList);

    return WLAN_SUCCESS;
}

// try to find 802.11n information
bool Win32WlanScanner::Find_80211n_IE(UCHAR *ucIE, unsigned int uiSize)
{
	unsigned int i = 0;
	bool result = false;
	while ( (i < uiSize) && !result )
	{
		if ( ucIE[i] == 45 )   // 45 = HT Capabilities IE which is only in 802.11n capable beacons.
		{
			result = true;
		}
		if ( !result )
		{
		    i += (UCHAR)(ucIE[i + 1] + 2);
		}
	}
	return result;
}

WlanResult Win32WlanScanner::GetNetworkFromNdis(AccessPointList* pList)
{
    if ( pList == NULL )
        return WLAN_INVALID_PARAMETER;

    pList->nCount = 0;
    Win32NetworkDeviceController* pDevCtrl = Win32NetworkDeviceController::GetCurrentInstance();
    NetworkDeviceInfoArray devArray = pDevCtrl->EnumNetworkDevice();
    if ( devArray.IsEmpty() )
    {
        return WLAN_SCAN_ERROR;
    }
    size_t wlanDevIndex = wxNOT_FOUND;
    for ( size_t i = 0; i < devArray.GetCount(); i++ )
    {
        NetworkDeviceInfo dev = devArray.Item(i);
        if ( dev.GetDeviceType() == ADAPTER_WIREDLESS )
        {
            wlanDevIndex = i;
            break;
        }
    }
    if ( wlanDevIndex == wxNOT_FOUND )
    {
        devArray.Clear();
        return WLAN_NO_INTERFACE;
    }

    HANDLE devHandle = pDevCtrl->OpenNetworkDevice(devArray.Item(wlanDevIndex));
    if ( devHandle == INVALID_HANDLE_VALUE )
    {
        devArray.Clear();
        return WLAN_INVALID_HANDLE;
    }

    memset(m_pBssidList, 0, BSSID_LIST_LENGTH);
    NDIS_802_11_BSSID_LIST_EX* pBssidList = (NDIS_802_11_BSSID_LIST_EX*)m_pBssidList;
    if ( !GetBssidList(devHandle, pBssidList) || pBssidList->NumberOfItems == 0 )
    {
        devArray.Clear();
        pDevCtrl->CloseNetworkDevice(devHandle);
        return WLAN_SCAN_ERROR;
    }

    pList->nCount = (pBssidList->NumberOfItems < MAX_AP_NUMBER) ? pBssidList->NumberOfItems : MAX_AP_NUMBER;
    memset(pList->apList, 0, sizeof(APStatus) * pList->nCount);
    for ( int i = 0; i < pList->nCount; i++ )
    {
        size_t temp = i;
        PNDIS_WLAN_BSSID_EX cpSsid = pBssidList->Bssid;
        ULONG ulIELen = 0;

        while( temp != 0 )
        {
            cpSsid = (PNDIS_WLAN_BSSID_EX)((char*)cpSsid + cpSsid->Length);
            temp --;
        }
        memcpy(&pList->apList[i].macAddress[0], &cpSsid->MacAddress[0], 6);
        pList->apList[i].ssid.length = cpSsid->Ssid.SsidLength;
        memcpy(&pList->apList[i].ssid.ucSsid[0], &cpSsid->Ssid.Ssid[0], AP_SSID_MAX_LENGTH);
        pList->apList[i].channel = cpSsid->Configuration.DSConfig % 2412000 / 5000 + 1;

        if ( cpSsid->Rssi <= -100 )
        {
            pList->apList[i].signalQuality = 0;
        }
        else if ( cpSsid->Rssi >= -50 )
        {
            pList->apList[i].signalQuality = 100;
        }
        else
        {
            pList->apList[i].signalQuality = 200 + 2 * cpSsid->Rssi;
        }

#ifndef GETTYPE_THROUGH_WLANAPI
        // get the AP phy type from its ie,rate,mode
        pList->apList[i].type = WLAN_UNKNOWN;
        if (cpSsid->NetworkTypeInUse == Ndis802_11OFDM5)
        {
            // 802.11a
        }
        else if (cpSsid->NetworkTypeInUse == Ndis802_11FH)
        {
            // Straight 802.11
        }
        else if ( (cpSsid->NetworkTypeInUse == Ndis802_11DS) || (cpSsid->NetworkTypeInUse == Ndis802_11OFDM24) )
        {
            float fRate = 0;
            // judge type through its rate
            for (int j = 0; j < NDIS_802_11_LENGTH_RATES_EX; j++)
            {
                fRate = ((cpSsid->SupportedRates[j] & 0x7f)*0.5);

                if ((fRate == 1.0) || (fRate == 2.0))
                {
					// Straight 802.11
                }
				else if ((fRate == 5.5) || (fRate == 11.0))
                {
					//	802.11b
					pList->apList[i].type = WLAN_802_11_B;
                }
                else if ((fRate > 11.0) && (fRate <= 54.0))
                {
					// 802.11g
					pList->apList[i].type = WLAN_802_11_G;
                }
            }
        }
        else
        {
			// Unknown type
			pList->apList[i].type = WLAN_UNKNOWN;
        }

        if (cpSsid->IELength > cpSsid->Length)
        {
            if ( cpSsid->Length > sizeof(NDIS_WLAN_BSSID_EX) )
            {
                ulIELen = cpSsid->Length - sizeof(NDIS_WLAN_BSSID_EX);
            }
            else
            {
                ulIELen = 0;
            }
        }
        else
        {
            ulIELen = cpSsid->IELength;
        }

        if ((ulIELen > 0) && Find_80211n_IE((UCHAR *)&cpSsid->IEs[sizeof(NDIS_802_11_FIXED_IEs)], ulIELen))
        {
            // 802.11n
            pList->apList[i].type = WLAN_802_11_N;
        }
        // end of getting type of AP
#endif
    }
    pDevCtrl->CloseNetworkDevice(devHandle);

    return WLAN_SUCCESS;
}

bool Win32WlanScanner::IsWlanInterfaceConnected()
{
    if (!m_isHandleOpen)
        return false;

    PWLAN_INTERFACE_INFO_LIST pInterfaceList = NULL;
	if (WlanEnumInterfaces(m_hClientHandle, NULL, &pInterfaceList) != ERROR_SUCCESS)
	{
		return false;
	}
	if (pInterfaceList->dwNumberOfItems == 0)
	{
	    // no WLAN interfaces
		WlanFreeMemory(pInterfaceList);
		return false;
	}
	PWLAN_INTERFACE_INFO pInterface = &pInterfaceList->InterfaceInfo[0];

	WLAN_INTERFACE_STATE state = pInterface->isState;
    WlanFreeMemory(pInterfaceList);
    return state == wlan_interface_state_connected;
}

bool Win32WlanScanner::CloseWlanConnection()
{
    if (!m_isHandleOpen)
        return false;

    PWLAN_INTERFACE_INFO_LIST pInterfaceList = NULL;
	if (WlanEnumInterfaces(m_hClientHandle, NULL, &pInterfaceList) != ERROR_SUCCESS)
	{
		return false;
	}
	if (pInterfaceList->dwNumberOfItems == 0)
	{
		WlanFreeMemory(pInterfaceList);
		return false;
	}
	PWLAN_INTERFACE_INFO pInterface = &pInterfaceList->InterfaceInfo[0];

	DWORD result = WlanDisconnect(m_hClientHandle, &pInterface->InterfaceGuid, NULL);
	if ( result == ERROR_SUCCESS )
	{
	    DataCenter::GetInstance()->UpdateCurrentAP(NULL, false);
	}

	WlanFreeMemory(pInterfaceList);
	return result == ERROR_SUCCESS;
}

wxString Win32WlanScanner::GetCurrentWlanAdapterId()
{
    if ( !m_isHandleOpen )
        return WlanScanner::GetCurrentWlanAdapterId();

    PWLAN_INTERFACE_INFO_LIST pInterfaceList = NULL;
	if ( WlanEnumInterfaces(m_hClientHandle, NULL, &pInterfaceList) != ERROR_SUCCESS )
	{
		return WlanScanner::GetCurrentWlanAdapterId();
	}
	if ( pInterfaceList->dwNumberOfItems == 0 )
	{
		WlanFreeMemory(pInterfaceList);
		return WlanScanner::GetCurrentWlanAdapterId();
	}

	PWLAN_INTERFACE_INFO pInterface = &pInterfaceList->InterfaceInfo[0];
	wxString strGuid = GuidToString(pInterface->InterfaceGuid);
	WlanFreeMemory(pInterfaceList);
    return strGuid;
}

bool Win32WlanScanner::GetCurrentConnectedNetwork(APStatus* pAPStatus)
{
    ENMUtil::Log(logFile, wxString::Format(_T("Enter Win32WlanScanner::GetCurrentConnectedNetwork(%08X)"), pAPStatus));
    if ( pAPStatus == NULL )
    {
        ENMUtil::Log(logFile, _T("Return Win32WlanScanner::GetCurrentConnectedNetwork, pAPStatus=NULL\n"));
        return false;
    }

    if ( !m_isHandleOpen )
    {
        ENMUtil::Log(logFile, _T("Return Win32WlanScanner::GetCurrentConnectedNetwork, handle not opened\n"));
        return false;
    }

    PWLAN_INTERFACE_INFO_LIST pInterfaceList = NULL;
	if ( WlanEnumInterfaces(m_hClientHandle, NULL, &pInterfaceList) != ERROR_SUCCESS )
	{
	    ENMUtil::Log(logFile, _T("Return Win32WlanScanner::GetCurrentConnectedNetwork, enum interface error\n"));
		return false;
	}
	if ( pInterfaceList->dwNumberOfItems == 0 )
	{
		WlanFreeMemory(pInterfaceList);
		ENMUtil::Log(logFile, _T("Return Win32WlanScanner::GetCurrentConnectedNetwork, no interface\n"));
		return false;
	}

	PWLAN_INTERFACE_INFO pInterface = &pInterfaceList->InterfaceInfo[0];
	DWORD dataSize = 0;
	PWLAN_CONNECTION_ATTRIBUTES pAttr = NULL;
	WLAN_OPCODE_VALUE_TYPE opCodeValueType = wlan_opcode_value_type_query_only;
	DWORD result = WlanQueryInterface(m_hClientHandle, &pInterface->InterfaceGuid, wlan_intf_opcode_current_connection, NULL, &dataSize, (PVOID*)&pAttr, &opCodeValueType);
	if ( result == ERROR_SUCCESS )
	{
	    DOT11_SSID ssid = pAttr->wlanAssociationAttributes.dot11Ssid;
	    ENMUtil::Log(logFile, _T("\t Get SSID: ") + wxString::FromUTF8((char*)ssid.ucSSID, ssid.uSSIDLength));
	    mbstowcs(pAPStatus->name, (const char*)ssid.ucSSID, ssid.uSSIDLength);
        pAPStatus->ssid.length = ssid.uSSIDLength;
        memcpy(&pAPStatus->ssid.ucSsid[0], &ssid.ucSSID[0], sizeof(ssid.ucSSID));
        memcpy(&pAPStatus->macAddress[0], &pAttr->wlanAssociationAttributes.dot11Bssid, WLAN_MAC_ADDRESS_LENGTH);
//        pAPStatus->signalQuality = pAttr->wlanAssociationAttributes.wlanSignalQuality;
	}
	if ( pAttr != NULL )
	{
        WlanFreeMemory(pAttr);
	}
    WlanFreeMemory(pInterfaceList);

    AccessPointList list;
    memset(&list, 0, sizeof(AccessPointList));
//    WlanResult scanResult = ScanWlanNetwork(&list);
    WlanResult scanResult = GetNetworkFromNdis(&list);
    if ( scanResult != WLAN_SUCCESS )
        return false;

    bool updated = false;
    ENMUtil::Log(logFile, wxString::Format(_T("\t Scan %d networks from NDIS"), list.nCount));
    for ( int i = 0 ; i < list.nCount; i++ )
    {
        if ( CompareMacAddress(pAPStatus->macAddress, list.apList[i].macAddress) == 0 )
        {
            updated = true;
            pAPStatus->signalQuality = list.apList[i].signalQuality;
            pAPStatus->type = list.apList[i].type;
            pAPStatus->bssType = list.apList[i].bssType;
            pAPStatus->auth = list.apList[i].auth;
            break;
        }
    }
    ENMUtil::Log(logFile, wxString::Format(_T("Return Win32WlanScanner::GetCurrentConnectedNetwork: %d\n"), updated));
    return updated;
}

#endif // defined(__WXMSW__)
