/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <string.h>
#include <NetworkManager.h>
#include <nm-connection.h>
#include <nm-setting-connection.h>
#include <nm-setting-wireless.h>
#include <nm-setting-8021x.h>
#include <nm-utils.h>
#include "nmn-wifi-item.h"
#include "nma-gconf-settings.h"
#include "nma-gconf-connection.h"
#include "wireless-dialog.h"
#include "utils.h"

G_DEFINE_TYPE (NmnWifiItem, nmn_wifi_item, NMN_TYPE_NETWORK_ITEM)

enum {
    PROP_0,
    PROP_ACCESS_POINT,

    LAST_PROP
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_WIFI_ITEM, NmnWifiItemPrivate))

typedef struct {
    NMAccessPoint *ap;
    gulong notify_id;

    gboolean disposed;
} NmnWifiItemPrivate;

static void connect (NmnItem *item);

GtkWidget *
nmn_wifi_item_new (NmnNMData *nm_data,
                   NMDeviceWifi *device,
                   NMAccessPoint *ap)
{
    g_return_val_if_fail (NMN_IS_NM_DATA (nm_data), NULL);
    g_return_val_if_fail (NM_IS_DEVICE_WIFI (device), NULL);
    g_return_val_if_fail (NM_IS_ACCESS_POINT (ap), NULL);

    return GTK_WIDGET (g_object_new (NMN_TYPE_WIFI_ITEM,
                                     NMN_NETWORK_ITEM_NM_DATA, nm_data,
                                     NMN_NETWORK_ITEM_DEVICE, device,
                                     NMN_WIFI_ITEM_ACCESS_POINT, ap,
                                     NULL));
}

static void
update_item (NmnWifiItem *self)
{
    NmnWifiItemPrivate *priv = GET_PRIVATE (self);
    NmnItem *item = NMN_ITEM (self);
    const char *icon;
    const char *security_string;
    guint strength;
	guint32 flags;
	guint32 wpa_flags;
	guint32 rsn_flags;

    strength = nm_access_point_get_strength (priv->ap);
    strength = CLAMP (strength, 0, 100);

    if (strength > 71)
        icon = "nm-signal-100";
    else if (strength > 38)
        icon = "nm-signal-66";
    else if (strength > 5)
        icon = "nm-signal-33";
    else
        icon = "nm-signal-00";

	flags = nm_access_point_get_flags (priv->ap);
	wpa_flags = nm_access_point_get_wpa_flags (priv->ap);
	rsn_flags = nm_access_point_get_rsn_flags (priv->ap);

    if (rsn_flags & NM_802_11_AP_SEC_KEY_MGMT_PSK ||
        wpa_flags & NM_802_11_AP_SEC_KEY_MGMT_PSK ||
        rsn_flags & NM_802_11_AP_SEC_KEY_MGMT_802_1X ||
        wpa_flags & NM_802_11_AP_SEC_KEY_MGMT_802_1X)
        security_string = "WPA encrypted";
    else if (flags & NM_802_11_AP_FLAGS_PRIVACY)
        security_string = "WEP encrypted";
    else
        security_string = "";

    nmn_item_set_icon (item, icon);
    nmn_item_set_security (item, security_string);

    if (nmn_network_item_get_connection (NMN_NETWORK_ITEM (self)) == NULL) {
        /* We're an AP list item, set the item's name */
        const GByteArray *ssid;
        char *printable_ssid;

        ssid = nm_access_point_get_ssid (priv->ap);
        printable_ssid = nm_utils_ssid_to_utf8 ((char *) ssid->data, ssid->len);
        nmn_item_set_name (item, printable_ssid);
        g_free (printable_ssid);

        nmn_item_set_status_visible (item, FALSE);
    }
}

static void
updated (NMAccessPoint *ap,
         GParamSpec *pspec,
         gpointer user_data)
{
    update_item (NMN_WIFI_ITEM (user_data));
}

void
nmn_wifi_item_set_ap (NmnWifiItem *self,
                      NMAccessPoint *ap)
{
    NmnWifiItemPrivate *priv;

    g_return_if_fail (NMN_IS_WIFI_ITEM (self));

    priv = GET_PRIVATE (self);

    if (priv->ap) {
        g_signal_handler_disconnect (priv->ap, priv->notify_id);
        g_object_unref (priv->ap);
    }

    if (ap) {
        priv->ap = g_object_ref (ap);
        priv->notify_id = g_signal_connect (ap, "notify", G_CALLBACK (updated), self);
        update_item (self);
    } else
        priv->ap = NULL;
}

NMAccessPoint *
nmn_wifi_item_get_ap (NmnWifiItem *self)
{
    g_return_val_if_fail (NMN_IS_WIFI_ITEM (self), NULL);

    return GET_PRIVATE (self)->ap;
}

static const char * default_ssid_list[] = {
    "linksys",
    "linksys-a",
    "linksys-g",
    "default",
    "belkin54g",
    "NETGEAR",
    NULL
};

static gboolean
is_manufacturer_default_ssid (const GByteArray *ssid)
{
    const char **default_ssid = default_ssid_list;

    while (*default_ssid) {
        if (ssid->len == strlen (*default_ssid)) {
            if (!memcmp (*default_ssid, ssid->data, ssid->len))
                return TRUE;
        }
        default_ssid++;
    }
    return FALSE;
}

static void
add_ciphers_from_flags (NMSettingWirelessSecurity *sec,
                        guint32 flags,
                        gboolean pairwise)
{
    if (pairwise) {
        if (flags & NM_802_11_AP_SEC_PAIR_TKIP)
            nm_setting_wireless_security_add_pairwise (sec, "tkip");
        if (flags & NM_802_11_AP_SEC_PAIR_CCMP)
            nm_setting_wireless_security_add_pairwise (sec, "ccmp");
    } else {
        if (flags & NM_802_11_AP_SEC_GROUP_WEP40)
            nm_setting_wireless_security_add_group (sec, "wep40");
        if (flags & NM_802_11_AP_SEC_GROUP_WEP104)
            nm_setting_wireless_security_add_group (sec, "wep104");
        if (flags & NM_802_11_AP_SEC_GROUP_TKIP)
            nm_setting_wireless_security_add_group (sec, "tkip");
        if (flags & NM_802_11_AP_SEC_GROUP_CCMP)
            nm_setting_wireless_security_add_group (sec, "ccmp");
    }
}

static NMSettingWirelessSecurity *
get_security_for_ap (NMAccessPoint *ap,
                     guint32 dev_caps,
                     gboolean *supported,
                     NMSetting8021x **s_8021x)
{
    NMSettingWirelessSecurity *sec;
    NM80211Mode mode;
    guint32 flags;
    guint32 wpa_flags;
    guint32 rsn_flags;

    g_return_val_if_fail (NM_IS_ACCESS_POINT (ap), NULL);
    g_return_val_if_fail (supported != NULL, NULL);
    g_return_val_if_fail (*supported == TRUE, NULL);
    g_return_val_if_fail (s_8021x != NULL, NULL);
    g_return_val_if_fail (*s_8021x == NULL, NULL);

    sec = (NMSettingWirelessSecurity *) nm_setting_wireless_security_new ();

    mode = nm_access_point_get_mode (ap);
    flags = nm_access_point_get_flags (ap);
    wpa_flags = nm_access_point_get_wpa_flags (ap);
    rsn_flags = nm_access_point_get_rsn_flags (ap);

    /* No security */
    if (   !(flags & NM_802_11_AP_FLAGS_PRIVACY)
           && (wpa_flags == NM_802_11_AP_SEC_NONE)
           && (rsn_flags == NM_802_11_AP_SEC_NONE))
        goto none;

    /* Static WEP, Dynamic WEP, or LEAP */
    if (flags & NM_802_11_AP_FLAGS_PRIVACY) {
        if ((dev_caps & NM_WIFI_DEVICE_CAP_RSN) || (dev_caps & NM_WIFI_DEVICE_CAP_WPA)) {
            /* If the device can do WPA/RSN but the AP has no WPA/RSN informatoin
             * elements, it must be LEAP or static/dynamic WEP.
             */
            if ((wpa_flags == NM_802_11_AP_SEC_NONE) && (rsn_flags == NM_802_11_AP_SEC_NONE)) {
                g_object_set (sec,
                              NM_SETTING_WIRELESS_SECURITY_KEY_MGMT, "none",
                              NM_SETTING_WIRELESS_SECURITY_WEP_TX_KEYIDX, 0,
                              NULL);
                return sec;
            }
            /* Otherwise, the AP supports WPA or RSN, which is preferred */
        } else {
            /* Device can't do WPA/RSN, but can at least pass through the
             * WPA/RSN information elements from a scan.  Since Privacy was
             * advertised, LEAP or static/dynamic WEP must be in use.
             */
            g_object_set (sec,
                          NM_SETTING_WIRELESS_SECURITY_KEY_MGMT, "none",
                          NM_SETTING_WIRELESS_SECURITY_WEP_TX_KEYIDX, 0,
                          NULL);
            return sec;
        }
    }

    /* Stuff after this point requires infrastructure */
    if (mode != NM_802_11_MODE_INFRA) {
        *supported = FALSE;
        goto none;
    }

    /* WPA2 PSK first */
    if (   (rsn_flags & NM_802_11_AP_SEC_KEY_MGMT_PSK)
           && (dev_caps & NM_WIFI_DEVICE_CAP_RSN)) {
        g_object_set (sec, NM_SETTING_WIRELESS_SECURITY_KEY_MGMT, "wpa-psk", NULL);
        nm_setting_wireless_security_add_proto (sec, "rsn");
        add_ciphers_from_flags (sec, rsn_flags, TRUE);
        add_ciphers_from_flags (sec, rsn_flags, FALSE);
        return sec;
    }

    /* WPA PSK */
    if (   (wpa_flags & NM_802_11_AP_SEC_KEY_MGMT_PSK)
           && (dev_caps & NM_WIFI_DEVICE_CAP_WPA)) {
        g_object_set (sec, NM_SETTING_WIRELESS_SECURITY_KEY_MGMT, "wpa-psk", NULL);
        nm_setting_wireless_security_add_proto (sec, "wpa");
        add_ciphers_from_flags (sec, wpa_flags, TRUE);
        add_ciphers_from_flags (sec, wpa_flags, FALSE);
        return sec;
    }

    /* WPA2 Enterprise */
    if (   (rsn_flags & NM_802_11_AP_SEC_KEY_MGMT_802_1X)
           && (dev_caps & NM_WIFI_DEVICE_CAP_RSN)) {
        g_object_set (sec, NM_SETTING_WIRELESS_SECURITY_KEY_MGMT, "wpa-eap", NULL);
        nm_setting_wireless_security_add_proto (sec, "rsn");
        add_ciphers_from_flags (sec, rsn_flags, TRUE);
        add_ciphers_from_flags (sec, rsn_flags, FALSE);

        *s_8021x = NM_SETTING_802_1X (nm_setting_802_1x_new ());
        nm_setting_802_1x_add_eap_method (*s_8021x, "ttls");
        g_object_set (*s_8021x, NM_SETTING_802_1X_PHASE2_AUTH, "mschapv2", NULL);
        return sec;
    }

    /* WPA Enterprise */
    if (   (wpa_flags & NM_802_11_AP_SEC_KEY_MGMT_802_1X)
           && (dev_caps & NM_WIFI_DEVICE_CAP_WPA)) {
        g_object_set (sec, NM_SETTING_WIRELESS_SECURITY_KEY_MGMT, "wpa-eap", NULL);
        nm_setting_wireless_security_add_proto (sec, "wpa");
        add_ciphers_from_flags (sec, wpa_flags, TRUE);
        add_ciphers_from_flags (sec, wpa_flags, FALSE);

        *s_8021x = NM_SETTING_802_1X (nm_setting_802_1x_new ());
        nm_setting_802_1x_add_eap_method (*s_8021x, "ttls");
        g_object_set (*s_8021x, NM_SETTING_802_1X_PHASE2_AUTH, "mschapv2", NULL);
        return sec;
    }

    *supported = FALSE;

 none:
    g_object_unref (sec);
    return NULL;
}

static NMConnection *
create_new_connection (NMDeviceWifi *device, NMAccessPoint *ap)
{
    NMConnection *connection;
    NMSettingConnection *s_con;
    NMSettingWireless *s_wireless;
    NMSettingWirelessSecurity *s_wireless_sec;
    NMSetting8021x *s_8021x = NULL;
    const GByteArray *ap_ssid;
    char *id;
    char buf[33];
    int buf_len;
    NM80211Mode mode;
    guint32 dev_caps;
    gboolean supported = TRUE;

    dev_caps = nm_device_wifi_get_capabilities (device);
    s_wireless_sec = get_security_for_ap (ap, dev_caps, &supported, &s_8021x);
    if (!supported)
        return NULL;

    s_wireless = NM_SETTING_WIRELESS (nm_setting_wireless_new ());
    ap_ssid = nm_access_point_get_ssid (ap);
    g_object_set (s_wireless, NM_SETTING_WIRELESS_SSID, ap_ssid, NULL);

    mode = nm_access_point_get_mode (ap);
    if (mode == NM_802_11_MODE_ADHOC)
        g_object_set (s_wireless, NM_SETTING_WIRELESS_MODE, "adhoc", NULL);
    else if (mode == NM_802_11_MODE_INFRA)
        g_object_set (s_wireless, NM_SETTING_WIRELESS_MODE, "infrastructure", NULL);
    else
        g_assert_not_reached ();

    connection = nm_connection_new ();
    nm_connection_add_setting (connection, NM_SETTING (s_wireless));
    if (s_wireless_sec) {
        g_object_set (s_wireless, NM_SETTING_WIRELESS_SEC, NM_SETTING_WIRELESS_SECURITY_SETTING_NAME, NULL);
        nm_connection_add_setting (connection, NM_SETTING (s_wireless_sec));
    }
    if (s_8021x)
        nm_connection_add_setting (connection, NM_SETTING (s_8021x));

    s_con = NM_SETTING_CONNECTION (nm_setting_connection_new ());
    g_object_set (s_con,
                  NM_SETTING_CONNECTION_TYPE, nm_setting_get_name (NM_SETTING (s_wireless)),
                  NM_SETTING_CONNECTION_AUTOCONNECT, !is_manufacturer_default_ssid (ap_ssid),
                  NULL);

    memset (buf, 0, sizeof (buf));
    buf_len = MIN (ap_ssid->len, sizeof (buf) - 1);
    memcpy (buf, ap_ssid->data, buf_len);
    id = nm_utils_ssid_to_utf8 (buf, buf_len);
    g_object_set (s_con, NM_SETTING_CONNECTION_ID, id, NULL);
    g_free (id);

    id = nm_utils_uuid_generate ();
    g_object_set (s_con, NM_SETTING_CONNECTION_UUID, id, NULL);
    g_free (id);

    nm_connection_add_setting (connection, NM_SETTING (s_con));

    return connection;
}

static void
wifi_item_created (NmnNetworkItem *item, NMConnection *connection)
{
    NmnNMData *nm_data;
    NMExportedConnection *exported;

    nm_data = nmn_network_item_get_nm_data (item);

    /* FIXME: Maybe it already exists, try harder */
    exported = NM_EXPORTED_CONNECTION 
        (nma_gconf_settings_add_connection (NMA_GCONF_SETTINGS (nmn_nm_data_get_user_settings (nm_data)),
                                            connection));

    if (!exported)
        return;

    nmn_network_item_set_connection (item, exported);
    g_object_unref (exported);

    connect (NMN_ITEM (item));
    nmn_network_item_set_connection (item, NULL);
}

static void
wireless_dialog_response_cb (NMAWirelessDialog *dialog,
                             gint response,
                             gpointer user_data)
{
    if (response == GTK_RESPONSE_OK) {
        wifi_item_created (NMN_NETWORK_ITEM (user_data), nma_wireless_dialog_get_connection (dialog));
    }

    gtk_widget_hide (GTK_WIDGET (dialog));
    gtk_widget_destroy (GTK_WIDGET (dialog));
}

static gboolean
update_autoconnect (NmnNetworkItem *item, gboolean connect_automatically)
{
    NMExportedConnection *exported;
    NMConnection *wrapped;
    NMSettingConnection *s_con;
    GHashTable *new_settings;

    exported = nmn_network_item_get_connection (item);
    wrapped = nm_exported_connection_get_connection (exported);
    s_con = NM_SETTING_CONNECTION (nm_connection_get_setting (wrapped, NM_TYPE_SETTING_CONNECTION));

    if (nm_setting_connection_get_autoconnect (s_con) == connect_automatically)
        return FALSE;

    g_object_set (s_con, NM_SETTING_CONNECTION_AUTOCONNECT, connect_automatically, NULL);
    new_settings = nm_connection_to_hash (wrapped);

    nm_exported_connection_update (exported, new_settings, NULL);
    g_hash_table_unref (new_settings);

    return TRUE;
}

static void
connect (NmnItem *item)
{
    NmnNetworkItem *network_item = NMN_NETWORK_ITEM (item);

    if (!nmn_network_item_get_connection (network_item)) {
        NMDevice *device;
        NMAccessPoint *ap;
        NMConnection *connection;

        device = nmn_network_item_get_device (network_item);
        ap = nmn_wifi_item_get_ap (NMN_WIFI_ITEM (item));
        connection = create_new_connection (NM_DEVICE_WIFI (device), ap);
        if (!connection)
            return;

        if (nm_connection_need_secrets (connection, NULL)) {
            NmnNMData *nm_data;
            GtkWidget *dialog;

            nm_data = nmn_network_item_get_nm_data (network_item);
            dialog = nma_wireless_dialog_new (NM_CLIENT (nm_data), connection, device, ap);
            g_signal_connect (dialog, "done", G_CALLBACK (wireless_dialog_response_cb), item);
            nma_wireless_dialog_show (NMA_WIRELESS_DIALOG (dialog));
        } else
            wifi_item_created (network_item, connection);

        g_object_unref (connection);
    } else {
        update_autoconnect (network_item, TRUE);
        NMN_ITEM_CLASS (nmn_wifi_item_parent_class)->connect (item);
    }
}

static void
disconnect (NmnItem *item)
{
    /* Turn off autoconnect, otherwise it would reconnect right back. */
    update_autoconnect (NMN_NETWORK_ITEM (item), FALSE);
    NMN_ITEM_CLASS (nmn_wifi_item_parent_class)->disconnect (item);
}

static guint
get_priority (NmnItem *item)
{
    guint priority;

    priority = NMN_ITEM_CLASS (nmn_wifi_item_parent_class)->get_priority (item) + 20;

    /* FIXME: Add some more based on SSID? signal strength? */

    return priority;
}

static const char *
get_specific_object (NmnNetworkItem *network_item)
{
    NmnWifiItemPrivate *priv = GET_PRIVATE (network_item);

    return nm_object_get_path (NM_OBJECT (priv->ap));
}

static gboolean
add_one_setting (GHashTable *settings,
                 NMConnection *connection,
                 NMSetting *setting,
                 GError **error)
{
    GHashTable *secrets;

    utils_fill_connection_certs (connection);
    secrets = nm_setting_to_hash (setting);
    utils_clear_filled_connection_certs (connection);

    if (secrets) {
        g_hash_table_insert (settings, g_strdup (nm_setting_get_name (setting)), secrets);
    } else {
        g_set_error (error, NM_SETTINGS_ERROR, NM_SETTINGS_ERROR_INTERNAL_ERROR,
                     G_STRLOC ": failed to hash setting '%s'.",
                     nm_setting_get_name (setting));
    }

    return secrets ? TRUE : FALSE;
}

typedef struct {
    NMExportedConnection *exported;
    DBusGMethodInvocation *context;
} SecretsRequestInfo;

static void
secrets_requested_response_cb (NMAWirelessDialog *dialog,
                               gint response,
                               gpointer user_data)
{
    SecretsRequestInfo *info = (SecretsRequestInfo *) user_data;
    NMSettingWirelessSecurity *s_wireless_sec;
    const char *key_mgmt;
    NMConnection *wrapped = NULL;
    GHashTable *settings = NULL;
    GError *error = NULL;

    if (response != GTK_RESPONSE_OK) {
        g_set_error_literal (&error, NM_SETTINGS_ERROR, NM_SETTINGS_ERROR_SECRETS_REQUEST_CANCELED,
                             G_STRLOC ": canceled");
        goto done;
    }

    wrapped = nm_exported_connection_get_connection (info->exported);

    /* Second-guess which setting NM wants secrets for. */
    s_wireless_sec = NM_SETTING_WIRELESS_SECURITY (nm_connection_get_setting (wrapped, NM_TYPE_SETTING_WIRELESS_SECURITY));
    if (!s_wireless_sec) {
        g_set_error_literal (&error, NM_SETTINGS_ERROR, NM_SETTINGS_ERROR_INVALID_CONNECTION,
                             G_STRLOC ": requested setting '802-11-wireless-security'"
                             " didn't exist in the connection.");
        goto done;  /* Unencrypted */
    }

    /* Returned secrets are a{sa{sv}}; this is the outer a{s...} hash that
     * will contain all the individual settings hashes.
     */
    settings = g_hash_table_new_full (g_str_hash, g_str_equal,
                                      g_free, (GDestroyNotify) g_hash_table_destroy);

    /* If the user chose an 802.1x-based auth method, return 802.1x secrets,
     * not wireless secrets.  Can happen with Dynamic WEP, because NM doesn't
     * know the capabilities of the AP (since Dynamic WEP APs don't broadcast
     * beacons), and therefore defaults to requesting WEP secrets from the
     * wireless-security setting, not the 802.1x setting.
     */
    key_mgmt = nm_setting_wireless_security_get_key_mgmt (s_wireless_sec);
    if (!strcmp (key_mgmt, "ieee8021x") || !strcmp (key_mgmt, "wpa-eap")) {
        const char *auth_alg;

        /* LEAP secrets aren't in the 802.1x setting */
        auth_alg = nm_setting_wireless_security_get_auth_alg (s_wireless_sec);
        if (!auth_alg || strcmp (auth_alg, "leap")) {
            NMSetting8021x *s_8021x;

            s_8021x = (NMSetting8021x *) nm_connection_get_setting (wrapped, NM_TYPE_SETTING_802_1X);
            if (!s_8021x) {
                g_set_error_literal (&error, NM_SETTINGS_ERROR, NM_SETTINGS_ERROR_INVALID_CONNECTION,
                                     G_STRLOC ": requested setting '802-1x' didn't"
                                     " exist in the connection.");
                goto done;
            }

            /* Add the 802.1x setting */
            if (!add_one_setting (settings, wrapped, NM_SETTING (s_8021x), &error))
                goto done;
        }
    }

    /* Add the 802-11-wireless-security setting no matter what */
    if (!add_one_setting (settings, wrapped, NM_SETTING (s_wireless_sec), &error))
        goto done;

    dbus_g_method_return (info->context, settings);

    /* Save the connection back to GConf _after_ hashing it, because
     * saving to GConf might trigger the GConf change notifiers, resulting
     * in the connection being read back in from GConf which clears secrets.
     */
    nma_gconf_connection_save (NMA_GCONF_CONNECTION (info->exported));

 done:
    if (settings)
        g_hash_table_destroy (settings);

    if (error) {
        g_warning ("%s", error->message);
        dbus_g_method_return_error (info->context, error);
        g_error_free (error);
    }

    if (wrapped)
        nm_connection_clear_secrets (wrapped);

    gtk_widget_hide (GTK_WIDGET (dialog));
    gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
secrets_requested (NmnNetworkItem *network_item,
                   const char *setting_name,
                   const char **hints,
                   gboolean ask_user,
                   DBusGMethodInvocation *context)
{
    GtkWidget *dialog;
    NMExportedConnection *exported;
    SecretsRequestInfo *info;

    exported = nmn_network_item_get_connection (network_item);
    dialog = nma_wireless_dialog_new (NM_CLIENT (nmn_network_item_get_nm_data (network_item)),
                                      nm_exported_connection_get_connection (exported),
                                      nmn_network_item_get_device (network_item),
                                      nmn_wifi_item_get_ap (NMN_WIFI_ITEM (network_item)));

    info = g_new (SecretsRequestInfo, 1);
    info->exported = exported;
    info->context = context;

    g_signal_connect_data (dialog, "done",
                           G_CALLBACK (secrets_requested_response_cb),
                           info, (GClosureNotify) g_free, 0);

    nma_wireless_dialog_show (NMA_WIRELESS_DIALOG (dialog));
}

static void
nmn_wifi_item_init (NmnWifiItem *item)
{
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    NmnWifiItemPrivate *priv;

    object = G_OBJECT_CLASS (nmn_wifi_item_parent_class)->constructor
        (type, n_construct_params, construct_params);

    if (!object)
        return NULL;

    priv = GET_PRIVATE (object);

    if (!priv->ap) {
        g_warning ("Missing constructor arguments");
        g_object_unref (object);
        return NULL;
    }

    return object;
}

static void
set_property (GObject *object, guint prop_id,
              const GValue *value, GParamSpec *pspec)
{
    NmnWifiItem *self = NMN_WIFI_ITEM (object);

    switch (prop_id) {
    case PROP_ACCESS_POINT:
        nmn_wifi_item_set_ap (self, (NMAccessPoint *) g_value_get_object (value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object, guint prop_id,
              GValue *value, GParamSpec *pspec)
{
    NmnWifiItem *self = NMN_WIFI_ITEM (object);

    switch (prop_id) {
    case PROP_ACCESS_POINT:
        g_value_set_object (value, nmn_wifi_item_get_ap (self));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
dispose (GObject *object)
{
    NmnWifiItemPrivate *priv = GET_PRIVATE (object);

    if (priv->disposed)
        return;

    nmn_wifi_item_set_ap (NMN_WIFI_ITEM (object), NULL);

    priv->disposed = TRUE;

    G_OBJECT_CLASS (nmn_wifi_item_parent_class)->dispose (object);
}

static void
nmn_wifi_item_class_init (NmnWifiItemClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);
    NmnItemClass *item_class = NMN_ITEM_CLASS (class);
	NmnNetworkItemClass *network_item_class = NMN_NETWORK_ITEM_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnWifiItemPrivate));

    /* methods */
    object_class->constructor = constructor;
    object_class->set_property = set_property;
    object_class->get_property = get_property;
    object_class->dispose = dispose;

    item_class->connect = connect;
    item_class->disconnect = disconnect;
    item_class->get_priority = get_priority;

    network_item_class->get_specific_object = get_specific_object;
    network_item_class->secrets_requested = secrets_requested;

    /* properties */
    g_object_class_install_property
        (object_class, PROP_ACCESS_POINT,
         g_param_spec_object (NMN_WIFI_ITEM_ACCESS_POINT,
                              "NMAccessPoint",
                              "NMAccessPoint",
                              NM_TYPE_ACCESS_POINT,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
}
