/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <arpa/inet.h>
#include <NetworkManager.h>
#include <nm-gsm-device.h>
#include <nm-cdma-device.h>
#include <nm-connection.h>
#include <nm-setting-connection.h>
#include <nm-setting-serial.h>
#include <nm-setting-ppp.h>
#include <nm-setting-gsm.h>
#include <nm-setting-cdma.h>
#include <nm-setting-ip4-config.h>
#include <nm-utils.h>
#include "nmn-new-connection.h"
#include "nma-gconf-settings.h"
#include "nmn-wifi-list.h"
#include "nmn-mobile-providers.h"
#include "wireless-dialog.h"

G_DEFINE_TYPE (NmnNewConnection, nmn_new_connection, GTK_TYPE_EVENT_BOX)

enum {
    PROP_0,
    PROP_BUILDER,
    PROP_NM_DATA,

    LAST_PROP
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), NMN_TYPE_NEW_CONNECTION, NmnNewConnectionPrivate))

typedef struct {
    GtkBuilder *builder;
    NmnNMData *nm_data;

    /* WiFi widgets */
    GtkWidget *wifi_list;
    GtkWidget *wifi_hidden_network;

    /* 3G widgets */
    GtkWidget *mobile_label;
    GtkWidget *mobile_list;
    GtkWidget *mobile_save;
    gboolean mobile_list_populated;
    gulong mobile_toggled_id;

    GtkWidget *close;

    gboolean disposed;
} NmnNewConnectionPrivate;

GtkWidget *
nmn_new_connection_create (GtkBuilder *builder,
                           NmnNMData *nm_data)
{
    g_return_val_if_fail (GTK_IS_BUILDER (builder), NULL);
    g_return_val_if_fail (NMN_IS_NM_DATA (nm_data), NULL);

    return GTK_WIDGET (g_object_new (NMN_TYPE_NEW_CONNECTION,
                                     NMN_NEW_CONNECTION_BUILDER, builder,
                                     NMN_NEW_CONNECTION_NM_DATA, nm_data,
                                     NULL));
}

static void
connect_requested (gpointer instance, gpointer user_data)
{
    gtk_widget_hide (GTK_WIDGET (user_data));
}

static void
connect_cb (gpointer user_data,
            const char *object_path,
            GError *error)
{
    /* FIXME: Report the error somewhere */
    g_debug ("connect_cb: %s %s", object_path,
             error ? error->message : "success");
}

static gboolean
save_connection (NmnNMData *data, NMConnection *connection)
{
    NMAGConfConnection *exported;
    NMAGConfSettings *settings;

    settings = NMA_GCONF_SETTINGS (nmn_nm_data_get_user_settings (data));

	exported = nma_gconf_settings_get_by_connection (settings, connection);
	if (exported) {
		/* Not a new or system connection, save the updated settings to GConf */
		nma_gconf_connection_save (exported);
        return TRUE;
    }

    /* FIXME: Try fuzzy match */

    exported = nma_gconf_settings_add_connection (settings, connection);

    return exported != NULL;
}

static void
wifi_dialog_done (NMAWirelessDialog *dialog,
                  gint response,
                  gpointer user_data)
{
    NmnNewConnectionPrivate *priv = GET_PRIVATE (user_data);
    NMConnection *connection;

    if (response != GTK_RESPONSE_OK)
        goto done;

    connection = nma_wireless_dialog_get_connection (dialog);
    save_connection (priv->nm_data, connection);

    nm_client_activate_connection (NM_CLIENT (priv->nm_data),
                                   NM_DBUS_SERVICE_USER_SETTINGS,
                                   nm_connection_get_path (connection),
                                   nma_wireless_dialog_get_device (dialog),
                                   "/",
                                   connect_cb,
                                   NULL);

 done:
    gtk_widget_hide (GTK_WIDGET (dialog));
    gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
wifi_search (GtkButton *button, gpointer user_data)
{
    NmnNewConnectionPrivate *priv = GET_PRIVATE (user_data);
    GtkWidget *dialog;

    dialog = nma_wireless_dialog_hidden_new (NM_CLIENT (priv->nm_data));
    g_signal_connect (dialog, "done", G_CALLBACK (wifi_dialog_done), user_data);
    nma_wireless_dialog_show (NMA_WIRELESS_DIALOG (dialog));
}

static void
wifi_page_init (NmnNewConnection *connection)
{
    NmnNewConnectionPrivate *priv = GET_PRIVATE (connection);
    GtkWidget *container;

    container = GTK_WIDGET (gtk_builder_get_object (priv->builder, "new_wifi_container"));
    priv->wifi_list = nmn_wifi_list_new (priv->nm_data);
    g_signal_connect (priv->wifi_list, "connect-requested", G_CALLBACK (connect_requested), connection);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (container), priv->wifi_list);
    gtk_widget_show_all (priv->wifi_list);

    priv->wifi_hidden_network = GTK_WIDGET (gtk_builder_get_object (priv->builder, "new_wifi_hidden"));
    g_signal_connect (priv->wifi_hidden_network, "clicked", G_CALLBACK (wifi_search), connection);
    g_signal_connect (priv->wifi_hidden_network, "clicked", G_CALLBACK (connect_requested), connection);
}

/* Mobile */

#define MOBILE_COL_NAME     0
#define MOBILE_COL_PROVIDER 1
#define MOBILE_COL_METHOD   2

static gboolean
mobile_have_device (NmnNewConnection *connection, NmnMobileProviderType *type)
{
    NmnNewConnectionPrivate *priv = GET_PRIVATE (connection);
    const GPtrArray *devices;
    int i;
    gboolean found = FALSE;

    devices = nm_client_get_devices (NM_CLIENT (priv->nm_data));
    for (i = 0; !found && devices && i < devices->len; i++) {
        NMDevice *device = NM_DEVICE (g_ptr_array_index (devices, i));

        if (NM_IS_GSM_DEVICE (device)) {
            found = TRUE;
            *type = NMN_MOBILE_ACCESS_METHOD_TYPE_GSM;
        }

        if (NM_IS_CDMA_DEVICE (device)) {
            found = TRUE;
            *type = NMN_MOBILE_ACCESS_METHOD_TYPE_CDMA;
        }
    }

    return found;
}

static void
mobile_status_update (NmnNewConnection *connection)
{
    NmnNewConnectionPrivate *priv = GET_PRIVATE (connection);
    NmnMobileProviderType type;
    gboolean found;

    if (!nmn_nm_data_modems_get_active (priv->nm_data)) {
        found = FALSE;
        gtk_label_set_text (GTK_LABEL (priv->mobile_label), "3G disabled");
    } else {
        found = mobile_have_device (connection, &type);

        if (found)
            gtk_label_set_text (GTK_LABEL (priv->mobile_label), "Internal 3G modem and SIM card detected");
        else
            gtk_label_set_text (GTK_LABEL (priv->mobile_label), "No modems detected");
    }

    gtk_widget_set_sensitive (priv->mobile_list, found);
    gtk_widget_set_sensitive (priv->mobile_save, found);
}

static gboolean
method_list_has_type (GSList *list, NmnMobileProviderType type)
{
    GSList *iter;

    for (iter = list; iter; iter = iter->next) {
        NmnMobileAccessMethod *method = iter->data;

        if (method->type == type)
            return TRUE;
    }

    return FALSE;
}

static gboolean
provider_list_has_type (GSList *list, NmnMobileProviderType type)
{
    GSList *iter;

    for (iter = list; iter; iter = iter->next) {
        NmnMobileProvider *provider = (NmnMobileProvider *) iter->data;

        if (method_list_has_type (provider->methods, type))
            return TRUE;
    }

    return FALSE;
}

typedef struct {
    GtkTreeStore *store;
    NmnMobileProviderType provider_type;

    GtkTreeIter *parent_iter;
} MobilePopulateInfo;

static void
add_one_method (gpointer data, gpointer user_data)
{
    NmnMobileAccessMethod *method = (NmnMobileAccessMethod *) data;
    MobilePopulateInfo *info = (MobilePopulateInfo *) user_data;
    GtkTreeIter iter;

    if (method->type != info->provider_type)
        return;

    gtk_tree_store_append (info->store, &iter, info->parent_iter);

    if (method->type == NMN_MOBILE_ACCESS_METHOD_TYPE_GSM) {
        char *txt;

        txt = g_strdup_printf ("%s (APN %s)", method->name, method->gsm_apn);
        gtk_tree_store_set (info->store, &iter,
                            MOBILE_COL_NAME, txt,
                            MOBILE_COL_METHOD, method,
                            -1);

        g_free (txt);
    } else {
        gtk_tree_store_set (info->store, &iter,
                            MOBILE_COL_NAME, method->name,
                            MOBILE_COL_METHOD, method,
                            -1);
    }
}

static void
add_one_provider (gpointer data, gpointer user_data)
{
    NmnMobileProvider *provider = (NmnMobileProvider *) data;
    MobilePopulateInfo *info = (MobilePopulateInfo *) user_data;
    GtkTreeIter *country_iter;
    GtkTreeIter iter;

    if (!method_list_has_type (provider->methods, info->provider_type))
        return;

    gtk_tree_store_append (info->store, &iter, info->parent_iter);
    gtk_tree_store_set (info->store, &iter,
                        MOBILE_COL_NAME, provider->name,
                        MOBILE_COL_PROVIDER, provider,
                        -1);

    country_iter = info->parent_iter;
    info->parent_iter = &iter;
    g_slist_foreach (provider->methods, add_one_method, info);
    info->parent_iter = country_iter;
}

static void
add_one_country (gpointer key, gpointer value, gpointer user_data)
{
    GSList *providers = (GSList *) value;
    MobilePopulateInfo *info = (MobilePopulateInfo *) user_data;
    GtkTreeIter iter;

    if (!provider_list_has_type (providers, info->provider_type))
        return;

    gtk_tree_store_append (info->store, &iter, NULL);
    gtk_tree_store_set (info->store, &iter,
                        MOBILE_COL_NAME, key,
                        -1);

    info->parent_iter = &iter;
    g_slist_foreach (providers, add_one_provider, info);
}

static void
mobile_list_populate (NmnNewConnection *connection)
{
    NmnNewConnectionPrivate *priv = GET_PRIVATE (connection);
    GHashTable *mobile_providers;
    GtkTreeStore *store;
    GtkTreeModel *sort_model;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    NmnMobileProviderType provider_type;
    MobilePopulateInfo info;

    if (priv->mobile_list_populated || !mobile_have_device (connection, &provider_type))
        return;

    mobile_providers = nmn_mobile_providers_parse (NULL);
    if (!mobile_providers)
        return;

    store = gtk_tree_store_new (3, G_TYPE_STRING, NMN_TYPE_MOBILE_PROVIDER, NMN_TYPE_MOBILE_ACCESS_METHOD);

    info.store = store;
    info.provider_type = provider_type;
    g_hash_table_foreach (mobile_providers, add_one_country, &info);
    g_hash_table_destroy (mobile_providers);

    sort_model = gtk_tree_model_sort_new_with_model (GTK_TREE_MODEL (store));
    gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (sort_model),
                                          MOBILE_COL_NAME, GTK_SORT_ASCENDING);

    gtk_tree_view_set_model (GTK_TREE_VIEW (priv->mobile_list), sort_model);

    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes ("Country",
                                                       renderer,
                                                       "text", MOBILE_COL_NAME,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (priv->mobile_list), column);
    gtk_tree_view_column_set_clickable (column, TRUE);

    priv->mobile_list_populated = TRUE;
}

static void
mobile_tree_selection_changed (GtkTreeSelection *selection,
                               gpointer user_data)
{
    NmnNewConnection *connection = (NmnNewConnection *) user_data;
    NmnNewConnectionPrivate *priv = GET_PRIVATE (connection);
    GtkTreeModel *model;
    NmnMobileAccessMethod *method = NULL;
    GtkTreeIter iter;

    if (gtk_tree_selection_get_selected (selection, &model, &iter))
        gtk_tree_model_get (model, &iter, MOBILE_COL_METHOD, &method, -1);

    gtk_widget_set_sensitive (priv->mobile_save, method != NULL);
}

static NMConnection *
mobile_new_connection (NmnMobileProvider *provider,
                       NmnMobileAccessMethod *method)
{
    NMConnection *connection;
    NMSetting *type_setting;
    NMSetting *setting;
    char *name;
    char *uuid;

    connection = nm_connection_new ();

    if (method->type == NMN_MOBILE_ACCESS_METHOD_TYPE_GSM) {
        type_setting = nm_setting_gsm_new ();
        g_object_set (type_setting,
                      NM_SETTING_GSM_NUMBER,   "*99#",
                      NM_SETTING_GSM_USERNAME, method->username,
                      NM_SETTING_GSM_PASSWORD, method->password,
                      NM_SETTING_GSM_APN,      method->gsm_apn,
                      NULL);

        /* FIXME: Choose the network_id more intelligently */
        if (g_slist_length (method->gsm_mcc_mnc) == 1) {
            NmnGsmMccMnc *mcc_mnc = (NmnGsmMccMnc *) method->gsm_mcc_mnc->data;
            char *network_id;

            network_id = g_strconcat (mcc_mnc->mcc, mcc_mnc->mnc, NULL);
            g_object_set (type_setting,
                          NM_SETTING_GSM_NETWORK_ID, network_id,
                          NULL);

            g_free (network_id);
        }
    } else if (method->type == NMN_MOBILE_ACCESS_METHOD_TYPE_CDMA) {
        type_setting = nm_setting_cdma_new ();
        g_object_set (type_setting, NM_SETTING_CDMA_NUMBER, "#777", NULL);
    } else
        g_assert_not_reached ();

    nm_connection_add_setting (connection, type_setting);

    if (method->gateway || method->dns) {
        GSList *iter;

        setting = nm_setting_ip4_config_new ();
        g_object_set (setting, NM_SETTING_IP4_CONFIG_METHOD, NM_SETTING_IP4_CONFIG_METHOD_AUTO, NULL);

        for (iter = method->dns; iter; iter = iter->next) {
            struct in_addr addr;

            if (inet_pton (AF_INET, (char *) iter->data, &addr) > 0)
                nm_setting_ip4_config_add_dns (NM_SETTING_IP4_CONFIG (setting), addr.s_addr);
        }

        /* I'd rather not use the gateway from providers database */
        /* if (method->gateway) */

        nm_connection_add_setting (connection, setting);
    }

    /* Serial setting */
    setting = nm_setting_serial_new ();
    g_object_set (setting,
                  NM_SETTING_SERIAL_BAUD, 115200,
                  NM_SETTING_SERIAL_BITS, 8,
                  NM_SETTING_SERIAL_PARITY, 'n',
                  NM_SETTING_SERIAL_STOPBITS, 1,
                  NULL);

    nm_connection_add_setting (connection, setting);

    setting = nm_setting_ppp_new ();
    nm_connection_add_setting (connection, setting);

    setting = nm_setting_connection_new ();
    uuid = nm_utils_uuid_generate ();
    name = g_strdup_printf ("Mobile %s", provider->name);

    g_object_set (setting,
                  NM_SETTING_CONNECTION_ID, name,
                  NM_SETTING_CONNECTION_TYPE, nm_setting_get_name (type_setting),
                  NM_SETTING_CONNECTION_AUTOCONNECT, FALSE,
                  NM_SETTING_CONNECTION_UUID, uuid,
                  NULL);
    g_free (uuid);
    g_free (name);
    nm_connection_add_setting (connection, setting);

    return connection;
}

static void
mobile_save_clicked (GtkButton *button, gpointer user_data)
{
    NmnNewConnection *self = (NmnNewConnection *) user_data;
    NmnNewConnectionPrivate *priv = GET_PRIVATE (self);
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    NmnMobileProvider *provider = NULL;
    NmnMobileAccessMethod *method = NULL;
    GtkTreeIter method_iter;
    GtkTreeIter provider_iter;

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->mobile_list));
    if (!gtk_tree_selection_get_selected (selection, &model, &method_iter))
        return;

    if (!gtk_tree_model_iter_parent (model, &provider_iter, &method_iter))
        return;

    gtk_tree_model_get (model, &method_iter, MOBILE_COL_METHOD, &method, -1);
    gtk_tree_model_get (model, &provider_iter, MOBILE_COL_PROVIDER, &provider, -1);

    if (provider && method) {
        NMConnection *connection;

        connection = NULL;
        connection = mobile_new_connection (provider, method);
        nmn_mobile_provider_unref (provider);

        if (connection) {
            nma_gconf_settings_add_connection (NMA_GCONF_SETTINGS (nmn_nm_data_get_user_settings (priv->nm_data)),
                                               connection);
            g_object_unref (connection);
        }

        gtk_widget_hide (GTK_WIDGET (self));
    }
}

static void
mobile_device_added (NMClient *client,
                     NMDevice *device,
                     gpointer user_data)
{
    if (NM_IS_GSM_DEVICE (device) || NM_IS_CDMA_DEVICE (device)) {
        NmnNewConnection *connection = NMN_NEW_CONNECTION (user_data);

        mobile_list_populate (connection);
        mobile_status_update (connection);
    }
}

static void
mobile_page_init (NmnNewConnection *connection)
{
    NmnNewConnectionPrivate *priv = GET_PRIVATE (connection);
    GtkTreeSelection *selection;

    g_signal_connect (connection, "realize", G_CALLBACK (mobile_list_populate), NULL);

    priv->mobile_label = GTK_WIDGET (gtk_builder_get_object (priv->builder, "new_3g_label"));
    priv->mobile_list = GTK_WIDGET (gtk_builder_get_object (priv->builder, "new_3g_list"));
    priv->mobile_save = GTK_WIDGET (gtk_builder_get_object (priv->builder, "new_3g_save"));
    g_signal_connect (priv->mobile_save, "clicked", G_CALLBACK (mobile_save_clicked), connection);

    g_signal_connect (priv->nm_data, "device-added", G_CALLBACK (mobile_device_added), connection);
    g_signal_connect_swapped (priv->nm_data, "device-removed", G_CALLBACK (mobile_status_update), connection);
    
    priv->mobile_toggled_id = g_signal_connect_swapped (priv->nm_data, "modems-toggled",
                                                        G_CALLBACK (mobile_status_update),
                                                        connection);

    mobile_status_update (connection);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->mobile_list));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
    g_signal_connect (selection, "changed", G_CALLBACK (mobile_tree_selection_changed), connection);
}

static void
nmn_new_connection_init (NmnNewConnection *connection)
{
}

static GObject*
constructor (GType type,
             guint n_construct_params,
             GObjectConstructParam *construct_params)
{
    GObject *object;
    NmnNewConnectionPrivate *priv;
    GtkWidget *container;

    object = G_OBJECT_CLASS (nmn_new_connection_parent_class)->constructor
        (type, n_construct_params, construct_params);

    if (!object)
        return NULL;

    priv = GET_PRIVATE (object);

    if (!priv->builder || !priv->nm_data) {
        g_warning ("Missing constructor arguments");
        g_object_unref (object);
        return NULL;
    }

    container = GTK_WIDGET (gtk_builder_get_object (priv->builder, "new_main_container"));
    gtk_widget_reparent (container, GTK_WIDGET (object));

    wifi_page_init (NMN_NEW_CONNECTION (object));
    mobile_page_init (NMN_NEW_CONNECTION (object));

    priv->close = GTK_WIDGET (gtk_builder_get_object (priv->builder, "new_close"));
    g_signal_connect_swapped (priv->close, "clicked", G_CALLBACK (gtk_widget_hide), object);

    return object;
}

static void
set_property (GObject *object, guint prop_id,
              const GValue *value, GParamSpec *pspec)
{
    NmnNewConnectionPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_BUILDER:
        /* Construct only */
        priv->builder = g_value_dup_object (value);
        break;
    case PROP_NM_DATA:
        /* Construct only */
        priv->nm_data = g_value_dup_object (value);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object, guint prop_id,
              GValue *value, GParamSpec *pspec)
{
    NmnNewConnectionPrivate *priv = GET_PRIVATE (object);

    switch (prop_id) {
    case PROP_BUILDER:
        g_value_set_object (value, priv->builder);
        break;
    case PROP_NM_DATA:
        g_value_set_object (value, priv->nm_data);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
dispose (GObject *object)
{
    NmnNewConnectionPrivate *priv = GET_PRIVATE (object);

    if (priv->disposed)
        return;

    priv->disposed = TRUE;

    if (priv->wifi_list)
        gtk_widget_destroy (priv->wifi_list);

    if (priv->builder)
        g_object_unref (priv->builder);

    g_signal_handler_disconnect (priv->nm_data, priv->mobile_toggled_id);

    if (priv->nm_data)
        g_object_unref (priv->nm_data);

    G_OBJECT_CLASS (nmn_new_connection_parent_class)->dispose (object);
}

static void
nmn_new_connection_class_init (NmnNewConnectionClass *class)
{
    GObjectClass *object_class = G_OBJECT_CLASS (class);

    g_type_class_add_private (object_class, sizeof (NmnNewConnectionPrivate));

    /* methods */
    object_class->constructor = constructor;
    object_class->set_property = set_property;
    object_class->get_property = get_property;
    object_class->dispose = dispose;

    /* properties */
    g_object_class_install_property
        (object_class, PROP_BUILDER,
         g_param_spec_object (NMN_NEW_CONNECTION_BUILDER,
                              "GtkBuilder",
                              "GtkBuilder",
                              GTK_TYPE_BUILDER,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

    g_object_class_install_property
        (object_class, PROP_NM_DATA,
         g_param_spec_object (NMN_NEW_CONNECTION_NM_DATA,
                              "NmnNMData",
                              "NmnNMData",
                              NMN_TYPE_NM_DATA,
                              G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
}
