/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-volumes-source.h"

#include <glib.h>
#include <gio/gio.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <clutk/clutk.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <launcher/launcher.h>
#include <netbook-launcher/netbook-launcher.h>

G_DEFINE_TYPE (NlVolumesSource, nl_volumes_source, G_TYPE_OBJECT);

#define NL_VOLUMES_SOURCE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_VOLUMES_SOURCE, \
  NlVolumesSourcePrivate))

struct _NlVolumesSourcePrivate
{
  NlShell       *shell;
  GVolumeMonitor      *monitor;
  NlPixbufCache *pixbuf_cache;

  ClutterActor *vbox;
  ClutterActor *text;
  ClutterActor *iconview;

  GSList *excluded_volumes;
};

enum
{
  PROP_0,
  PROP_SHELL
};

/* Forwards */
static gboolean reload_volumes (NlVolumesSource *source);
static void     volume_added   (GVolumeMonitor  *monitor,
                                GVolume         *volume,
                                NlVolumesSource *source);
static void     update_emblem  (NlIconTile *tile, GVolume *volume);

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlVolumesSourcePrivate *priv;

  g_return_if_fail (NL_IS_VOLUMES_SOURCE (object));
  priv = NL_VOLUMES_SOURCE_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlVolumesSourcePrivate *priv;

  g_return_if_fail (NL_IS_VOLUMES_SOURCE (object));
  priv = NL_VOLUMES_SOURCE_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}
static void
nl_volumes_source_finalize (GObject *object)
{
  NlVolumesSourcePrivate *priv;

  priv = NL_VOLUMES_SOURCE_GET_PRIVATE (object);

  if (priv->pixbuf_cache)
    {
      g_object_unref (priv->pixbuf_cache);
      priv->pixbuf_cache = NULL;
    }

  if (priv->monitor)
    {
      g_object_unref (priv->monitor);
      priv->monitor = NULL;
    }

  if (priv->excluded_volumes)
    {
      g_slist_foreach (priv->excluded_volumes, (GFunc)g_free, NULL);
      g_slist_free (priv->excluded_volumes);
      priv->excluded_volumes = NULL;
    }

  G_OBJECT_CLASS (nl_volumes_source_parent_class)->finalize (object);
}

static void
nl_volumes_source_constructed (GObject *object)
{
  NlVolumesSourcePrivate *priv;
  ClutterActor *button;

  priv = NL_VOLUMES_SOURCE_GET_PRIVATE (object);

  /* Build the view */
  priv->vbox = ctk_vbox_new (6);
  g_object_ref_sink (priv->vbox);

  priv->text = ctk_text_new (_("<big><b>Volumes</b></big>"));
  ctk_text_set_alignment (CTK_TEXT (priv->text), PANGO_ALIGN_CENTER);
  ctk_box_pack (CTK_BOX (priv->vbox), priv->text, FALSE, FALSE);

  priv->iconview = ctk_icon_view_new ();
  ctk_box_pack (CTK_BOX (priv->vbox), priv->iconview, FALSE, FALSE);

  button = ctk_text_new (" ");
  clutter_actor_set_opacity (button, 0);
  ctk_box_pack (CTK_BOX (priv->vbox), button, FALSE, FALSE);
  
  nl_shell_add_places_source (priv->shell, priv->vbox);

  g_idle_add ((GSourceFunc)reload_volumes, object);
}

static void
nl_volumes_source_class_init (NlVolumesSourceClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);
  GParamSpec   *pspec;

  obj_class->finalize     = nl_volumes_source_finalize;
  obj_class->constructed  = nl_volumes_source_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  g_type_class_add_private (obj_class, sizeof (NlVolumesSourcePrivate));
}

static void
nl_volumes_source_init (NlVolumesSource *source)
{
  NlVolumesSourcePrivate *priv;
  GConfClient *client;

  priv = source->priv = NL_VOLUMES_SOURCE_GET_PRIVATE (source);

  priv->pixbuf_cache = nl_pixbuf_cache_get_default ();

  client = gconf_client_get_default ();
  priv->excluded_volumes = gconf_client_get_list (client,
                           "/apps/netbook-launcher/volume_exclude_list",
                           GCONF_VALUE_STRING,
                           NULL);
  g_object_unref (client);

  priv->monitor = g_volume_monitor_get ();
  g_signal_connect (priv->monitor, "volume-added",
                    G_CALLBACK (volume_added), source);


}

/*
 * Public methods
 */
GObject *
nl_volumes_source_new (NlShell *shell)

{
  GObject *volumes_source;

  g_return_val_if_fail (NL_IS_SHELL (shell), NULL);

  volumes_source = g_object_new (NL_TYPE_VOLUMES_SOURCE,
                                 "shell", shell,
                                 NULL);

  return volumes_source;
}

/*
 * Private methods
 */
static gboolean
volume_is_excluded (NlVolumesSource *source, GVolume *volume)
{
  NlVolumesSourcePrivate *priv = source->priv;
  GSList *v;
  gchar *uid;

  uid = g_volume_get_identifier (volume, G_VOLUME_IDENTIFIER_KIND_HAL_UDI);

  if (!uid)
    return FALSE;

  for (v = priv->excluded_volumes; v; v = v->next)
    {
      if (v->data && strstr (uid, v->data))
        {
          g_free (uid);
          return TRUE;
        }
    }

  g_free (uid);
  return FALSE;
}

static NlIconTile *
make_tile_for_volume (NlVolumesSource *source, GVolume *volume)
{
  ClutterActor *tile;
  GIcon        *icon;
  GdkPixbuf    *pixbuf;

  icon = g_volume_get_icon (volume);
  pixbuf = nl_pixbuf_cache_icon_for_gicon (source->priv->pixbuf_cache,
           icon, 48);
  tile = nl_icon_tile_new (g_volume_get_name (volume),
                           g_volume_get_name (volume),
                           pixbuf);
  if (pixbuf)
    g_object_unref (pixbuf);
  g_object_unref (icon);

  return NL_ICON_TILE (tile);
}

static void
remove_child (ClutterActor *actor, ClutterContainer *container)
{
  clutter_container_remove_actor (container, actor);
}

static void
on_volume_mounted (GVolume          *volume,
                   GAsyncResult     *res,
                   ClutterActor     *shortcut)
{
  gboolean success;
  GError *error = NULL;
  GMount *mount;
  GFile *root;
  gchar *path;

  g_return_if_fail (G_IS_VOLUME (volume));

  success = g_volume_mount_finish (volume, res, NULL);

  if (!success)
    {
      g_debug ("Unsuccessful mount operation");
      return;
    }

  mount = g_volume_get_mount (volume);
  if (!G_IS_MOUNT (mount))
    {
      g_debug ("Mount attempt failed");
      return;
    }

  root = g_mount_get_root (mount);
  path = g_file_get_uri (root);

  g_app_info_launch_default_for_uri (path, NULL, &error);

  if (error)
    {
      GtkWidget *dialog;
      gchar     *title;

      dialog = gtk_message_dialog_new (NULL,
                                       0,
                                       GTK_MESSAGE_ERROR,
                                       GTK_BUTTONS_CLOSE,
                                       "%s",
                                       error->message);

      title = g_strdup_printf (_("Unable to open %s"), path);
      gtk_window_set_title (GTK_WINDOW (dialog), title);
      g_free (title);

      g_signal_connect (dialog, "close", G_CALLBACK (gtk_widget_destroy), NULL);
      g_signal_connect (dialog, "response", G_CALLBACK(gtk_widget_destroy),NULL);

      gtk_widget_show (dialog);

      g_warning ("Unable to mount %s: %s", path, error->message);

      g_error_free (error);
    }
  else
    {
      CtkImage     *image;
      ClutterActor *icon;

      image = ctk_button_get_image (CTK_BUTTON (shortcut));
      icon = ctk_image_new_from_pixbuf (ctk_image_get_size (image),
                                        ctk_image_get_pixbuf (image));
      nl_notify_popup (nl_notify_get_default (),
                       ctk_button_get_label (CTK_BUTTON (shortcut)),
                       CTK_IMAGE (icon),
                       0);
    }
  g_free (path);
  g_object_unref (root);
  g_object_unref (mount);

  update_emblem (NL_ICON_TILE (shortcut), volume);
}

static void
on_volume_clicked (ClutterActor *shortcut, GVolume *volume)
{
  GMount *mount;
  GFile *root;
  gchar *path;
  GError *error = NULL;

  g_return_if_fail (G_IS_VOLUME (volume));

  mount = g_volume_get_mount (volume);
  if (!G_IS_MOUNT (mount))
    {
      if (!g_volume_can_mount (volume))
        return;

      g_volume_mount (volume, 0, NULL, NULL,
                      (GAsyncReadyCallback)on_volume_mounted, shortcut);

      return;
    }

  root = g_mount_get_root (mount);
  path = g_file_get_uri (root);

  g_app_info_launch_default_for_uri (path, NULL, &error);

  if (error)
    {
      GtkWidget *dialog;
      gchar     *title;

      dialog = gtk_message_dialog_new (NULL,
                                       0,
                                       GTK_MESSAGE_ERROR,
                                       GTK_BUTTONS_CLOSE,
                                       "%s",
                                       error->message);

      title = g_strdup_printf (_("Unable to open %s"), path);
      gtk_window_set_title (GTK_WINDOW (dialog), title);
      g_free (title);

      g_signal_connect (dialog, "close", G_CALLBACK (gtk_widget_destroy), NULL);
      g_signal_connect (dialog, "response", G_CALLBACK(gtk_widget_destroy),NULL);

      gtk_widget_show (dialog);

      g_warning ("Unable to mount %s: %s", path, error->message);

      g_error_free (error);
    }
  else
    {
      CtkImage     *image;
      ClutterActor *icon;

      image = ctk_button_get_image (CTK_BUTTON (shortcut));
      icon = ctk_image_new_from_pixbuf (ctk_image_get_size (image),
                                        ctk_image_get_pixbuf (image));
      nl_notify_popup (nl_notify_get_default (),
                       ctk_button_get_label (CTK_BUTTON (shortcut)),
                       CTK_IMAGE (icon),
                       0);
    }

  g_free (path);
  g_object_unref (root);
  g_object_unref (mount);
}

static void
on_volume_ejected (GVolume          *volume,
                   GAsyncResult     *res)
{
  g_volume_eject_finish (volume, res, NULL);
}

static void
on_mount_unmounted (GMount           *mount,
                    GAsyncResult     *res)
{
  g_mount_unmount_finish (mount, res, NULL);
}

static void
on_eject_clicked (GtkWidget *item, GVolume *volume)
{
  g_return_if_fail (G_IS_VOLUME (volume));

  if (g_volume_can_eject (volume))
    {
      g_volume_eject (volume, 0, NULL,
                      (GAsyncReadyCallback)on_volume_ejected, NULL);
    }
  else
    {
      GMount *mount = g_volume_get_mount (volume);

      if (!mount)
        return;
      g_mount_unmount (mount, 0, NULL,
                       (GAsyncReadyCallback)on_mount_unmounted, NULL);
    }
}

static void
show_menu (ClutterActor *actor, guint32 event_time, GVolume *volume)
{
  GtkWidget *menu, *item;

  g_return_if_fail (G_IS_VOLUME (volume));

  menu = gtk_menu_new ();

  item = gtk_image_menu_item_new_from_stock (GTK_STOCK_OPEN, NULL);
  gtk_widget_show (item);
  g_signal_connect(item, "activate",
                   G_CALLBACK (on_volume_clicked), volume);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  if (g_volume_can_eject (volume))
    {
      item = gtk_image_menu_item_new_with_label (_("Eject"));
      g_object_set (item,
                    "image", gtk_image_new_from_stock ("media-eject",
                                                       GTK_ICON_SIZE_MENU),
                    NULL);
      gtk_widget_show (item);
      g_signal_connect (item, "activate", G_CALLBACK (on_eject_clicked), volume);
      gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
    }
  else if (g_volume_get_mount (volume))
    {
      item = gtk_image_menu_item_new_with_label (_("Unmount"));
      g_object_set (item,
                    "image", gtk_image_new_from_stock ("media-eject",
                                                       GTK_ICON_SIZE_MENU),
                    NULL);
      gtk_widget_show (item);
      g_signal_connect (item, "activate", G_CALLBACK (on_eject_clicked), volume);
      gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
    }

  gtk_menu_popup (GTK_MENU (menu),
                  NULL, NULL,
                  NULL, NULL,
                  3, event_time);
}

static void
update_emblem (NlIconTile *tile, GVolume *volume)
{
  ClutterActor  *emblem = NULL;
  GMount        *mount = NULL;
  NlPixbufCache *cache;

  g_return_if_fail (NL_IS_ICON_TILE (tile));
  g_return_if_fail (G_IS_VOLUME (volume));

  cache = nl_pixbuf_cache_get_default ();

  if (g_volume_can_eject (volume) || ((mount = g_volume_get_mount (volume))))
    {
      GdkPixbuf *pixbuf;

      pixbuf = nl_pixbuf_cache_icon_for_name (cache, PKGDATADIR"/eject.png", -1);
      emblem = ctk_image_new_from_pixbuf (24, pixbuf);

      g_object_unref (pixbuf);
    }

  nl_icon_tile_set_emblem (tile, NL_ICON_TILE_EMBLEM_TYPE_ALWAYS, emblem);
  ctk_button_set_label (CTK_BUTTON (tile), g_volume_get_name (volume));

  g_object_unref (cache);
  if (mount)
    g_object_unref (mount);
}

static gboolean
reload_volumes (NlVolumesSource *source)
{
  NlVolumesSourcePrivate *priv;
  GList *volumes, *v;
  GList *children;

  g_return_val_if_fail (NL_IS_VOLUMES_SOURCE (source), FALSE);
  priv = source->priv;

  children = clutter_container_get_children (CLUTTER_CONTAINER (priv->iconview));
  g_list_foreach (children, (GFunc)remove_child, priv->iconview);
  g_list_free (children);

  volumes = g_volume_monitor_get_volumes (priv->monitor);

  for (v = volumes; v; v = v->next)
    {
      GVolume *volume = v->data;
      NlIconTile *tile;

      if (volume_is_excluded (source, volume))
        {
          g_object_unref (volume);
          continue;
        }

      tile = make_tile_for_volume (source, volume);
      clutter_container_add_actor (CLUTTER_CONTAINER (priv->iconview),
                                   CLUTTER_ACTOR (tile));

      update_emblem (tile, volume);

      g_signal_connect (tile, "clicked", G_CALLBACK (on_volume_clicked), volume);
      g_signal_connect (tile, "show-context-menu",
                        G_CALLBACK (show_menu), volume);
      g_signal_connect (tile, "emblem-clicked",
                        G_CALLBACK (on_eject_clicked), volume);

      g_signal_connect_swapped (volume, "changed",
                                G_CALLBACK (update_emblem), tile);
      g_signal_connect_swapped (volume, "removed",
                                G_CALLBACK (reload_volumes), source);

      g_object_set_data (G_OBJECT (volume), "tile", tile);
      g_object_unref (volume);
    }

  g_list_free (volumes);

  return FALSE;
}

static gboolean
is_running (gchar *process_name)
{
  gchar    *argv[] = { "pgrep", "-lfu", NULL, process_name, NULL };
  gchar    *output = NULL;
  gchar    *error = NULL;
    
  argv[2] = (gchar*)g_get_user_name ();

  g_spawn_sync (NULL, argv, NULL, 
                G_SPAWN_SEARCH_PATH, NULL, NULL, 
                &output, &error, NULL, NULL);
  if (output)
    {
      if (g_strstr_len (output, strlen (output), process_name))
        return TRUE;
    }

  return FALSE;
}

static gboolean
launch_volume (GVolume *volume)
{
  GMount       *mount;
  ClutterActor *tile;

  g_return_val_if_fail (G_IS_VOLUME (volume), FALSE);

  tile = (ClutterActor *)g_object_get_data (G_OBJECT (volume), "tile");
  if (!CLUTTER_IS_ACTOR (tile))
    {
      g_debug ("Volume has no tile associated with it");
      return FALSE;
    }

  mount = g_volume_get_mount (volume);
  
  if (!G_IS_MOUNT (mount)
      && !is_running ("gnome-volume-manager")
      && !is_running ("nautilus"))
    {
      g_debug ("Automounting volume");
      on_volume_clicked (tile, volume);
      update_emblem (NL_ICON_TILE (tile), volume);
    }
  else
    {
      if (G_IS_MOUNT (mount))
        g_object_unref (mount);
    }

  return FALSE;
}

static void
volume_added (GVolumeMonitor  *monitor,
              GVolume         *volume,
              NlVolumesSource *self)
{
  g_return_if_fail (NL_IS_VOLUMES_SOURCE (self));

  reload_volumes (self);

  g_idle_add ((GSourceFunc)launch_volume, volume);
}

/*
 * Public Methods
 */
