/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-favorite-app.h"

#include <glib.h>
#include <gio/gio.h>
#include <gio/gdesktopappinfo.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <clutk/clutk.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <launcher/launcher.h>

G_DEFINE_TYPE (NlFavoriteApp, nl_favorite_app,
               NL_TYPE_FAVORITE);

#define NL_FAVORITE_APP_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE \
        ((obj), NL_TYPE_FAVORITE_APP, NlFavoriteAppPrivate))

struct _NlFavoriteAppPrivate
{
  LauncherFavorites *favorites;
  GKeyFile      *desktop_file;
  gchar         *uid;
  gchar         *desktop_path;

  ClutterActor *button;
  ClutterActor *image;
};

enum
{
  PROP_0,
  PROP_UID
};

/* Forwards */
static void on_clicked (ClutterActor *button, NlFavoriteApp *self);

static const gchar        * get_name (NlFavorite *self);
static GdkPixbuf          * get_icon (NlFavorite   *self);
static NlFavoriteViewType   get_view_type (NlFavorite *self);

static gboolean      nl_favorite_app_motion_event (ClutterActor       *actor,
                                                   ClutterMotionEvent *event);
static void          on_show_context_menu (ClutterActor  *button,
                                           guint32        event_time,
                                           NlFavoriteApp *self);
static void          removed (NlFavorite *self);
static gboolean      on_activated (ClutterActor *actor,
                                   ClutterEvent *event,
                                   NlFavorite *self);

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlFavoriteAppPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_APP (object));
  priv = NL_FAVORITE_APP_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_UID:
      if (priv->uid)
        g_free (priv->uid);
      priv->uid = g_value_dup_string (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlFavoriteAppPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_APP (object));
  priv = NL_FAVORITE_APP_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_UID:
      g_value_set_string (value, priv->uid);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
nl_favorite_app_finalize (GObject *object)
{
  NlFavoriteAppPrivate *priv;

  priv = NL_FAVORITE_APP_GET_PRIVATE (object);

  if (priv->uid)
    {
      g_free (priv->uid);
      priv->uid = NULL;
    }
  if (priv->desktop_path)
    {
      g_free (priv->desktop_path);
      priv->desktop_path = NULL;
    }
  if (priv->desktop_file)
    {
      g_key_file_free (priv->desktop_file);
      priv->desktop_file = NULL;
    }

  G_OBJECT_CLASS (nl_favorite_app_parent_class)->finalize (object);
}

static void
nl_favorite_app_constructed (GObject *object)
{
  NlFavoriteAppPrivate *priv;
  NlPixbufCache *cache = nl_pixbuf_cache_get_default ();
  gchar     *desktop_path = NULL;
  gchar     *name = NULL;
  gchar     *temp = NULL;
  gchar     *icon_name = NULL;
  GError    *error = NULL;
  GdkPixbuf *pixbuf = NULL;

  priv = NL_FAVORITE_APP_GET_PRIVATE (object);

  priv->favorites = launcher_favorites_get_default ();

  /* Grab the desktop file path */
  desktop_path = launcher_favorites_get_string (priv->favorites,
                 priv->uid,
                 "desktop_file");
  if (!desktop_path)
    {
      g_warning ("Favorite %s type 'application' requires 'desktop_file' key",
                 priv->uid);
      goto nl_favorite_app_constructed_out;
    }
  priv->desktop_path = desktop_path;

  /* Load the key file */
  priv->desktop_file = g_key_file_new ();

  g_key_file_load_from_file (priv->desktop_file, desktop_path, 0, &error);
  if (error)
    {
      g_warning ("Unable to read favorite desktop file %s: %s",
                 desktop_path,
                 error->message);
      g_error_free (error);
      goto nl_favorite_app_constructed_out;
    }

  temp = g_key_file_get_locale_string (priv->desktop_file,
                                       G_KEY_FILE_DESKTOP_GROUP,
                                       G_KEY_FILE_DESKTOP_KEY_NAME,
                                       NULL,
                                       &error);
  if (error)
    {
      g_warning ("Unable to read favorite desktop file %s: %s",
                 desktop_path,
                 error->message);
      g_error_free (error);
      goto nl_favorite_app_constructed_out;
    }
  name = g_strdup_printf ("<big><b>%s</b></big>", temp);

  icon_name = g_key_file_get_string (priv->desktop_file,
                                     G_KEY_FILE_DESKTOP_GROUP,
                                     G_KEY_FILE_DESKTOP_KEY_ICON,
                                     NULL);
  if (icon_name)
    {
      pixbuf = nl_pixbuf_cache_icon_for_name (cache, icon_name, 64);
    }
  if (!pixbuf)
    {
      pixbuf = nl_pixbuf_cache_icon_for_name (cache,
                                              "application-x-executable",
                                              64);
    }

  priv->button = nl_icon_tile_new (temp, NULL, pixbuf);
  clutter_container_add_actor (CLUTTER_CONTAINER (object), priv->button);
  clutter_actor_show (priv->button);

  priv->image = (ClutterActor*)ctk_button_get_image (CTK_BUTTON (priv->button));
  ctk_image_set_size (CTK_IMAGE (priv->image), 64);
  ctk_image_set_from_pixbuf (CTK_IMAGE (priv->image), pixbuf);


#if 0
  priv->button = ctk_button_new (CTK_ORIENTATION_VERTICAL);
  ctk_button_set_label (CTK_BUTTON (priv->button), name);
  clutter_text_set_markup (CLUTTER_TEXT (ctk_button_get_text
                                         (CTK_BUTTON (priv->button))), name);
  clutter_container_add_actor (CLUTTER_CONTAINER (object), priv->button);
  clutter_actor_show (priv->button);
  priv->image = (ClutterActor*)ctk_button_get_image (CTK_BUTTON (priv->button));
  ctk_image_set_size (CTK_IMAGE (priv->image), 64);
  ctk_image_set_from_pixbuf (CTK_IMAGE (priv->image), pixbuf);
#endif

  g_signal_connect (priv->button, "clicked", G_CALLBACK (on_clicked), object);
  g_signal_connect (object, "button-press-event", G_CALLBACK (on_activated),NULL);
  g_signal_connect (priv->button, "show-context-menu",
                    G_CALLBACK (on_show_context_menu), object);

nl_favorite_app_constructed_out:
  g_free (temp);
  g_free (name);
  g_free (icon_name);
  if (pixbuf) g_object_unref (pixbuf);
  g_object_unref (cache);
}

static void
nl_favorite_app_class_init (NlFavoriteAppClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *act_class = CLUTTER_ACTOR_CLASS (klass);
  NlFavoriteClass   *fav_class = NL_FAVORITE_CLASS (klass);
  GParamSpec        *pspec;

  obj_class->finalize     = nl_favorite_app_finalize;
  obj_class->constructed  = nl_favorite_app_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  act_class->motion_event = nl_favorite_app_motion_event;

  fav_class->get_name      = get_name;
  fav_class->get_icon      = get_icon;
  fav_class->removed       = removed;
  fav_class->get_view_type = get_view_type;

  pspec = g_param_spec_string ("uid", "uid", "uid",  "",
                               G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_UID, pspec);

  g_type_class_add_private (obj_class, sizeof (NlFavoriteAppPrivate));
}

static void
nl_favorite_app_init (NlFavoriteApp *self)
{
  NlFavoriteAppPrivate *priv;

  priv = self->priv = NL_FAVORITE_APP_GET_PRIVATE (self);

  clutter_actor_set_reactive (CLUTTER_ACTOR (self), TRUE);
}

/*
 * Public methods
 */
NlFavorite *
nl_favorite_app_new (const gchar *uid)

{
  NlFavorite *favorite_app;

  favorite_app = g_object_new (NL_TYPE_FAVORITE_APP,
                               "uid", uid,
                               NULL);

  return favorite_app;
}

/*
 * Private methods
 */
static void
removed (NlFavorite *self)
{
  NlFavoriteAppPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE (self));
  priv = NL_FAVORITE_APP (self)->priv;

  launcher_favorites_remove_favorite (priv->favorites, priv->uid);
}

static void
on_clicked (ClutterActor *button, NlFavoriteApp *self)
{
  NlFavoriteAppPrivate *priv;
  LauncherMenu        *menu;
  GSList          *apps, *a;
  GDesktopAppInfo *info;
  GError          *error = NULL;

  g_return_if_fail (NL_IS_FAVORITE_APP (self));
  priv = self->priv;

  menu = launcher_menu_get_default ();

  apps = launcher_menu_get_applications (menu);
  for (a = apps; a; a = a->next)
    {
      LauncherApplication *app = a->data;
      const gchar *app_path = launcher_application_get_desktop_file (app);

      if (g_strcmp0 (app_path, priv->desktop_path) == 0)
        {
          launcher_application_launch (app, &error);

          if (error)
            {
              GtkWidget *dialog;
              gchar     *title;

              dialog = gtk_message_dialog_new (NULL,
                                               0,
                                               GTK_MESSAGE_ERROR,
                                               GTK_BUTTONS_CLOSE,
                                               "%s",
                                               error->message);

              title = g_strdup_printf (_("Unable to open %s"),
                                       launcher_application_get_name (app));
              gtk_window_set_title (GTK_WINDOW (dialog), title);
              g_free (title);

              g_signal_connect (dialog, "close", G_CALLBACK (gtk_widget_destroy),
                                NULL);
              g_signal_connect (dialog, "response", G_CALLBACK(gtk_widget_destroy),
                                NULL);

              gtk_widget_show (dialog);

              g_error_free (error);

              return;
            }
          else
            {
              CtkImage *image = ctk_button_get_image (CTK_BUTTON (priv->button));
              ClutterActor *icon;

              icon = ctk_image_new_from_pixbuf (ctk_image_get_size (image),
                                                ctk_image_get_pixbuf (image));
              nl_notify_popup (nl_notify_get_default (),
                               launcher_application_get_name (app),
                               CTK_IMAGE (icon),
                               0);
            }
          return;
        }
    }

  /* Couldn't find the app, so launch the desktop file directly */
  info = g_desktop_app_info_new_from_keyfile (priv->desktop_file);
  if (!info)
    {
      g_warning ("Unable to load GDesktopAppInfo for %s", priv->desktop_path);
      return;
    }
  g_app_info_launch ((GAppInfo *)info, NULL, NULL, &error);

  if (error)
    {
      g_warning ("Unable to launch %s: %s", priv->desktop_path, error->message);
      g_error_free (error);
    }

  g_object_unref (info);
}

static void
on_user_removed (GtkMenuItem *item, NlFavoriteApp *self)
{
  g_return_if_fail (NL_IS_FAVORITE_APP (self));
  g_signal_emit_by_name (self, "remove-me");
}

static void
on_show_context_menu (ClutterActor  *button,
                      guint32        event_time,
                      NlFavoriteApp *self)
{
  GtkWidget *menu, *item;

  g_return_if_fail (NL_IS_FAVORITE_APP (self));

  menu = gtk_menu_new ();

  item = gtk_image_menu_item_new_from_stock (GTK_STOCK_OPEN, NULL);
  gtk_widget_show (item);
  g_signal_connect (item, "activate", G_CALLBACK (on_clicked), self);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  item = gtk_image_menu_item_new_from_stock (GTK_STOCK_REMOVE, NULL);
  gtk_widget_show (item);
  g_signal_connect (item, "activate", G_CALLBACK (on_user_removed), self);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  gtk_menu_popup (GTK_MENU (menu),
                  NULL, NULL,
                  NULL, NULL,
                  3, event_time);

}

static const gchar *
get_name (NlFavorite *self)
{
  NlFavoriteAppPrivate *priv;

  g_return_val_if_fail (NL_IS_FAVORITE_APP (self), NULL);
  priv = NL_FAVORITE_APP (self)->priv;

  return ctk_button_get_label (CTK_BUTTON (priv->button));
}

static GdkPixbuf *
get_icon (NlFavorite   *self)
{
  NlFavoriteAppPrivate *priv;

  g_return_val_if_fail (NL_IS_FAVORITE_APP (self), NULL);
  priv = NL_FAVORITE_APP (self)->priv;

  return ctk_image_get_pixbuf (CTK_IMAGE (priv->image));
}

static NlFavoriteViewType
get_view_type (NlFavorite *self)
{
  return NL_FAVORITE_VIEW_FIXED;
}

static gboolean
nl_favorite_app_motion_event (ClutterActor       *actor,
                              ClutterMotionEvent *event)
{
  g_return_val_if_fail (NL_IS_FAVORITE_APP (actor), FALSE);

  if (event->modifier_state & CLUTTER_BUTTON1_MASK)
    {
      g_signal_emit_by_name (actor, "begin-drag-move", event);
      return TRUE;
    }
  return FALSE;
}

static gboolean
on_activated (ClutterActor *actor,
              ClutterEvent *event,
              NlFavorite *self)
{
  g_signal_emit_by_name (actor, "active");

  return FALSE;
}


/*
 * Public Methods
 */
