## -*- coding: utf-8 -*-
#
# «changer» - Mythbuntu Control Centre class for making change
#
# This script:
# Copyright (C) 2007, Mario Limonciello, for Mythbuntu
#
#
# Mythbuntu is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free
# Software Foundation; either version 2 of the License, or at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this application; if not, write to the Free Software Foundation, Inc., 51
# Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##################################################################################

#Package install support
import warnings
warnings.filterwarnings("ignore", "apt API not stable yet", FutureWarning)
import apt
import apt_pkg

#GUI support
import pygtk
pygtk.require('2.0')
import gtk

#Terminal Widget support
import vte

#File, String, & Process manipulation
import os
import shutil
import re
import string
import subprocess
import time
import sys
import tempfile

from popen2 import Popen3

#Package reloading support
import thread

#Borrow GDebi's install methods
from GDebi.GDebi import GDebi

#Glade directory
GLADEDIR = '/usr/share/mythbuntu-control-centre/glade'

ltsp_base_dir='/opt/ltsp/'

#Translation Support
from gettext import gettext as _

#common functions
from mythbuntu_common.vnc import VNCHandler
from mythbuntu_common.mysql import MySQLHandler
from mythbuntu_common.lirc import LircHandler
from mythbuntu_common.debconftalk import debconftalk

def utf8(str):
  return unicode(str, 'latin1').encode('utf-8')


# needed for paul cannon's interface magic
# eg http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/439093
import socket
import fcntl
import struct
import array

def get_ip_address(ifname):
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    return socket.inet_ntoa(fcntl.ioctl(
        s.fileno(),
        0x8915,  # SIOCGIFADDR
        struct.pack('256s', ifname[:15])
    )[20:24])

class Applicator():
    def __init__(self,to_install,to_remove,to_reconfigure):
        """Takes input from core and passes it on to Applicator"""

        #Initialize common frameworks
        self.mysql=MySQLHandler()
        self.vnc=VNCHandler()
        self.lirc=LircHandler()

        #Initialize GUI & Terminal
        self.glade = gtk.glade.XML(GLADEDIR + '/' + 'progress_dialogs.glade')
        for widget in self.glade.get_widget_prefix(""):
            setattr(self, widget.get_name(), widget)
            if isinstance(widget, gtk.Label):
                widget.set_property('can-focus', False)
        self.glade.signal_autoconnect(self)
        self._term = vte.Terminal()
        self._term.show()
        self._term.set_font_from_string("monospace 10")
        self.expander_install.add(self._term)
        self.progress_dialog.show()

        #Don't show anything during package installs
        os.putenv('DEBIAN_FRONTEND','noninteractive')

        #Apt Cache
        apt_pkg.Config.Set("APT::Install-Recommends","0")
        self.cache=apt.Cache()

        #Other Changes
        self.to_reconfigure=to_reconfigure

        #debconf communicator
        self.communicator = debconftalk()

        #for tracking diskless stuff
        self.lines = 0

        #Packages to install/remove
        try:
            self._init_error=False
            self.package_changes=False
            for pkg in to_install:
                self.cache[pkg].markInstall()
                self.package_changes=True
            for pkg in to_remove:
                self.cache[pkg].markDelete()
                self.package_changes=True
        except KeyError, msg:
            self._init_error = True
            header = _("Could not mark all packages")
            body = _("Check your repository lists for universe,  "
                    "multiverse, and main.  Perform a package"
                    "list update as well.")
            self.show_alert(gtk.MESSAGE_ERROR, header, body, msg,
                parent=self.progress_dialog)
            self.progress_dialog.hide()
        except SystemError, msg:
            self._init_error = True
            header = _("Broken apt package cache")
            body = _("Check your repository lists for universe,  "
                    "multiverse, and main.  Perform a package"
                    "list update as well.")
            self.show_alert(gtk.MESSAGE_ERROR, header, body, msg,
                parent=self.progress_dialog)
            self.progress_dialog.hide()

        while gtk.events_pending():
            gtk.main_iteration()

    def get_init_error(self):
        return self._init_error

    def show_alert(self, type, header, body=None, details=None, parent=None):
        if parent is not None:
             self.dialog_hig.set_transient_for(parent)
        else:
             self.dialog_hig.set_transient_for(self.progress_dialog)

        message = "<b><big>%s</big></b>" % header
        if not body == None:
             message = "%s\n\n%s" % (message, body)
        self.label_hig.set_markup(message)

        if not details == None:
             buffer = self.textview_hig.get_buffer()
             buffer.set_text(str(details))
             self.expander_hig.set_expanded(False)
             self.expander_hig.show()

        if type == gtk.MESSAGE_ERROR:
             self.image_hig.set_property("stock", "gtk-dialog-error")
        elif type == gtk.MESSAGE_WARNING:
             self.image_hig.set_property("stock", "gtk-dialog-warning")
        elif type == gtk.MESSAGE_INFO:
             self.image_hig.set_property("stock", "gtk-dialog-info")

        res = self.dialog_hig.run()
        self.dialog_hig.hide()
        if res == gtk.RESPONSE_CLOSE:
            return True
        return False

    def acquire_lock(self):
        " lock the pkgsystem for install "

        # check if we can lock the apt database
        try:
            apt_pkg.PkgSystemLock()
        except SystemError:
            self.error_header = _("Only one software management tool is allowed to"
                                  " run at the same time")
            self.error_body = _("Please close the other application e.g. 'Update "
                                "Manager', 'aptitude' or 'Synaptic' first.")
            return False
        return True

    def release_lock(self):
        " release the pkgsystem lock "
        apt_pkg.PkgSystemLock()
        return True

    def commit_changes(self):
        """Called to launch things"""
        if self.package_changes:
            if not self.acquire_lock():
                  self.show_alert(gtk.MESSAGE_ERROR, self.error_header, self.error_body)
                  return False

            self.fprogress = GDebi.FetchProgressAdapter(self.progressbar,
                                                  self.action,
                                                  self.progress_dialog)
            self.iprogress = self.MythbuntuInstallProgressAdapter(self.progressbar,
                                                    self._term,
                                                    self.action,
                                                    self.expander_install)
            errMsg = None
            try:

                res = self.cache.commit(self.fprogress,self.iprogress)
            except IOError, msg:
                res = False
                errMsg = "%s" % msg
                header = _("Could not download all required files")
                body = _("Please check your internet connection or "
                        "installation medium.")
            except SystemError, msg:
                res = False
                header = _("Could not install all dependencies"),
                body = _("Usually this is related to an error of the "
                        "software distributor. See the terminal window for "
                        "more details.")
            if not res:
                self.show_alert(gtk.MESSAGE_ERROR, header, body, msg,
                                parent=self.progress_dialog)
                self.progress_dialog.hide()
                return

        self.prepare_configuration(self.progressbar,self.expander_install)
        self.process_configuration()
        self.progress_dialog.hide()

#Overrideen GDebi Methods
    class MythbuntuInstallProgressAdapter(GDebi.InstallProgressAdapter):
        def startUpdate(self):
            #print "startUpdate"
            apt_pkg.PkgSystemUnLock()
            self.term_expander.show()
            self.action.set_markup("<i>"+_("Installing and Removing packages...")+"</i>")
            self.progress.set_fraction(0.0)
            self.progress.set_text("")

    def prepare_configuration(self,progressbar,expander):
        """Prepares the GUI for the post install steps"""
        expander.hide()
        progressbar.set_text("")
        self.update_gui(0.0,"Post-install Mythbuntu Configuration")
        while gtk.events_pending():
            gtk.main_iteration()

    def update_gui(self,progress,new_text=None):
        """Updates the GUI to show what we are working on"""
        self.progressbar.set_fraction(progress)
        if new_text != None:
            self.action.set_markup("<i>"+_(new_text)+"</i>")
        time.sleep(0.5)
        while gtk.events_pending():
            gtk.main_iteration()

#Action Functions
    def process_configuration(self):
        """Interprets the to_reconfigure variable and does everything described"""
        mysql_config = {"user":None,"password":None,"database":None,"server":None}
        mplayer = None
        xine_ui = None
        vlc = None
        mythtv_backend = None
        mythtv_frontend = None
        mythvideo = None
        mythgallery = None
        mythmusic = None
        autologin = None
        autologin_user = None
        samba = None
        nfs_kernel_server = None
        x11vnc = None
        vnc_password = None
        mysql_server = None
        remote_config = {"remote":None}
        transmitter_config = {"transmitter":None}
        lircrc = None
        mythweb_auth_activated = None
        mythweb_auth_username = None
        mythweb_auth_password = None
        mysql_tweaks_enabled = None
        enable_nightly_defrag = None
        new_mysql_repair = None
        medibuntu_change = None
        enable_ivtv_tweak = None
        enable_rtc_tweak = None
        diskless_build = None
        diskless_delete = None
        diskless_pen = None
        diskless_arch = None
        diskless_sign = None
        diskless_wol = None
        diskless_block_device = None
        diskless_nbd_device = None
        diskless_server_nfs = None
        diskless_server_dhcp = None
        master_backend_address = None

        for item in self.to_reconfigure:
            #Mysql.txt
            if item == "mythtv_mysql_pass":
                mysql_config["password"] = self.to_reconfigure[item]
            elif item == "mythtv_mysql_user":
                mysql_config["user"] = self.to_reconfigure[item]
            elif item == "mythtv_mysql_database":
                mysql_config["database"] = self.to_reconfigure[item]
            elif item == "mythtv_mysql_server":
                mysql_config["server"] = self.to_reconfigure[item]
            #Media apps
            elif item == "mplayer":
                mplayer = self.to_reconfigure[item]
            elif item == "xine-ui":
                xine_ui = self.to_reconfigure[item]
            elif item == "vlc":
                vlc = self.to_reconfigure[item]
            #frontend and backend
            elif item == "mythtv-backend":
                mythtv_backend = self.to_reconfigure[item]
            elif item == "mythtv-frontend":
                mythtv_frontend = self.to_reconfigure[item]
            #Plugins
            elif item == "mythvideo":
                mythvideo = self.to_reconfigure[item]
            elif item == "mythgallery":
                mythgallery = self.to_reconfigure[item]
            elif item == "mythmusic":
                mythmusic = self.to_reconfigure[item]
            #Autologin
            elif item == "autologin":
                autologin = self.to_reconfigure[item]
            elif item == "autologin_user":
                autologin_user = self.to_reconfigure[item]
            #Services
            elif item == "samba":
                samba = self.to_reconfigure[item]
            elif item == "nfs-kernel-server":
                nfs_kernel_server = self.to_reconfigure[item]
            elif item == "x11vnc":
                x11vnc = self.to_reconfigure[item]
            elif item == "vnc_password":
                vnc_password = self.to_reconfigure[item]
            elif item == "mysql-server":
                mysql_server = self.to_reconfigure[item]
            #Remotes
            elif item == "remote" or item == "remote_modules" or item == "remote_device" or item == "remote_driver" or item == "remote_lircd_conf":
                remote_config[item] = self.to_reconfigure[item]
            elif item == "transmitter" or item == "transmitter_modules" or item == "transmitter_device" or item == "transmitter_driver" or item == "transmitter_lircd_conf":
                transmitter_config[item] = self.to_reconfigure[item]
            elif item == "lircrc":
                lircrc = self.to_reconfigure[item]
            #mythweb
            elif item == "mythweb_auth_activated":
                 mythweb_auth_activated = self.to_reconfigure[item]
            elif item == "mythweb_auth_username":
                mythweb_auth_username = self.to_reconfigure[item]
            elif item == "mythweb_auth_password":
                mythweb_auth_password = self.to_reconfigure[item]
            elif item == "mysql_tweaks_enabled":
                mysql_tweaks_enabled = self.to_reconfigure[item]
            elif item == "enable_nightly_defrag":
                enable_nightly_defrag = self.to_reconfigure[item]
            elif item == "new_mysql_repair":
                new_mysql_repair = self.to_reconfigure[item]
            #Extra Repos
            elif item == "medibuntu":
                medibuntu_change = self.to_reconfigure[item]
            #tweaks
            elif item == "enable_ivtv_tweak":
                enable_ivtv_tweak = self.to_reconfigure[item]
            elif item == "enable_rtc_tweak":
                enable_rtc_tweak = self.to_reconfigure[item]
            #diskless support
            elif item == "diskless_build":
                diskless_build = self.to_reconfigure[item]
            elif item == "diskless_delete":
                diskless_delete = self.to_reconfigure[item]
            elif item == "diskless_pen":
                diskless_pen = self.to_reconfigure[item]
            elif item == "diskless_sign":
                diskless_signed = self.to_reconfigure[item]
            elif item == "diskless_arch":
                diskless_arch = self.to_reconfigure[item]
            elif item == "diskless_wol":
                diskless_wol = self.to_reconfigure[item]
            elif item == "diskless_nbd_device":
                diskless_nbd_device = self.to_reconfigure[item]
            elif item == "diskless_block_device":
                diskless_block_device = self.to_reconfigure[item]
            elif item == "mythbuntu-diskless-server":
                diskless_server_nfs = self.to_reconfigure[item]
            elif item == "mythbuntu-diskless-server-standalone":
                diskless_server_dhcp = self.to_reconfigure[item]
            elif item == "master-backend-address":
                master_backend_address = self.to_reconfigure[item]
               

        #diskless stuff
        if diskless_build is not None and diskless_build:
            self.update_gui(0.05,"Building diskless image")
            self.build_diskless(diskless_build,diskless_arch,diskless_sign)
        else:
            self.update_gui(0.05)
        if diskless_delete is not None and diskless_delete:
            self.update_gui(0.1,"Deleting diskless image")
            self.delete_diskless(diskless_delete,diskless_arch)
        else:
            self.update_gui(0.1)
        if diskless_pen is not None and diskless_pen:
            self.update_gui(0.15,"Building bootable pen drive")
            self.diskless_pen(diskless_pen,diskless_arch,diskless_wol,diskless_block_device,diskless_nbd_device)
        else:
            self.update_gui(0.15)
        
        if diskless_server_nfs is not None and master_backend_address is not None and diskless_server_nfs:
            self.update_gui(0.16,"Enabling overlay directory for mythbuntu-diskless")
            # note: the NFS share will be exported to the world which is a very bad thing
            # this should be fixed, eg by making the user select the network in the UI
            self.communicator.run("set","mythbuntu-diskless/create_share true")
            tmp = master_backend_address.split('.')
            network =  tmp[0] + '.' + tmp[1] + '.' + tmp[2] + '.0/24'
            self.communicator.run("set","mythbuntu-diskless/share_host " + network)
            subprocess.call(["dpkg-reconfigure", "-fnoninteractive", "mythbuntu-diskless-server"])
        else:
            self.update_gui(0.16)

        if diskless_server_dhcp is not None and master_backend_address is not None and diskless_server_dhcp:
            self.update_gui(0.17,"Configuring DHCP server")
            self.diskless_dhcp(master_backend_address)
        else:
            self.update_gui(0.17)

        #Services
        if samba is not None and samba:
            self.update_gui(.175,"Configuring Samba Service")
            self.configure_samba(mythtv_backend,mythvideo,mythmusic,mythgallery)
        else:
            self.update_gui(.175)
        if nfs_kernel_server is not None and nfs_kernel_server:
            self.update_gui(.250,"Configuring NFS Service")
            self.configure_nfs(master_backend_address,mythtv_backend,mythvideo,mythmusic,mythgallery)
        else:
            self.update_gui(.250)
        if x11vnc is not None:
            self.update_gui(.375,"Configuring VNC Service")
            #Because vnc4server broke, don't enable vnc, but always disable it
            #If we are reconfiguring
            #self.vnc.toggle_xorg(x11vnc)
            self.vnc.toggle_xorg(False)
            if x11vnc:
                self.vnc.create_password(vnc_password)
        else:
            self.update_gui(.375)
        if mysql_server is not None:
            self.update_gui(.400,"Configuring MySQL Service")
            self.mysql.toggle_mysql_service_config(mysql_server)
            self.mysql.restart_mysql_service()
        else:
            self.update_gui(.400)
        #Mysql.txt
        if mysql_config["password"] is not None and mysql_config["user"] is not None and mysql_config["database"] is not None and mysql_config["server"] is not None:
            self.update_gui(.500,"Configuring Remote MythTV Connection Information...")
            self.mysql.update_config(mysql_config)
            self.mysql.write_mysql_txt("/etc/mythtv/mysql.txt")
        else:
            self.update_gui(.500)
        # mysqld tweaks
        if mysql_tweaks_enabled is not None:
            self.update_gui(.550,"Configuring MySQL tweaks")
            self.mysql_tweaks(mysql_tweaks_enabled)
        else:
            self.update_gui(.550)
        if new_mysql_repair is not None:
            self.update_gui(.575,"Configuring MySQL optimization/repair cron job")
            self.mysql_repair(new_mysql_repair)
        else:
            self.update_gui(.575)
        #Extra Repos
        if medibuntu_change is not None:
            self.update_gui(.600,"Configuring Medibuntu Repository")
            self.update_medibuntu()
            self.update_gui(.640,"Updating Package Lists")
            self.update_package_lists()
        else:
            self.update_gui(.625)

        #Autologin
        if autologin is not None:
            self.update_gui(.650,"Configuring Automatic Login...")
            self.configure_autologin(autologin,autologin_user)
        else:
            self.update_gui(.650)
        # xfs defrag
        if enable_nightly_defrag is not None:
            self.update_gui(.725,"Configuring daily XFS defragmentation")
            self.xfs_defrag(enable_nightly_defrag)
        else:
            self.update_gui(.725)

        #Remote
        if remote_config["remote"] is not None:
            self.update_gui(.800,"Configuring Remote Control...")
            self.lirc.set_device(remote_config,"remote")
        else:
            self.update_gui(.800)
        #transmitter
        if transmitter_config["transmitter"] is not None:
            self.update_gui(.850,"Configuring IR transmitter...")
            self.lirc.set_device(transmitter_config,"transmitter")
        else:
            self.update_gui(.850)
        #remote/transmitter common
        if remote_config["remote"] is not None or transmitter_config["transmitter"] is not None:
            #For advanced remote controls, we will need an interactive
            #debconf frontend
            try:
                interactive = self.lirc.query_advanced()
                self.lirc.write_hardware_conf()
                self.lirc.reconfigure_lirc(interactive)
            except:
                header = _("Error reconfiguring LIRC")
                body =("There was a problem reconfiguring your IR devices. "
                       "Please manually reconfigure lirc via: "
                       "sudo dpkg-reconfigure lirc")
                self.show_alert(gtk.MESSAGE_ERROR, header, body, None,
                    parent=self.progress_dialog)
        # mythweb auth
        if mythweb_auth_activated is not None or mythweb_auth_username is not None or mythweb_auth_password is not None:
            self.update_gui(.925,"Setting up MythWeb authentication")
            self.config_mythweb_auth(mythweb_auth_activated,mythweb_auth_username,mythweb_auth_password)
        else:
            self.update_gui(.925)
        # IVTV Tweaks
        if enable_ivtv_tweak is not None:
            self.update_gui(.975,"Configuring IVTV Memory Tweak...")
            self.ivtv_tweak(enable_ivtv_tweak)
        else:
            self.update_gui(.975)
        # RTC Video Tweak
        if enable_rtc_tweak is not None:
            self.update_gui(.995,"Configuring RTC Video Tweak...")
            self.rtc_tweak(enable_rtc_tweak)
        else:
            self.update_gui(.995)
        #Lircrc
        if lircrc is not None and lircrc:
            self.update_gui(1.00,"Configuring Application Specific Remote Buttons...")
            self.lirc.create_lircrc()
        else:
            self.update_gui(1.00)

        if x11vnc is not None:
            header = _("Please logout for changes to take effect")
            body = _("The x11vnc has been modified or enabled. "
                     "In order for the changes to take effect, "
                     "you will need to log out/in.")
            self.show_alert(gtk.MESSAGE_INFO, header, body, None,
                parent=self.progress_dialog)

    def mysql_tweaks(self,mysql_tweaks):
        # string comparison. No, it's not a boolean variable
        if mysql_tweaks == "True":
            out_mt = open("/etc/mysql/conf.d/mythtv-tweaks.cnf", "w")
            # some of these seem to be in gutsy proper already
            out_mt.write("[mysqld]\n")
            out_mt.write("# The following values were partly taken from:\n")
            out_mt.write("# http://www.gossamer-threads.com/lists/mythtv/users/90942#90942\n")
            # out_mt.write("key_buffer = 48M\n")
            # out_mt.write("max_allowed_packet = 8M\n")
            out_mt.write("table_cache = 128\n")
            out_mt.write("sort_buffer_size = 48M\n")
            out_mt.write("net_buffer_length = 8M\n")
            # out_mt.write("thread_cache_size = 4\n")
            out_mt.write("query_cache_type = 1\n")
            out_mt.write("query_cache_size = 32M\n")
            out_mt.write("# don't do binary logs for mythconverg\n")
            out_mt.write("binlog_ignore_db = mythconverg\n")
            out_mt.close()
        elif mysql_tweaks == "False":
            os.remove("/etc/mysql/conf.d/mythtv-tweaks.cnf")

    def configure_autologin(self,enable,user):
        """Configures autologin for user if enable is True"""
        #Login Session Option
        home = ''
        uid = ''
        gid = ''
        in_f=open("/etc/passwd")
        for line in in_f:
            fields =  string.split(line,":")
            id=int(fields[2])
            if fields[0] == user:
                uid = int(fields[2])
                gid = int(fields[3])
                home = fields[5]
                break
        if enable:
            #dmrc, used to define which session is autologin
            if os.path.exists(home + '/.dmrc') and not os.path.exists(home + '/.dmrc.mythbuntu-old'):
                shutil.move(home + '/.dmrc', home + '/.dmrc.mythbuntu-old')
            shutil.copy('/etc/skel/.dmrc', home + '/.dmrc')
            #start myth frontend automatically
            if not os.path.exists(home + '/.config/autostart'):
                os.makedirs(home + '/.config/autostart')
            if not os.path.exists(home + '/.config/autostart/mythtv.desktop'):
                try:
                    os.symlink('/usr/share/applications/mythtv.desktop',home + '/.config/autostart/mythtv.desktop')
                except OSError:
                    os.unlink(home + '/.config/autostart/mythtv.desktop')
                    os.symlink('/usr/share/applications/mythtv.desktop',home + '/.config/autostart/mythtv.desktop')
            if uid != '' and gid != '':
                os.chown(home + '/.dmrc',uid,gid)
                os.chown(home + '/.config',uid,gid)
                os.chown(home + '/.config/autostart',uid,gid)
        else:
            if os.path.exists(home + '/.dmrc'):
                os.remove(home + '/.dmrc')
            if os.path.exists(home + '/.dmrc.mythbuntu-old'):
                shutil.move(home + '/.dmrc.mythbuntu-old', home + '/.dmrc')
            if os.path.exists(home + '/.config/autostart/mythtv.desktop'):
                os.remove(home + '/.config/autostart/mythtv.desktop')
        #GDM Configuration
        if os.path.exists("/etc/gdm/gdm-cdd.conf"):
            shutil.copy("/etc/gdm/gdm-cdd.conf","/etc/gdm/gdm-cdd.conf.mythbuntu-old")
            print "Copying /etc/gdm/gdm-cdd.conf to /etc/gdm/gdm-cdd.conf.mythbuntu-old"
            in_f=open("/etc/gdm/gdm-cdd.conf.mythbuntu-old")
            out_f=open("/etc/gdm/gdm-cdd.conf","w")
            pattern=re.compile("^AutomaticLoginEnable=|^AutomaticLogin=|^TimedLoginEnable=|^TimedLogin=")
            for line in in_f:
                if re.compile("^AutomaticLoginEnable=").search(line) is not None:
                    if enable:
                        out_f.write("AutomaticLoginEnable=true\n")
                    else:
                        out_f.write("AutomaticLoginEnable=false\n")
                elif re.compile("^AutomaticLogin=").search(line):
                    if enable:
                        out_f.write("AutomaticLogin=" + user + "\n")
                    else:
                        out_f.write(line)
                elif re.compile("^TimedLoginEnable=").search(line) is not None:
                    if enable:
                        out_f.write("TimedLoginEnable=true\n")
                    else:
                        out_f.write("TimedLoginEnable=false\n")
                elif re.compile("^TimedLogin=").search(line):
                    if enable:
                        out_f.write("TimedLogin=" + user + "\n")
                    else:
                        out_f.write(line)
                else:
                    out_f.write(line)
            out_f.close()

    def configure_samba(self,mythtv,videos,music,pictures):
        """Writes a sane default samba configuration"""
        if os.path.exists("/etc/samba/smb.conf"):
            print "Copying existing /etc/samba/smb.conf to /etc/samba/smb.conf.mythbuntu-old"
            shutil.copy("/etc/samba/smb.conf","/etc/samba/smb.conf.mythbuntu-old")
        out_f=open("/etc/samba/smb.conf","w")
        out_f.write("[global]\n")
        out_f.write("workgroup = MSHOME\n")
        out_f.write("server string = %h server (Samba, Mythbuntu)\n")
        out_f.write("log file = /var/log/samba/log.%m\n")
        out_f.write("max log size = 1000\n")
        out_f.write("syslog = 0\n")
        out_f.write("panic action = /usr/share/samba/panic-action %d\n")
        out_f.write("dns proxy = no\n")
        out_f.write("security = share\n")
        out_f.write("\n")
        if mythtv:
            out_f.write("[recordings]\n")
            out_f.write("comment = TV Recordings\n")
 # FIXME: this value should be pulled from the mysql DB, IMHO
 # Note: it'd be host-specific.
 # Current code also doesn't account for storage groups. --laga
            out_f.write("path = /var/lib/mythtv/recordings\n")
            out_f.write("public = yes\n")
            out_f.write("writable = no\n")
            out_f.write("create mask = 0777\n")
            out_f.write("directory mask = 0777\n")
            out_f.write("force user = nobody\n")
            out_f.write("force group = nogroup\n")
            out_f.write("\n")
        if videos:
            out_f.write("[videos]\n")
            out_f.write("comment = Videos\n")
            out_f.write("path = /var/lib/mythtv/videos\n")
            out_f.write("public = yes\n")
            out_f.write("writable = yes\n")
            out_f.write("create mask = 0660\n")
            out_f.write("directory mask = 0770\n")
            out_f.write("force user = mythtv\n")
            out_f.write("force group = mythtv\n")
            out_f.write("\n")
        if music:
            out_f.write("[music]\n")
            out_f.write("comment = Music\n")
            out_f.write("path = /var/lib/mythtv/music\n")
            out_f.write("public = yes\n")
            out_f.write("writable = yes\n")
            out_f.write("create mask = 0660\n")
            out_f.write("directory mask = 0770\n")
            out_f.write("force user = mythtv\n")
            out_f.write("force group = mythtv\n")
            out_f.write("\n")
        if pictures:
            out_f.write("[pictures]\n")
            out_f.write("comment = Pictures\n")
            out_f.write("path = /var/lib/mythtv/pictures\n")
            out_f.write("public = yes\n")
            out_f.write("writable = yes\n")
            out_f.write("create mask = 0660\n")
            out_f.write("directory mask = 0770\n")
            out_f.write("force user = mythtv\n")
            out_f.write("force group = mythtv\n")
            out_f.write("\n")
        out_f.close()

    def configure_nfs(self,network,mythtv,videos,music,pictures):
        if network is None:
            net = "*"
        else:
            tmp = network.split('.')
            # we could also get the netmask from ifconfig
            net = tmp[0] + '.' + tmp[1] + '.' + tmp[2] + '.0/24'
        if os.path.exists("/etc/exports"):
            print "Copying /etc/exports to /etc/exports.mythbuntu-old"
            shutil.copy("/etc/exports","/etc/exports.mythbuntu-old")
        out_f=open("/etc/exports","a")
        if mythtv:
            pattern = re.compile("^/var/lib/mythtv/recordings")
            in_f = open("/etc/exports.mythbuntu-old")
            sys_in = in_f.read()
            in_f.close()
            if not pattern.search(sys_in):
                out_f.write("/var/lib/mythtv/recordings    " + net + "(ro,async,no_root_squash,no_subtree_check)\n")
        if videos:
            pattern = re.compile("^/var/lib/mythtv/videos")
            in_f = open("/etc/exports.mythbuntu-old")
            sys_in = in_f.read()
            in_f.close()
            if not pattern.search(sys_in):
                out_f.write("/var/lib/mythtv/videos    " + net + "(ro,async,no_root_squash,no_subtree_check)\n")
        if music:
            pattern = re.compile("^/var/lib/mythtv/music")
            in_f = open("/etc/exports.mythbuntu-old")
            sys_in = in_f.read()
            in_f.close()
            if not pattern.search(sys_in):
                out_f.write("/var/lib/mythtv/music    " + net + "(ro,async,no_root_squash,no_subtree_check)\n")
        if pictures:
            pattern = re.compile("^/var/lib/mythtv/pictures")
            in_f = open("/etc/exports.mythbuntu-old")
            sys_in = in_f.read()
            in_f.close()
            if not pattern.search(sys_in):
                out_f.write("/var/lib/mythtv/pictures    " + net + "(ro,async,no_root_squash,no_subtree_check)\n")
        out_f.close()

    def config_mythweb_auth(self,mythweb_auth_activated,mythweb_auth_username,mythweb_auth_password):

        # python does not want to mix boolean and strings, that's why we convert
        # mythweb_auth_activated here
        if mythweb_auth_activated == True:
            self.communicator.run('set', 'mythweb/enable true')
            # that extra whitespace in there is probably a good idea
            self.communicator.run('set', 'mythweb/username ' + mythweb_auth_username)
            self.communicator.run('set', 'mythweb/password ' + mythweb_auth_password)
        elif mythweb_auth_activated == False:
            self.communicator.run('set', 'mythweb/enable false')

        # this should be made more robust
        # eg check for errors and print a nice warning message
        subprocess.call(["dpkg-reconfigure", "-fnoninteractive", "mythweb"])

    def update_medibuntu(self):
        #toggles medibuntu
        medibuntuDeb = 'deb http://packages.medibuntu.org/ hardy free non-free'
        #find it
        in_f=open("/etc/apt/sources.list")
        pattern=re.compile("^deb http://packages.medibuntu.org/ hardy free non-free")
        medibuntu_exists=False
        for line in in_f:
            if pattern.search(line
            ):
                medibuntu_exists=True
        if not medibuntu_exists and os.path.exists("/etc/apt/sources.list.d/medibuntu.list"):
            medibuntu_exists=True
        #act
        if medibuntu_exists == False and self.check_medibuntu_key() == False:
            os.system("wget -q http://packages.medibuntu.org/medibuntu-key.gpg -O- | apt-key add -")
        self.update_source(medibuntuDeb, medibuntu_exists)

    def update_source(self, line, remove):
        from aptsources import distro
        from aptsources.sourceslist import SourcesList, SourceEntry
        from softwareproperties.SoftwareProperties import SoftwareProperties
        SoftwareProperties = SoftwareProperties()
        if remove == False:
            SoftwareProperties.sourceslist.list.append(SourceEntry(line))
        else:
            SoftwareProperties.sourceslist.list.remove(SourceEntry(line))
        SoftwareProperties.save_sourceslist()
        SoftwareProperties.reload_sourceslist()


    def xfs_defrag(self,enable_nightly_defrag):
        # string comparison. No, it's not a boolean variable
        if enable_nightly_defrag == "True":
            out_defrag = open("/etc/cron.daily/mythtv-xfs-defrag", "w")
            # the cron entry could handle failures more gracefully, but I don't see why currently
            out_defrag.write("#!/bin/sh\n")
            out_defrag.write("/usr/bin/ionice -c3 /usr/sbin/xfs_fsr\n")
            out_defrag.close()
            os.chmod("/etc/cron.daily/mythtv-xfs-defrag",0755)
        elif enable_nightly_defrag == "False":
            os.remove("/etc/cron.daily/mythtv-xfs-defrag")

    def mysql_repair(self,enable_mysql_repair):
        if enable_mysql_repair == "True":
            shutil.copy("/usr/share/doc/mythtv-backend/contrib/optimize_mythdb.pl", "/etc/cron.daily/optimize_mythdb.pl")
            os.chmod("/etc/cron.daily/optimize_mythdb.pl",0755)
        elif enable_mysql_repair == "False":
            os.remove("/etc/cron.daily/optimize_mythdb.pl")

    def update_package_lists(self):
        lock = thread.allocate_lock()
        lock.acquire()
        t = thread.start_new_thread(self.update_cache,
                                   (self.progress_dialog.window.xid, lock))
        while lock.locked():
            while gtk.events_pending():
                gtk.main_iteration()
                time.sleep(0.05)

    def check_medibuntu_key(self):
        #Checks if the medibuntu key is installed
        import gettext
        from subprocess import PIPE
        gpg = ["/usr/bin/gpg"]
        base_opt = gpg + ["--no-options", "--no-default-keyring",
                                    "--secret-keyring", "/etc/apt/secring.gpg",
                                    "--trustdb-name", "/etc/apt/trustdb.gpg",
                                    "--keyring", "/etc/apt/trusted.gpg"]
        list_opt = base_opt + ["--with-colons", "--batch",
                               "--list-keys"]
        p = subprocess.Popen(list_opt,stdout=PIPE).stdout
        for line in p.readlines():
            fields = line.split(":")
            if fields[4][-8:] == "0C5A2783":
                return True
        return False

    def update_cache(self, window_id, lock):
        """start synaptic to update the package cache"""
        try:
            apt_pkg.PkgSystemUnLock()
        except SystemError:
            pass
        cmd = ["/usr/sbin/synaptic", "--hide-main-window",
               "--non-interactive",
               "--parent-window-id", "%s" % (window_id),
               "--update-at-startup"]
        subprocess.call(cmd)
        lock.release()

    def ivtv_tweak(self,enable_ivtv_tweak):
        if enable_ivtv_tweak == "True":
            try:
                in_f = open("/etc/sysctl.conf")
                lines = in_f.readlines()
                in_f.close()
            except IOError:
                print "Unable to Open File!"
            out_f = open("/etc/sysctl.conf","w")
            for line in lines:
                if "#vm.min_free_kbytes=16384" in line:
                    line = line.replace(line, "vm.min_free_kbytes=16384\n")
                    out_f.write(line)
                elif "vm.min_free_kbytes=16384" in line:
                    out_f.write(line)
                else:
                    out_f.write(line)
            out_f.close()
            pattern = re.compile("vm.min_free_kbytes=16384$")
            in_f = open("/etc/sysctl.conf")
            sys_in = in_f.read()
            in_f.close()
            if not pattern.search(sys_in):
                out_f = open("/etc/sysctl.conf","a")
                out_f.write("vm.min_free_kbytes=16384\n")
                out_f.close()
                subprocess.call(['sysctl', '-w', 'vm.min_free_kbytes=16384'])
        elif enable_ivtv_tweak == "False":
            try:
                in_f = open("/etc/sysctl.conf")
                lines = in_f.readlines()
                in_f.close()
            except IOError:
                print "Unable to Open File!"
            out_f = open("/etc/sysctl.conf","w")
            for line in lines:
                if "vm.min_free_kbytes=16384" in line:
                    line = line.replace(line, "#vm.min_free_kbytes=16384\n")
                    out_f.write(line)
                else:
                    out_f.write(line)
            out_f.close()
            subprocess.call(['sysctl', '-w', 'vm.min_free_kbytes=16384'])

    def rtc_tweak(self,enable_rtc_tweak):
        if enable_rtc_tweak == "True":
            try:
                in_f = open("/etc/sysctl.conf")
                lines = in_f.readlines()
                in_f.close()
            except IOError:
                print "Unable to Open File!"
            out_f = open("/etc/sysctl.conf","w")
            for line in lines:
                if "#dev.rtc.max-user-freq=1024" in line:
                    subprocess.call(['sysctl', '-w', 'dev.rtc.max-user-freq=1024'])
                    line = line.replace(line, "dev.rtc.max-user-freq=1024\n")
                    out_f.write(line)
                elif "dev.rtc.max-user-freq=1024" in line:
                    subprocess.call(['sysctl', '-w', 'dev.rtc.max-user-freq=1024'])
                    out_f.write(line)
                else:
                    out_f.write(line)
            out_f.close()
            pattern = re.compile("dev.rtc.max-user-freq=1024$")
            in_f = open("/etc/sysctl.conf")
            sys_in = in_f.read()
            in_f.close()
            if not pattern.search(sys_in):
                out_f = open("/etc/sysctl.conf","a")
                out_f.write("\ndev.rtc.max-user-freq=1024\n")
                out_f.close()
                subprocess.call(['sysctl', '-w', 'dev.rtc.max-user-freq=1024'])
        elif enable_rtc_tweak == "False":
            try:
                in_f = open("/etc/sysctl.conf")
                lines = in_f.readlines()
                in_f.close()
            except IOError:
                print "Unable to Open File!"
            out_f = open("/etc/sysctl.conf","w")
            for line in lines:
                if "dev.rtc.max-user-freq=1024" in line:
                    subprocess.call(['sysctl', '-w', 'dev.rtc.max-user-freq=64'])
                    line = line.replace(line, "#dev.rtc.max-user-freq=1024\n")
                    out_f.write(line)
                else:
                    out_f.write(line)
            out_f.close()

    def delete_diskless(self,enable,arch):
        if enable:
            try:
                #in case LRM got started
                os.system("umount " + ltsp_base_dir + arch + "/lib/modules/*/volatile")
                if os.path.exists(ltsp_base_dir + arch):
                    shutil.rmtree(ltsp_base_dir + arch)
                if os.path.exists(ltsp_base_dir + "/images/" + arch + ".img"):
                    os.remove(ltsp_base_dir + "/images/" + arch + ".img")
            except OSError,e:
                header = _("Error deleting image")
                self.show_alert(gtk.MESSAGE_INFO, header, e, None,
                parent=self.progress_dialog)

    def build_diskless(self,enable,arch,signed=False):
        if enable:
            ltsp_args = '--mythbuntu --mythbuntu-copy-user-credentials --copy-sourceslist --arch ' + arch + ' --base ' + ltsp_base_dir
            if signed:
                ltsp_args = ltsp_args + ' --accept-unsigned-packages'

            progbar = self.progressbar
            fraction=0

            # other locales break parsing
            oldlocale = os.getenv('LC_ALL')
            os.putenv('LC_ALL','en_GB') 
            pp=Popen3('/usr/sbin/ltsp-build-client ' + ltsp_args + ' 2>/dev/null')
            if oldlocale is None:
                oldlocale = ''
            os.putenv('LC_ALL',oldlocale)
            #for nn in xrange(3472): # this seems to be number of lines spit out by ltsp-build-client
            for nn in xrange(4083):
                while gtk.events_pending():
                    gtk.main_iteration(True)
                line = pp.fromchild.readline().strip()
                if line.startswith('I: '):
                    line = line.lstrip('I: ')
                out = self.line_filter(line)
                if out:
                    progbar.set_text(out)
                    #step = fraction+0.000666667 #0.000731529 for http, 0.000877193 for cdroms
                    step = fraction+0.000389863 # mythbuntu-diskless in hardy
                    # reset when we hit 100 percent.  this shouldn't happen
                    # old behavior: reset before it starts to install things
                    if step > 1.0:
                        step = 0
                    progbar.set_fraction(step)
                    fraction = step

            progbar.set_fraction(1.0)
            progbar.set_text('LTSP Setup Done...')

    def diskless_pen(self,enable,arch,wol,block,nbd):
        """Creates a pen drive to use with diskless stuff"""
        if self.mcc_debug():
            print 'ltsp_base_dir: ' + ltsp_base_dir
            print 'arch: ' + arch
            print 'block: ' + block
            print 'nbd: ' + nbd
        def diskless_pendrive_dowork():
            #install syslinux
            os.system('syslinux /dev/' + partition)
            #give them their MBR
            os.system('dd if=/usr/lib/syslinux/mbr.bin of=/dev/' + block)
            #mount partition
            if os.system('mount /dev/' + partition + ' ' + temp) != 0:
                e = 'Sorry, ' + block + ' could not be mounted'
                header = _("Error preparing pen drive")
                self.show_alert(gtk.MESSAGE_INFO, header, e, None,
                parent=self.progress_dialog)
            else:
                self.diskless_pendrive_mounted=True
            conf = open(temp + 'SYSLINUX.CFG', 'w')
            try:
                if shutil.copy(kerneldir + 'vmlinuz',temp) != None:
                    raise IOError("52","copying kernel")
                if shutil.copy(kerneldir + 'initrd.img',temp) != None:
                    raise IOError("51","copying initrd")
                if conf.write(syslinux_conf) != None:
                    raise IOError("50","writing syslinux to partition")
            except IOError,e:
                header = _("Error preparing pen drive")
                self.show_alert(gtk.MESSAGE_INFO, header, e, None,
                parent=self.progress_dialog)
            finally:
                conf.close()
                os.system('umount /dev/' + partition)
        if enable:
            if  wol is not None:
                wol_arg = 'wol=' + wol
            else:
                wol_arg = ''
            try:
                server = get_ip_address(nbd)

            except IOError,e:
                header = _("Error detecting IP address")
                self.show_alert(gtk.MESSAGE_INFO, header, e, None,
                parent=self.progress_dialog)

            # first partition on device
            partition = block + '1'
            temp = tempfile.mkdtemp() + '/'
            kerneldir = '/var/lib/tftpboot/ltsp/' + arch + '/'
            nbdport = self.find_nbd_port(arch)
            if nbdport is None:
                header= _("Error detecting NBD server port")
                message = _("Make sure /etc/inetd.conf is up to date.\nIf you think it's correct, please submit a bug report")
                self.show_alert(gtk.MESSAGE_INFO, header, message, None,
                parent=self.progress_dialog)
                # we should abort if we get an error, but return() will just let the script continue
                # otherwise, we will
                # self.check_mount_n_umount(block)
                # return
            
            syslinux_conf = 'default diskless\ntimeout 3\nprompt 1\nlabel diskless\nkernel vmlinuz\nappend initrd=initrd.img nbdroot=' + server  + ' ' + wol_arg + ' nbdport=' + nbdport + ' ip=dhcp\n'


            self.check_mount_n_umount(block)
            diskless_pendrive_dowork()
            #don't clean up.  it's in /tmp, let someone else handle it.
            #shutil.rmtree(temp)

    def diskless_dhcp(self,addr):
        """Sets up /etc/ltsp/dhcpd.conf"""
        tmp = addr.split('.')
        network =  tmp[0] + '.' + tmp[1] + '.' + tmp[2]
        if not os.path.isfile('/etc/ltsp/dhcpd.conf'):
            try:
                shutil.copyfile('/usr/share/doc/ltsp-server/examples/dhcpd.conf','/etc/ltsp/dhcpd.conf')
                subprocess.check_call(['sed', '-i','/etc/ltsp/dhcpd.conf','-e','s/192.168.0/' + network + '/g'])
            except:
                pass
                return
        else:
            header= _("/etc/ltsp/dhcpd.conf exists")
            message = _("The configuration file for the DHCP server (/etc/ltsp/dhcpd.conf)\nalready exists.\nYou'll have to edit it manually to suit your network setup")
            self.show_alert(gtk.MESSAGE_INFO, header, message, None,
            parent=self.progress_dialog)
    

    def check_mount_n_umount(self,item):
        """Checks whether a device or path is mounted somewhere"""
        mounts = open('/proc/mounts')
        mount=None
        pattern = re.compile('.*/dev/' + item + '.*' )
        for line in mounts:
            # FIXME: how do escape the whole string?
            if pattern.search(line) != None:
                mount=string.split(line," ")[0]
        mounts.close()
        if mount is not None:
            os.system("umount " + mount)

    def find_nbd_port(self,arch):
        if not os.path.exists('/etc/inetd.conf'):
            self.display_error('/etc/inetd.conf couldn\'t be found. Please make sure\n that LTSP is installed properly')
        else:
            file = open('/etc/inetd.conf')
            pattern = re.compile( '.*nbdrootd' + '.*' + ltsp_base_dir + 'images/' + arch + '.img' + '.*')
            pport = re.compile('^[0-9]+')
            for line in file:
                if pattern.match(line) != None:
                    m = pport.match(line)
                    if m != None:
                        # FIXME: add more sanity checking
                        return m.group()
            file.close()

    def line_filter(self, line):
        if self.mcc_debug():
            print line
        if line.startswith('Retrieving') or line.startswith('Validating') or line.startswith('Extracting') or line.startswith('Installing') or line.startswith('Unpacking') or line.startswith('Configuring') or line.startswith('Need to get') or line.startswith('Setting up') or line.startswith('Cleaning'):
            line = line.split('(')[0]
            if not line.endswith('...'):
                line=line+'...'
            self.lines=self.lines+1
            return line
        elif line.startswith('Get:'):
            line = 'Retrieving '+line.split()[3]+'...'
            self.lines=self.lines+1
            return line
        elif line.startswith('Parallel mksquashfs:'):
            line = 'Compressing image. This can take a while.'
            self.lines=self.lines+1
            return line
        else:
            return

    def mcc_debug(self):
        env = os.getenv('MCC_DEBUG')
        return env == 'true' or env == '1' or env == 'True' or env == 'TRUE' or env == 'Yes' or env == 'YES' 
