/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*- */

#define _ISOC99_SOURCE /* for roundf */
#include <math.h>

#include "mutter-window-private.h"
#include "mutter-window-group.h"

struct _MutterWindowGroupClass
{
  ClutterGroupClass parent_class;
};

struct _MutterWindowGroup
{
  ClutterGroup parent;

  MetaScreen   *screen;
  ClutterActor *above;
  ClutterActor *normal;
  ClutterActor *below;
};

G_DEFINE_TYPE (MutterWindowGroup, mutter_window_group, CLUTTER_TYPE_GROUP);

ClutterActor *
mutter_window_group_get_above (MutterWindowGroup *group)
{
  g_return_val_if_fail (MUTTER_IS_WINDOW_GROUP (group), NULL);
  
  return group->above;
}

ClutterActor *
mutter_window_group_get_normal (MutterWindowGroup *group)
{
  g_return_val_if_fail (MUTTER_IS_WINDOW_GROUP (group), NULL);
  
  return group->normal;
}

ClutterActor *
mutter_window_group_get_below (MutterWindowGroup *group)
{
  g_return_val_if_fail (MUTTER_IS_WINDOW_GROUP (group), NULL);
  
  return group->below;
}

void
mutter_window_group_pack_window (MutterWindowGroup *group, 
                                 MutterWindow *window)
{
  MetaCompWindowType type;
  ClutterContainer  *container;
  
  g_return_if_fail (MUTTER_IS_WINDOW_GROUP (group));
  g_return_if_fail (MUTTER_IS_WINDOW (window));
  
  type = mutter_window_get_window_type (window);
  
  if (type == META_COMP_WINDOW_DROPDOWN_MENU ||
      type == META_COMP_WINDOW_POPUP_MENU ||
      type == META_COMP_WINDOW_TOOLTIP ||
      type == META_COMP_WINDOW_NOTIFICATION ||
      type == META_COMP_WINDOW_COMBO ||
      type == META_COMP_WINDOW_DND ||
      type == META_COMP_WINDOW_OVERRIDE_OTHER)
    {
      container = CLUTTER_CONTAINER (group->above);
    }
  /*else if (type == META_COMP_WINDOW_DESKTOP)
    {
      container = CLUTTER_CONTAINER (group->below);
    }*/
  else
    {
      container = CLUTTER_CONTAINER (group->normal);
    }
  
  clutter_container_add_actor (container, CLUTTER_ACTOR (window));
}

/* We want to find out if the window is "close enough" to
 * 1:1 transform. We do that by converting the transformed coordinates
 * to 24.8 fixed-point before checking if they look right.
 */
static inline int
round_to_fixed (float x)
{
  return roundf (x * 256);
}

/* We can only (easily) apply our logic for figuring out what a window
 * obscures if is not transformed. This function does that check and
 * as a side effect gets the position of the upper-left corner of the
 * actors.
 *
 * (We actually could handle scaled and non-integrally positioned actors
 * too as long as they weren't shaped - no filtering is done at the
 * edges so a rectangle stays a rectangle. But the gain from that is
 * small, especally since most of our windows are shaped. The simple
 * case we handle here is the case that matters when the user is just
 * using the desktop normally.)
 *
 * If we assume that the window group is untransformed (it better not
 * be!) then we could also make this determination by checking directly
 * if the actor itself is rotated, scaled, or at a non-integral position.
 * However, the criterion for "close enough" in that case get trickier,
 * since, for example, the allowed rotation depends on the size of
 * actor. The approach we take here is to just require everything
 * to be within 1/256th of a pixel.
 */
static gboolean
actor_is_untransformed (ClutterActor *actor,
                        int          *x_origin,
                        int          *y_origin)
{
  
  if (clutter_actor_is_scaled (actor) || clutter_actor_is_rotated (actor))
    return FALSE;

  gfloat out_x, out_y;
  clutter_actor_get_position (actor, &out_x, &out_y);
 
  *x_origin = round_to_fixed (out_x) >> 8;
  *y_origin = round_to_fixed (out_y) >> 8;

  return TRUE;
}

static GList *
mutter_window_group_get_windows (MutterWindowGroup *window_group)
{
  GList *children, *other;
  
  children = clutter_container_get_children (CLUTTER_CONTAINER (window_group->below));
  
  other = clutter_container_get_children (CLUTTER_CONTAINER (window_group->normal));
  children = g_list_concat (children, other);
  
  other = clutter_container_get_children (CLUTTER_CONTAINER (window_group->above));
  children = g_list_concat (children, other);
  
  return children;
}

static void
mutter_window_group_paint (ClutterActor *actor)
{
  MutterWindowGroup *window_group = MUTTER_WINDOW_GROUP (actor);
  GdkRegion *visible_region;
  GdkRectangle screen_rect = { 0 };
  GList *children, *l;

  /* We walk the list from top to bottom (opposite of painting order),
   * and subtract the opaque area of each window out of the visible
   * region that we pass to the windows below.
   */
  children = mutter_window_group_get_windows (window_group);
  children = g_list_reverse (children);

  /* Start off with the full screen area (for a multihead setup, we
   * might want to use a more accurate union of the monitors to avoid
   * painting in holes from mismatched monitor sizes. That's just an
   * optimization, however.)
   */
  meta_screen_get_size (window_group->screen, &screen_rect.width, &screen_rect.height);
  visible_region = gdk_region_rectangle (&screen_rect);

  for (l = children; l; l = l->next)
    {
      MutterWindow *cw;
      gboolean x, y;

      if (!MUTTER_IS_WINDOW (l->data) || !CLUTTER_ACTOR_IS_VISIBLE (l->data))
        continue;

      cw = l->data;

      if (!actor_is_untransformed (CLUTTER_ACTOR (cw), &x, &y))
        continue;

      /* Temporarily move to the coordinate system of the actor */
      gdk_region_offset (visible_region, - x, - y);

      mutter_window_set_visible_region (cw, visible_region);

      if (clutter_actor_get_paint_opacity (CLUTTER_ACTOR (cw)) == 0xff)
        {
          GdkRegion *obscured_region = mutter_window_get_obscured_region (cw);
          if (obscured_region)
            gdk_region_subtract (visible_region, obscured_region);
        }

      mutter_window_set_visible_region_beneath (cw, visible_region);
      gdk_region_offset (visible_region, x, y);
    }

  gdk_region_destroy (visible_region);

  CLUTTER_ACTOR_CLASS (mutter_window_group_parent_class)->paint (actor);

  /* Now that we are done painting, unset the visible regions (they will
   * mess up painting clones of our actors)
   */
  for (l = children; l; l = l->next)
    {
      MutterWindow *cw;

      if (!MUTTER_IS_WINDOW (l->data))
        continue;

      cw = l->data;
      mutter_window_reset_visible_regions (cw);
    }

  g_list_free (children);
}

static void
mutter_window_group_class_init (MutterWindowGroupClass *klass)
{
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  actor_class->paint = mutter_window_group_paint;
}

static void
mutter_window_group_init (MutterWindowGroup *window_group)
{
  window_group->below  = clutter_group_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (window_group), window_group->below);
  
  window_group->normal = clutter_group_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (window_group), window_group->normal);
  
  window_group->above  = clutter_group_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (window_group), window_group->above);
  
  clutter_container_raise_child (CLUTTER_CONTAINER (window_group), window_group->normal, NULL);
  clutter_container_raise_child (CLUTTER_CONTAINER (window_group), window_group->above, NULL);
}

ClutterActor *
mutter_window_group_new (MetaScreen *screen)
{
  MutterWindowGroup *window_group;

  window_group = g_object_new (MUTTER_TYPE_WINDOW_GROUP, NULL);

  window_group->screen = screen;

  return CLUTTER_ACTOR (window_group);
}
