/* mhs: A GObject wrapper for the Mozilla Mhs API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>

#include "mhs-error.h"
#include "mhs-error-private.h"

#include "mhs-cookies.h"
#include "mhs-cookies-bindings.h"
#include "mhs-service.h"
#include "mhs-marshal.h"

G_DEFINE_TYPE (MhsCookies, mhs_cookies, G_TYPE_OBJECT)

#define COOKIES_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MHS_TYPE_COOKIES, MhsCookiesPrivate))

struct _MhsCookiesPrivate
{
  DBusGProxy *proxy;
};

enum
{
  COOKIE_CHANGED_SIGNAL,
  COOKIE_REJECTED_SIGNAL,

  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

static void
mhs_cookies_dispose (GObject *object)
{
  MhsCookies *self = (MhsCookies *) object;
  MhsCookiesPrivate *priv = self->priv;

  if (priv->proxy)
    {
      g_object_unref (priv->proxy);
      priv->proxy = NULL;
    }

  G_OBJECT_CLASS (mhs_cookies_parent_class)->dispose (object);
}

static void
mhs_cookies_class_init (MhsCookiesClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MhsCookiesPrivate));

  object_class->dispose = mhs_cookies_dispose;

  signals[COOKIE_CHANGED_SIGNAL] =
    g_signal_new ("cookie-changed",
                  G_TYPE_FROM_CLASS (object_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsCookiesClass, cookie_changed),
                  NULL, NULL,
                  mhs_marshal_VOID__STRING_STRING,
                  G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_STRING);

  signals[COOKIE_REJECTED_SIGNAL] =
    g_signal_new ("cookie-rejected",
                  G_TYPE_FROM_CLASS (object_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MhsCookiesClass, cookie_rejected),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  dbus_g_object_register_marshaller (mhs_marshal_VOID__STRING_STRING,
                                     G_TYPE_NONE,
                                     G_TYPE_STRING,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID);
}

static void
mhs_cookies_cookie_changed_cb (DBusGProxy  *proxy,
                               const gchar *detail,
                               const gchar *cookie,
                               gpointer data)
{
  g_signal_emit (data, signals[COOKIE_CHANGED_SIGNAL], 0, detail, cookie);
}

static void
mhs_cookies_cookie_rejected_cb (DBusGProxy  *proxy,
                                const gchar *uri,
                                gpointer     data)
{
  g_signal_emit (data, signals[COOKIE_REJECTED_SIGNAL], 0, uri);
}

static void
mhs_cookies_init (MhsCookies *self)
{
  DBusGConnection *connection;

  GError *error = NULL;
  MhsCookiesPrivate *priv = self->priv = COOKIES_PRIVATE (self);

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      priv->proxy =
        dbus_g_proxy_new_for_name (connection,
                                   MHS_SERVICE_COOKIES,
                                   MHS_SERVICE_COOKIES_PATH,
                                   MHS_SERVICE_COOKIES_INTERFACE);

      dbus_g_connection_unref (connection);

      dbus_g_proxy_add_signal (priv->proxy, "CookieChanged",
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "CookieRejected",
                               G_TYPE_STRING,
                               G_TYPE_INVALID);

      dbus_g_proxy_connect_signal (priv->proxy, "CookieChanged",
                                   G_CALLBACK (mhs_cookies_cookie_changed_cb),
                                   self, NULL);
      dbus_g_proxy_connect_signal (priv->proxy, "CookieRejected",
                                   G_CALLBACK (mhs_cookies_cookie_rejected_cb),
                                   self, NULL);
    }
}

MhsCookies *
mhs_cookies_new (void)
{
  return (MhsCookies *)g_object_new (MHS_TYPE_COOKIES, NULL);
}

static gboolean
mhs_cookies_check_proxy (MhsCookies  *self,
                         GError     **error)
{
  if (self->priv->proxy)
    return TRUE;

  g_set_error (error, MHS_ERROR, MHS_ERROR_PROXY,
               "Failed to initialize DBUS proxy");

  return FALSE;
}

#define CHECK_ERROR_AND_PROXY \
  MhsCookiesPrivate *priv; \
  gboolean ret; \
  g_return_val_if_fail (MHS_IS_COOKIES (self), FALSE); \
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE); \
  if (!mhs_cookies_check_proxy (self, error)) \
    return FALSE; \
  priv = self->priv;

/* nsICookieService bindings */
gboolean
mhs_cookies_get_cookie_string (MhsCookies   *self,
                               const gchar  *uri,
                               gchar       **cookie,
                               GError      **error)
{
  CHECK_ERROR_AND_PROXY

  ret = org_moblin_mhs_Cookies_get_cookie_string (priv->proxy,
                                                  uri,
                                                  cookie,
                                                  error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_cookies_get_cookie_string_from_http (MhsCookies   *self,
                                         const gchar  *uri,
                                         const gchar  *first_uri,
                                         gchar       **cookie,
                                         GError      **error)
{
  CHECK_ERROR_AND_PROXY

  ret = org_moblin_mhs_Cookies_get_cookie_string_from_http (priv->proxy,
                                                            uri,
                                                            first_uri,
                                                            cookie,
                                                            error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_cookies_set_cookie_string (MhsCookies   *self,
                               const gchar  *uri,
                               const gchar  *cookie,
                               GError      **error)
{
  CHECK_ERROR_AND_PROXY

  ret = org_moblin_mhs_Cookies_set_cookie_string (priv->proxy,
                                                  uri,
                                                  cookie,
                                                  error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_cookies_set_cookie_string_from_http (MhsCookies   *self,
                                         const gchar  *uri,
                                         const gchar  *first_uri,
                                         const gchar  *cookie,
                                         const gchar  *time_string,
                                         GError      **error)
{
  CHECK_ERROR_AND_PROXY

  ret = org_moblin_mhs_Cookies_set_cookie_string_from_http (priv->proxy,
                                                            uri,
                                                            first_uri,
                                                            cookie,
                                                            time_string,
                                                            error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

/* nsICookieManager bindings */
gboolean
mhs_cookies_remove_all (MhsCookies   *self,
                        GError     **error)
{
  CHECK_ERROR_AND_PROXY

  ret = org_moblin_mhs_Cookies_remove_all (priv->proxy,
                                           error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_cookies_get_all (MhsCookies  *self,
                     GPtrArray  **cookies, // GPtrArray{GValueArray}
                     GError     **error)
{
  CHECK_ERROR_AND_PROXY

  ret = org_moblin_mhs_Cookies_get_all (priv->proxy,
                                        cookies,
                                        error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_cookies_remove (MhsCookies   *self,
                    const gchar  *domain,
                    const gchar  *name,
                    const gchar  *path,
                    gboolean      blocked,
                    GError      **error)
{
  CHECK_ERROR_AND_PROXY

  ret = org_moblin_mhs_Cookies_remove (priv->proxy,
                                       domain,
                                       name,
                                       path,
                                       blocked,
                                       error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

/* nsICookieManager2 bindings */
gboolean
mhs_cookies_add (MhsCookies   *self,
                 const gchar  *domain,
                 const gchar  *path,
                 const gchar  *name,
                 const gchar  *value,
                 gboolean      is_secure,
                 gboolean      is_http_only,
                 gboolean      is_session,
                 gint64        expiry,
                 GError      **error)
{
  CHECK_ERROR_AND_PROXY

  ret = org_moblin_mhs_Cookies_add (priv->proxy,
                                    domain,
                                    path,
                                    name,
                                    value,
                                    is_secure,
                                    is_http_only,
                                    is_session,
                                    expiry,
                                    error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_cookies_count_cookies_from_host (MhsCookies   *self,
                                     const gchar  *host,
                                     guint        *count,
                                     GError      **error)
{
  CHECK_ERROR_AND_PROXY

  ret = org_moblin_mhs_Cookies_count_cookies_from_host (priv->proxy,
                                                        host,
                                                        count,
                                                        error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

gboolean
mhs_cookies_import_cookies (MhsCookies   *self,
                            const gchar  *path,
                            GError      **error)
{
  CHECK_ERROR_AND_PROXY

  ret = org_moblin_mhs_Cookies_import_cookies (priv->proxy,
                                               path,
                                               error);

  _mhs_error_translate_from_dbus (error);

  return ret;
}

