using System;
using System.Collections;
using System.Globalization;
using System.Reflection;
using System.Reflection.Emit;

namespace System.Text.RegularExpressions {

	//
	// Compiler which generates IL bytecode to perform the matching instead of
	// interpreting a program.
	// For simplicity, we inherit from RxCompiler, and generate the IL code based
	// on the program generated by it. This also allows us to fallback to interpretation
	// if we can't handle something.
	// This is net 2.0, since 1.0 doesn't support DynamicMethods
	// FIXME: Add support for 1.0, and CompileToAssembly
	//

#if NET_2_0
	class CILCompiler : RxCompiler, ICompiler {
		DynamicMethod[] eval_methods;
		bool[] eval_methods_defined;
		int group_count;

		static FieldInfo fi_str = typeof (RxInterpreter).GetField ("str", BindingFlags.Instance|BindingFlags.NonPublic);
		static FieldInfo fi_string_start = typeof (RxInterpreter).GetField ("string_start", BindingFlags.Instance|BindingFlags.NonPublic);
		static FieldInfo fi_string_end = typeof (RxInterpreter).GetField ("string_end", BindingFlags.Instance|BindingFlags.NonPublic);
		static FieldInfo fi_match_start = typeof (RxInterpreter).GetField ("match_start", BindingFlags.Instance|BindingFlags.NonPublic);
		static FieldInfo fi_program = typeof (RxInterpreter).GetField ("program", BindingFlags.Instance|BindingFlags.NonPublic);
		static FieldInfo fi_marks = typeof (RxInterpreter).GetField ("marks", BindingFlags.Instance|BindingFlags.NonPublic);
		static FieldInfo fi_groups = typeof (RxInterpreter).GetField ("groups", BindingFlags.Instance|BindingFlags.NonPublic);
		static FieldInfo fi_mark_start = typeof (Mark).GetField ("Start", BindingFlags.Instance|BindingFlags.Public|BindingFlags.NonPublic);
		static FieldInfo fi_mark_end = typeof (Mark).GetField ("End", BindingFlags.Instance|BindingFlags.Public|BindingFlags.NonPublic);
		static MethodInfo mi_is_word_char = typeof (RxInterpreter).GetMethod ("IsWordChar", BindingFlags.Static|BindingFlags.NonPublic);
		static MethodInfo mi_reset_groups = typeof (RxInterpreter).GetMethod ("ResetGroups", BindingFlags.Instance|BindingFlags.NonPublic);
		static MethodInfo mi_open = typeof (RxInterpreter).GetMethod ("Open", BindingFlags.Instance|BindingFlags.NonPublic);
		static MethodInfo mi_close = typeof (RxInterpreter).GetMethod ("Close", BindingFlags.Instance|BindingFlags.NonPublic);

		IMachineFactory ICompiler.GetMachineFactory () {
			byte[] code = new byte [curpos];
			Buffer.BlockCopy (program, 0, code, 0, curpos);

			eval_methods = new DynamicMethod [code.Length];
			eval_methods_defined = new bool [code.Length];

			group_count = 1 + (code [1] | (code [2] << 8));

			// The main eval method
		    DynamicMethod main = GetEvalMethod (code, 11);
			if (main != null)
				return new RxInterpreterFactory (code, (EvalDelegate)main.CreateDelegate (typeof (EvalDelegate)));
			else
				return new RxInterpreterFactory (code, null);
		}

		DynamicMethod GetEvalMethod (byte[] program, int pc) {
			if (eval_methods_defined [pc])
				return eval_methods [pc];

			// FIXME: Recursion ?
			eval_methods_defined [pc] = true;

			eval_methods [pc] = CreateEvalMethod (program, pc);
			return eval_methods [pc];
		}

		private MethodInfo GetInterpreterMethod (string name) {
			return typeof (RxInterpreter).GetMethod (name, BindingFlags.Instance|BindingFlags.NonPublic);
		}

		private int ReadInt (byte[] code, int pc)
		{
			int val = code [pc];
			val |= code [pc + 1] << 8;
			val |= code [pc + 2] << 16;
			val |= code [pc + 3] << 24;
			return val;
		}

		class Frame {
			public Label label_pass, label_fail;
			public LocalBuilder local_strpos_res;

			public Frame (ILGenerator ilgen) {
				label_fail = ilgen.DefineLabel ();
				label_pass = ilgen.DefineLabel ();
				local_strpos_res = ilgen.DeclareLocal (typeof (int));
			}				
		}

		/*
		 * Create a dynamic method which is equivalent to the RxInterpreter.EvalByteCode 
		 * method specialized to the given program and a given pc. Return the newly
		 * created method or null if a not-supported opcode was encountered.
		 */
		DynamicMethod CreateEvalMethod (byte[] program, int pc) {
			DynamicMethod m = new DynamicMethod ("Eval_" + pc, typeof (bool), new Type [] { typeof (RxInterpreter), typeof (int), typeof (int).MakeByRefType () }, typeof (RxInterpreter), true);
			ILGenerator ilgen = m.GetILGenerator ();

			/* 
			   Args:
			   interp - 0
			   strpos - 1
			   strpos_result - 2
			*/

			/*
			 * Recursive calls to EvalByteCode are inlined manually by calling 
			 * EmitEvalMethodBody with the pc of the recursive call. Frame objects hold
			 * the information required to link together the code generated by the recursive
			 * call with the rest of the code.
			 */
			Frame frame = new Frame (ilgen);

			m = EmitEvalMethodBody (m, ilgen, frame, program, pc, false, out pc);
			if (m == null)
				return null;
				
			ilgen.MarkLabel (frame.label_pass);
			ilgen.Emit (OpCodes.Ldarg_2);
			ilgen.Emit (OpCodes.Ldloc, frame.local_strpos_res);
			ilgen.Emit (OpCodes.Stind_I4);
			ilgen.Emit (OpCodes.Ldc_I4_1);
			ilgen.Emit (OpCodes.Ret);

			ilgen.MarkLabel (frame.label_fail);
			ilgen.Emit (OpCodes.Ldc_I4_0);
			ilgen.Emit (OpCodes.Ret);

			return m;
		}

		/*
		 * Emit IL code for a sequence of opcodes starting at pc. If there is a match,
		 * set frame.local_strpos_res to the position of the match, then branch to 
		 * frame.label_pass. Else branch to frame.label_fail. If one_op is true, only
		 * generate code for one opcode and set out_pc to the next pc after the opcode.
		 * Keep this in synch with RxInterpreter.EvalByteCode (). It it is sync with
		 * the version in r96072.
		 * FIXME: In the new interpreter and the IL compiler, '<.+>' does not match '<FOO>'
		 * Also, '<.+?' matches '<FOO>', but the match is '<FOO>' instead of '<F'
		 * FIXME: Modify the regex tests so they are run with RegexOptions.Compiled as
		 * well.
		 */
		private DynamicMethod EmitEvalMethodBody (DynamicMethod m, ILGenerator ilgen,
												  Frame frame, byte[] program, int pc,
												  bool one_op, out int out_pc)
		{
			int start, length, end;

			out_pc = 0;

			int group_count = 1 + (program [1] | (program [2] << 8));

			while (true) {
				RxOp op = (RxOp)program [pc];

				switch (op) {
				case RxOp.Anchor: {
					length = program [pc + 3] | (program [pc + 4] << 8);
					pc += program [pc + 1] | (program [pc + 2] << 8);

					// Optimize some common cases by inlining the code generated for the
					// anchor body 
					RxOp anch_op = (RxOp)program [pc];
					// FIXME: Do this even if the archor op is not the last in the regex
					if (group_count == 1 && anch_op == RxOp.Char && (RxOp)program [pc + 2] == RxOp.True) {

						/*
						 * while (strpos < string_end) {
						 *   if (str [strpos] == program [pc + 1]) {
						 *     match_start = strpos;
						 *     strpos_result = strpos + 1;
						 *     marks [groups [0]].Start = strpos;
						 *     if (groups.Length > 1)
						 *		marks [groups [0]].End = res;
						 *     return true;
						 *   }
						 *   strpos ++;
						 * }
						 * return false;
						 */
						// Add some locals to avoid an indirection
						LocalBuilder local_string_end = ilgen.DeclareLocal (typeof (int));
						ilgen.Emit (OpCodes.Ldarg_0);
						ilgen.Emit (OpCodes.Ldfld, fi_string_end);
						ilgen.Emit (OpCodes.Stloc, local_string_end);
						LocalBuilder local_str = ilgen.DeclareLocal (typeof (string));
						ilgen.Emit (OpCodes.Ldarg_0);
						ilgen.Emit (OpCodes.Ldfld, fi_str);
						ilgen.Emit (OpCodes.Stloc, local_str);

						//while (strpos < string_end) {
						// -> Done at the end of the loop like mcs does
						Label l1 = ilgen.DefineLabel ();
						Label l2 = ilgen.DefineLabel ();
						ilgen.Emit (OpCodes.Br, l2);
						ilgen.MarkLabel (l1);

						//  if (str [strpos] == program [pc + 1]) {
						Label l3 = ilgen.DefineLabel ();
						ilgen.Emit (OpCodes.Ldloc, local_str);
						ilgen.Emit (OpCodes.Ldarg_1);
						ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
						ilgen.Emit (OpCodes.Conv_I4);
						ilgen.Emit (OpCodes.Ldc_I4, (int)program [pc + 1]);
						ilgen.Emit (OpCodes.Beq, l3);

						// The true case is done after the loop

						//  }
						//  strpos++;
						ilgen.Emit (OpCodes.Ldarg_1);
						ilgen.Emit (OpCodes.Ldc_I4_1);
						ilgen.Emit (OpCodes.Add);
						ilgen.Emit (OpCodes.Starg, 1);
						//}
						ilgen.MarkLabel (l2);
						ilgen.Emit (OpCodes.Ldarg_1);
						ilgen.Emit (OpCodes.Ldloc, local_string_end);
						ilgen.Emit (OpCodes.Blt, l1);

						//return false;
						ilgen.Emit (OpCodes.Br, frame.label_fail);

						// True case
						ilgen.MarkLabel (l3);
						//    match_start = strpos;
						// match_start doesn't seem to be used
						/* 
						ilgen.Emit (OpCodes.Ldarg_0);
						ilgen.Emit (OpCodes.Ldarg_1);
						ilgen.Emit (OpCodes.Stfld, fi_match_start);
						*/
						//    strpos_result = strpos + 1;
						ilgen.Emit (OpCodes.Ldarg_1);
						ilgen.Emit (OpCodes.Ldc_I4_1);
						ilgen.Emit (OpCodes.Add);
						ilgen.Emit (OpCodes.Stloc, frame.local_strpos_res);
						// call SetStartOfMatch (strpos)
						ilgen.Emit (OpCodes.Ldarg_0);
						ilgen.Emit (OpCodes.Ldarg_1);
						ilgen.Emit (OpCodes.Call, GetInterpreterMethod ("SetStartOfMatch"));
						//    return true;
						ilgen.Emit (OpCodes.Br, frame.label_pass);

					} else {
						// General case

						//Console.WriteLine ("Anchor op " + anch_op);

						// Add some locals to avoid an indirection
						LocalBuilder local_string_end = ilgen.DeclareLocal (typeof (int));
						ilgen.Emit (OpCodes.Ldarg_0);
						ilgen.Emit (OpCodes.Ldfld, fi_string_end);
						ilgen.Emit (OpCodes.Stloc, local_string_end);

						//while (strpos < string_end) {
						// -> Done at the end of the loop like mcs does
						Label l1 = ilgen.DefineLabel ();
						Label l2 = ilgen.DefineLabel ();
						ilgen.Emit (OpCodes.Br, l2);
						ilgen.MarkLabel (l1);

						//if (groups.Length > 1) {
						//	ResetGroups ();
						//	marks [groups [0]].Start = strpos;
						//}
						if (group_count > 1) {
							ilgen.Emit (OpCodes.Ldarg_0);
							ilgen.Emit (OpCodes.Call, mi_reset_groups);

							ilgen.Emit (OpCodes.Ldarg_0);
							ilgen.Emit (OpCodes.Ldfld, fi_marks);
							ilgen.Emit (OpCodes.Ldarg_0);
							ilgen.Emit (OpCodes.Ldfld, fi_groups);
							ilgen.Emit (OpCodes.Ldc_I4_0);
							ilgen.Emit (OpCodes.Ldelem_I4);
							ilgen.Emit (OpCodes.Ldelema, typeof (Mark));
							ilgen.Emit (OpCodes.Ldarg_1);
							ilgen.Emit (OpCodes.Stfld, fi_mark_start);
						}

						//  if (EvalByteCode (pc, strpos, ref res)) {

						Frame new_frame = new Frame (ilgen);

						//  old_stros = strpos;
						LocalBuilder local_old_strpos = ilgen.DeclareLocal (typeof (int));
						ilgen.Emit (OpCodes.Ldarg_1);
						ilgen.Emit (OpCodes.Stloc, local_old_strpos);

						m = EmitEvalMethodBody (m, ilgen, new_frame, program, pc, false, out out_pc);
						if (m == null)
							return null;

						// Pass
						ilgen.MarkLabel (new_frame.label_pass);
						//    match_start = old_strpos;
						// match_start doesn't seem to be used
						/*
						ilgen.Emit (OpCodes.Ldarg_0);
						ilgen.Emit (OpCodes.Ldloc, local_old_strpos);
						ilgen.Emit (OpCodes.Stfld, fi_match_start);
						*/
						//    strpos_result = res;
						ilgen.Emit (OpCodes.Ldloc, new_frame.local_strpos_res);
						ilgen.Emit (OpCodes.Stloc, frame.local_strpos_res);
						//    marks [groups [0]].Start = old_strpos;
						ilgen.Emit (OpCodes.Ldarg_0);
						ilgen.Emit (OpCodes.Ldfld, fi_marks);
						ilgen.Emit (OpCodes.Ldarg_0);
						ilgen.Emit (OpCodes.Ldfld, fi_groups);
						ilgen.Emit (OpCodes.Ldc_I4_0);
						ilgen.Emit (OpCodes.Ldelem_I4);
						ilgen.Emit (OpCodes.Ldelema, typeof (Mark));
						ilgen.Emit (OpCodes.Ldloc, local_old_strpos);
						ilgen.Emit (OpCodes.Stfld, fi_mark_start);
						//    if (groups.Length > 1)
						//		marks [groups [0]].End = res;
						if (group_count > 1) {
							ilgen.Emit (OpCodes.Ldarg_0);
							ilgen.Emit (OpCodes.Ldfld, fi_marks);
							ilgen.Emit (OpCodes.Ldarg_0);
							ilgen.Emit (OpCodes.Ldfld, fi_groups);
							ilgen.Emit (OpCodes.Ldc_I4_0);
							ilgen.Emit (OpCodes.Ldelem_I4);
							ilgen.Emit (OpCodes.Ldelema, typeof (Mark));
							ilgen.Emit (OpCodes.Ldloc, new_frame.local_strpos_res);
							ilgen.Emit (OpCodes.Stfld, fi_mark_end);
						}

						//    return true;
						ilgen.Emit (OpCodes.Br, frame.label_pass);

						// Fail
						ilgen.MarkLabel (new_frame.label_fail);
						//  strpos = old_strpos + 1;
						ilgen.Emit (OpCodes.Ldloc, local_old_strpos);
						ilgen.Emit (OpCodes.Ldc_I4_1);
						ilgen.Emit (OpCodes.Add);
						ilgen.Emit (OpCodes.Starg, 1);
						//}
						ilgen.MarkLabel (l2);
						ilgen.Emit (OpCodes.Ldarg_1);
						ilgen.Emit (OpCodes.Ldloc, local_string_end);
						ilgen.Emit (OpCodes.Blt, l1);
						//return false;
						ilgen.Emit (OpCodes.Br, frame.label_fail);
					}

					goto End;
				}
				case RxOp.Branch: {
					//if (EvalByteCode (pc + 3, strpos, ref res)) {

					// Emit the rest of the code inline instead of making a recursive call
					Frame new_frame = new Frame (ilgen);

					//  old_strpos = strpos;
					LocalBuilder local_old_strpos = ilgen.DeclareLocal (typeof (int));
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Stloc, local_old_strpos);

					m = EmitEvalMethodBody (m, ilgen, new_frame, program, pc + 3, false, out out_pc);
					if (m == null)
						return null;

					// Pass
					ilgen.MarkLabel (new_frame.label_pass);
					//  strpos_result = res;
					ilgen.Emit (OpCodes.Ldloc, new_frame.local_strpos_res);
					ilgen.Emit (OpCodes.Stloc, frame.local_strpos_res);
					//  return true;
					ilgen.Emit (OpCodes.Br, frame.label_pass);

					// Fail
					ilgen.MarkLabel (new_frame.label_fail);
					//  strpos = old_strpos;
					ilgen.Emit (OpCodes.Ldloc, local_old_strpos);
					ilgen.Emit (OpCodes.Starg, 1);

					pc += program [pc + 1] | (program [pc + 2] << 8);
					break;
				}
				case RxOp.Char:
				case RxOp.NoChar:
				case RxOp.CharIgnoreCase:
				case RxOp.NoCharIgnoreCase: {
					bool invert = (op == RxOp.NoChar) || (op == RxOp.NoCharIgnoreCase);
					bool ignore = (op == RxOp.CharIgnoreCase) || (op == RxOp.NoCharIgnoreCase);

					//if (strpos < string_end && (str [strpos] == program [pc + 1])) {
					Label l1 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Bge, l1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					if (ignore)
						ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("ToLower", new Type [] { typeof (char) }));
					ilgen.Emit (OpCodes.Conv_I4);
					ilgen.Emit (OpCodes.Ldc_I4, (int)program [pc + 1]);
					ilgen.Emit (invert ? OpCodes.Beq : OpCodes.Bne_Un, l1);
					//  strpos++;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Add);
					ilgen.Emit (OpCodes.Starg, 1);
					Label l2 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Br, l2);
					//}
					ilgen.MarkLabel (l1);
					//return false;
					ilgen.Emit (OpCodes.Br, frame.label_fail);
					ilgen.MarkLabel (l2);

					pc += 2;
					break;
				}
				case RxOp.Range:
				case RxOp.RangeIgnoreCase: {
					bool ignore = (op == RxOp.RangeIgnoreCase) || (op == RxOp.NoRangeIgnoreCase);

					//if (strpos < string_end) {
					Label l1 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Bge, l1);

					//  int c = str [strpos];
					LocalBuilder local_c = ilgen.DeclareLocal (typeof (char));
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					if (ignore)
						ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("ToLower", new Type [] { typeof (char) }));
					ilgen.Emit (OpCodes.Stloc, local_c);

					//  if (c >= program [pc + 1] && c <= program [pc + 2]) {
					ilgen.Emit (OpCodes.Ldloc, local_c);
					ilgen.Emit (OpCodes.Ldc_I4, (int)program [pc + 1]);
					ilgen.Emit (OpCodes.Blt, l1);
					ilgen.Emit (OpCodes.Ldloc, local_c);
					ilgen.Emit (OpCodes.Ldc_I4, (int)program [pc + 2]);
					ilgen.Emit (OpCodes.Bgt, l1);

					//    strpos++;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Add);
					ilgen.Emit (OpCodes.Starg, 1);
					//  }
					Label l2 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Br, l2);
					//}
					ilgen.MarkLabel (l1);
					//return false;
					ilgen.Emit (OpCodes.Br, frame.label_fail);
					ilgen.MarkLabel (l2);

					pc += 3;
					break;
				}
				case RxOp.NoRange:
				case RxOp.NoRangeIgnoreCase: {
					bool ignore = (op == RxOp.RangeIgnoreCase) || (op == RxOp.NoRangeIgnoreCase);

					//if (strpos >= string_end) {
					//  return false;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Bge, frame.label_fail);

					//  int c = str [strpos];
					LocalBuilder local_c = ilgen.DeclareLocal (typeof (char));
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					if (ignore)
						ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("ToLower", new Type [] { typeof (char) }));
					ilgen.Emit (OpCodes.Stloc, local_c);

					//  if (c >= program [pc + 1] && c <= program [pc + 2]) {
					//     return false;
					Label l3 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Ldloc, local_c);
					ilgen.Emit (OpCodes.Ldc_I4, (int)program [pc + 1]);
					ilgen.Emit (OpCodes.Blt, l3);
					ilgen.Emit (OpCodes.Ldloc, local_c);
					ilgen.Emit (OpCodes.Ldc_I4, (int)program [pc + 2]);
					ilgen.Emit (OpCodes.Bgt, l3);
					ilgen.Emit (OpCodes.Br, frame.label_fail);
					ilgen.MarkLabel (l3);

					//    strpos++;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Add);
					ilgen.Emit (OpCodes.Starg, 1);

					pc += 3;
					break;
				}
				case RxOp.True: {
					//strpos_result = strpos;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Stloc, frame.local_strpos_res);
					//  return true;
					ilgen.Emit (OpCodes.Br, frame.label_pass);
					pc++;
					goto End;
				}
				case RxOp.False: {
					//  return false;
					ilgen.Emit (OpCodes.Br, frame.label_fail);
					pc++;
					goto End;
				}
				case RxOp.AnyPosition: {
					pc++;
					break;
				}
				case RxOp.StartOfString: {
					//if (strpos != 0)
					//	return false;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_0);
					ilgen.Emit (OpCodes.Bgt, frame.label_fail);
					pc++;
					break;
				}
				case RxOp.StartOfLine: {
					// FIXME: windows line endings
					//if (!(strpos == 0 || str [strpos - 1] == '\n'))
					//	return false;
					Label l = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_0);
					ilgen.Emit (OpCodes.Beq, l);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Sub);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					ilgen.Emit (OpCodes.Ldc_I4, (int)'\n');
					ilgen.Emit (OpCodes.Beq, l);
					ilgen.Emit (OpCodes.Br, frame.label_fail);
					ilgen.MarkLabel (l);

					pc++;
					break;
				}
				case RxOp.StartOfScan: {
					//if (strpos != string_start)
					//	return false;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_start);
					ilgen.Emit (OpCodes.Bne_Un, frame.label_fail);
					pc++;
					break;
				}
				case RxOp.End: {
					//if (!(strpos == string_end || (strpos == string_end - 1 && str [strpos] == '\n')))
					//	return false;
					Label l = ilgen.DefineLabel ();

					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Beq, l);

					Label l2 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Sub);
					ilgen.Emit (OpCodes.Bne_Un, l2);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					ilgen.Emit (OpCodes.Ldc_I4, (int)'\n');
					ilgen.Emit (OpCodes.Bne_Un, l2);
					ilgen.Emit (OpCodes.Br, l);
					ilgen.MarkLabel (l2);

					ilgen.Emit (OpCodes.Br, frame.label_fail);
					ilgen.MarkLabel (l);

					pc++;
					break;
				}
				case RxOp.EndOfString: {
					//if (strpos != string_end)
					//	return false;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Bne_Un, frame.label_fail);
					pc++;
					break;
				}
				case RxOp.EndOfLine: {
					//if (!(strpos == string_end || str [strpos] == '\n'))
					//	return false;
					Label l_match = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Beq, l_match);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					ilgen.Emit (OpCodes.Ldc_I4, (int)'\n');
					ilgen.Emit (OpCodes.Beq, l_match);
					ilgen.Emit (OpCodes.Br, frame.label_fail);
					ilgen.MarkLabel (l_match);
					
					pc++;
					break;
				}
				case RxOp.WordBoundary:
				case RxOp.NoWordBoundary: {
					bool invert = op == RxOp.NoWordBoundary;

					//if (string_end == 0)
					//	return false;
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Ldc_I4_0);
					ilgen.Emit (OpCodes.Beq, frame.label_fail);

					Label l_match = ilgen.DefineLabel ();

					//if (strpos == 0) {
					Label l1 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_0);
					ilgen.Emit (OpCodes.Bne_Un, l1);
					//if (!IsWordChar (str [strpos])) {
					//  return false;
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					ilgen.Emit (OpCodes.Call, mi_is_word_char);
					ilgen.Emit (invert ? OpCodes.Brtrue : OpCodes.Brfalse, frame.label_fail);
					ilgen.Emit (OpCodes.Br, l_match);

					//} else if (strpos == string_end) {
					ilgen.MarkLabel (l1);
					Label l2 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Bne_Un, l2);
					//if (!IsWordChar (str [strpos - 1])) {
					//  return false;
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Sub);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					ilgen.Emit (OpCodes.Call, mi_is_word_char);
					ilgen.Emit (invert ? OpCodes.Brtrue : OpCodes.Brfalse, frame.label_fail);
					ilgen.Emit (OpCodes.Br, l_match);

					//} else {
					ilgen.MarkLabel (l2);
					//if (IsWordChar (str [strpos]) == IsWordChar (str [strpos - 1])) {
					//  return false;
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					ilgen.Emit (OpCodes.Call, mi_is_word_char);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Sub);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					ilgen.Emit (OpCodes.Call, mi_is_word_char);
					ilgen.Emit (invert ? OpCodes.Bne_Un : OpCodes.Beq, frame.label_fail);
					ilgen.Emit (OpCodes.Br, l_match);

					ilgen.MarkLabel (l_match);

					pc++;
					break;
				}
				case RxOp.Bitmap:
				case RxOp.BitmapIgnoreCase: {
					bool ignore = (op == RxOp.BitmapIgnoreCase);

					//if (strpos < string_end) {
					Label l1 = ilgen.DefineLabel ();
					Label l2 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Bge, l1);
					//  int c = str [strpos];
					LocalBuilder local_c = ilgen.DeclareLocal (typeof (int));
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					ilgen.Emit (OpCodes.Conv_I4);
					if (ignore)
						ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("ToLower", new Type [] { typeof (char) }));
					//  c -= program [pc + 1];
					ilgen.Emit (OpCodes.Ldc_I4, (int)program [pc + 1]);
					ilgen.Emit (OpCodes.Sub);
					ilgen.Emit (OpCodes.Stloc, local_c);
					length =  program [pc + 2];
					//  if (c < 0 || c >= (length << 3))
					//    return false;
					ilgen.Emit (OpCodes.Ldloc, local_c);
					ilgen.Emit (OpCodes.Ldc_I4_0);
					ilgen.Emit (OpCodes.Blt, frame.label_fail);
					ilgen.Emit (OpCodes.Ldloc, local_c);
					ilgen.Emit (OpCodes.Ldc_I4, length << 3);
					ilgen.Emit (OpCodes.Bge, frame.label_fail);
					pc += 3;

					// Optimized version for small bitmaps
					if (length <= 4) {
						uint bitmap = program [pc];
						
						if (length > 1)
							bitmap |= ((uint)program [pc + 1] << 8);
						if (length > 2)
							bitmap |= ((uint)program [pc + 2] << 16);
						if (length > 3)
							bitmap |= ((uint)program [pc + 3] << 24);

						//if ((bitmap >> c) & 1)
						ilgen.Emit (OpCodes.Ldc_I4, bitmap);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Shr_Un);
						ilgen.Emit (OpCodes.Ldc_I4_1);
						ilgen.Emit (OpCodes.And);
						ilgen.Emit (OpCodes.Brfalse, l1);
					} else {
						//  if ((program [pc + (c >> 3)] & (1 << (c & 0x7))) != 0) {
						ilgen.Emit (OpCodes.Ldarg_0);
						ilgen.Emit (OpCodes.Ldfld, fi_program);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4_3);
						ilgen.Emit (OpCodes.Shr);
						ilgen.Emit (OpCodes.Ldc_I4, pc);
						ilgen.Emit (OpCodes.Add);
						ilgen.Emit (OpCodes.Ldelem_I1);
						ilgen.Emit (OpCodes.Ldc_I4_1);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, 7);
						ilgen.Emit (OpCodes.And);
						ilgen.Emit (OpCodes.Shl);
						ilgen.Emit (OpCodes.And);
						ilgen.Emit (OpCodes.Ldc_I4_0);
						ilgen.Emit (OpCodes.Beq, l1);
					}
					//    strpos++;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Add);
					ilgen.Emit (OpCodes.Starg, 1);
					//    continue;
					ilgen.Emit (OpCodes.Br, l2);
					//  }
					//}
					//return false;
					ilgen.MarkLabel (l1);
					ilgen.Emit (OpCodes.Br, frame.label_fail);

					ilgen.MarkLabel (l2);

					pc += length;
					break;
				}
				case RxOp.NoBitmap:
				case RxOp.NoBitmapIgnoreCase: {
					// Not currently used
					Console.WriteLine ("Opcode " + op + " not supported.");
					return null;
				}
				case RxOp.String:
				case RxOp.StringIgnoreCase: {
					bool ignore = (op == RxOp.StringIgnoreCase);

					start = pc + 2;
					length = program [pc + 1];
					//if (strpos + length > string_end)
					//	return false;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4, length);
					ilgen.Emit (OpCodes.Add);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Bgt, frame.label_fail);

					/* Avoid unsafe code in Moonlight build */
#if false && !NET_2_1
					int i;
					LocalBuilder local_strptr = ilgen.DeclareLocal (typeof (char).MakePointerType ());
					// char *strptr = &str.start_char + strpos
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldflda, typeof (String).GetField ("start_char", BindingFlags.Instance|BindingFlags.NonPublic));
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Shl);
					ilgen.Emit (OpCodes.Add);
					ilgen.Emit (OpCodes.Stloc, local_strptr);

					end = start + length;
					for (i = 0; i < length; ++i) {
						// if (*(strptr + i) != program [start + i])
						//   return false;
						ilgen.Emit (OpCodes.Ldloc, local_strptr);
						ilgen.Emit (OpCodes.Ldc_I4, i * 2);
						ilgen.Emit (OpCodes.Add);
						ilgen.Emit (OpCodes.Ldind_I2);
						if (ignore)
							ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("ToLower", new Type [] { typeof (char) }));
						ilgen.Emit (OpCodes.Ldc_I4, (int)program [start + i]);
						ilgen.Emit (OpCodes.Bne_Un, frame.label_fail);
					}

					// strpos += length
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4, length);
					ilgen.Emit (OpCodes.Add);
					ilgen.Emit (OpCodes.Starg, 1);

#else
					// Allocate a local for 'str' to save an indirection
					LocalBuilder local_str = ilgen.DeclareLocal (typeof (string));
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Stloc, local_str);

					// FIXME: Emit a loop for long strings
					end = start + length;
					for (; start < end; ++start) {
						//if (str [strpos] != program [start])
						//	return false;
						ilgen.Emit (OpCodes.Ldloc, local_str);
						ilgen.Emit (OpCodes.Ldarg_1);
						ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
						if (ignore)
							ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("ToLower", new Type [] { typeof (char) }));
						ilgen.Emit (OpCodes.Ldc_I4, (int)program [start]);
						ilgen.Emit (OpCodes.Bne_Un, frame.label_fail);
						//strpos++;
						ilgen.Emit (OpCodes.Ldarg_1);
						ilgen.Emit (OpCodes.Ldc_I4_1);
						ilgen.Emit (OpCodes.Add);
						ilgen.Emit (OpCodes.Starg, 1);
					}
#endif

					pc = end;
					break;
				}
				case RxOp.OpenGroup: {
					//Open (program [pc + 1] | (program [pc + 2] << 8), strpos);
					int group_id = program [pc + 1] | (program [pc + 2] << 8);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldc_I4, group_id);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Call, mi_open);

					pc += 3;
					break;
				}
				case RxOp.CloseGroup: {
					//Close (program [pc + 1] | (program [pc + 2] << 8), strpos);
					int group_id = program [pc + 1] | (program [pc + 2] << 8);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldc_I4, group_id);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Call, mi_close);

					pc += 3;
					break;
				}
				case RxOp.Jump: {
					pc += program [pc + 1] | (program [pc + 2] << 8);
					break;
				}
				case RxOp.TestCharGroup: {
					int char_group_end = pc + program [pc + 1] | (program [pc + 2] << 8);
					pc += 3;

					Label label_match = ilgen.DefineLabel ();

					/*
					 * Generate code for all the matching ops in the group
					 */
					while (pc < char_group_end) {
						Frame new_frame = new Frame (ilgen);
						m = EmitEvalMethodBody (m, ilgen, new_frame, program, pc, true, out pc);
						if (m == null)
							return null;						
						
						// Pass
						// strpos is already updated by the code generated by the
						// recursive call
						ilgen.MarkLabel (new_frame.label_pass);
						ilgen.Emit (OpCodes.Br, label_match);
						
						// Fail
						// Just fall through to the next test
						ilgen.MarkLabel (new_frame.label_fail);
					}

					// If we reached here, all the matching ops have failed
					ilgen.Emit (OpCodes.Br, frame.label_fail);

					ilgen.MarkLabel (label_match);

					break;
				}
				case RxOp.Repeat: {
					start = ReadInt (program, pc + 3);
					end = ReadInt (program, pc + 7);

					LocalBuilder local_length = ilgen.DeclareLocal (typeof (int));

					Label label_repeat_success = ilgen.DefineLabel ();

					//length = 0;

					//while (length < end) {
					// -> done at the end of the loop
					Label l1 = ilgen.DefineLabel ();
					Label l2 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Br, l2);
					ilgen.MarkLabel (l1);

					//if (!EvalByteCode (pc + 11, strpos, ref res)) {

					Frame new_frame = new Frame (ilgen);

					//  old_strpos = strpos;
					LocalBuilder local_old_strpos = ilgen.DeclareLocal (typeof (int));
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Stloc, local_old_strpos);

					m = EmitEvalMethodBody (m, ilgen, new_frame, program, pc + 11, false, out out_pc);
					if (m == null)
						return null;

					// Fail
					ilgen.MarkLabel (new_frame.label_fail);
					//if (length >= start) {
					//  goto repeat_success;
				    //}
					ilgen.Emit (OpCodes.Ldloc, local_length);
					ilgen.Emit (OpCodes.Ldc_I4, start);
					ilgen.Emit (OpCodes.Bge, label_repeat_success);
					//return false;
					ilgen.Emit (OpCodes.Br, frame.label_fail);

					// Pass
					ilgen.MarkLabel (new_frame.label_pass);
					//  strpos = res;
					ilgen.Emit (OpCodes.Ldloc, new_frame.local_strpos_res);
					ilgen.Emit (OpCodes.Starg, 1);
					//  length++;
					ilgen.Emit (OpCodes.Ldloc, local_length);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Add);
					ilgen.Emit (OpCodes.Stloc, local_length);
					//}
					ilgen.MarkLabel (l2);
					ilgen.Emit (OpCodes.Ldloc, local_length);
					ilgen.Emit (OpCodes.Ldc_I4, end);
					ilgen.Emit (OpCodes.Blt, l1);

					//if (length != end)
					//	return false;
					ilgen.Emit (OpCodes.Ldloc, local_length);
					ilgen.Emit (OpCodes.Ldc_I4, end);
					ilgen.Emit (OpCodes.Bne_Un, frame.label_fail);

					//repeat_success:
					ilgen.MarkLabel (label_repeat_success);
					
					pc += program [pc + 1] | (program [pc + 2] << 8);
					break;
				}
				case RxOp.CategoryAny: {
					//if (strpos < string_end && str [strpos] != '\n') {
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Bge, frame.label_fail);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					ilgen.Emit (OpCodes.Ldc_I4, (int)'\n');
					ilgen.Emit (OpCodes.Beq, frame.label_fail);
					//	strpos++;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Add);
					ilgen.Emit (OpCodes.Starg, 1);

					pc++;
					break;
				}
				case RxOp.CategoryWord:
				case RxOp.CategoryDigit:
				case RxOp.CategoryWhiteSpace:
				case RxOp.CategoryEcmaWord:
				case RxOp.CategoryEcmaWhiteSpace:
				case RxOp.CategoryUnicodeSpecials:
				case RxOp.CategoryUnicode:
				case RxOp.NoCategoryWord:
				case RxOp.NoCategoryDigit:
				case RxOp.NoCategoryWhiteSpace:
				case RxOp.NoCategoryEcmaWord:
				case RxOp.NoCategoryEcmaWhiteSpace:
				case RxOp.NoCategoryUnicodeSpecials:
				case RxOp.NoCategoryUnicode: {
					bool invert = (op == RxOp.NoCategoryWord || op == RxOp.NoCategoryWhiteSpace || op == RxOp.NoCategoryEcmaWord || op == RxOp.NoCategoryEcmaWhiteSpace || op == RxOp.NoCategoryUnicodeSpecials || op == RxOp.NoCategoryUnicode);

					//if (strpos < string_end) {
					Label l_nomatch = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_string_end);
					ilgen.Emit (OpCodes.Bge, l_nomatch);

					//  int c = str [strpos];
					LocalBuilder local_c = ilgen.DeclareLocal (typeof (char));
					ilgen.Emit (OpCodes.Ldarg_0);
					ilgen.Emit (OpCodes.Ldfld, fi_str);
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Callvirt, typeof (string).GetMethod ("get_Chars"));
					ilgen.Emit (OpCodes.Stloc, local_c);

					Label l_match = ilgen.DefineLabel ();

					Label l_true, l_false;

					l_true = invert ? l_nomatch : l_match;
					l_false = invert ? l_match : l_nomatch;

					switch (op) {
					case RxOp.CategoryWord:
					case RxOp.NoCategoryWord:
						//  if (Char.IsLetterOrDigit (c) || Char.GetUnicodeCategory (c) == UnicodeCategory.ConnectorPunctuation) {
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("IsLetterOrDigit", new Type [] { typeof (char) }));
						ilgen.Emit (OpCodes.Brtrue, l_true);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("GetUnicodeCategory", new Type [] { typeof (char) }));
						ilgen.Emit (OpCodes.Ldc_I4, (int)UnicodeCategory.ConnectorPunctuation);
						ilgen.Emit (OpCodes.Beq, l_true);
						break;
					case RxOp.CategoryDigit:
					case RxOp.NoCategoryDigit:
						// if (Char.IsDigit (c)) {
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("IsDigit", new Type [] { typeof (char) }));
						ilgen.Emit (OpCodes.Brtrue, l_true);
						break;
					case RxOp.CategoryWhiteSpace:
					case RxOp.NoCategoryWhiteSpace:
						// if (Char.IsWhiteSpace (c)) {
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("IsWhiteSpace", new Type [] { typeof (char) }));
						ilgen.Emit (OpCodes.Brtrue, l_true);
						break;
					case RxOp.CategoryEcmaWord:
					case RxOp.NoCategoryEcmaWord:
						// if ('a' <= c && c <= 'z' || 'A' <= c && c <= 'Z' || '0' <= c && c <= '9' || c == '_') {
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'a' - 1);
						ilgen.Emit (OpCodes.Cgt);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'z' + 1);
						ilgen.Emit (OpCodes.Clt);
						ilgen.Emit (OpCodes.And);
						ilgen.Emit (OpCodes.Brtrue, l_true);

						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'A' - 1);
						ilgen.Emit (OpCodes.Cgt);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'Z' + 1);
						ilgen.Emit (OpCodes.Clt);
						ilgen.Emit (OpCodes.And);
						ilgen.Emit (OpCodes.Brtrue, l_true);

						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'0' - 1);
						ilgen.Emit (OpCodes.Cgt);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'9' + 1);
						ilgen.Emit (OpCodes.Clt);
						ilgen.Emit (OpCodes.And);
						ilgen.Emit (OpCodes.Brtrue, l_true);

						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'_');
						ilgen.Emit (OpCodes.Beq, l_true);
						break;
					case RxOp.CategoryEcmaWhiteSpace:
					case RxOp.NoCategoryEcmaWhiteSpace:
						// if (c == ' ' || c == '\t' || c == '\n' || c == '\r' || c == '\f' || c == '\v') {
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)' ');
						ilgen.Emit (OpCodes.Beq, l_true);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'\t');
						ilgen.Emit (OpCodes.Beq, l_true);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'\n');
						ilgen.Emit (OpCodes.Beq, l_true);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'\r');
						ilgen.Emit (OpCodes.Beq, l_true);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'\f');
						ilgen.Emit (OpCodes.Beq, l_true);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'\v');
						ilgen.Emit (OpCodes.Beq, l_true);
						break;
					case RxOp.CategoryUnicodeSpecials:
					case RxOp.NoCategoryUnicodeSpecials:
						// if ('\uFEFF' <= c && c <= '\uFEFF' || '\uFFF0' <= c && c <= '\uFFFD') {
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'\uFEFF' - 1);
						ilgen.Emit (OpCodes.Cgt);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'\uFEFF' + 1);
						ilgen.Emit (OpCodes.Clt);
						ilgen.Emit (OpCodes.And);
						ilgen.Emit (OpCodes.Brtrue, l_true);

						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'\uFFF0' - 1);
						ilgen.Emit (OpCodes.Cgt);
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Ldc_I4, (int)'\uFFFD' + 1);
						ilgen.Emit (OpCodes.Clt);
						ilgen.Emit (OpCodes.And);
						ilgen.Emit (OpCodes.Brtrue, l_true);
						break;
					case RxOp.CategoryUnicode:
					case RxOp.NoCategoryUnicode:
						// if (Char.GetUnicodeCategory (c) == (UnicodeCategory)program [pc + 1]) {						
						ilgen.Emit (OpCodes.Ldloc, local_c);
						ilgen.Emit (OpCodes.Call, typeof (Char).GetMethod ("GetUnicodeCategory", new Type [] { typeof (char) }));
						ilgen.Emit (OpCodes.Ldc_I4, (int)program [pc + 1]);
						ilgen.Emit (OpCodes.Beq, l_true);
						break;
					}

					ilgen.Emit (OpCodes.Br, l_false);

					ilgen.MarkLabel (l_match);

					//    strpos++;
					ilgen.Emit (OpCodes.Ldarg_1);
					ilgen.Emit (OpCodes.Ldc_I4_1);
					ilgen.Emit (OpCodes.Add);
					ilgen.Emit (OpCodes.Starg, 1);
					//  }
					Label l2 = ilgen.DefineLabel ();
					ilgen.Emit (OpCodes.Br, l2);
					//}
					ilgen.MarkLabel (l_nomatch);
					//return false;
					ilgen.Emit (OpCodes.Br, frame.label_fail);

					ilgen.MarkLabel (l2);

					if (op == RxOp.CategoryUnicode || op == RxOp.NoCategoryUnicode)
						pc += 2;
					else
						pc++;
					break;
				}
				default:
					Console.WriteLine ("Opcode " + op + " not supported.");
					return null;
			    }

				if (one_op)
					break;
			}

			End:

			out_pc = pc;

			return m;
		}
	}
#else
	class CILCompiler : RxCompiler {
	}
#endif

}

