/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

const MwbIPagesObserver
  = Components.interfaces.MwbIPagesObserver;
const nsISupports = Components.interfaces.nsISupports;
const nsISupportsWeakReference = Components.interfaces.nsISupportsWeakReference;

const N_PINNED_PAGES = 5;

var pagesService;

function FetchFavsObserver()
{
}

FetchFavsObserver.prototype =
{
  favoritesReceived : function(favorites, nFavorites)
  {
    // Remove all existing favorite links
    var favDiv = document.getElementById("favorites");
    while (favDiv.childNodes.length > 0)
      favDiv.removeChild(favDiv.childNodes[0]);

    if (nFavorites < 1)
      {
        // If there are no favorites then display a helpful page instead
        var req = new XMLHttpRequest();
        req.open('GET',
                 'no-history-fallback.xhtml',
                 false);
        req.send(null);
        favDiv.innerHTML = "";
        favDiv.appendChild(req.responseXML.getElementById("no-history-root"));
      }
    else
      {
        var h1 = document.createElement('h1');
        h1.appendChild(document.createTextNode(_("Most visited")));
        favDiv.appendChild(h1);

        for (fav = 0; fav < nFavorites; fav++)
          {
            var div = document.createElement('div');
            div.setAttribute('class', 'favorite');

            // Create elements for the close button. The outer div is
            // just used to make the position for the inner element
            // relative to the top of the favorite
            var closeOuter = document.createElement('div');
            closeOuter.setAttribute('class', 'close-button-outer');
            var closeInner = document.createElement('a');
            closeInner.setAttribute('class', 'close-button-inner');
            closeInner.setAttribute('href', 'javascript:');
            closeOuter.appendChild(closeInner);
            div.appendChild(closeOuter);

            // Create a closure with the url and use it as a callback
            // to remove the favorite when the link is clicked
            let (url = favorites[fav].url) {
              closeInner.addEventListener("click",
                                          function(e) {
                                            pagesService.removeFavorite(url);
                                            pagesService.getFavorites();
                                            e.stopPropagation();
                                          },
                                          false);
            }

            var a = document.createElement('a');
            div.appendChild(a);
            var thumbnailDiv = document.createElement('div');
            thumbnailDiv.setAttribute('class', 'thumbnail');
            img = document.createElement('img');
            img.setAttribute('class', 'thumbnail');
            thumbnailDiv.appendChild(img);
            a.appendChild(thumbnailDiv);
            var p = document.createElement('p');
            a.appendChild(p);

            p.appendChild(document.createTextNode(favorites[fav].title));
            a.setAttribute("href", favorites[fav].url);

            // Get a local URI for the thumbnail based on the page URL
            var thumbnailURI =
              pagesService.getThumbnailURL (favorites[fav].url);
            img.setAttribute("src", thumbnailURI.spec);

            favDiv.appendChild(div);
          }
      }
  },

  linkVisited : function(uri, visit_time, is_pinned)
  {
    if (is_pinned)
      rebuildPinnedPagesList();
  },

  pinnedPagesChanged : function()
  {
    rebuildPinnedPagesList();
  },

  QueryInterface : function(aIID)
  {
    if (!aIID.equals(MwbIPagesObserver)
        && !aIID.equals(nsISupports)
        && !aIID.equals(nsISupportsWeakReference))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

function rebuildPinnedPagesList()
{
  var div = document.getElementById('pinned_pages_div');
  var i;

  // Clear all existing entries
  while (div.childNodes.length > 0)
    div.removeChild(div.childNodes[0]);

  var nPages = pagesService.numPinnedPages;
  if (nPages > N_PINNED_PAGES)
    nPages = N_PINNED_PAGES;

  // If there are no pages then display some instructions instead
  if (nPages <= 0)
    div.innerHTML = ("<p>" + _("No pinned pages. Push the pin in the " +
                               "automagic bar on a page you wish to " +
                               "remember.") + "</p>");
  else
    {
      // Add each new entry
      for (i = 0; i < nPages; i++)
        {
          var page = pagesService.getPinnedPageByTime(i);
          var a = document.createElement('a');
          var pageDiv = document.createElement('div');
          var imgDiv = document.createElement('div');
          var img = document.createElement('img');
          var innerDiv = document.createElement('div');
          var titleSpan = document.createElement('span');

          a.setAttribute('href', page.uri);
          pageDiv.setAttribute('class', 'pinned_page_div');
          imgDiv.setAttribute('class', 'pinned_favicon_div');
          img.setAttribute('class', 'pinned_favicon');
          img.setAttribute('src', 'mwb-favicon://' + page.uri);
          imgDiv.appendChild(img);
          pageDiv.appendChild(imgDiv);
          innerDiv.setAttribute('class', 'pinned_inner_div');
          titleSpan.setAttribute('class', 'pinned_title');
          titleSpan.appendChild(document.createTextNode(page.title + " "));
          innerDiv.appendChild(titleSpan);
          if (page.visitTime > 0)
            {
              var visitTime = pagesService.formatTime(page.visitTime);
              var visitTimeSpan = document.createElement('span');
              visitTimeSpan.setAttribute('class', 'pinned_visit_time');
              visitTimeSpan.appendChild(document.createTextNode(visitTime));
              innerDiv.appendChild(visitTimeSpan);
            }
          pageDiv.appendChild(innerDiv);
          a.appendChild(pageDiv);
          div.appendChild(a);
        }

      if (pagesService.numPinnedPages >= 1)
        {
          var a = document.createElement('a');
          a.setAttribute('class', 'pinned_more_link');
          a.appendChild(document.createTextNode(_("Manage pinned pages")));
          a.setAttribute('href', 'pinnedpages.xhtml');
          var p = document.createElement('p');
          p.appendChild(a);
          div.appendChild(p);
        }
    }
}

function makeTabPrivate()
{
  pagesService.startPrivateBrowsing();
}

var observer;

// Run the initialisation once the page has finished loading
document.addEventListener("DOMContentLoaded", function(event)
{
  pagesService = (Components.classes["@moblin.org/mwb-pages-service;1"].
                  getService(Components.interfaces.MwbIPagesService));

  observer = new FetchFavsObserver();

  // Add our observer for when the favorites become available
  pagesService.addPagesObserver(observer);

  // Start fetching the favorites immediately
  pagesService.getFavorites();

  // Generate the pinned pages list
  rebuildPinnedPagesList();

  var privateButton = document.getElementById("make_private_button");
  privateButton.addEventListener("click", makeTabPrivate, false);

  // Set a style on the body tag if we are in private mode so that we
  // can style the page differently
  var pbService = (Components.classes["@mozilla.org/privatebrowsing;1"].
                   getService(Components.interfaces.nsIPrivateBrowsingService));
  if (pbService.privateBrowsingEnabled)
    {
      var body = document.getElementsByTagName("body")[0];
      body.setAttribute("class", "private_mode");
    }

  // Unregister the load event so we don't initialize more than once
  event.currentTarget.removeEventListener("DOMContentLoaded",
                                          arguments.callee,
                                          false);
}, false);
