/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n.h>
#include <mhs/mhs.h>

#include "mwb-mozembed.h"
#include "mwb-utils.h"

/*#define MWB_NATIVE_SCROLL 1*/

static void scrollable_interface_init (NbtkScrollableInterface *iface);

static void scrollable_set_adjustments (NbtkScrollable *scrollable,
                                        NbtkAdjustment *hadjustment,
                                        NbtkAdjustment *vadjustment);

static void scrollable_get_adjustments (NbtkScrollable  *scrollable,
                                        NbtkAdjustment **hadjustment,
                                        NbtkAdjustment **vadjustment);

G_DEFINE_TYPE_WITH_CODE (MwbMozEmbed, mwb_mozembed, CLUTTER_TYPE_MOZEMBED,
                         G_IMPLEMENT_INTERFACE(NBTK_TYPE_SCROLLABLE,
                                               scrollable_interface_init))

#define MOZEMBED_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_MOZEMBED, MwbMozEmbedPrivate))

enum
{
  NEW_TAB,
  PIN_PAGE,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

static const char const *
mwb_mozembed_comp_paths[] =
{
  PKGDATADIR "/components",
  NULL
};

static const char const *
mwb_mozembed_chrome_paths[] =
{
  PKGDATADIR "/chrome",
  NULL
};

struct _MwbMozEmbedPrivate
{
  NbtkAdjustment *hadjust;
  NbtkAdjustment *vadjust;
  gboolean        sync_adjust;
  guint           adjust_idle;

  NbtkWidget     *tooltip_hack;
  NbtkWidget     *popup;

  gfloat         last_right_click_x;
  gfloat         last_right_click_y;
  gchar          *context_info_uri;
  gchar          *context_info_href;
  gchar          *context_info_img_href;
  gchar          *context_info_selected_txt;

  GtkWidget      *mwb_window;
};

enum
{
  PROP_0,

  PROP_HADJUST,
  PROP_VADJUST,
  PROP_SYNC_ADJUSTMENTS
};

static void
mwb_mozembed_get_property (GObject *object, guint property_id,
                           GValue *value, GParamSpec *pspec)
{
  NbtkAdjustment *adjustment;

  switch (property_id)
    {
    case PROP_HADJUST :
      scrollable_get_adjustments (NBTK_SCROLLABLE (object), &adjustment, NULL);
      g_value_set_object (value, adjustment);
      break;

    case PROP_VADJUST :
      scrollable_get_adjustments (NBTK_SCROLLABLE (object), NULL, &adjustment);
      g_value_set_object (value, adjustment);
      break;

    case PROP_SYNC_ADJUSTMENTS :
      g_value_set_boolean (value, mwb_mozembed_get_sync_adjustments (
                                    MWB_MOZEMBED (object)));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_mozembed_set_property (GObject *object, guint property_id,
                           const GValue *value, GParamSpec *pspec)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (object)->priv;

  switch (property_id)
    {
    case PROP_HADJUST :
      scrollable_set_adjustments (NBTK_SCROLLABLE (object),
                                  g_value_get_object (value),
                                  priv->vadjust);
      break;

    case PROP_VADJUST :
      scrollable_set_adjustments (NBTK_SCROLLABLE (object),
                                  priv->hadjust,
                                  g_value_get_object (value));
      break;

    case PROP_SYNC_ADJUSTMENTS :
      mwb_mozembed_set_sync_adjustments (MWB_MOZEMBED (object),
                                         g_value_get_boolean (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
hadjust_value_notify_cb (NbtkAdjustment *adjustment,
                         GParamSpec     *pspec,
                         MwbMozEmbed    *mozembed)
{
  gdouble value = nbtk_adjustment_get_value (adjustment);
  g_object_set (G_OBJECT (mozembed), "scroll-x", (gint)value, NULL);
}

static void
vadjust_value_notify_cb (NbtkAdjustment *adjustment,
                         GParamSpec     *pspec,
                         MwbMozEmbed    *mozembed)
{
  gdouble value = nbtk_adjustment_get_value (adjustment);
  g_object_set (G_OBJECT (mozembed), "scroll-y", (gint)value, NULL);
}

static void
mwb_mozembed_dispose (GObject *object)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (object)->priv;

  if (priv->adjust_idle)
    {
      g_source_remove (priv->adjust_idle);
      priv->adjust_idle = 0;
    }

  if (priv->hadjust)
    {
      g_signal_handlers_disconnect_by_func (priv->hadjust,
                                            hadjust_value_notify_cb,
                                            object);
      g_object_unref (priv->hadjust);
      priv->hadjust = NULL;
    }

  if (priv->vadjust)
    {
      g_signal_handlers_disconnect_by_func (priv->vadjust,
                                            vadjust_value_notify_cb,
                                            object);
      g_object_unref (priv->vadjust);
      priv->vadjust = NULL;
    }

  if (priv->popup)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->popup));
      priv->popup = NULL;
    }

  if (priv->tooltip_hack)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->tooltip_hack));
      priv->tooltip_hack = NULL;
    }

  G_OBJECT_CLASS (mwb_mozembed_parent_class)->dispose (object);
}

static void
mwb_mozembed_finalize (GObject *object)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (object)->priv;

  g_free (priv->context_info_uri);
  g_free (priv->context_info_href);
  g_free (priv->context_info_img_href);
  g_free (priv->context_info_selected_txt);

  G_OBJECT_CLASS (mwb_mozembed_parent_class)->finalize (object);
}

#ifdef MWB_NATIVE_SCROLL
/* NOTE: This is really working around an Nbtk bug, but quicker
 *       to fix it here for now.
 */
static gboolean
mwb_mozembed_update_adjustments_cb (MwbMozEmbed *self)
{
  ClutterActorBox box;
  MwbMozEmbedPrivate *priv = self->priv;

  clutter_actor_get_allocation_box (CLUTTER_ACTOR (self), &box);

  if (priv->hadjust)
    g_object_set (G_OBJECT (priv->hadjust),
                  "page-size", (gdouble)(box.x2 - box.x1),
                  "page-increment", (box.x2 - box.x1)/2.0, NULL);
  if (priv->vadjust)
    g_object_set (G_OBJECT (priv->vadjust),
                  "page-size", (gdouble)(box.y2 - box.y1),
                  "page-increment", (box.y2 - box.y1)/2.0, NULL);

  priv->adjust_idle = 0;

  return FALSE;
}
#endif

static void
mwb_mozembed_allocate (ClutterActor           *actor,
                       const ClutterActorBox  *box,
                       ClutterAllocationFlags  flags)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->allocate (actor, box, flags);

  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (priv->tooltip_hack),
                                         flags);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (priv->popup),
                                         flags);

#ifdef MWB_NATIVE_SCROLL
  if (!priv->adjust_idle)
    priv->adjust_idle =
      g_idle_add_full (CLUTTER_PRIORITY_REDRAW,
                       (GSourceFunc)mwb_mozembed_update_adjustments_cb,
                       actor,
                       NULL);
#endif
}

static gchar*
context_get_download_filename(MwbMozEmbed *self, const gchar* uri)
{
  GFile *file;
  gint pref_id;
  MhsPrefs *prefs;
  GtkWidget *dialog;
  gchar *suggest_filename;

  GError *error = NULL;
  gchar *filename = NULL;
  gchar *default_dir = NULL;
  gboolean disable_file_picker = FALSE;
  MwbMozEmbedPrivate *priv = self->priv;

  file = g_file_new_for_uri (uri);
  suggest_filename = g_file_get_basename (file);

  /* See if we need to use a file chooser */
  prefs = mhs_prefs_new ();
  if (mhs_prefs_get_branch (prefs,
                            "clutter_mozembed",
                            &pref_id,
                            &error))
    {
      if (!mhs_prefs_branch_get_bool (prefs,
                                      pref_id,
                                      ".disable_download_file_picker",
                                      &disable_file_picker,
                                      &error))
        {
          g_warning ("Error getting disable_download_file_picker: %s",
                     error->message);
          g_clear_error (&error);
        }

      if (!mhs_prefs_branch_get_char (prefs,
                                      pref_id,
                                      ".download_directory",
                                      &default_dir,
                                      &error))
        {
          g_warning ("Error getting download_directory: %s", error->message);
          g_clear_error (&error);
        }

      if (!mhs_prefs_release_branch (prefs, pref_id, &error))
        {
          g_warning ("Error releasing clutter_mozembed pref branch: %s",
                     error->message);
          g_clear_error (&error);
        }
    }
  else
    {
      g_warning ("Error retrieving clutter_mozembed pref branch: %s", error->message);
      g_clear_error (&error);
    }
  g_object_unref (G_OBJECT (prefs));

  /* If we do, use a Gtk file chooser and run it in-line */
  if (!disable_file_picker)
    {
      if (!suggest_filename || !g_strcmp0 (suggest_filename, "/"))
        suggest_filename = g_strdup ("Untitled document");

      dialog =
        gtk_file_chooser_dialog_new ("Save as",
                                     GTK_WINDOW(priv->mwb_window),
                                     GTK_FILE_CHOOSER_ACTION_SAVE,
                                     GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                     GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                                     NULL);
      gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog),
                                                      TRUE);

      gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog),
                                           default_dir);
      gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialog),
                                         suggest_filename);

      if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      else
        filename = NULL;

      gtk_widget_destroy (dialog);
    }
  else
    {
      if (!suggest_filename || !g_strcmp0 (suggest_filename, "/"))
        suggest_filename = NULL;
      else
        filename = g_strdup_printf("%s/%s", default_dir, suggest_filename);
    }

  g_free (suggest_filename);
  g_free (default_dir);

  return filename;
}

static void
context_link_open_new_tab_cb (NbtkAction  *action,
                              MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->context_info_href && *priv->context_info_href)
    g_signal_emit (self, signals[NEW_TAB], 0, priv->context_info_href);
}

static void
context_link_copy_cb (NbtkAction  *action,
                      MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->context_info_href && *priv->context_info_href)
    nbtk_clipboard_set_text (nbtk_clipboard_get_default (),
                             priv->context_info_href);
}

static void
context_link_save_as_cb (NbtkAction *action,
                         MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->context_info_href && *priv->context_info_href)
    {
      gchar *filename =
        context_get_download_filename (self, priv->context_info_href);

      if (filename)
        {
          clutter_mozembed_save_uri (CLUTTER_MOZEMBED (self),
                                     priv->context_info_href,
                                     filename);
          g_free (filename);
        }
    }
}

static void
context_image_view_cb (NbtkAction *action,
                       MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->context_info_img_href && *priv->context_info_img_href)
    g_signal_emit (self, signals[NEW_TAB], 0, priv->context_info_img_href);
}

static void
context_image_copy_location_cb (NbtkAction *action,
                                MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->context_info_img_href && *priv->context_info_img_href)
    nbtk_clipboard_set_text (nbtk_clipboard_get_default (),
                             self->priv->context_info_img_href);
}

static void
context_image_save_as_cb (NbtkAction *action,
                          MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->context_info_img_href && *priv->context_info_img_href)
    {
      gchar *filename =
        context_get_download_filename (self, priv->context_info_img_href);

      if (filename)
        {
          clutter_mozembed_save_uri (CLUTTER_MOZEMBED (self),
                                     priv->context_info_img_href,
                                     filename);
          g_free (filename);
        }
    }
}

static void
context_text_copy_cb (NbtkAction *action,
                      MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->context_info_selected_txt && *priv->context_info_selected_txt)
    nbtk_clipboard_set_text (nbtk_clipboard_get_default (),
                             self->priv->context_info_selected_txt);
}

static void
context_page_back_cb (NbtkAction *action,
                      MwbMozEmbed *self)
{
  if (nbtk_action_get_active (action))
      clutter_mozembed_back (CLUTTER_MOZEMBED (self));
}

static void
context_page_forward_cb (NbtkAction *action,
                         MwbMozEmbed *self)
{
  if (nbtk_action_get_active (action))
    clutter_mozembed_forward (CLUTTER_MOZEMBED (self));
}

static void
context_page_stop_cb (NbtkAction *action,
                      MwbMozEmbed *self)
{
  if (nbtk_action_get_active (action))
    clutter_mozembed_stop (CLUTTER_MOZEMBED (self));
}

static void
context_page_reload_cb (NbtkAction *action,
                        MwbMozEmbed *self)
{
  if (nbtk_action_get_active (action))
    clutter_mozembed_refresh (CLUTTER_MOZEMBED (self));
}

static void
context_page_pin_cb (NbtkAction *action,
                     MwbMozEmbed *self)
{
  g_signal_emit (self, signals[PIN_PAGE], 0);
}

static void
context_page_save_as_cb (NbtkAction *action,
                         MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->context_info_uri && *priv->context_info_uri)
    {
      gchar *filename =
        context_get_download_filename (self, priv->context_info_uri);

      if (filename)
        {
          clutter_mozembed_save_uri (CLUTTER_MOZEMBED (self),
                                     priv->context_info_uri,
                                     filename);
          g_free (filename);
        }
    }
}

static void
mwb_mozembed_context_info_cb (ClutterMozEmbed *mozembed,
                              guint            type,
                              const gchar     *uri,
                              const gchar     *href,
                              const gchar     *img_href,
                              const gchar     *sel_text)
{
  gfloat x, y;

  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (mozembed)->priv;

  g_free (priv->context_info_uri);
  g_free (priv->context_info_href);
  g_free (priv->context_info_img_href);
  g_free (priv->context_info_selected_txt);

  priv->context_info_uri = g_strdup (uri);
  priv->context_info_href = g_strdup (href);
  priv->context_info_img_href = g_strdup (img_href);
  priv->context_info_selected_txt = g_strdup (sel_text);

  if (clutter_actor_transform_stage_point (CLUTTER_ACTOR (mozembed),
                                           priv->last_right_click_x,
                                           priv->last_right_click_y,
                                           &x,
                                           &y))
    {
      NbtkAction *action;
      nbtk_popup_clear (NBTK_POPUP (priv->popup));

      if (type & MOZ_HEADLESS_CTX_LINK)
        {
          action =
            nbtk_action_new_full (_("Open link in new tab"),
                                  G_CALLBACK (context_link_open_new_tab_cb),
                                  mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);

          action =
            nbtk_action_new_full (_("Copy link location"),
                                  G_CALLBACK (context_link_copy_cb),
                                  mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);

          action =
            nbtk_action_new_full (_("Save link"),
                                  G_CALLBACK (context_link_save_as_cb),
                                  mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);
        }

      if (type & MOZ_HEADLESS_CTX_IMAGE)
        {
          action =
            nbtk_action_new_full (_("View image"),
                                  G_CALLBACK (context_image_view_cb),
                                  mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);

          action =
            nbtk_action_new_full (_("Copy image location"),
                                  G_CALLBACK (context_image_copy_location_cb),
                                  mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);

          action =
            nbtk_action_new_full (_("Save image"),
                                  G_CALLBACK (context_image_save_as_cb),
                                  mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);
        }

      if (type & MOZ_HEADLESS_CTX_SELECTION)
        {
          action =
            nbtk_action_new_full (_("Copy text"),
                                  G_CALLBACK (context_text_copy_cb),
                                  mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);
        }

      if (!(type &
            (MOZ_HEADLESS_CTX_LINK |
             MOZ_HEADLESS_CTX_IMAGE |
             MOZ_HEADLESS_CTX_SELECTION)))
        {
          action = nbtk_action_new_full (_("Back"),
                                         G_CALLBACK (context_page_back_cb),
                                         mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);

          if (!clutter_mozembed_can_go_back (CLUTTER_MOZEMBED (mozembed)))
            nbtk_action_set_active (action, FALSE);
          else
            nbtk_action_set_active (action, TRUE);

          action = nbtk_action_new_full (_("Forward"),
                                         G_CALLBACK (context_page_forward_cb),
                                         mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);

          if (!clutter_mozembed_can_go_forward (CLUTTER_MOZEMBED (mozembed)))
            nbtk_action_set_active (action, FALSE);
          else
            nbtk_action_set_active (action, TRUE);

          action = nbtk_action_new_full (_("Stop"),
                                         G_CALLBACK (context_page_stop_cb),
                                         mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);

          if (clutter_mozembed_is_loading (CLUTTER_MOZEMBED (mozembed)))
            nbtk_action_set_active (action, FALSE);
          else
            nbtk_action_set_active (action, TRUE);

          action = nbtk_action_new_full (_("Reload"),
                                         G_CALLBACK (context_page_reload_cb),
                                         mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);

          if (!clutter_mozembed_is_loading (CLUTTER_MOZEMBED (mozembed)))
            nbtk_action_set_active (action, FALSE);
          else
            nbtk_action_set_active (action, TRUE);

          action = nbtk_action_new_full (_("Pin this page"),
                                         G_CALLBACK (context_page_pin_cb),
                                         mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);

          action = nbtk_action_new_full (_("Save page"),
                                         G_CALLBACK (context_page_save_as_cb),
                                         mozembed);
          nbtk_popup_add_action (NBTK_POPUP (priv->popup), action);
        }

      clutter_actor_set_position (CLUTTER_ACTOR (priv->popup), x, y);
      mwb_utils_show_popup (NBTK_POPUP (priv->popup));
    }
}

static gboolean
mwb_mozembed_button_press_event (ClutterActor       *actor,
                                 ClutterButtonEvent *event)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;

  if (!CLUTTER_ACTOR_IS_VISIBLE (CLUTTER_ACTOR (priv->popup)) &&
      (event->button == 3))
    {
      priv->last_right_click_x = event->x;
      priv->last_right_click_y = event->y;
    }

  return CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->
    button_press_event (actor, event);
}

static void
mwb_mozembed_paint (ClutterActor *actor)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->paint (actor);

  if (CLUTTER_ACTOR_IS_MAPPED (priv->popup))
    clutter_actor_paint (CLUTTER_ACTOR (priv->popup));
}

static void
mwb_mozembed_pick (ClutterActor       *actor,
                   const ClutterColor *color)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->pick (actor, color);

  if (CLUTTER_ACTOR_IS_MAPPED (priv->popup))
    clutter_actor_paint (CLUTTER_ACTOR (priv->popup));
}

static void
mwb_mozembed_map (ClutterActor *actor)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;
  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->map (actor);
  clutter_actor_map (CLUTTER_ACTOR (priv->tooltip_hack));
  clutter_actor_map (CLUTTER_ACTOR (priv->popup));
}

static void
mwb_mozembed_unmap (ClutterActor *actor)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;
  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->unmap (actor);
  clutter_actor_unmap (CLUTTER_ACTOR (priv->tooltip_hack));
  clutter_actor_unmap (CLUTTER_ACTOR (priv->popup));
}

static void
mwb_mozembed_key_focus_out (ClutterActor *actor)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;

  clutter_actor_hide (CLUTTER_ACTOR (priv->popup));

  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->key_focus_out (actor);
}

static void
mwb_mozembed_class_init (MwbMozEmbedClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbMozEmbedPrivate));

  object_class->get_property = mwb_mozembed_get_property;
  object_class->set_property = mwb_mozembed_set_property;
  object_class->dispose = mwb_mozembed_dispose;
  object_class->finalize = mwb_mozembed_finalize;

  actor_class->allocate = mwb_mozembed_allocate;
  actor_class->button_press_event = mwb_mozembed_button_press_event;
  actor_class->paint = mwb_mozembed_paint;
  actor_class->pick = mwb_mozembed_pick;
  actor_class->map = mwb_mozembed_map;
  actor_class->unmap = mwb_mozembed_unmap;
  actor_class->key_focus_out = mwb_mozembed_key_focus_out;

  g_object_class_install_property (object_class,
                                   PROP_SYNC_ADJUSTMENTS,
                                   g_param_spec_boolean ("sync-adjustments",
                                                         "Sync adjustments",
                                                         "Keep adjustment values "
                                                         "synchronised with "
                                                         "internal state.",
                                                         TRUE,
                                                         G_PARAM_READWRITE |
                                                         G_PARAM_STATIC_NAME |
                                                         G_PARAM_STATIC_NICK |
                                                         G_PARAM_STATIC_BLURB));

  signals[NEW_TAB] =
    g_signal_new ("new-tab",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbMozEmbedClass, new_tab),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  signals[PIN_PAGE] =
    g_signal_new ("pin-page",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbMozEmbedClass, pin_page),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  g_object_class_override_property (object_class,
                                    PROP_HADJUST,
                                    "hadjustment");

  g_object_class_override_property (object_class,
                                    PROP_VADJUST,
                                    "vadjustment");
}

static void
mwb_mozembed_scroll_x_notify_cb (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

#ifdef MWB_NATIVE_SCROLL
  if (priv->hadjust && priv->sync_adjust)
    {
      gint scroll_x;
      g_object_get (G_OBJECT (self), "scroll-x", &scroll_x, NULL);
      nbtk_adjustment_set_value (priv->hadjust, scroll_x);
    }
#endif

  nbtk_widget_hide_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_scroll_y_notify_cb (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

#ifdef MWB_NATIVE_SCROLL
  if (priv->vadjust && priv->sync_adjust)
    {
      gint scroll_y;
      g_object_get (G_OBJECT (self), "scroll-y", &scroll_y, NULL);
      nbtk_adjustment_set_value (priv->vadjust, scroll_y);
    }
#endif

  nbtk_widget_hide_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_doc_width_notify_cb (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

#ifdef MWB_NATIVE_SCROLL
  if (priv->hadjust)
    {
      gint doc_width;
      g_object_get (G_OBJECT (self), "doc-width", &doc_width, NULL);
      g_object_set (G_OBJECT (priv->hadjust),
                    "upper", (gdouble)doc_width, NULL);
    }
#endif

  nbtk_widget_hide_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_doc_height_notify_cb (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

#ifdef MWB_NATIVE_SCROLL
  if (priv->vadjust)
    {
      gint doc_height;
      g_object_get (G_OBJECT (self), "doc-height", &doc_height, NULL);
      g_object_set (G_OBJECT (priv->vadjust),
                    "upper", (gdouble)doc_height, NULL);
    }
#endif

  nbtk_widget_hide_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_show_tooltip_cb (MwbMozEmbed *self,
                              const gchar *text,
                              gint         x,
                              gint         y)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (CLUTTER_ACTOR_IS_VISIBLE (CLUTTER_ACTOR (priv->popup)))
    return;

  clutter_actor_set_position (CLUTTER_ACTOR (priv->tooltip_hack), x, y);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (priv->tooltip_hack), 0);
  nbtk_widget_set_tooltip_text (priv->tooltip_hack, text);
  nbtk_widget_show_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_hide_tooltip_cb (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;
  nbtk_widget_hide_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_init (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv = MOZEMBED_PRIVATE (self);

  priv->sync_adjust = TRUE;

  g_signal_connect (self, "notify::scroll-x",
                    G_CALLBACK (mwb_mozembed_scroll_x_notify_cb), NULL);
  g_signal_connect (self, "notify::scroll-y",
                    G_CALLBACK (mwb_mozembed_scroll_y_notify_cb), NULL);
  g_signal_connect (self, "notify::doc-width",
                    G_CALLBACK (mwb_mozembed_doc_width_notify_cb), NULL);
  g_signal_connect (self, "notify::doc-height",
                    G_CALLBACK (mwb_mozembed_doc_height_notify_cb), NULL);
  g_signal_connect (self, "show-tooltip",
                    G_CALLBACK (mwb_mozembed_show_tooltip_cb), NULL);
  g_signal_connect (self, "hide-tooltip",
                    G_CALLBACK (mwb_mozembed_hide_tooltip_cb), NULL);
  g_signal_connect (self, "context-info",
                    G_CALLBACK (mwb_mozembed_context_info_cb), NULL);

  priv->tooltip_hack = nbtk_bin_new ();
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->tooltip_hack),
                            CLUTTER_ACTOR (self));
  clutter_actor_set_height (CLUTTER_ACTOR (priv->tooltip_hack), 24);

  priv->popup = nbtk_popup_new ();
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->popup),
                            CLUTTER_ACTOR (self));
  clutter_actor_hide (CLUTTER_ACTOR (priv->popup));
}

ClutterActor *
mwb_mozembed_new (gboolean private)
{
  return g_object_new (MWB_TYPE_MOZEMBED,
                       "comp-paths", mwb_mozembed_comp_paths,
                       "chrome-paths", mwb_mozembed_chrome_paths,
                       "user-chrome-path", PKGDATADIR "/chrome",
#ifdef MWB_NATIVE_SCROLL
                       "scrollbars", FALSE,
#endif
                       "private", private,
                       NULL);
}

ClutterActor *
mwb_mozembed_new_with_parent (ClutterMozEmbed *parent)
{
  return g_object_new (MWB_TYPE_MOZEMBED,
                       "comp-paths", mwb_mozembed_comp_paths,
                       "chrome-paths", mwb_mozembed_chrome_paths,
                       "user-chrome-path", PKGDATADIR "/chrome",
                       "parent", parent,
#ifdef MWB_NATIVE_SCROLL
                       "scrollbars", FALSE,
#endif
                       NULL);
}

ClutterActor *
mwb_mozembed_new_for_new_window ()
{
  return g_object_new (MWB_TYPE_MOZEMBED,
                       "comp-paths", mwb_mozembed_comp_paths,
                       "chrome-paths", mwb_mozembed_chrome_paths,
                       "user-chrome-path", PKGDATADIR "/chrome",
                       "spawn", FALSE,
#ifdef MWB_NATIVE_SCROLL
                       "scrollbars", FALSE,
#endif
                       NULL);
}

static void
scrollable_set_adjustments (NbtkScrollable *scrollable,
                            NbtkAdjustment *hadjust,
                            NbtkAdjustment *vadjust)
{
  NbtkAdjustment *old_hadjust, *old_vadjust;

  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (scrollable)->priv;

  old_hadjust = priv->hadjust;
  old_vadjust = priv->vadjust;

  if (priv->hadjust)
    g_signal_handlers_disconnect_by_func (priv->hadjust,
                                          hadjust_value_notify_cb,
                                          scrollable);

  if (priv->vadjust)
    g_signal_handlers_disconnect_by_func (priv->vadjust,
                                          vadjust_value_notify_cb,
                                          scrollable);

  priv->hadjust = hadjust;
  priv->vadjust = vadjust;

  if (priv->hadjust)
    {
      g_object_ref (priv->hadjust);
#ifdef MWB_NATIVE_SCROLL
      g_signal_connect (priv->hadjust, "notify::value",
                        G_CALLBACK (hadjust_value_notify_cb), scrollable);
#else
      g_object_set (G_OBJECT (priv->hadjust),
                    "lower", 0.0,
                    "upper", 1.0,
                    "page-size", 1.0,
                    "page-increment", 1.0,
                    "value", 0.0,
                    NULL);
#endif
    }

  if (priv->vadjust)
    {
      g_object_ref (priv->vadjust);
#ifdef MWB_NATIVE_SCROLL
      g_signal_connect (priv->vadjust, "notify::value",
                        G_CALLBACK (vadjust_value_notify_cb), scrollable);
#else
      g_object_set (G_OBJECT (priv->vadjust),
                    "lower", 0.0,
                    "upper", 1.0,
                    "page-size", 1.0,
                    "page-increment", 1.0,
                    "value", 0.0,
                    NULL);
#endif
    }

  if (old_hadjust)
    g_object_unref (old_hadjust);
  if (old_vadjust)
    g_object_unref (old_vadjust);
}

static void
mwb_mozembed_ensure_adjustments (MwbMozEmbed *mozembed)
{
  gint doc_width, doc_height, scroll_x, scroll_y, width, height;

  MwbMozEmbedPrivate *priv = mozembed->priv;

  if (priv->hadjust && priv->vadjust)
    return;

  g_object_get (G_OBJECT (mozembed),
                "doc-width", &doc_width,
                "doc-height", &doc_height,
                "scroll-x", &scroll_x,
                "scroll-y", &scroll_y,
                NULL);
  clutter_texture_get_base_size (CLUTTER_TEXTURE (mozembed),
                                 &width, &height);

  if (!priv->hadjust)
    {
      priv->hadjust = nbtk_adjustment_new (scroll_x,
                                           0.0,
                                           doc_width,
                                           8.0,
                                           width/2.0,
                                           width);
    }

  if (!priv->vadjust)
    {
      priv->vadjust = nbtk_adjustment_new (scroll_y,
                                           0.0,
                                           doc_height,
                                           8.0,
                                           height/2.0,
                                           height);
    }

  scrollable_set_adjustments (NBTK_SCROLLABLE (mozembed),
                              priv->hadjust,
                              priv->vadjust);
}

static void
scrollable_get_adjustments (NbtkScrollable *scrollable,
                            NbtkAdjustment **hadjustment,
                            NbtkAdjustment **vadjustment)
{
  MwbMozEmbed *mozembed = MWB_MOZEMBED (scrollable);
  MwbMozEmbedPrivate *priv = mozembed->priv;

  mwb_mozembed_ensure_adjustments (mozembed);

  if (hadjustment)
    *hadjustment = priv->hadjust;

  if (vadjustment)
    *vadjustment = priv->vadjust;
}

static void
scrollable_interface_init (NbtkScrollableInterface *iface)
{
  iface->set_adjustments = scrollable_set_adjustments;
  iface->get_adjustments = scrollable_get_adjustments;
}

gboolean
mwb_mozembed_get_sync_adjustments (MwbMozEmbed *mozembed)
{
  return mozembed->priv->sync_adjust;
}

void
mwb_mozembed_set_sync_adjustments (MwbMozEmbed *mozembed, gboolean sync)
{
  MwbMozEmbedPrivate *priv = mozembed->priv;

  if (priv->sync_adjust != sync)
    {
      priv->sync_adjust = sync;

      if (sync)
        {
          gint scroll_x, scroll_y;
          g_object_get (G_OBJECT (mozembed),
                        "scroll-x", &scroll_x,
                        "scroll-y", &scroll_y,
                        NULL);
          if (priv->hadjust)
            nbtk_adjustment_set_value (priv->hadjust, scroll_x);
          if (priv->vadjust)
            nbtk_adjustment_set_value (priv->vadjust, scroll_y);
        }
    }
}

void
mwb_mozembed_set_mwb_window (MwbMozEmbed *mozembed, GtkWidget *window)
{
  MwbMozEmbedPrivate *priv = mozembed->priv;
  priv->mwb_window = window;
}

