/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <clutter/clutter.h>
#include <clutter/x11/clutter-x11.h>
#include <clutter-gtk/clutter-gtk.h>
#include <clutter-mozembed/clutter-mozembed.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <nbtk/nbtk.h>
#include <string.h>
#include <unique/unique.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <sys/file.h>
#include <unistd.h>
#include <glib/gstdio.h>
#include <mhs/mhs.h>

#define MWB_SN_SUPPORT 1

#ifdef MWB_SN_SUPPORT
#  define SN_API_NOT_YET_FROZEN
#  include <libsn/sn.h>
#endif

#include "mwb-window.h"
#include "mwb-browser.h"
#include "mwb-dbus.h"

enum
{
  COMMAND_0,

  COMMAND_OPEN_USER_INITIATED
};

static UniqueResponse
mwb_message_received_cb (UniqueApp         *app,
                         gint               command,
                         UniqueMessageData *message_data,
                         guint              time_,
                         gpointer           user_data)
{
  gchar *url;

  MwbDbus *mwb_dbus = MWB_DBUS (user_data);

  switch (command)
    {
    case UNIQUE_ACTIVATE :
      mwb_dbus_raise (mwb_dbus, NULL);
      return UNIQUE_RESPONSE_OK;

    case UNIQUE_NEW :
      mwb_dbus_new_tab (mwb_dbus, NULL, FALSE, FALSE, NULL);
      mwb_dbus_raise (mwb_dbus, NULL);
      return UNIQUE_RESPONSE_OK;

    case UNIQUE_OPEN :
    case COMMAND_OPEN_USER_INITIATED :
      url = unique_message_data_get_text (message_data);
      if (url && (url[0] != '\0'))
        {
          MwbBrowserOpenFlags flags =
            MWB_BROWSER_OPEN_NEW_TAB | MWB_BROWSER_OPEN_USE_EXISTING |
            MWB_BROWSER_OPEN_ACTIVATE;
          if (command == COMMAND_OPEN_USER_INITIATED)
            flags |= MWB_BROWSER_OPEN_USER_INITIATED;
          mwb_dbus_new_tab_with_flags (mwb_dbus, url, flags, NULL);
        }
      g_free (url);
      mwb_dbus_raise (mwb_dbus, NULL);
      return UNIQUE_RESPONSE_OK;
    }

  return UNIQUE_RESPONSE_PASSTHROUGH;
}

#ifdef MWB_SN_SUPPORT
static gboolean
startup_cb (MwbBrowser *mwb_browser)
{
  SnLauncheeContext *context;
  ClutterActor *stage;
  Window xwindow;
  
  stage = clutter_actor_get_stage (CLUTTER_ACTOR (mwb_browser));
  if (!stage || !CLUTTER_ACTOR_IS_REALIZED (stage))
    return FALSE;

  context = g_object_get_data (G_OBJECT (mwb_browser), "sn-context");
  xwindow = clutter_x11_get_stage_window (CLUTTER_STAGE (stage));
  sn_launchee_context_setup_window (context, xwindow);
  sn_launchee_context_complete (context);
  sn_launchee_context_unref (context);

  return FALSE;
}
#endif

static void
assert_session_dir (void)
{
    char *session_dir =
      g_build_filename (g_get_home_dir (), ".moblin-web-browser", NULL);
    g_mkdir (session_dir, 0777);
    g_free (session_dir);
}

static gboolean
mwb_load_session (MwbBrowser  *browser,
                  gboolean     crash)
{
  int fd;
  GKeyFile *keys;
  gboolean success;
  gchar *session_file;

  gboolean opened_something = FALSE;

  assert_session_dir ();

  session_file =
    g_build_filename (g_get_home_dir (),
                      ".moblin-web-browser",
                      crash ? "crash-session" : "session",
                      NULL);

  if (!g_file_test (session_file, G_FILE_TEST_EXISTS))
    {
      g_free (session_file);
      return FALSE;
    }

  fd = g_open (session_file, O_RDWR|O_CREAT, S_IRUSR|S_IWUSR);
  if (fd == -1)
    {
      g_warning ("Failed to open session file %s", session_file);
      g_free (session_file);
      return FALSE;
    }

  if (!crash)
    {
      if (flock (fd,  LOCK_EX|LOCK_NB) == -1)
        {
          g_warning ("Failed to lock session file %s", session_file);
          g_free (session_file);
          return FALSE;
        }
    }

  keys = g_key_file_new ();

  success = g_key_file_load_from_file (keys, session_file, 0, NULL);
  if (!success)
    {
      g_warning ("Failed to load keys from session file %s", session_file);
    }
  else
    {
      int i;
      char **groups = g_key_file_get_groups (keys, NULL);

      for (i = 0; groups[i]; i++)
        {
          char *url = g_key_file_get_string (keys, groups[i], "url", NULL);
          if (url)
            {
              mwb_browser_open (browser, strcmp (url, "NULL") ? url : NULL,
                                MWB_BROWSER_OPEN_NEW_TAB |
                                MWB_BROWSER_OPEN_ACTIVATE);
              opened_something = TRUE;
              g_free (url);
            }
        }
      g_strfreev (groups);
    }

  g_key_file_free (keys);
  g_free (session_file);

  if (crash)
    {
      close (fd);
      g_remove (session_file);
    }

  return opened_something;
}

void
mwb_save_session (MwbBrowser *browser)
{
  char *session_file, *keys_data, *home;
  GList *p;
  gsize keys_len;

  GList *pages = mwb_browser_get_pages (browser);
  GKeyFile *keys = g_key_file_new ();
  GError *error = NULL;
  gint page_count = 0;

  g_object_get (G_OBJECT (browser), "home", &home, NULL);

  for (p = pages; p; p = p->next)
    {
      const gchar *url;
      gchar *group_name;
      ClutterMozEmbed *mozembed = p->data;

      if (clutter_mozembed_get_private (mozembed))
        continue;

      url = clutter_mozembed_get_location (mozembed);

      /* Don't save tabs on the home-page */
      if (url && (!home || g_str_equal (url, home)))
        continue;

      group_name = g_strdup_printf ("page %d", page_count);
      g_key_file_set_string (keys, group_name, "url", url ? url : "NULL");
      g_free (group_name);
      page_count++;
    }

  g_free (home);
  g_list_free (pages);

  session_file =
    g_build_filename (g_get_home_dir (), ".moblin-web-browser", "crash-session", NULL);
  keys_data = g_key_file_to_data (keys, &keys_len, NULL);
  if (!g_file_set_contents (session_file, keys_data, keys_len, &error))
    {
      g_warning ("Failed to save to session file %s: %s",
                 session_file,
                 error->message);
    }

  g_free (session_file);
  g_key_file_free (keys);
}

static void
mwb_window_destroy_cb (GtkObject *object, gpointer data)
{
  gboolean *mwb_window_destroyed = data;

  *mwb_window_destroyed = TRUE;
  gtk_main_quit ();
}

int
main (int argc, char **argv)
{
  gchar *crash_session, *new_session;
  gulong destroy_handler_id;
  MwbBrowser *mwb_browser;
  GtkWidget *mwb_window;
  gboolean opened_tabs;
  MwbToolbar *toolbar;
  MwbDbus *mwb_dbus;
  MhsPrefs *prefs;
  UniqueApp *app;
  gint pref_id;

#ifdef MWB_SN_SUPPORT
  SnLauncheeContext *context;
  SnDisplay *display;
  Display *xdisplay;
  int screen;
#endif

  GError *error = NULL;
  gboolean restore_session = FALSE;
  gboolean mwb_window_destroyed = FALSE;

  static gboolean windowed = FALSE;
  static gboolean user_initiated = FALSE;
  static GOptionEntry entries[] = {
    { "windowed", 'w', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_NONE, &windowed,
      N_("Run the browser in a window"), NULL },
    { "user-initiated", 'I', G_OPTION_FLAG_NO_ARG, G_OPTION_ARG_NONE,
      &user_initiated,
      N_("Add the URL to the recent files list"), NULL },
    { NULL }
  };

  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  if (!gtk_clutter_init_with_args (&argc, &argv, "Moblin web browser",
                                   entries, NULL, &error) || error)
    {
      g_error ("Error initialising GtkClutter: %s", error->message);
      g_error_free (error);
      return 1;
    }

  g_set_application_name (_("Moblin Web Browser"));

#ifdef MWB_SN_SUPPORT
  xdisplay = clutter_x11_get_default_display ();

  if (g_getenv ("LIBSN_SYNC") != NULL)
    XSynchronize (xdisplay, True);

  display = sn_display_new (xdisplay, NULL, NULL);
  screen = clutter_x11_get_default_screen ();
  context = sn_launchee_context_new_from_environment (display, screen);
#endif

  app = unique_app_new_with_commands ("org.moblin.MoblinWebBrowser",
                                      NULL,
                                      "open-user-initiated",
                                      COMMAND_OPEN_USER_INITIATED,
                                      NULL);
  if (unique_app_is_running (app))
    {
      if (argc < 2)
        unique_app_send_message (app, UNIQUE_NEW, NULL);
      else
        {
          UniqueMessageData *data = unique_message_data_new ();
          unique_message_data_set_text (data, argv[1], -1);
          unique_app_send_message (app,
                                   user_initiated ?
                                   COMMAND_OPEN_USER_INITIATED :
                                   UNIQUE_OPEN,
                                   data);
          unique_message_data_free (data);
        }

      return 0;
    }

  /*g_log_set_always_fatal (G_LOG_LEVEL_WARNING);*/

  nbtk_style_load_from_file (nbtk_style_get_default (),
                             PKGDATADIR "/default.css", NULL);

  gtk_window_set_default_icon_name ("moblin-web-browser");

  mwb_window = mwb_window_new ();
  /* NB: we can't simply call gtk_main_quit here because we need to be able
   * to determine once the mainloop has quit if the mwb_window has already
   * been destroyed or not. */
  destroy_handler_id =
    g_signal_connect (mwb_window, "destroy",
                      G_CALLBACK (mwb_window_destroy_cb),
                      &mwb_window_destroyed);
  mwb_browser =
    MWB_BROWSER (mwb_window_get_internal_actor (MWB_WINDOW (mwb_window)));

#ifdef MWB_SN_SUPPORT
  if (context)
    {
      g_object_set_data (G_OBJECT (mwb_browser), "sn-context", context);
      g_idle_add ((GSourceFunc)startup_cb, mwb_browser);
    }
#endif

  gtk_window_resize (GTK_WINDOW (mwb_window), 1024, 600);
  if (!windowed)
    mwb_window_fullscreen (MWB_WINDOW (mwb_window));

  mwb_dbus = mwb_dbus_create (MWB_WINDOW (mwb_window), mwb_browser);

  g_signal_connect (app, "message-received",
                    G_CALLBACK (mwb_message_received_cb), mwb_dbus);

  /* Set the user agent */
  prefs = mhs_prefs_new ();
  if (mhs_prefs_get_default_branch (prefs,
                                    "general.useragent.extra",
                                    &pref_id,
                                    &error))
    {
      if (!mhs_prefs_branch_set_char (prefs,
                                      pref_id,
                                      ".moblin",
                                      "MoblinWebBrowser/" MWB_VERSION_STRING,
                                      &error) ||
          !mhs_prefs_branch_set_char (prefs,
                                      pref_id,
                                      ".firefox",
                                      "Firefox/3.5",
                                      &error))
        {
          g_warning ("Error setting user-agent: %s", error->message);
          g_clear_error (&error);
        }
      if (!mhs_prefs_release_branch (prefs, pref_id, &error))
        {
          g_warning ("Error releasing user-agent pref branch: %s",
                     error->message);
          g_clear_error (&error);
        }
    }
  else
    {
      g_warning ("Error retrieving user-agent pref branch: %s", error->message);
      g_clear_error (&error);
    }
  if (!mhs_prefs_branch_get_bool (prefs,
                                  0,
                                  "mwb.restore_session",
                                  &restore_session,
                                  &error))
    {
      g_warning ("Error retrieving session restoring preference: %s",
                 error->message);
      g_clear_error (&error);
    }
  g_object_unref (G_OBJECT (prefs));

  opened_tabs = mwb_load_session (mwb_browser, TRUE);
  if (!opened_tabs && restore_session)
    opened_tabs = mwb_load_session (mwb_browser, FALSE);

  if (argc >= 2)
    {
      MwbBrowserOpenFlags flags = (MWB_BROWSER_OPEN_USE_EXISTING |
                                   MWB_BROWSER_OPEN_NEW_TAB |
                                   MWB_BROWSER_OPEN_ACTIVATE);
      if (user_initiated)
        flags |= MWB_BROWSER_OPEN_USER_INITIATED;
      mwb_browser_open (mwb_browser, argv[1], flags);
    }
  else if (!opened_tabs)
    mwb_browser_open (mwb_browser, NULL,
                      MWB_BROWSER_OPEN_NEW_TAB |
                      MWB_BROWSER_OPEN_ACTIVATE);

  gtk_widget_show_all (GTK_WIDGET (mwb_window));

  /* Focus location entry */
  toolbar = mwb_browser_get_toolbar (mwb_browser);
  mwb_radical_bar_focus (mwb_toolbar_get_radical_bar (toolbar));

  gtk_main ();

  /* We need to cleanly dispose everything so we get the chance to unmap and
   * reparent the plugin viewport window (which is owned by the moz-headless
   * process) back to the root window before our X client connection closes.
   * (otherwise the X SaveSet mechanism will kick in and reparent the viewport
   * back to the root window, but also map it, which we don't want.)
   */
  if (!mwb_window_destroyed)
    {
      g_signal_handler_disconnect (mwb_window, destroy_handler_id);
      gtk_widget_destroy (mwb_window);
    }

  /* Rename session file */
  crash_session = g_build_filename (g_get_home_dir (),
                                    ".moblin-web-browser",
                                    "crash-session",
                                    NULL);
  new_session = g_build_filename (g_get_home_dir (),
                                  ".moblin-web-browser",
                                  "session",
                                  NULL);
  g_rename (crash_session, new_session);
  g_free (crash_session);
  g_free (new_session);

  return 0;
}

