/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "mwb-popup.h"

G_DEFINE_TYPE (MwbPopup, mwb_popup, NBTK_TYPE_WIDGET)

#define POPUP_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_POPUP, MwbPopupPrivate))

typedef struct
{
  MwbAction  *action;
  NbtkWidget *button;
} MwbPopupChild;

struct _MwbPopupPrivate
{
  GArray   *children;
  gboolean  transition_out;
};

enum
{
  ACTION_ACTIVATED,

  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

static void
mwb_popup_get_property (GObject *object, guint property_id,
                              GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_popup_set_property (GObject *object, guint property_id,
                              const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_popup_free_action_at (MwbPopup *popup, gint index, gboolean remove)
{
  MwbPopupPrivate *priv = popup->priv;
  MwbPopupChild *child = &g_array_index (priv->children, MwbPopupChild, index);

  clutter_actor_unparent (CLUTTER_ACTOR (child->button));
  g_object_unref (child->action);

  if (remove)
    g_array_remove_index (priv->children, index);
}

static void
mwb_popup_dispose (GObject *object)
{
  MwbPopup *popup = MWB_POPUP (object);
  MwbPopupPrivate *priv = popup->priv;

  if (priv->children)
    {
      gint i;
      for (i = 0; i < priv->children->len; i++)
        mwb_popup_free_action_at (popup, i, FALSE);
      g_array_free (priv->children, TRUE);
      priv->children = NULL;
    }

  G_OBJECT_CLASS (mwb_popup_parent_class)->dispose (object);
}

static void
mwb_popup_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_popup_parent_class)->finalize (object);
}

static void
mwb_popup_get_preferred_width (ClutterActor *actor,
                               gfloat        for_height,
                               gfloat       *min_width_p,
                               gfloat       *natural_width_p)
{
  gint i;
  NbtkPadding padding;
  gfloat min_width, nat_width;

  MwbPopupPrivate *priv = MWB_POPUP (actor)->priv;

  /* Add padding and the size of the widest child */
  nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);
  min_width = nat_width = 0;
  for (i = 0; i < priv->children->len; i++)
    {
      gfloat child_min_width, child_nat_width;

      MwbPopupChild *child = &g_array_index (priv->children, MwbPopupChild, i);

      clutter_actor_get_preferred_width (CLUTTER_ACTOR (child->button),
                                         for_height,
                                         &child_min_width,
                                         &child_nat_width);

      if (child_min_width > min_width)
        min_width = child_min_width;
      if (child_nat_width > nat_width)
        nat_width = child_nat_width;
    }

  if (min_width_p)
    *min_width_p += min_width + padding.left + padding.right;
  if (natural_width_p)
    *natural_width_p += nat_width + padding.left + padding.right;
}

static void
mwb_popup_get_preferred_height (ClutterActor *actor,
                                gfloat        for_width,
                                gfloat       *min_height_p,
                                gfloat       *natural_height_p)
{
  gint i;
  NbtkPadding padding;
  gfloat min_height, nat_height;

  MwbPopupPrivate *priv = MWB_POPUP (actor)->priv;

  /* Add padding and the cumulative height of the children */
  nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);
  min_height = nat_height = padding.top + padding.bottom;
  for (i = 0; i < priv->children->len; i++)
    {
      gfloat child_min_height, child_nat_height;

      MwbPopupChild *child = &g_array_index (priv->children, MwbPopupChild, i);

      clutter_actor_get_preferred_height (CLUTTER_ACTOR (child->button),
                                          for_width,
                                          &child_min_height,
                                          &child_nat_height);

      min_height += child_min_height;
      nat_height += child_nat_height;
    }

  if (min_height_p)
    *min_height_p += min_height;
  if (natural_height_p)
    *natural_height_p += nat_height;
}

static void
mwb_popup_allocate (ClutterActor           *actor,
                    const ClutterActorBox  *box,
                    ClutterAllocationFlags  flags)
{
  gint i;
  NbtkPadding padding;
  ClutterActorBox child_box;
  MwbPopupPrivate *priv = MWB_POPUP (actor)->priv;

  /* Allocate children */
  nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);
  child_box.x1 = padding.left;
  child_box.y1 = padding.top;
  child_box.x2 = box->x2 - box->x1 - padding.right;
  for (i = 0; i < priv->children->len; i++)
    {
      gfloat natural_height;

      MwbPopupChild *child = &g_array_index (priv->children, MwbPopupChild, i);

      clutter_actor_get_preferred_height (CLUTTER_ACTOR (child->button),
                                          child_box.x2 - child_box.x1,
                                          NULL,
                                          &natural_height);
      child_box.y2 = child_box.y1 + natural_height;

      clutter_actor_allocate (CLUTTER_ACTOR (child->button), &child_box, flags);

      child_box.y1 = child_box.y2 + 1;
    }

  /* Chain up and allocate background */
  CLUTTER_ACTOR_CLASS (mwb_popup_parent_class)-> allocate (actor, box, flags);
}

static void
mwb_popup_paint (ClutterActor *actor)
{
  gint i;
  MwbPopupPrivate *priv = MWB_POPUP (actor)->priv;

  /* Chain up to get background */
  CLUTTER_ACTOR_CLASS (mwb_popup_parent_class)->paint (actor);

  /* Paint children */
  for (i = 0; i < priv->children->len; i++)
    {
      MwbPopupChild *child = &g_array_index (priv->children, MwbPopupChild, i);
      clutter_actor_paint (CLUTTER_ACTOR (child->button));
    }
}

static void
mwb_popup_pick (ClutterActor       *actor,
                const ClutterColor *color)
{
  ClutterActorBox box;

  cogl_set_source_color4ub (color->red,
                            color->green,
                            color->blue,
                            color->alpha);
  clutter_actor_get_allocation_box (actor, &box);
  cogl_rectangle (0, 0, box.x2 - box.x1, box.y2 - box.y1);

  /* Use paint to pick children */
  mwb_popup_paint (actor);
}

static void
mwb_popup_map (ClutterActor *actor)
{
  gint i;
  MwbPopupPrivate *priv = MWB_POPUP (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_popup_parent_class)->map (actor);

  for (i = 0; i < priv->children->len; i++)
    {
      MwbPopupChild *child = &g_array_index (priv->children, MwbPopupChild, i);
      clutter_actor_map (CLUTTER_ACTOR (child->button));
    }
}

static void
mwb_popup_unmap (ClutterActor *actor)
{
  gint i;
  MwbPopupPrivate *priv = MWB_POPUP (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_popup_parent_class)->unmap (actor);

  for (i = 0; i < priv->children->len; i++)
    {
      MwbPopupChild *child = &g_array_index (priv->children, MwbPopupChild, i);
      clutter_actor_unmap (CLUTTER_ACTOR (child->button));
    }
}

static gboolean
mwb_popup_event (ClutterActor *actor,
                 ClutterEvent *event)
{
  /* We swallow mouse events so that they don't fall through to whatever's
   * beneath us.
   */
  switch (event->type)
    {
    case CLUTTER_MOTION:
    case CLUTTER_BUTTON_PRESS:
    case CLUTTER_BUTTON_RELEASE:
    case CLUTTER_SCROLL:
      return TRUE;
    default:
      return FALSE;
    }
}

static void
mwb_popup_class_init (MwbPopupClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbPopupPrivate));

  object_class->get_property = mwb_popup_get_property;
  object_class->set_property = mwb_popup_set_property;
  object_class->dispose = mwb_popup_dispose;
  object_class->finalize = mwb_popup_finalize;

  actor_class->get_preferred_width = mwb_popup_get_preferred_width;
  actor_class->get_preferred_height = mwb_popup_get_preferred_height;
  actor_class->allocate = mwb_popup_allocate;
  actor_class->paint = mwb_popup_paint;
  actor_class->pick = mwb_popup_pick;
  actor_class->map = mwb_popup_map;
  actor_class->unmap = mwb_popup_unmap;
  actor_class->event = mwb_popup_event;

  signals[ACTION_ACTIVATED] =
    g_signal_new ("action-activated",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbPopupClass, action_activated),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__OBJECT,
                  G_TYPE_NONE, 1, MWB_TYPE_ACTION);
}

static void
mwb_popup_init (MwbPopup *self)
{
  MwbPopupPrivate *priv = self->priv = POPUP_PRIVATE (self);

  priv->children = g_array_new (FALSE, FALSE, sizeof (MwbPopupChild));

  g_object_set (G_OBJECT (self),
                "scale-gravity", CLUTTER_GRAVITY_CENTER,
                "scale-x", 0.0,
                "scale-y", 0.0,
                "show-on-set-parent", FALSE,
                "reactive", TRUE,
                NULL);
}

NbtkWidget *
mwb_popup_new (void)
{
  return g_object_new (MWB_TYPE_POPUP, NULL);
}

static void
mwb_popup_button_clicked_cb (NbtkButton *button,
                             MwbAction  *action)
{
  MwbPopup *popup =
    MWB_POPUP (clutter_actor_get_parent (CLUTTER_ACTOR (button)));

  g_signal_emit (popup, signals[ACTION_ACTIVATED], 0, action);
  g_signal_emit_by_name (action, "activated");

  mwb_popup_hide (popup);
}

void
mwb_popup_add_action (MwbPopup  *popup,
                      MwbAction *action)
{
  MwbPopupChild child;

  MwbPopupPrivate *priv = popup->priv;

  child.action = g_object_ref_sink (action);
  /* TODO: Connect to notify signals in case action properties change */
  child.button = nbtk_button_new_with_label (mwb_action_get_name (action));
  nbtk_bin_set_alignment (NBTK_BIN (child.button),
                          NBTK_ALIGN_LEFT,
                          NBTK_ALIGN_CENTER);
  g_signal_connect (child.button, "clicked",
                    G_CALLBACK (mwb_popup_button_clicked_cb), action);
  clutter_actor_set_parent (CLUTTER_ACTOR (child.button),
                            CLUTTER_ACTOR (popup));

  g_array_append_val (priv->children, child);

  clutter_actor_queue_relayout (CLUTTER_ACTOR (popup));
}

void
mwb_popup_remove_action (MwbPopup  *popup,
                         MwbAction *action)
{
  gint i;

  MwbPopupPrivate *priv = popup->priv;

  for (i = 0; i < priv->children->len; i++)
    {
      MwbPopupChild *child = &g_array_index (priv->children, MwbPopupChild, i);

      if (child->action == action)
        {
          mwb_popup_free_action_at (popup, i, TRUE);
          break;
        }
    }
}

void
mwb_popup_clear (MwbPopup *popup)
{
  gint i;

  MwbPopupPrivate *priv = popup->priv;

  if (!priv->children->len)
    return;

  for (i = 0; i < priv->children->len; i++)
    mwb_popup_free_action_at (popup, i, FALSE);

  g_array_remove_range (priv->children, 0, priv->children->len);
}

static void
mwb_popup_anim_out_cb (ClutterAnimation *animation,
                       MwbPopup         *popup)
{
  MwbPopupPrivate *priv = popup->priv;
  clutter_actor_hide (CLUTTER_ACTOR (popup));
  priv->transition_out = FALSE;
}

void
mwb_popup_show (MwbPopup *popup)
{
  MwbPopupPrivate *priv = popup->priv;

  if (!CLUTTER_ACTOR_IS_VISIBLE (popup) || priv->transition_out)
    {
      clutter_actor_show (CLUTTER_ACTOR (popup));

      priv->transition_out = FALSE;
      g_signal_handlers_disconnect_by_func (
        clutter_actor_animate (CLUTTER_ACTOR (popup),
                               CLUTTER_EASE_OUT_BOUNCE,
                               150,
                               "scale-x", 1.0,
                               "scale-y", 1.0,
                               NULL),
        mwb_popup_anim_out_cb,
        popup);
    }
}

void
mwb_popup_hide (MwbPopup *popup)
{
  MwbPopupPrivate *priv = popup->priv;

  if (CLUTTER_ACTOR_IS_VISIBLE (popup) && !priv->transition_out)
    {
      priv->transition_out = TRUE;
      clutter_actor_animate (CLUTTER_ACTOR (popup),
                             CLUTTER_EASE_OUT_SINE,
                             150,
                             "scale-x", 0.0,
                             "scale-y", 0.0,
                             "signal::completed", mwb_popup_anim_out_cb, popup,
                             NULL);
    }
}

gboolean
mwb_popup_is_visible (MwbPopup *popup)
{
  if (CLUTTER_ACTOR_IS_VISIBLE (popup))
    return !popup->priv->transition_out;
  else
    return FALSE;
}

