/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <mhs/mhs.h>
#include <nsCOMArray.h>
#include <nsWeakPtr.h>
#include <nsIWeakReferenceUtils.h>
#include <dbus/dbus-glib.h>
#include "mwb-start-page.h"

NS_IMPL_ISUPPORTS1(MwbFavorite, MwbIFavorite)

MwbFavorite::MwbFavorite (const gchar *a_url, const gchar *a_title)
: url (g_strdup (a_url)),
  title (g_strdup (a_title))
{
}

MwbFavorite::~MwbFavorite ()
{
  g_free (url);
  g_free (title);
}

NS_IMETHODIMP
MwbFavorite::GetUrl (nsACString &a_url)
{
  a_url.Assign (url);

  return NS_OK;
}

NS_IMETHODIMP
MwbFavorite::GetTitle (nsACString &a_title)
{
  a_title.Assign (title);

  return NS_OK;
}

NS_INTERFACE_MAP_BEGIN(MwbStartPage)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, MwbIStartPage)
  NS_INTERFACE_MAP_ENTRY(MwbIStartPage)
NS_INTERFACE_MAP_END

MwbStartPage *MwbStartPage::start_page = nsnull;

MwbStartPage *
MwbStartPage::GetSingleton(void)
{
  if (!start_page)
    start_page = new MwbStartPage ();

  return start_page;
}

MwbStartPage::MwbStartPage ()
{
  history = mhs_history_new ();

  g_signal_connect (history, "favorites-received",
                    G_CALLBACK (StaticFavoritesReceivedCb),
                    this);
  g_signal_connect (history, "link-visited",
                    G_CALLBACK (StaticLinkVisitedCb),
                    this);
  g_signal_connect (history, "pinned-page",
                    G_CALLBACK (StaticPinnedPageCb),
                    this);
  g_signal_connect (history, "unpinned-page",
                    G_CALLBACK (StaticUnpinnedPageCb),
                    this);
}

MwbStartPage::~MwbStartPage ()
{
  g_signal_handlers_disconnect_by_func (history,
                                        (gpointer) StaticFavoritesReceivedCb,
                                        this);
  g_object_unref (history);
}

NS_IMETHODIMP_(nsrefcnt)
MwbStartPage::AddRef ()
{
  return 1;
}

NS_IMETHODIMP_(nsrefcnt)
MwbStartPage::Release ()
{
  return 1;
}

nsresult
MwbStartPage::GetFavorites ()
{
  mhs_history_get_favorites (history);

  return NS_OK;
}

nsresult
MwbStartPage::GetPinnedPages ()
{
  mhs_history_get_pinned_pages (history);

  return NS_OK;
}

nsresult
MwbStartPage::StartPrivateBrowsing ()
{
  DBusGConnection *connection;
  DBusGProxy *proxy;
  GError *error = NULL;
  nsresult rv = NS_OK;

  connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
  if (connection == NULL)
    {
      g_warning ("Failed to connect to session bus: %s", error->message);
      g_error_free (error);
      rv = NS_ERROR_UNEXPECTED;
    }
  else
    {
      proxy = dbus_g_proxy_new_for_name (connection,
                                         "org.moblin.MoblinWebBrowser",
                                         "/org/moblin/MoblinWebBrowser",
                                         "org.moblin.MoblinWebBrowser");

      if (!dbus_g_proxy_call (proxy, "StartPrivateBrowsing", &error,
                              G_TYPE_INVALID, G_TYPE_INVALID))
        {
          g_warning ("Failed to start private browsing: %s", error->message);
          g_error_free (error);
          rv = NS_ERROR_UNEXPECTED;
        }

      g_object_unref (proxy);
    }

  return rv;
}

nsresult
MwbStartPage::AddStartPageObserver (MwbIStartPageObserver *observer)
{
  nsresult rv;

  nsCOMPtr<nsIWeakReference> weak_ptr
    = getter_AddRefs (NS_GetWeakReference (observer, &rv));
  if (NS_FAILED (rv))
    return rv;
  favorites_observers.AppendObject (weak_ptr);

  return NS_OK;
}

nsresult
MwbStartPage::RemoveStartPageObserver (MwbIStartPageObserver *observer)
{
  PRInt32 i;

  for (i = 0; i < favorites_observers.Count (); i++)
    {
      nsWeakPtr weak_ptr = favorites_observers[i];
      nsCOMPtr<MwbIStartPageObserver> strong_ptr = do_QueryReferent (weak_ptr);
      if (strong_ptr == observer)
        {
          favorites_observers.RemoveObjectAt(i);
          break;
        }
    }

  return NS_OK;
}

void
MwbStartPage::FavoritesReceivedCb (gchar **urls, gchar **titles)
{
  int n_favorites = 0;
  gchar **url_p, **title_p;
  MwbIFavorite **favorites;
  int i = 0;

  /* Count the number of favorites */
  for (url_p = urls, title_p = titles; *url_p && *title_p; url_p++, title_p++)
    n_favorites++;

  favorites = (MwbIFavorite **) g_malloc (n_favorites
                                          * sizeof (MwbIFavorite *));

  for (url_p = urls, title_p = titles; *url_p && *title_p; url_p++, title_p++)
    {
      favorites[i] = new MwbFavorite (*url_p, *title_p);
      favorites[i++]->AddRef ();
    }

  for (i = 0; i < favorites_observers.Count ();)
    {
      nsWeakPtr weak_ptr = favorites_observers[i];
      nsCOMPtr<MwbIStartPageObserver> strong_ptr = do_QueryReferent (weak_ptr);
      if (strong_ptr)
        {
          strong_ptr->FavoritesReceived (favorites, n_favorites);
          i++;
        }
      else
        /* Observer has gone away, remove it */
        favorites_observers.RemoveObjectAt (i);
    }

  for (i = 0; i < n_favorites; i++)
    favorites[i]->Release ();

  g_free (favorites);
}

void
MwbStartPage::LinkVisitedCb (const gchar *uri,
                             gint visit_time)
{
  int i;

  for (i = 0; i < favorites_observers.Count ();)
    {
      nsWeakPtr weak_ptr = favorites_observers[i];
      nsCOMPtr<MwbIStartPageObserver> strong_ptr = do_QueryReferent (weak_ptr);
      if (strong_ptr)
        {
          strong_ptr->LinkVisited (nsDependentCString (uri),
                                   visit_time);
          i++;
        }
      else
        /* Observer has gone away, remove it */
        favorites_observers.RemoveObjectAt (i);
    }
}

void
MwbStartPage::PinnedPageCb (const gchar *title, const gchar *uri,
                            gint visit_time)
{
  int i;

  for (i = 0; i < favorites_observers.Count ();)
    {
      nsWeakPtr weak_ptr = favorites_observers[i];
      nsCOMPtr<MwbIStartPageObserver> strong_ptr = do_QueryReferent (weak_ptr);
      if (strong_ptr)
        {
          strong_ptr->PinnedPage (nsDependentCString (title),
                                  nsDependentCString (uri),
                                  visit_time);
          i++;
        }
      else
        /* Observer has gone away, remove it */
        favorites_observers.RemoveObjectAt (i);
    }
}

void
MwbStartPage::UnpinnedPageCb (const gchar *uri)
{
  int i;

  for (i = 0; i < favorites_observers.Count ();)
    {
      nsWeakPtr weak_ptr = favorites_observers[i];
      nsCOMPtr<MwbIStartPageObserver> strong_ptr = do_QueryReferent (weak_ptr);
      if (strong_ptr)
        {
          strong_ptr->UnpinnedPage (nsDependentCString (uri));
          i++;
        }
      else
        /* Observer has gone away, remove it */
        favorites_observers.RemoveObjectAt (i);
    }
}

void
MwbStartPage::StaticFavoritesReceivedCb (MhsHistory *history,
                                         gchar **urls,
                                         gchar **titles,
                                         MwbStartPage *self)
{
  self->FavoritesReceivedCb (urls, titles);
}

void
MwbStartPage::StaticLinkVisitedCb (MhsHistory *history,
                                   const gchar *uri,
                                   gint visit_time,
                                   MwbStartPage *self)
{
  self->LinkVisitedCb (uri, visit_time);
}

void
MwbStartPage::StaticPinnedPageCb (MhsHistory *history,
                                  const gchar *title,
                                  const gchar *uri,
                                  gint visit_time,
                                  MwbStartPage *self)
{
  self->PinnedPageCb (title, uri, visit_time);
}

void
MwbStartPage::StaticUnpinnedPageCb (MhsHistory *history,
                                    const gchar *uri,
                                    MwbStartPage *self)
{
  self->UnpinnedPageCb (uri);
}
