/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mwb-download-manager.h"
#include "mwb-download-model.h"
#include "mwb-download.h"
#include "mwb-utils.h"
#include <moz-headless.h>

G_DEFINE_TYPE (MwbDownloadManager, mwb_download_manager, NBTK_TYPE_WIDGET)

#define DOWNLOAD_MANAGER_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_DOWNLOAD_MANAGER, MwbDownloadManagerPrivate))

enum
{
  DOWNLOAD_ADDED,

  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

struct _MwbDownloadManagerPrivate
{
  NbtkWidget *scroll_view;
  NbtkWidget *list_view;
  GList      *downloads;
};

static void
mwb_download_manager_get_property (GObject *object, guint property_id,
                                   GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_download_manager_set_property (GObject *object, guint property_id,
                                   const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_download_manager_dispose (GObject *object)
{
  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (object)->priv;

  if (priv->scroll_view)
    {
      /* Unparenting the scroll view will destroy everything else */
      clutter_actor_unparent (CLUTTER_ACTOR (priv->scroll_view));
      priv->scroll_view = NULL;
    }

  while (priv->downloads)
    {
      g_object_unref (G_OBJECT (priv->downloads->data));
      priv->downloads = g_list_delete_link (priv->downloads, priv->downloads);
    }

  G_OBJECT_CLASS (mwb_download_manager_parent_class)->dispose (object);
}

static void
mwb_download_manager_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_download_manager_parent_class)->finalize (object);
}

static void
mwb_download_manager_paint (ClutterActor *actor)
{
  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (actor)->priv;

  /* Chain up to get background */
  CLUTTER_ACTOR_CLASS (mwb_download_manager_parent_class)->paint (actor);

  clutter_actor_paint (CLUTTER_ACTOR (priv->scroll_view));
}

static void
mwb_download_manager_pick (ClutterActor *actor, const ClutterColor *color)
{
  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (actor)->priv;

  clutter_actor_paint (CLUTTER_ACTOR (priv->scroll_view));
}

static void
mwb_download_manager_allocate (ClutterActor           *actor,
                               const ClutterActorBox  *box,
                               ClutterAllocationFlags  flags)
{
  NbtkPadding padding;
  ClutterActorBox child_box;

  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_download_manager_parent_class)->
    allocate (actor, box, flags);

  nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);

  child_box.x1 = padding.left;
  child_box.y1 = padding.top;
  child_box.x2 = box->x2 - box->x1 - padding.right;
  child_box.y2 = box->y2 - box->y1 - padding.bottom;

  clutter_actor_allocate (CLUTTER_ACTOR (priv->scroll_view), &child_box, flags);
}

static void
mwb_download_manager_map (ClutterActor *actor)
{
  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (actor)->priv;
  CLUTTER_ACTOR_CLASS (mwb_download_manager_parent_class)->map (actor);
  clutter_actor_map (CLUTTER_ACTOR (priv->scroll_view));
}

static void
mwb_download_manager_unmap (ClutterActor *actor)
{
  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (actor)->priv;
  CLUTTER_ACTOR_CLASS (mwb_download_manager_parent_class)->unmap (actor);
  clutter_actor_unmap (CLUTTER_ACTOR (priv->scroll_view));
}

static void
mwb_download_manager_class_init (MwbDownloadManagerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbDownloadManagerPrivate));

  object_class->get_property = mwb_download_manager_get_property;
  object_class->set_property = mwb_download_manager_set_property;
  object_class->dispose = mwb_download_manager_dispose;
  object_class->finalize = mwb_download_manager_finalize;

  actor_class->paint = mwb_download_manager_paint;
  actor_class->pick = mwb_download_manager_pick;
  actor_class->allocate = mwb_download_manager_allocate;
  actor_class->map = mwb_download_manager_map;
  actor_class->unmap = mwb_download_manager_unmap;

  signals[DOWNLOAD_ADDED] =
    g_signal_new ("download-added",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbDownloadManagerClass, download_added),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);
}

static void
mwb_download_manager_init (MwbDownloadManager *self)
{
  ClutterModel *model;
  MwbDownloadManagerPrivate *priv = self->priv = DOWNLOAD_MANAGER_PRIVATE (self);

  priv->scroll_view = nbtk_scroll_view_new ();
  priv->list_view = nbtk_list_view_new ();

  model = mwb_download_model_new ();
  nbtk_list_view_set_model (NBTK_LIST_VIEW (priv->list_view), model);
  g_object_unref (G_OBJECT (model));

  nbtk_list_view_set_item_type (NBTK_LIST_VIEW (priv->list_view),
                                MWB_TYPE_DOWNLOAD);

  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "download",
                                MWB_COL_DOWNLOAD);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "source",
                                MWB_COL_SRC);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "destination",
                                MWB_COL_DEST);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "progress",
                                MWB_COL_PROGRESS);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "max-progress",
                                MWB_COL_MAX_PROGRESS);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "completed",
                                MWB_COL_COMPLETED);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "cancelled",
                                MWB_COL_CANCELLED);

  clutter_actor_set_parent (CLUTTER_ACTOR (priv->scroll_view),
                            CLUTTER_ACTOR (self));

  clutter_container_add_actor (CLUTTER_CONTAINER (priv->scroll_view),
                               CLUTTER_ACTOR (priv->list_view));
}

NbtkWidget *
mwb_download_manager_new (void)
{
  return g_object_new (MWB_TYPE_DOWNLOAD_MANAGER, NULL);
}

void
mwb_download_manager_add (MwbDownloadManager      *dlman,
                          ClutterMozEmbedDownload *download)
{
  MwbDownloadManagerPrivate *priv = dlman->priv;
  ClutterModel *model =
    nbtk_list_view_get_model (NBTK_LIST_VIEW (priv->list_view));

  mwb_download_model_add (MWB_DOWNLOAD_MODEL (model), download);
  g_signal_emit (dlman, signals[DOWNLOAD_ADDED], 0);
}

MwbDownloadModel *
mwb_download_manager_get_model (MwbDownloadManager *dlman)
{
  return MWB_DOWNLOAD_MODEL (
    nbtk_list_view_get_model (NBTK_LIST_VIEW (dlman->priv->list_view)));
}

