/**************************************************************************
 * 
 * Copyright 2007 Tungsten Graphics, Inc., Cedar Park, Texas.
 * All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL TUNGSTEN GRAPHICS AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 **************************************************************************/

/* Author:
 *    Keith Whitwell <keith@tungstengraphics.com>
 */


#include "pipe/p_defines.h"
#include "draw/draw_context.h"
#include "sp_flush.h"
#include "sp_context.h"
#include "sp_state.h"
#include "sp_tile_cache.h"
#include "sp_tex_tile_cache.h"


void
softpipe_flush( struct pipe_context *pipe,
		unsigned flags,
                struct pipe_fence_handle **fence )
{
   struct softpipe_context *softpipe = softpipe_context(pipe);
   uint i;

   draw_flush(softpipe->draw);

   if (flags & PIPE_FLUSH_TEXTURE_CACHE) {
      for (i = 0; i < softpipe->num_sampler_views; i++) {
         sp_flush_tex_tile_cache(softpipe->tex_cache[i]);
      }
      for (i = 0; i < softpipe->num_vertex_sampler_views; i++) {
         sp_flush_tex_tile_cache(softpipe->vertex_tex_cache[i]);
      }
   }

   if (flags & PIPE_FLUSH_SWAPBUFFERS) {
      /* If this is a swapbuffers, just flush color buffers.
       *
       * The zbuffer changes are not discarded, but held in the cache
       * in the hope that a later clear will wipe them out.
       */
      for (i = 0; i < softpipe->framebuffer.nr_cbufs; i++)
         if (softpipe->cbuf_cache[i])
            sp_flush_tile_cache(softpipe->cbuf_cache[i]);

      /* Need this call for hardware buffers before swapbuffers.
       *
       * there should probably be another/different flush-type function
       * that's called before swapbuffers because we don't always want
       * to unmap surfaces when flushing.
       */
      softpipe_unmap_transfers(softpipe);
   }
   else if (flags & PIPE_FLUSH_RENDER_CACHE) {
      for (i = 0; i < softpipe->framebuffer.nr_cbufs; i++)
         if (softpipe->cbuf_cache[i])
            sp_flush_tile_cache(softpipe->cbuf_cache[i]);

      if (softpipe->zsbuf_cache)
         sp_flush_tile_cache(softpipe->zsbuf_cache);
     
      softpipe->dirty_render_cache = FALSE;
   }

   /* Enable to dump BMPs of the color/depth buffers each frame */
#if 0
   if(flags & PIPE_FLUSH_FRAME) {
      static unsigned frame_no = 1;
      static char filename[256];
      util_snprintf(filename, sizeof(filename), "cbuf_%u.bmp", frame_no);
      debug_dump_surface_bmp(softpipe, filename, softpipe->framebuffer.cbufs[0]);
      util_snprintf(filename, sizeof(filename), "zsbuf_%u.bmp", frame_no);
      debug_dump_surface_bmp(softpipe, filename, softpipe->framebuffer.zsbuf);
      ++frame_no;
   }
#endif
   
   if (fence)
      *fence = NULL;
}


/**
 * Flush context if necessary.
 *
 * Returns FALSE if it would have block, but do_not_block was set, TRUE
 * otherwise.
 *
 * TODO: move this logic to an auxiliary library?
 */
boolean
softpipe_flush_resource(struct pipe_context *pipe,
                        struct pipe_resource *texture,
                        unsigned face,
                        unsigned level,
                        unsigned flush_flags,
                        boolean read_only,
                        boolean cpu_access,
                        boolean do_not_block)
{
   unsigned referenced;

   referenced = pipe->is_resource_referenced(pipe, texture, face, level);

   if ((referenced & PIPE_REFERENCED_FOR_WRITE) ||
       ((referenced & PIPE_REFERENCED_FOR_READ) && !read_only)) {

      /*
       * TODO: The semantics of these flush flags are too obtuse. They should
       * disappear and the pipe driver should just ensure that all visible
       * side-effects happen when they need to happen.
       */
      if (referenced & PIPE_REFERENCED_FOR_WRITE)
         flush_flags |= PIPE_FLUSH_RENDER_CACHE;

      if (referenced & PIPE_REFERENCED_FOR_READ)
         flush_flags |= PIPE_FLUSH_TEXTURE_CACHE;

      if (cpu_access) {
         /*
          * Flush and wait.
          */

         struct pipe_fence_handle *fence = NULL;

         if (do_not_block)
            return FALSE;

         pipe->flush(pipe, flush_flags, &fence);

         if (fence) {
            /*
             * This is for illustrative purposes only, as softpipe does not
             * have fences.
             */
            pipe->screen->fence_finish(pipe->screen, fence, 0);
            pipe->screen->fence_reference(pipe->screen, &fence, NULL);
         }
      } else {
         /*
          * Just flush.
          */

         pipe->flush(pipe, flush_flags, NULL);
      }
   }

   return TRUE;
}
