/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */

#include <glib.h>
#include <glib-object.h>
#include "zeitgeist-data-source.h"
#include "zeitgeist-event.h"
#include "zeitgeist-subject.h"
#include "zeitgeist-eggdbusconversions.h"
#include "zeitgeist-timestamp.h"

typedef struct
{
  
} Fixture;

static void setup    (Fixture *fix, gconstpointer data);
static void teardown (Fixture *fix, gconstpointer data);

static void
setup (Fixture *fix, gconstpointer data)
{
  
}

static void
teardown (Fixture *fix, gconstpointer data)
{
  
}

static void
test_z_to_egg_two_subjects (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent     *zev;
  ZeitgeistSubject  *zsu;
  EggZeitgeistEvent  *eggev;
  gchar             **ev_data, **su_data;
  EggDBusArraySeq    *subjects_data;
  
  zev = zeitgeist_event_new ();
  zeitgeist_event_set_timestamp (zev, 68);
  zeitgeist_event_set_interpretation (zev, "foo://interp");
  zeitgeist_event_set_manifestation (zev, "foo://manif");
  zeitgeist_event_set_actor (zev, "app://firefox.desktop");

  zsu = g_object_new (ZEITGEIST_TYPE_SUBJECT, NULL);
  zeitgeist_subject_set_uri (zsu, "file:///tmp/foo.txt");
  zeitgeist_subject_set_interpretation (zsu, "foo://TextDoc");
  zeitgeist_subject_set_manifestation (zsu, "foo://File");
  zeitgeist_subject_set_mimetype (zsu, "text/plain");
  zeitgeist_subject_set_origin (zsu, "file:///tmp");
  zeitgeist_subject_set_text (zsu, "foo.txt");
  zeitgeist_subject_set_storage (zsu, "51e63f15-0672-49f9-9cae-87786460bc97");
  zeitgeist_event_add_subject (zev, zsu);

  zsu = g_object_new (ZEITGEIST_TYPE_SUBJECT, NULL);
  zeitgeist_subject_set_uri (zsu, "file:///tmp/foo.txt2");
  zeitgeist_subject_set_interpretation (zsu, "foo://TextDoc2");
  zeitgeist_subject_set_manifestation (zsu, "foo://File2");
  zeitgeist_subject_set_mimetype (zsu, "text/plain2");
  zeitgeist_subject_set_origin (zsu, "file:///tmp2");
  zeitgeist_subject_set_text (zsu, "foo.txt2");
  zeitgeist_subject_set_storage (zsu, "51e63f15-0672-49f9-9cae-87786460bc972");
  zeitgeist_event_add_subject (zev, zsu);
  
  eggev = _zeitgeist_event_to_egg_zeitgeist_event (zev);

  /* We unref the zev here to assert that we have completely separated
   * the memory of the converted event */
  g_object_unref (zev);
  
  ev_data = egg_zeitgeist_event_get_event_data (eggev);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID]);
  g_assert_cmpstr ("68", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP]);
  g_assert_cmpstr ("foo://interp", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION]);
  g_assert_cmpstr ("foo://manif", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION]);
  g_assert_cmpstr ("app://firefox.desktop", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR]);

  subjects_data = egg_zeitgeist_event_get_subjects (eggev);
  g_assert_cmpint (2, ==, egg_dbus_array_seq_get_size (subjects_data));  

  su_data = egg_dbus_array_seq_get (subjects_data, 0);
  g_assert_cmpint (EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE + 1,
                   ==, g_strv_length (su_data));
  g_assert_cmpstr ("file:///tmp/foo.txt", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_URI]);
  g_assert_cmpstr ("foo://TextDoc", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_INTERPRETATION]);
  g_assert_cmpstr ("foo://File", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MANIFESTATION]);
  g_assert_cmpstr ("text/plain", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MIMETYPE]);
  g_assert_cmpstr ("file:///tmp", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_ORIGIN]);
  g_assert_cmpstr ("foo.txt", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_TEXT]);
  g_assert_cmpstr ("51e63f15-0672-49f9-9cae-87786460bc97", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE]);

  su_data = egg_dbus_array_seq_get (subjects_data, 1);
  g_assert_cmpint (EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE + 1,
                   ==, g_strv_length (su_data));
  g_assert_cmpstr ("file:///tmp/foo.txt2", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_URI]);
  g_assert_cmpstr ("foo://TextDoc2", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_INTERPRETATION]);
  g_assert_cmpstr ("foo://File2", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MANIFESTATION]);
  g_assert_cmpstr ("text/plain2", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MIMETYPE]);
  g_assert_cmpstr ("file:///tmp2", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_ORIGIN]);
  g_assert_cmpstr ("foo.txt2", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_TEXT]);
  g_assert_cmpstr ("51e63f15-0672-49f9-9cae-87786460bc972", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE]);
}

static void
test_z_to_egg_one_subject (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent     *zev;
  ZeitgeistSubject  *zsu;
  EggZeitgeistEvent  *eggev;
  gchar             **ev_data, **su_data;
  EggDBusArraySeq    *subjects_data;
  
  zev = zeitgeist_event_new ();
  zeitgeist_event_set_timestamp (zev, 68);
  zeitgeist_event_set_interpretation (zev, "foo://interp");
  zeitgeist_event_set_manifestation (zev, "foo://manif");
  zeitgeist_event_set_actor (zev, "app://firefox.desktop");

  zsu = g_object_new (ZEITGEIST_TYPE_SUBJECT, NULL);
  zeitgeist_subject_set_uri (zsu, "file:///tmp/foo.txt");
  zeitgeist_subject_set_interpretation (zsu, "foo://TextDoc");
  zeitgeist_subject_set_manifestation (zsu, "foo://File");
  zeitgeist_subject_set_mimetype (zsu, "text/plain");
  zeitgeist_subject_set_origin (zsu, "file:///tmp");
  zeitgeist_subject_set_text (zsu, "foo.txt");
  zeitgeist_subject_set_storage (zsu, "51e63f15-0672-49f9-9cae-87786460bc97");
  zeitgeist_event_add_subject (zev, zsu);
  
  eggev = _zeitgeist_event_to_egg_zeitgeist_event (zev);

  /* We unref the zev here to assert that we have completely separated
   * the memory of the converted event */
  g_object_unref (zev);
  
  ev_data = egg_zeitgeist_event_get_event_data (eggev);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID]);
  g_assert_cmpstr ("68", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP]);
  g_assert_cmpstr ("foo://interp", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION]);
  g_assert_cmpstr ("foo://manif", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION]);
  g_assert_cmpstr ("app://firefox.desktop", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR]);

  subjects_data = egg_zeitgeist_event_get_subjects (eggev);
  g_assert_cmpint (1, ==, egg_dbus_array_seq_get_size (subjects_data));  
  su_data = egg_dbus_array_seq_get (subjects_data, 0);
  g_assert_cmpint (EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE + 1,
                   ==, g_strv_length (su_data));
  g_assert_cmpstr ("file:///tmp/foo.txt", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_URI]);
  g_assert_cmpstr ("foo://TextDoc", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_INTERPRETATION]);
  g_assert_cmpstr ("foo://File", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MANIFESTATION]);
  g_assert_cmpstr ("text/plain", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MIMETYPE]);
  g_assert_cmpstr ("file:///tmp", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_ORIGIN]);
  g_assert_cmpstr ("foo.txt", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_TEXT]);
  g_assert_cmpstr ("51e63f15-0672-49f9-9cae-87786460bc97", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE]);
}

static void
test_z_to_egg_empty_subject (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent     *zev;
  ZeitgeistSubject  *zsu;
  EggZeitgeistEvent  *eggev;
  gchar             **ev_data, **su_data;
  EggDBusArraySeq    *subjects_data;
  
  zev = zeitgeist_event_new ();

  zsu = g_object_new (ZEITGEIST_TYPE_SUBJECT, NULL);
  zeitgeist_event_add_subject (zev, zsu);
  
  eggev = _zeitgeist_event_to_egg_zeitgeist_event (zev);

  /* We unref the zev here to assert that we have completely separated
   * the memory of the converted event */
  g_object_unref (zev);
  
  ev_data = egg_zeitgeist_event_get_event_data (eggev);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID]);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP]);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION]);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION]);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR]);

  subjects_data = egg_zeitgeist_event_get_subjects (eggev);
  g_assert_cmpint (1, ==, egg_dbus_array_seq_get_size (subjects_data));  
  su_data = egg_dbus_array_seq_get (subjects_data, 0);
  g_assert_cmpint (EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE + 1,
                   ==, g_strv_length (su_data));
  g_assert_cmpstr ("", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_URI]);
  g_assert_cmpstr ("", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_INTERPRETATION]);
  g_assert_cmpstr ("", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MANIFESTATION]);
  g_assert_cmpstr ("", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MIMETYPE]);
  g_assert_cmpstr ("", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_ORIGIN]);
  g_assert_cmpstr ("", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_TEXT]);
  g_assert_cmpstr ("", ==, su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE]);
}

static void
test_z_to_egg_no_subjects (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent     *zev;
  EggZeitgeistEvent  *eggev;
  gchar             **ev_data;
  EggDBusArraySeq    *subjects_data, *payload_data;
  
  zev = zeitgeist_event_new ();
  zeitgeist_event_set_id (zev, 27);
  zeitgeist_event_set_timestamp (zev, 68);
  zeitgeist_event_set_interpretation (zev, "foo://interp");
  zeitgeist_event_set_manifestation (zev, "foo://manif");
  zeitgeist_event_set_actor (zev, "app://firefox.desktop");

  eggev = _zeitgeist_event_to_egg_zeitgeist_event (zev);

  /* We unref the zev here to assert that we have completely separated
   * the memory of the converted event */
  g_object_unref (zev);
  
  ev_data = egg_zeitgeist_event_get_event_data (eggev);
  g_assert_cmpstr ("27", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID]);
  g_assert_cmpstr ("68", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP]);
  g_assert_cmpstr ("foo://interp", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION]);
  g_assert_cmpstr ("foo://manif", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION]);
  g_assert_cmpstr ("app://firefox.desktop", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR]);

  subjects_data = egg_zeitgeist_event_get_subjects (eggev);
  g_assert_cmpint (0, ==, egg_dbus_array_seq_get_size (subjects_data));
  g_assert_cmpint (G_TYPE_STRV, ==, egg_dbus_array_seq_get_element_type(subjects_data));

  payload_data = egg_zeitgeist_event_get_payload (eggev);
  g_assert_cmpint (0, ==, egg_dbus_array_seq_get_size (payload_data));
  g_assert_cmpint (G_TYPE_UCHAR, ==, egg_dbus_array_seq_get_element_type(payload_data));
}

static void
test_z_to_egg_empty (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent     *zev;
  EggZeitgeistEvent  *eggev;
  gchar             **ev_data;
  EggDBusArraySeq    *subjects_data, *payload_data;
  
  zev = zeitgeist_event_new ();
  eggev = _zeitgeist_event_to_egg_zeitgeist_event (zev);

  /* We unref the zev here to assert that we have completely separated
   * the memory of the converted event */
  g_object_unref (zev);
  
  ev_data = egg_zeitgeist_event_get_event_data (eggev);
  g_assert_cmpint (EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR + 1, ==, g_strv_length (ev_data));
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID]);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP]);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION]);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION]);
  g_assert_cmpstr ("", ==, ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR]);

  subjects_data = egg_zeitgeist_event_get_subjects (eggev);
  g_assert_cmpint (0, ==, egg_dbus_array_seq_get_size (subjects_data));
  g_assert_cmpint (G_TYPE_STRV, ==, egg_dbus_array_seq_get_element_type(subjects_data));

  payload_data = egg_zeitgeist_event_get_payload (eggev);
  g_assert_cmpint (0, ==, egg_dbus_array_seq_get_size (payload_data));
  g_assert_cmpint (G_TYPE_UCHAR, ==, egg_dbus_array_seq_get_element_type(payload_data));
}

static void
test_egg_to_z_one_subject (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent     *zev;
  ZeitgeistSubject   *zsu;
  EggZeitgeistEvent  *eggev;
  gchar             **ev_data, **su_data;
  EggDBusArraySeq    *subjects_data, *payload_data;

  ev_data = g_new0 (gchar*, EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR + 2);
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID] = "27";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP] = "68";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION] = "foo://Interp";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION] = "foo://Manif";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR] = "app://firefox.desktop";

  su_data = g_new0 (gchar*, EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE + 2);
  su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_URI] = "file:///tmp/foo.txt";
  su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_INTERPRETATION] = "foo://TextDoc";
  su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MANIFESTATION] = "foo://File";
  su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_MIMETYPE] = "text/plain";
  su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_ORIGIN] = "file:///tmp";
  su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_TEXT] = "foo.txt";
  su_data[EGG_ZEITGEIST_SUBJECT_DATA_OFFSET_STORAGE] = "51e63f15-0672-49f9-9cae-87786460bc97";
  
    
  subjects_data = egg_dbus_array_seq_new (G_TYPE_STRV, NULL, NULL, NULL);
  egg_dbus_array_seq_add (subjects_data, su_data);
  payload_data = egg_dbus_array_seq_new (G_TYPE_UCHAR, NULL, NULL, NULL);
  
  eggev = egg_zeitgeist_event_new (ev_data, subjects_data, payload_data);
  zev = _egg_zeitgeist_event_to_zeitgeist_event (eggev);
  
  /* We unref the zev here to assert that we have completely separated
   * the memory of the converted event */
  g_object_unref (eggev);
  g_free (ev_data);
  g_free (su_data);
  
  g_assert_cmpint (27, ==, zeitgeist_event_get_id (zev));
  g_assert_cmpint (68, ==, zeitgeist_event_get_timestamp (zev));
  g_assert_cmpstr ("foo://Interp", ==, zeitgeist_event_get_interpretation (zev));
  g_assert_cmpstr ("foo://Manif", ==, zeitgeist_event_get_manifestation (zev));
  g_assert_cmpstr ("app://firefox.desktop", ==, zeitgeist_event_get_actor (zev));

  g_assert_cmpint (1, ==, zeitgeist_event_num_subjects (zev));
  zsu = zeitgeist_event_get_subject (zev, 0);
  g_assert_cmpstr ("file:///tmp/foo.txt", ==, zeitgeist_subject_get_uri (zsu));
  g_assert_cmpstr ("foo://TextDoc", ==, zeitgeist_subject_get_interpretation (zsu));
  g_assert_cmpstr ("foo://File", ==, zeitgeist_subject_get_manifestation (zsu));
  g_assert_cmpstr ("text/plain", ==, zeitgeist_subject_get_mimetype (zsu));
  g_assert_cmpstr ("file:///tmp", ==, zeitgeist_subject_get_origin (zsu));
  g_assert_cmpstr ("foo.txt", ==, zeitgeist_subject_get_text (zsu));
  g_assert_cmpstr ("51e63f15-0672-49f9-9cae-87786460bc97", ==, zeitgeist_subject_get_storage (zsu));

  g_assert (zeitgeist_event_get_payload (zev) == NULL);
}

static void
test_egg_to_z_no_subjects (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent     *zev;
  EggZeitgeistEvent  *eggev;
  gchar             **ev_data;
  EggDBusArraySeq    *subjects_data, *payload_data;

  ev_data = g_new0 (gchar*, EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR + 2);
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID] = "27";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP] = "68";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION] = "foo://Interp";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION] = "foo://Manif";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR] = "app://firefox.desktop";

  subjects_data = egg_dbus_array_seq_new (G_TYPE_STRV, NULL, NULL, NULL);
  payload_data = egg_dbus_array_seq_new (G_TYPE_UCHAR, NULL, NULL, NULL);
  
  eggev = egg_zeitgeist_event_new (ev_data, subjects_data, payload_data);
  zev = _egg_zeitgeist_event_to_zeitgeist_event (eggev);
  
  /* We unref the zev here to assert that we have completely separated
   * the memory of the converted event */
  g_object_unref (eggev);
  g_free (ev_data);
  
  g_assert_cmpint (27, ==, zeitgeist_event_get_id (zev));
  g_assert_cmpint (68, ==, zeitgeist_event_get_timestamp (zev));
  g_assert_cmpstr ("foo://Interp", ==, zeitgeist_event_get_interpretation (zev));
  g_assert_cmpstr ("foo://Manif", ==, zeitgeist_event_get_manifestation (zev));
  g_assert_cmpstr ("app://firefox.desktop", ==, zeitgeist_event_get_actor (zev));

  g_assert_cmpint (0, ==, zeitgeist_event_num_subjects (zev));
  g_assert (zeitgeist_event_get_payload (zev) == NULL);
}

static void
test_egg_to_z_empty (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent     *zev;
  EggZeitgeistEvent  *eggev;
  gchar             **ev_data;
  EggDBusArraySeq    *subjects_data, *payload_data;

  ev_data = g_new0 (gchar*, EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR + 2);
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID] = "";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP] = "";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION] = "";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION] = "";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR] = "";

  subjects_data = egg_dbus_array_seq_new (G_TYPE_STRV, NULL, NULL, NULL);
  payload_data = egg_dbus_array_seq_new (G_TYPE_UCHAR, NULL, NULL, NULL);
  
  eggev = egg_zeitgeist_event_new (ev_data, subjects_data, payload_data);
  zev = _egg_zeitgeist_event_to_zeitgeist_event (eggev);
  
  /* We unref the zev here to assert that we have completely separated
   * the memory of the converted event */
  g_object_unref (eggev);
  g_free (ev_data);
  
  g_assert_cmpint (0, ==, zeitgeist_event_get_id (zev));
  g_assert_cmpint (0, ==, zeitgeist_event_get_timestamp (zev));
  g_assert_cmpstr ("", ==, zeitgeist_event_get_interpretation (zev));
  g_assert_cmpstr ("", ==, zeitgeist_event_get_manifestation (zev));
  g_assert_cmpstr ("", ==, zeitgeist_event_get_actor (zev));

  g_assert_cmpint (0, ==, zeitgeist_event_num_subjects (zev));
  g_assert (zeitgeist_event_get_payload (zev) == NULL);
}

static void
test_egg_to_z_ds_empty (Fixture *fix, gconstpointer data)
{
  ZeitgeistDataSource    *zds;
  EggZeitgeistDataSource *eggds;
  EggDBusArraySeq        *templates;

  templates = egg_dbus_array_seq_new (EGG_ZEITGEIST_TYPE_DATA_SOURCE,
                                      g_object_unref,
                                      NULL,
                                      NULL);

  eggds = egg_zeitgeist_data_source_new ("", "", "", templates,
                                         FALSE, 0, FALSE);

  zds = _egg_zeitgeist_data_source_to_zeitgeist_data_source (eggds);

  /* We unref the eggds here to assert that we have completely separated
   * the memory of the converted data source */
  g_object_unref (eggds);
  g_object_unref (templates);

  g_assert_cmpstr ("", ==, zeitgeist_data_source_get_unique_id (zds));
  g_assert_cmpstr ("", ==, zeitgeist_data_source_get_name (zds));
  g_assert_cmpstr ("", ==, zeitgeist_data_source_get_description (zds));
  g_assert_cmpint (0, ==, zeitgeist_data_source_is_running (zds));
  g_assert_cmpint (0, ==, zeitgeist_data_source_is_enabled (zds));
  g_assert_cmpint (0, ==, zeitgeist_data_source_get_timestamp (zds));

  g_object_unref (zds);
}

static void
test_egg_to_z_ds_full (Fixture *fix, gconstpointer data)
{
  ZeitgeistDataSource    *zds;
  EggZeitgeistDataSource *eggds;
  EggDBusArraySeq        *templates;
  gint64                  timestamp;

  templates = egg_dbus_array_seq_new (EGG_ZEITGEIST_TYPE_DATA_SOURCE,
                                      g_object_unref,
                                      NULL,
                                      NULL);
  timestamp = zeitgeist_timestamp_for_now ();

  eggds = egg_zeitgeist_data_source_new ("id", "name", "desc", templates,
                                         FALSE,
                                         timestamp,
                                         TRUE);

  zds = _egg_zeitgeist_data_source_to_zeitgeist_data_source (eggds);

  /* We unref the eggds here to assert that we have completely separated
   * the memory of the converted data source */
  g_object_unref (eggds);
  g_object_unref (templates);

  g_assert_cmpstr ("id", ==, zeitgeist_data_source_get_unique_id (zds));
  g_assert_cmpstr ("name", ==, zeitgeist_data_source_get_name (zds));
  g_assert_cmpstr ("desc", ==, zeitgeist_data_source_get_description (zds));
  g_assert_cmpint (0, ==, zeitgeist_data_source_is_running (zds));
  g_assert_cmpint (0, !=, zeitgeist_data_source_is_enabled (zds));
  g_assert_cmpint (timestamp, ==, zeitgeist_data_source_get_timestamp (zds));

  g_object_unref (zds);
}

static void
test_timestamp (Fixture *fix, gconstpointer data)
{
  ZeitgeistEvent     *zev;
  EggZeitgeistEvent  *eggev;
  gchar             **ev_data;
  EggDBusArraySeq    *subjects_data, *payload_data;

  ev_data = g_new0 (gchar*, EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR + 2);
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ID] = "";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_TIMESTAMP] = "1271316978000";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_INTERPRETATION] = "";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_MANIFESTATION] = "";
  ev_data[EGG_ZEITGEIST_EVENT_DATA_OFFSET_ACTOR] = "";

  subjects_data = egg_dbus_array_seq_new (G_TYPE_STRV, NULL, NULL, NULL);
  payload_data = egg_dbus_array_seq_new (G_TYPE_UCHAR, NULL, NULL, NULL);
  
  eggev = egg_zeitgeist_event_new (ev_data, subjects_data, payload_data);
  zev = _egg_zeitgeist_event_to_zeitgeist_event (eggev);
  
  /* We unref the zev here to assert that we have completely separated
   * the memory of the converted event */
  g_object_unref (eggev);
  g_free (ev_data);
  
  g_assert_cmpint (1271316978000, ==, zeitgeist_event_get_timestamp (zev));
}

int
main (int   argc,
      char *argv[])
{
  g_type_init ();
  g_test_init (&argc, &argv, NULL);

  g_test_add ("/Zeitgeist/EggDBusConversions/ZToEggTwoSubjects", Fixture, NULL,
              setup, test_z_to_egg_two_subjects, teardown);
  
  g_test_add ("/Zeitgeist/EggDBusConversions/ZToEggOneSubject", Fixture, NULL,
              setup, test_z_to_egg_one_subject, teardown);

  g_test_add ("/Zeitgeist/EggDBusConversions/ZToEggEmptySubject", Fixture, NULL,
              setup, test_z_to_egg_empty_subject, teardown);
  
  g_test_add ("/Zeitgeist/EggDBusConversions/ZToEggNoSubjects", Fixture, NULL,
              setup, test_z_to_egg_no_subjects, teardown);

  g_test_add ("/Zeitgeist/EggDBusConversions/ZToEggEmpty", Fixture, NULL,
              setup, test_z_to_egg_empty, teardown);

  g_test_add ("/Zeitgeist/EggDBusConversions/EggToZOneSubject", Fixture, NULL,
              setup, test_egg_to_z_one_subject, teardown);
  
  g_test_add ("/Zeitgeist/EggDBusConversions/EggToZNoSubjects", Fixture, NULL,
              setup, test_egg_to_z_no_subjects, teardown);
  
  g_test_add ("/Zeitgeist/EggDBusConversions/EggToZEmpty", Fixture, NULL,
              setup, test_egg_to_z_empty, teardown);

  g_test_add ("/Zeitgeist/EggDBusConversions/EggToZDSFull", Fixture, NULL,
              setup, test_egg_to_z_ds_full, teardown);

  g_test_add ("/Zeitgeist/EggDBusConversions/EggToZDSEmpty", Fixture, NULL,
              setup, test_egg_to_z_ds_empty, teardown);

  g_test_add ("/Zeitgeist/EggDBusConversions/Timestamp", Fixture, NULL,
              setup, test_timestamp, teardown);
  
  return g_test_run();
}
