/*
 * Copyright (C) 2010 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by: Michal Hruby <michal.mhr@gmail.com>
 */

/**
 * SECTION:zeitgeist-data-source
 * @short_description: Abstracts data sources used by
 *                     the #ZeitgeistDataSourceRegistry extension
 * @include: zeitgeist.h
 *
 * #ZeitgeistDataSource represents a data source used to insert events into
 * Zeitgeist. The data sources are identified using the unique_id property,
 * and when querying the data source registry you get other interesting
 * information like timestamp of the last action of the data source, flag
 * whether it is currently running etc.
 */
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "zeitgeist-data-source.h"

G_DEFINE_TYPE (ZeitgeistDataSource, zeitgeist_data_source, G_TYPE_INITIALLY_UNOWNED);
#define ZEITGEIST_DATA_SOURCE_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE(obj, ZEITGEIST_TYPE_DATA_SOURCE, ZeitgeistDataSourcePrivate))

typedef struct
{
  gchar *unique_id;
  gchar *name;
  gchar *description;
  GPtrArray *templates;
  gboolean running;
  gint64 timestamp;
  gboolean enabled;
} ZeitgeistDataSourcePrivate;

/**
 * zeitgeist_data_source_new:
 * 
 * Create a new empty data source structure.
 *
 * Returns: A new instance of #ZeitgeistDataSource. The returned src will
 *          have a floating reference which will be consumed if you pass the
 *          data source to any of the methods provided by this library (like
 *          registering the data source). If you do not do that then you
 *          must free the data source youself with g_object_unref().
 */
ZeitgeistDataSource*
zeitgeist_data_source_new (void)
{
  return g_object_new (ZEITGEIST_TYPE_DATA_SOURCE, NULL);
}
/**
 * zeitgeist_data_source_new_full:
 * @id: Unique ID for the data source.
 * @name: Name of the data source (may be translated).
 * @desc: Data source description.
 * @event_templates: A #GPtrArray of #ZeitgeistEvent<!-- -->s. This parameter
 *                   is optional and purely informational to let data-source
 *                   management applications and other data-sources know what
 *                   sort of information the data source logs.
 *                   The data source will assume ownership of the events and
 *                   the pointer array.
 *                   If you want to keep a reference for yourself you must do a
 *                   g_ptr_array_ref() on @event_templates as well as reffing
 *                   the events held by it before calling this method.
 *
 * Creates a new instance of DataSource with the given ID, name and
 * description.
 *
 * Returns: New instance of #ZeitgeistDataSource with floating reference,
 *          if you do not pass the instance to method like 
 *          zeitgeist_data_source_registry_register_data_source(), then you 
 *          have to free the data source yourself using g_object_unref().
 */
ZeitgeistDataSource*
zeitgeist_data_source_new_full (const gchar *id,
                                const gchar *name,
                                const gchar *desc,
                                GPtrArray   *templates)
{
  ZeitgeistDataSource* src = zeitgeist_data_source_new ();

  zeitgeist_data_source_set_unique_id (src, id);
  zeitgeist_data_source_set_name (src, name);
  zeitgeist_data_source_set_description (src, desc);
  zeitgeist_data_source_set_event_templates (src, templates);

  return src;
}

const gchar*
zeitgeist_data_source_get_unique_id (ZeitgeistDataSource *src)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_val_if_fail (ZEITGEIST_IS_DATA_SOURCE (src), NULL);

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  return priv->unique_id;
}

void
zeitgeist_data_source_set_unique_id (ZeitgeistDataSource *src,
                                     const gchar* unique_id)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE (src));

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  if (priv->unique_id)
    {
      g_free (priv->unique_id);
    }
  
  priv->unique_id = g_strdup (unique_id);
}

const gchar*
zeitgeist_data_source_get_name (ZeitgeistDataSource *src)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_val_if_fail (ZEITGEIST_IS_DATA_SOURCE (src), NULL);

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  return priv->name;
}

void
zeitgeist_data_source_set_name (ZeitgeistDataSource *src,
                                const gchar         *name)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE (src));

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  if (priv->name)
    {
      g_free (priv->name);
    }
  
  priv->name = g_strdup (name);
}

const gchar*
zeitgeist_data_source_get_description (ZeitgeistDataSource *src)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_val_if_fail (ZEITGEIST_IS_DATA_SOURCE (src), NULL);

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  return priv->description;
}

void
zeitgeist_data_source_set_description (ZeitgeistDataSource *src,
                                       const gchar *description)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE (src));

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  if (priv->description)
    {
      g_free (priv->description);
    }

  priv->description = g_strdup (description);
}

GPtrArray*
zeitgeist_data_source_get_event_templates (ZeitgeistDataSource *src)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_val_if_fail (ZEITGEIST_IS_DATA_SOURCE (src), NULL);

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  return priv->templates;
}

/**
 * zeitgeist_data_source_set_event_templates:
 * @src: Instance of #ZeitgeistDataSource.
 * @event_templates: A #GPtrArray which contains elements of type
 *             #ZeitgeistEvent.
 *             The data source will assume ownership of the events and
 *             the pointer array.
 *             If you want to keep a reference for yourself you must do a
 *             g_ptr_array_ref() on @templates as well as reffing
 *             the events held by it before calling this method.
 *
 * Sets event templates which are logged by this #ZeitgeistDataSource
 * instance.
 */
void
zeitgeist_data_source_set_event_templates (ZeitgeistDataSource *src,
                                           GPtrArray *event_templates)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE (src));

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  if (priv->templates)
    {
      g_ptr_array_unref (priv->templates);
    }

  if (event_templates)
    {
      g_ptr_array_foreach (event_templates, (GFunc) g_object_ref_sink, NULL);
      g_ptr_array_set_free_func (event_templates, g_object_unref);
    }

  priv->templates = event_templates;
}

gboolean
zeitgeist_data_source_is_running (ZeitgeistDataSource *src)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_val_if_fail (ZEITGEIST_IS_DATA_SOURCE (src), FALSE);

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  return priv->running;
}

void
zeitgeist_data_source_set_running (ZeitgeistDataSource *src,
                                   gboolean running)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE (src));

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  priv->running = running;
}

gint64
zeitgeist_data_source_get_timestamp (ZeitgeistDataSource *src)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_val_if_fail (ZEITGEIST_IS_DATA_SOURCE (src), 0);

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  return priv->timestamp;
}

void
zeitgeist_data_source_set_timestamp (ZeitgeistDataSource *src,
                                     gint64 timestamp)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE (src));

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  priv->timestamp = timestamp;
}

gboolean
zeitgeist_data_source_is_enabled (ZeitgeistDataSource *src)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_val_if_fail (ZEITGEIST_IS_DATA_SOURCE (src), FALSE);

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  return priv->enabled;
}

void
zeitgeist_data_source_set_enabled (ZeitgeistDataSource *src,
                                   gboolean enabled)
{
  ZeitgeistDataSourcePrivate *priv;
  g_return_if_fail (ZEITGEIST_IS_DATA_SOURCE (src));

  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  priv->enabled = enabled;
}

static void
zeitgeist_data_source_init (ZeitgeistDataSource *object)
{
  ZeitgeistDataSourcePrivate *priv;

  priv =  ZEITGEIST_DATA_SOURCE_GET_PRIVATE (object);
  priv->unique_id = NULL;
  priv->name = NULL;
  priv->description = NULL;
  priv->templates = NULL;
  priv->running = FALSE;
  priv->timestamp = 0;
  priv->enabled = TRUE;
}

static void
zeitgeist_data_source_finalize (GObject *object)
{
  ZeitgeistDataSource *src = ZEITGEIST_DATA_SOURCE (object);
  ZeitgeistDataSourcePrivate *priv;
  
  priv = ZEITGEIST_DATA_SOURCE_GET_PRIVATE (src);

  zeitgeist_data_source_set_unique_id (src, NULL);
  zeitgeist_data_source_set_name (src, NULL);
  zeitgeist_data_source_set_description (src, NULL);
  zeitgeist_data_source_set_event_templates (src, NULL);
  
  G_OBJECT_CLASS (zeitgeist_data_source_parent_class)->finalize (object);
}

static void
zeitgeist_data_source_class_init (ZeitgeistDataSourceClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  
  object_class->finalize = zeitgeist_data_source_finalize;

  g_type_class_add_private (object_class, sizeof (ZeitgeistDataSourcePrivate));
}

