/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2011-2012 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 3 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gio/gio.h>
#include <glib/gi18n-lib.h>

#include "u1-codec-installer.h"

#define U1MS_PARTNER_LIST_FILE "u1ms-partner-mp3.list"
#define U1MS_SOURCES_LIST_DIR  "/etc/apt/sources.list.d/"
#define U1MS_SOURCES_LIST_FILE U1MS_SOURCES_LIST_DIR U1MS_PARTNER_LIST_FILE
#define U1MS_CODEC_PACKAGE     "gstreamer0.10-fluendo-plugins-mp3-partner"
#define FLUENDO_CODEC_EULA     "MPEG Layer-3 audio decoding technology " \
	"licensed from Fraunhofer IIS and Thomson. " \
	"This product cannot be installed in product other than Personal " \
	"Computers sold for general purpose usage, and not for set-top " \
	"boxes, embedded PCs, PCs which are sold and customized for " \
	"mainly audio or multimedia playback and/or registration, " \
	"unless the seller has received a license by Fraunhofer IIS " \
	"and Thomson and paid the relevant royalties to them."

/* glib < 2.30 still uses the deprecated g_format_size_for_display(), provide
 * compatibility shim */
#if (!GLIB_CHECK_VERSION(2, 30, 0))
    #define g_format_size g_format_size_for_display
#endif

G_DEFINE_TYPE (U1CodecInstaller, u1_codec_installer, G_TYPE_OBJECT)

struct _U1CodecInstallerPrivate {
	GDBusConnection *bus;
	GDBusProxy      *ad_proxy;
	GDBusProxy      *tr_proxy;
	gchar           *tr_object_path;

	/* Support information */
	gboolean         is_supported;
	gchar           *series;

	/* Widgets for the install process */
	GtkWidget       *progressbar;
};

enum {
	STARTED_SIGNAL,
	FINISHED_SIGNAL,
	ERROR_SIGNAL,
	LAST_SIGNAL
};

static guint installer_signals[LAST_SIGNAL] = { 0, };


static void
u1_codec_installer_init (U1CodecInstaller *installer)
{
	GError *error = NULL;

	installer->priv = G_TYPE_INSTANCE_GET_PRIVATE (installer, U1_TYPE_CODEC_INSTALLER, U1CodecInstallerPrivate);

	installer->priv->bus = g_bus_get_sync (G_BUS_TYPE_SYSTEM, NULL, &error);
	if (error != NULL) {
		g_warning ("Failed to connect to D-Bus system bus: %s",
			   error->message);
		g_error_free (error);
		return;
	}

	installer->priv->ad_proxy = g_dbus_proxy_new_sync (installer->priv->bus,
							   0, NULL,
							   "org.debian.apt",
							   "/org/debian/apt",
							   "org.debian.apt",
							   NULL,
							   &error);
	if (error != NULL) {
		g_warning ("Failed to connect to Apt daemon: %s",
			   error->message);
		g_error_free (error);
		return;
	}

	installer->priv->progressbar = gtk_progress_bar_new ();
	gtk_progress_bar_set_ellipsize (GTK_PROGRESS_BAR (installer->priv->progressbar), PANGO_ELLIPSIZE_END);
	gtk_progress_bar_set_pulse_step (GTK_PROGRESS_BAR (installer->priv->progressbar), 0.05);
	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (installer->priv->progressbar), "");
}

static void
_u1_codec_installer_finalize (GObject *object)
{
	U1CodecInstaller *installer = U1_CODEC_INSTALLER (object);

	if (installer->priv != NULL) {
		if (installer->priv->bus)
			g_object_unref (installer->priv->bus);

		if (installer->priv->tr_object_path)
			g_free (installer->priv->tr_object_path);

		if (installer->priv->series)
			g_free (installer->priv->series);

		if (GTK_IS_WIDGET (installer->priv->progressbar)) {
			gtk_widget_destroy (installer->priv->progressbar);
		}
	}

	G_OBJECT_CLASS (u1_codec_installer_parent_class)->finalize (object);
}

static void
u1_codec_installer_class_init (U1CodecInstallerClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private (klass, sizeof (U1CodecInstallerClass));

	/* Signals */
	installer_signals[STARTED_SIGNAL] = g_signal_new ("started",
							  G_TYPE_FROM_CLASS (klass),
							  (GSignalFlags) G_SIGNAL_RUN_LAST,
							  G_STRUCT_OFFSET (U1CodecInstallerClass, started),
							  NULL,
							  NULL,
							  g_cclosure_marshal_VOID__VOID,
							  G_TYPE_NONE, 0,
							  G_TYPE_NONE);
	installer_signals[FINISHED_SIGNAL] = g_signal_new ("finished",
							   G_TYPE_FROM_CLASS (klass),
							   (GSignalFlags) G_SIGNAL_RUN_LAST,
							   G_STRUCT_OFFSET (U1CodecInstallerClass, finished),
							   NULL,
							   NULL,
							   g_cclosure_marshal_VOID__VOID,
							   G_TYPE_NONE, 0,
							   G_TYPE_NONE);
	installer_signals[ERROR_SIGNAL] = g_signal_new ("error",
							G_TYPE_FROM_CLASS (klass),
							(GSignalFlags) G_SIGNAL_RUN_LAST,
							G_STRUCT_OFFSET (U1CodecInstallerClass, error),
							NULL,
							NULL,
							g_cclosure_marshal_VOID__POINTER,
							G_TYPE_NONE, 1,
							G_TYPE_POINTER);

	object_class->finalize = _u1_codec_installer_finalize;
}

static void
_get_ubuntu_release_series (U1CodecInstaller *installer)
{
	gchar *contents;
	gchar **lines;
	gint i;

	if (!g_file_get_contents ("/etc/lsb-release", &contents, NULL, NULL))
		return;

	lines = g_strsplit (contents, "\n", -1);
	g_free (contents);

	for (i = 0; lines[i] != NULL; i++) {
		gchar **keyval = g_strsplit (lines[i], "=", 2);
		if (g_strcmp0 (keyval[0], "DISTRIB_ID") == 0 &&
		    g_strcmp0 (keyval[1], "Ubuntu") == 0)
			installer->priv->is_supported = TRUE;
		else if (g_strcmp0 (keyval[0], "DISTRIB_CODENAME") == 0)
			installer->priv->series = g_strdup (keyval[1]);

		g_strfreev (keyval);
	}
	g_strfreev (lines);

	return;
}

/**
 * u1_codec_installer_new:
 *
 * Create a new #U1CodecInstaller object.
 *
 * Return value: A new #U1CodecInstaller object.
 */
U1CodecInstaller *
u1_codec_installer_new (void)
{
	U1CodecInstaller *installer = g_object_new (U1_TYPE_CODEC_INSTALLER,
						    NULL);
	_get_ubuntu_release_series (installer);
	return installer;
}

static void
_transaction_progress_changed_cb (GDBusConnection *bus,
				  const gchar *sender_name,
				  const gchar *object_path,
				  const gchar *interface_name,
				  const gchar *signal_name,
				  GVariant *parameters,
				  gpointer user_data)
{
	U1CodecInstaller *installer = U1_CODEC_INSTALLER (user_data);
	gchar *property;
	GVariant *value;

	g_variant_get (parameters, "(sv)", &property, &value);

	if (g_strcmp0 (property, "ProgressDetails") == 0) {
		gint transaction, items_done, items_total;
		gint bytes_done, bytes_total, speed;
		if (g_variant_is_of_type (value, G_VARIANT_TYPE ("(iixxdx)")))
			g_variant_get (value, "(iixxdx)", &transaction,
				       &items_done, &items_total,
				       &bytes_done, &bytes_total,
				       &speed);
		else
			g_variant_get (value, "(iiiiii)", &transaction,
				       &items_done, &items_total,
				       &bytes_done, &bytes_total,
				       &speed);

		if (items_total == 0 && bytes_total == 0)
			gtk_progress_bar_set_text (GTK_PROGRESS_BAR (installer->priv->progressbar), "");
		else {
			gchar *message;
			gchar *bd, *bt;

			bd = g_format_size (bytes_done);
			bt = g_format_size (bytes_total);
			if (speed != 0) {
				gchar *ks = g_format_size (speed);

				message = g_strdup_printf (_("Downloaded %s of %s at %s/s"), bd, bt, ks);
				g_free (ks);
			} else
				message = g_strdup_printf (_("Downloaded %s of %s"), bd, bt);

			gtk_progress_bar_set_text (GTK_PROGRESS_BAR (installer->priv->progressbar), message);

			g_free (bd);
			g_free (bt);
			g_free (message);
		}
	} else if (g_strcmp0 (property, "Progress") == 0) {
		gint progress;

		g_variant_get (value, "i", &progress);
		if (progress > 100)
			gtk_progress_bar_pulse (GTK_PROGRESS_BAR (installer->priv->progressbar));
		else
			gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (installer->priv->progressbar), (gdouble) progress / 100.0);
	} else if (g_strcmp0 (property, "ExitState") == 0) {
		gchar *exit_code;

		g_variant_get (value, "s", &exit_code);

		if (g_strcmp0 (exit_code, "exit-unfinished") != 0) {
			gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (installer->priv->progressbar), 1.0);
			if (g_strcmp0(object_path, installer->priv->tr_object_path) == 0) {
				g_signal_emit (installer, installer_signals[FINISHED_SIGNAL], 0, NULL);
			}
		}
		g_free (exit_code);
	} else if (g_strcmp0 (property, "Error") == 0) {
		gchar *error_code, *error_msg;

		g_variant_get (value, "(ss)", &error_code, &error_msg);

		if (g_strcmp0 (error_code, "error-package-already-installed") != 0) {
			GError *error = NULL;

			g_set_error (&error,
				     U1_CODEC_INSTALLER_ERROR,
				     U1_CODEC_INSTALLER_ERROR_UNKNOWN,
				     "%s", error_msg);
			g_signal_emit (installer,
				       installer_signals[ERROR_SIGNAL], 0,
				       error);
		}

		g_free (error_code);
		g_free (error_msg);
	}

	g_free (property);
	g_variant_unref (value);
}

static void
_transaction_connect_progress_changed (U1CodecInstaller *installer)
{
	g_dbus_connection_signal_subscribe (installer->priv->bus,
					    "org.debian.apt",
					    "org.debian.apt.transaction",
					    "PropertyChanged",
					    g_dbus_proxy_get_object_path (installer->priv->tr_proxy),
					    NULL,
					    G_DBUS_SIGNAL_FLAGS_NONE,
					    (GDBusSignalCallback) _transaction_progress_changed_cb,
					    installer,
					    NULL);
}

static void
_update_install_finished_cb (GObject *object, GAsyncResult *result, gpointer user_data)
{
	U1CodecInstaller *installer = U1_CODEC_INSTALLER (user_data);
	GVariant *finished;

	finished = g_dbus_proxy_call_finish (G_DBUS_PROXY (object),
					     result, NULL);
	g_object_unref (installer->priv->tr_proxy);
	if (finished != NULL)
		g_variant_unref (finished);
}

static void
_update_install_cb (GObject *object, GAsyncResult *result, gpointer user_data)
{
	U1CodecInstaller *installer = U1_CODEC_INSTALLER (user_data);
	GError *error = NULL;
	GVariant *finished;
	gchar *result_val;

	finished = g_dbus_proxy_call_finish (G_DBUS_PROXY (object),
					     result, &error);
	if (error != NULL) {
		g_signal_emit (installer,
			       installer_signals[ERROR_SIGNAL], 0,
			       error);
		return;
	}

	g_variant_get (finished, "(s)", &result_val);
	
	installer->priv->tr_object_path = g_strdup (result_val);
	installer->priv->tr_proxy = g_dbus_proxy_new_sync (installer->priv->bus,
							   0, NULL,
							   "org.debian.apt",
							   result_val,
							   "org.debian.apt.transaction",
							   NULL,
							   &error);

	g_free (result_val);
	g_variant_unref (finished);

	if (error != NULL) {
		g_signal_emit (installer,
			       installer_signals[ERROR_SIGNAL], 0,
			       error);
		return;
	}

	_transaction_connect_progress_changed (installer);

	g_dbus_proxy_call (installer->priv->tr_proxy,
			   "Run",
			   NULL,
			   G_DBUS_CALL_FLAGS_NONE,
			   -1,
			   NULL,
			   (GAsyncReadyCallback) _update_install_finished_cb,
			   installer);
}

static void
_update_cache_finished_cb (GObject *object, GAsyncResult *result, gpointer user_data)
{
	U1CodecInstaller *installer = U1_CODEC_INSTALLER (user_data);
	GError *error = NULL;
	GVariant *finished;
	GVariant *params;
	GVariantBuilder *builder;

	finished = g_dbus_proxy_call_finish (G_DBUS_PROXY (object),
					     result, &error);
	g_object_unref (installer->priv->tr_proxy);

	if (error != NULL) {
		g_signal_emit (installer,
			       installer_signals[ERROR_SIGNAL], 0,
			       error);
		return;
	}

	g_variant_unref (finished);

	builder = g_variant_builder_new (G_VARIANT_TYPE ("as"));
	g_variant_builder_add (builder, "s", U1MS_CODEC_PACKAGE);

	params = g_variant_new ("(as)", builder);

	g_dbus_proxy_call (installer->priv->ad_proxy,
			   "InstallPackages",
			   params, 0, 86400, NULL,
			   (GAsyncReadyCallback) _update_install_cb,
			   installer);

	g_variant_builder_unref (builder);
	g_variant_unref (params);
}

static void
_update_cache_cb (GObject *object, GAsyncResult *result, gpointer user_data)
{
	U1CodecInstaller *installer = U1_CODEC_INSTALLER (user_data);
	GError *error = NULL;
	GVariant *finished;
	gchar *result_val;

	finished = g_dbus_proxy_call_finish (G_DBUS_PROXY (object),
					     result, &error);
	if (error != NULL) {
		g_signal_emit (installer,
			       installer_signals[ERROR_SIGNAL], 0,
			       error);
		return;
	}

	g_variant_get (finished, "(s)", &result_val);
	
	installer->priv->tr_proxy = g_dbus_proxy_new_sync (installer->priv->bus,
							   0, NULL,
							   "org.debian.apt",
							   result_val,
							   "org.debian.apt.transaction",
							   NULL,
							   &error);

	g_free (result_val);
	g_variant_unref (finished);

	if (error != NULL) {
		g_signal_emit (installer,
			       installer_signals[ERROR_SIGNAL], 0,
			       error);
		return;
	}

	_transaction_connect_progress_changed (installer);

	g_dbus_proxy_call (installer->priv->tr_proxy,
			   "Run",
			   NULL,
			   G_DBUS_CALL_FLAGS_NONE,
			   -1,
			   NULL,
			   (GAsyncReadyCallback) _update_cache_finished_cb,
			   installer);

	g_signal_emit (installer, installer_signals[STARTED_SIGNAL], 0, NULL);
}

static void
_repository_add_finished_cb (GObject *object, GAsyncResult *result, gpointer user_data)
{
	U1CodecInstaller *installer = U1_CODEC_INSTALLER (user_data);
	GError *error = NULL;
	GVariant *finished;
	GVariant *params;
	GVariantBuilder *builder;

	finished = g_dbus_proxy_call_finish (G_DBUS_PROXY (object),
					     result, &error);
	g_object_unref (installer->priv->tr_proxy);

	if (error != NULL) {
		g_signal_emit (installer,
			       installer_signals[ERROR_SIGNAL], 0,
			       g_error_copy (error));
		return;
	}

	g_variant_unref (finished);

	builder = g_variant_builder_new (G_VARIANT_TYPE ("as"));
	g_variant_builder_add (builder, "s", U1MS_SOURCES_LIST_FILE);

	params = g_variant_new ("(s)", U1MS_SOURCES_LIST_FILE);

	g_dbus_proxy_call (installer->priv->ad_proxy,
			   "UpdateCachePartially",
			   params, 0, 86400, NULL,
			   (GAsyncReadyCallback) _update_cache_cb,
			   installer);

	g_variant_builder_unref (builder);
	g_variant_unref (params);
}

static void
_repository_add_cb (GObject *object, GAsyncResult *result, gpointer user_data)
{
	U1CodecInstaller *installer = U1_CODEC_INSTALLER (user_data);
	GError *error = NULL;
	GVariant *finished;
	gchar *result_val;

	finished = g_dbus_proxy_call_finish (G_DBUS_PROXY (object),
					     result, &error);
	if (error != NULL) {
		g_signal_emit (installer,
			       installer_signals[ERROR_SIGNAL], 0,
			       error);
		return;
	}

	g_variant_get (finished, "(s)", &result_val);

	installer->priv->tr_proxy = g_dbus_proxy_new_sync (installer->priv->bus,
							   0, NULL,
							   "org.debian.apt",
							   result_val,
							   "org.debian.apt.transaction",
							   NULL,
							   &error);
	g_free (result_val);
	g_variant_unref (finished);

	if (error != NULL) {
		g_signal_emit (installer,
			       installer_signals[ERROR_SIGNAL], 0,
			       error);
		return;
	}

	g_dbus_proxy_call (installer->priv->tr_proxy,
			   "Run",
			   NULL,
			   G_DBUS_CALL_FLAGS_NONE,
			   -1,
			   NULL,
			   (GAsyncReadyCallback) _repository_add_finished_cb,
			   installer);

}

enum {
	EULA_DIALOG_ACCEPT = -1,
	EULA_DIALOG_CANCEL = -2
};

/**
 * u1_codec_installer_install_codec:
 *
 * Begin the process to install the MP3 codec plug-in.
 **/
void
u1_codec_installer_install_codec (U1CodecInstaller *installer,
				  GtkWindow *parent_window)
{
	GtkWidget *eula_dialog;

	eula_dialog = gtk_message_dialog_new (parent_window,
					      GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
					      GTK_MESSAGE_OTHER,
					      GTK_BUTTONS_NONE,
					      "MPEG Layer-3 audio decoding technology notice");
	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (eula_dialog),
						  FLUENDO_CODEC_EULA);
	gtk_dialog_add_buttons (GTK_DIALOG (eula_dialog),
				_("Cancel"), EULA_DIALOG_CANCEL,
				_("Accept"), EULA_DIALOG_ACCEPT,
				NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (eula_dialog),
					 EULA_DIALOG_ACCEPT);

	switch (gtk_dialog_run (GTK_DIALOG (eula_dialog))) {
	case EULA_DIALOG_ACCEPT: {
		GVariant *params;
		GVariantBuilder *builder;

		builder = g_variant_builder_new (G_VARIANT_TYPE ("as"));
		g_variant_builder_add (builder, "s", "partner");

		params = g_variant_new ("(sssasss)",
					"deb",
					"http://archive.canonical.com/",
					installer->priv->series,
					builder,
					"added by Ubuntu One Music Store",
					U1MS_PARTNER_LIST_FILE);

		g_dbus_proxy_call (installer->priv->ad_proxy,
				   "AddRepository",
				   params, 0, 86400, NULL,
				   (GAsyncReadyCallback) _repository_add_cb,
				   installer);

		g_variant_builder_unref (builder);
		g_variant_unref (params);
	}
	case EULA_DIALOG_CANCEL:
	default:
		gtk_widget_destroy (eula_dialog);
		break;
	}
}

/**
 * u1_codec_installer_get_progress_widget:
 *
 * The widget used for displaying progress of codec installation.
 *
 * Return value: The #GtkWidget used to display progress.
 **/
GtkWidget *
u1_codec_installer_get_progress_widget (U1CodecInstaller *installer)
{
	return installer->priv->progressbar;
}

/**
 * U1_CODEC_INSTALLER_ERROR:
 *
 * A #GError domain representing an MP3 codec install error.
 * Used with #U1CodecInstallerError.
 **/
/**
 * u1_codec_installer_error_quark:
 *
 * The #GQuark used as %U1_CODEC_INSTALLER_ERROR.
 *
 * Return value: The #GQuark used as %U1_CODEC_INSTALLER_ERROR.
 **/
GQuark
u1_codec_installer_error_quark (void)
{
	return g_quark_from_static_string ("u1_codec_installer_error_quark");
}

/**
 * U1CodecInstallerError:
 * @U1_CODEC_INSTALLER_ERROR_UNKNOWN: Any otherwise unhandled error.
 * @U1_CODEC_INSTALLER_ERROR_UNSUPPORTED: Indicates an unsupported version of
 * Linux.
 *
 * Codec installation related errors.
 **/
