/*
 * Copyright (C) 2010-2011 Collabora Ltd.
 *   @author Marco Barisione <marco.barisione@collabora.co.uk>
 *   @author Travis Reitter <travis.reitter@collabora.co.uk>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <folks/folks-telepathy.h>
#include <QFile>
#include <QContactAddress>
#include <QContactAvatar>
#include <QContactBirthday>
#include <QContactEmailAddress>
#include <QContactFavorite>
#include <QContactGender>
#include <QContactGlobalPresence>
#include <QContactGuid>
#include <QContactName>
#include <QContactNickname>
#include <QContactNote>
#include <QContactOnlineAccount>
#include <QContactOrganization>
#include <QContactPhoneNumber>
#include <QContactUrl>
#include "managerengine.h"
#include "debug.h"
#include "contactcustomid.h"

QTM_USE_NAMESPACE

namespace
{

static void setFieldDetailsFromContexts(FolksAbstractFieldDetails *details,
                                        QStringList contexts)
{
    foreach(QString context, contexts) {
        QStringList parameters = context.split("=");
        foreach (QString value, parameters[1].split(",")) {
            folks_abstract_field_details_add_parameter (details,
                                                        parameters[0].toUtf8().data(),
                                                        value.toUtf8().data());
        }
    }
}

static void setFieldDetailsFromContexts(FolksAbstractFieldDetails *details,
                                        const QContactDetail &contactDetail)
{
    setFieldDetailsFromContexts(details, contactDetail.contexts());
}

static void setContextsFromFieldDetails(QContactDetail &contactDetail,
                                        FolksAbstractFieldDetails *details)
{
    if (details == NULL) {
        return;
    }

    GeeMultiMap *map = folks_abstract_field_details_get_parameters (details);
    GeeSet *keys = gee_multi_map_get_keys (map);
    GeeIterator *siter = gee_iterable_iterator (GEE_ITERABLE (keys));

    QStringList contexts;

    while (gee_iterator_next (siter)) {
        char *paramenter = (char*) gee_iterator_get (siter);

        GeeCollection *args = folks_abstract_field_details_get_parameter_values(
                details, paramenter);

        GeeIterator *iter = gee_iterable_iterator (GEE_ITERABLE (args));

        QStringList context;
        while (gee_iterator_next (iter)) {
            char *type = (char*) gee_iterator_get (iter);
            context << QLatin1String(type);
            g_free (type);
        }

        contexts << (QString(paramenter) + QString("=") + context.join(","));
    }

    contactDetail.setContexts(contexts);
}
} //namespace

namespace Folks
{

ManagerEngine::ManagerEngine(
        const QMap<QString, QString>& parameters,
        QContactManager::Error* error)
: m_currentLocalId(1)
{
    g_type_init();
    m_aggregator = folks_individual_aggregator_new();
    C_CONNECT(m_aggregator, "individuals-changed", individualsChangedCb);
    folks_individual_aggregator_prepare(
            m_aggregator,
            (GAsyncReadyCallback) STATIC_C_HANDLER_NAME(aggregatorPrepareCb),
            this);
}

ManagerEngine::~ManagerEngine()
{
    gObjectClear((GObject**) &m_aggregator);
}

#define SET_AFD_NEW() \
        GEE_SET(gee_hash_set_new( \
                    FOLKS_TYPE_ABSTRACT_FIELD_DETAILS, \
                    (GBoxedCopyFunc) g_object_ref, g_object_unref, \
                    (GHashFunc) folks_abstract_field_details_hash, \
                    (GEqualFunc) folks_abstract_field_details_equal))

void ManagerEngine::aggregatorPrepareCb()
{
}

void ManagerEngine::updateDisplayLabelFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    const char *displayLabel = folks_name_details_get_nickname(
            FOLKS_NAME_DETAILS(individual));
    if(!displayLabel || displayLabel[0] == '\0') {
        displayLabel = folks_alias_details_get_alias(
                FOLKS_ALIAS_DETAILS(individual));
    }

    QContactManagerEngine::setContactDisplayLabel(&contact,
            QString::fromUtf8(displayLabel));
}

void ManagerEngine::updateGuidFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    QContactGuid guid;
    guid.setGuid(QString::fromUtf8(folks_individual_get_id(individual)));
    contact.saveDetail(&guid);
}

void ManagerEngine::updateLocalIdsFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<ContactCustomId>(contact);

    GeeSet *ids = folks_local_id_details_get_local_ids(FOLKS_LOCAL_ID_DETAILS(individual));
    GeeIterator *iter = gee_iterable_iterator(GEE_ITERABLE(ids));

    QString id;
    while(gee_iterator_next(iter)) {
        id = QString::fromUtf8((const char*) gee_iterator_get(iter));
        // FIXME: check what to do when there is more than just one id.
    }
    ContactCustomId customId;
    customId.setCustomId(id);
    contact.saveDetail(&customId);
}

void ManagerEngine::updateAliasFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    updateDisplayLabelFromIndividual(contact, individual);
}

void ManagerEngine::updateNameFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    FolksStructuredName *sn = folks_name_details_get_structured_name(
            FOLKS_NAME_DETAILS(individual));
    const char *fullName = folks_name_details_get_full_name(
            FOLKS_NAME_DETAILS(individual));

    removeOldDetails<QContactName>(contact);

    if(sn || fullName) {
        QContactName name;

        if(fullName) {
            name.setCustomLabel(QString::fromUtf8(fullName));
            setContactDisplayLabel(&contact, QString::fromUtf8(fullName));
        }

        if(sn) {
            name.setFirstName(QString::fromUtf8(folks_structured_name_get_given_name(sn)));
            name.setLastName(QString::fromUtf8(folks_structured_name_get_family_name(sn)));
            name.setMiddleName(QString::fromUtf8(folks_structured_name_get_additional_names(sn)));
            name.setPrefix(QString::fromUtf8(folks_structured_name_get_prefixes(sn)));
            name.setSuffix(QString::fromUtf8(folks_structured_name_get_suffixes(sn)));
        }

        contact.saveDetail(&name);
    }
}

void ManagerEngine::updateStructuredNameFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    updateNameFromIndividual(contact, individual);
}

void ManagerEngine::updateFullNameFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    updateNameFromIndividual(contact, individual);
}

void ManagerEngine::updateNicknameFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactNickname>(contact);

    const char *nickname = folks_name_details_get_nickname(
            FOLKS_NAME_DETAILS(individual));
    if(nickname && nickname[0] != '\0') {
        QContactNickname detail;
        detail.setNickname(QString::fromUtf8(nickname));
        contact.saveDetail(&detail);
    }
}

void ManagerEngine::updatePresenceFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactGlobalPresence>(contact);

    QContactGlobalPresence presence;
    if(setPresenceDetail(presence, individual))
        contact.saveDetail(&presence);
}

QContactPresence ManagerEngine::getPresenceForPersona(
        QContact& contact,
        FolksPersona *persona)
{
    if(TPF_IS_PERSONA(persona)) {
        QString presenceUri = getPersonaPresenceUri(persona);
        foreach(const QContactPresence& presence,
                contact.details<QContactPresence>()) {
            if(presence.detailUri() == presenceUri)
                return presence;
        }
    }

    return QContactPresence();
}

void ManagerEngine::updatePresenceFromPersona(
        QContact& contact,
        FolksIndividual *individual,
        FolksPersona *persona)
{
    QContactPresence presence = getPresenceForPersona(contact, persona);
    if(!presence.isEmpty()) {
        if(setPresenceDetail(presence, persona))
            contact.saveDetail(&presence);
    }
}

void ManagerEngine::avatarReadyCB(GObject *source, GAsyncResult *res, gpointer user_data)
{
    AvatarLoadData *data = reinterpret_cast<AvatarLoadData*>(user_data);
    if(data->this_->m_allContacts.contains(data->localId)) {
        ContactPair &pair = data->this_->m_allContacts[data->localId];

        QContactAvatar avatar;
        avatar.setImageUrl(data->url);
        pair.contact.saveDetail(&avatar);
        debug() << "AvatarImage (UPDATE):" << data->url;
        emit data->this_->contactsChanged(QList<QContactLocalId>() << data->localId);
    }

    g_free(data);
}

void ManagerEngine::updateAvatarFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactAvatar>(contact);
    GLoadableIcon *avatarIcon = folks_avatar_details_get_avatar(
            FOLKS_AVATAR_DETAILS(individual));
    if(avatarIcon) {
        gchar *uri;
        if(G_IS_FILE_ICON(avatarIcon)) {
            GFile *avatarFile = g_file_icon_get_file(G_FILE_ICON(avatarIcon));
            uri = g_file_get_uri(avatarFile);

            QContactAvatar avatar;
            avatar.setImageUrl(QUrl(QLatin1String(uri)));
            contact.saveDetail(&avatar);
            g_free(uri);
        } else {
            FolksAvatarCache *cache = folks_avatar_cache_dup();
            const char *contactId = folks_individual_get_id(individual);
            uri = folks_avatar_cache_build_uri_for_avatar(cache, contactId);
            QUrl url = QUrl(QLatin1String(uri));
            if (QFile::exists(url.toLocalFile())) {
                QContactAvatar avatar;
                avatar.setImageUrl(url);
                contact.saveDetail(&avatar);
            } else {
                AvatarLoadData *data = new AvatarLoadData;
                data->url = url;
                data->localId = contact.localId();
                data->this_ = this;
                folks_avatar_cache_store_avatar(cache, contactId, avatarIcon, ManagerEngine::avatarReadyCB, data);
            }
        }
        g_free(uri);
    }
}

void ManagerEngine::updateBirthdayFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactBirthday>(contact);

    FolksBirthdayDetails *birthday_details = FOLKS_BIRTHDAY_DETAILS(individual);
    GDateTime *folks_birthday = folks_birthday_details_get_birthday(
            birthday_details);
    if(folks_birthday != NULL) {
        QContactBirthday birthday;
        QDateTime dt;
        dt.setTime_t(g_date_time_to_unix(folks_birthday));
        birthday.setDateTime(dt);
        contact.saveDetail(&birthday);
    }
}

void ManagerEngine::updateEmailAddressesFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactEmailAddress>(contact);

    GeeSet *addresses =
        folks_email_details_get_email_addresses(
                FOLKS_EMAIL_DETAILS(individual));
    GeeIterator *iter = gee_iterable_iterator(GEE_ITERABLE(addresses));

    while(gee_iterator_next(iter)) {
        FolksEmailFieldDetails *email_fd =
            FOLKS_EMAIL_FIELD_DETAILS(gee_iterator_get(iter));
        QContactEmailAddress addr;
        addr.setEmailAddress(
                QString::fromUtf8(
                    (const char*) folks_abstract_field_details_get_value(
                        FOLKS_ABSTRACT_FIELD_DETAILS(email_fd))));
        setContextsFromFieldDetails(addr,
                FOLKS_ABSTRACT_FIELD_DETAILS(email_fd));
        contact.saveDetail(&addr);
        g_object_unref(email_fd);
    }
    g_object_unref (iter);
}

void ManagerEngine::updateImAddressesFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactOnlineAccount>(contact);

    GeeMultiMap *address =
        folks_im_details_get_im_addresses (
                FOLKS_IM_DETAILS(individual));
    GeeSet *keys = gee_multi_map_get_keys (address);
    GeeIterator *iter = gee_iterable_iterator(GEE_ITERABLE(keys));

    while(gee_iterator_next(iter)) {
        const gchar *key = (const gchar*) gee_iterator_get(iter);

        GeeCollection *values = gee_multi_map_get(address, key);
        GeeIterator *iterValues = gee_iterable_iterator(GEE_ITERABLE(values));
        while(gee_iterator_next(iterValues)) {
            QContactOnlineAccount addr;
            FolksAbstractFieldDetails *fd =
                FOLKS_ABSTRACT_FIELD_DETAILS(gee_iterator_get(iterValues));

            addr.setAccountUri(QString::fromUtf8(
                    (const char*) folks_abstract_field_details_get_value(fd)));
            addr.setProtocol(QString::fromUtf8(key));

            setContextsFromFieldDetails(addr,
                                FOLKS_ABSTRACT_FIELD_DETAILS(fd));
            g_object_unref(fd);

            // Set Im type
            int i = 0;
            QStringList contexts = addr.contexts();
            foreach (QString context, contexts) {
                if (context.startsWith("type=")) {
                    QString types = context.replace("type=", "");
                    addr.setSubTypes(types.split(","));

                    contexts.removeAt(i);
                    addr.setContexts(contexts);
                    break;
                }
                i++;
            }
            contact.saveDetail(&addr);
        }
        g_object_unref (iterValues);
    }
    g_object_unref (iter);

}

void ManagerEngine::updateFavoriteFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactFavorite>(contact);

    QContactFavorite favorite;
    favorite.setFavorite(folks_favourite_details_get_is_favourite(
                FOLKS_FAVOURITE_DETAILS(individual)));

    contact.saveDetail(&favorite);
}

void ManagerEngine::updateGenderFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactGender>(contact);

    QContactGender gender;
    switch(folks_gender_details_get_gender(FOLKS_GENDER_DETAILS(individual))) {
    case FOLKS_GENDER_FEMALE:
        gender.setGender(QContactGender::GenderFemale);
        break;
    case FOLKS_GENDER_MALE:
        gender.setGender(QContactGender::GenderMale);
        break;
    default:
        // What's the difference between not having this field or having it
        // set to GenderUnspecified? Let's just not save the detail at all.
        return;
    }

    contact.saveDetail(&gender);
}

void ManagerEngine::updateNotesFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactNote>(contact);

    FolksNoteDetails *note_details = FOLKS_NOTE_DETAILS(individual);
    GeeSet *notes = folks_note_details_get_notes(note_details);
    GeeIterator *iter = gee_iterable_iterator(GEE_ITERABLE(notes));

    while(gee_iterator_next(iter)) {
        FolksAbstractFieldDetails *fd =
            FOLKS_ABSTRACT_FIELD_DETAILS(gee_iterator_get(iter));
        QContactNote note;
        note.setNote(
                QString::fromUtf8(
                    (const char*) folks_abstract_field_details_get_value(fd)));
        setContextsFromFieldDetails(note, fd);
        contact.saveDetail(&note);

        g_object_unref(fd);
    }
}

void ManagerEngine::updateOrganizationFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactOrganization>(contact);

    FolksRoleDetails *role_details = FOLKS_ROLE_DETAILS(individual);
    GeeSet *roles = folks_role_details_get_roles(role_details);
    GeeIterator *iter = gee_iterable_iterator(GEE_ITERABLE(roles));

    while(gee_iterator_next(iter)) {
        FolksAbstractFieldDetails *fd =
            FOLKS_ABSTRACT_FIELD_DETAILS(gee_iterator_get(iter));
        FolksRole *role =
            FOLKS_ROLE(folks_abstract_field_details_get_value(fd));

        QContactOrganization org;
        org.setName(QString::fromUtf8(folks_role_get_organisation_name(role)));
        org.setTitle(QString::fromUtf8(folks_role_get_title(role)));

        setContextsFromFieldDetails(org,
                FOLKS_ABSTRACT_FIELD_DETAILS(fd));
        contact.saveDetail(&org);
    }
}

void ManagerEngine::updatePhoneNumbersFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactPhoneNumber>(contact);

    GeeSet *numbers =
        folks_phone_details_get_phone_numbers(FOLKS_PHONE_DETAILS(individual));
    GeeIterator *iter = gee_iterable_iterator(GEE_ITERABLE(numbers));

    while(gee_iterator_next(iter)) {
        FolksAbstractFieldDetails *fd =
            FOLKS_ABSTRACT_FIELD_DETAILS(gee_iterator_get(iter));
        QContactPhoneNumber number;
        number.setNumber(
                QString::fromUtf8(
                    (const char*) folks_abstract_field_details_get_value(fd)));
        setContextsFromFieldDetails(number, fd);
        // Set phone type
        int i = 0;
        QStringList contexts = number.contexts();
        foreach (QString context, contexts) {
            if (context.startsWith("type=")) {
                QString types = context.replace("type=", "");
                number.setSubTypes(types.split(","));

                contexts.removeAt(i);
                number.setContexts(contexts);
                break;
            }
            i++;
        }
        contact.saveDetail(&number);
        g_object_unref(fd);
    }
    g_object_unref (iter);
}

void ManagerEngine::updateAddressesFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactAddress>(contact);

    GeeSet *addresses = folks_postal_address_details_get_postal_addresses(
            FOLKS_POSTAL_ADDRESS_DETAILS(individual));
    GeeIterator *iter = gee_iterable_iterator(GEE_ITERABLE(addresses));

    while(gee_iterator_next(iter)) {
        FolksPostalAddressFieldDetails *addrFd =
            FOLKS_POSTAL_ADDRESS_FIELD_DETAILS(gee_iterator_get(iter));
        FolksPostalAddress *addr = FOLKS_POSTAL_ADDRESS(
                folks_abstract_field_details_get_value (
                        FOLKS_ABSTRACT_FIELD_DETAILS(addrFd)));
        QContactAddress address;
        address.setCountry(
                QString::fromUtf8(folks_postal_address_get_country(addr)));
        address.setLocality(
                QString::fromUtf8(folks_postal_address_get_locality(addr)));
        address.setPostOfficeBox(
                QString::fromUtf8(folks_postal_address_get_po_box(addr)));
        address.setPostcode(
                QString::fromUtf8(folks_postal_address_get_postal_code(addr)));
        address.setRegion(
                QString::fromUtf8(folks_postal_address_get_region(addr)));
        address.setStreet(
                QString::fromUtf8(folks_postal_address_get_street(addr)));
        setContextsFromFieldDetails(address,
                FOLKS_ABSTRACT_FIELD_DETAILS(addrFd));

        // Set sub-types
        int i = 0;
        QStringList contexts = address.contexts();
        foreach (QString context, contexts) {
            if (context.startsWith("type=")) {
                QString types = context.replace("type=", "");
                address.setSubTypes(types.split(","));

                contexts.removeAt(i);
                address.setContexts(contexts);
                break;
            }
            i++;
        }

        contact.saveDetail(&address);

        g_object_unref(addrFd);
    }
    g_object_unref (iter);
}

void ManagerEngine::updateUrlsFromIndividual(
        QContact &contact,
        FolksIndividual *individual)
{
    removeOldDetails<QContactUrl>(contact);

    GeeSet *urls = folks_url_details_get_urls(FOLKS_URL_DETAILS(individual));
    GeeIterator *iter = gee_iterable_iterator(GEE_ITERABLE(urls));

    while(gee_iterator_next(iter)) {
        FolksAbstractFieldDetails *fd =
            FOLKS_ABSTRACT_FIELD_DETAILS(gee_iterator_get(iter));
        QContactUrl url;
        url.setUrl(
                QString::fromUtf8(
                    (const char*) folks_abstract_field_details_get_value(fd)));
        setContextsFromFieldDetails(url, fd);
        contact.saveDetail(&url);

        g_object_unref(fd);
    }
    g_object_unref (iter);
}

void ManagerEngine::updatePersonas(
        QContact &contact,
        FolksIndividual *individual,
        GeeSet *added,
        GeeSet *removed)
{
#define CONNECT_PERSONA_NOTIFY(propertyName) \
    signalHandlerId = C_NOTIFY_CONNECT(persona, propertyName, \
            personaPresenceChangedCb); \
    m_personasSignalHandlerIds.insert( \
            QPair<FolksIndividual *, FolksPersona *>(individual, persona), \
            signalHandlerId);

    /* this will be used throughout this function */
    GeeIterator *iter;

    iter = gee_iterable_iterator(GEE_ITERABLE(added));
    while(gee_iterator_next(iter)) {
        FolksPersona *persona = FOLKS_PERSONA(gee_iterator_get(iter));
        if(TPF_IS_PERSONA(persona)) {
            m_personasToIndividuals.insertMulti(persona, individual);

            gulong signalHandlerId = 0;

            if(FOLKS_IS_PRESENCE_DETAILS(persona)) {
                CONNECT_PERSONA_NOTIFY("presence-type");
                CONNECT_PERSONA_NOTIFY("presence-message");
            }

            if(FOLKS_IS_ALIAS_DETAILS(persona))
                /* The alias for a single account detail is set in the
                 * presence too */
                CONNECT_PERSONA_NOTIFY("alias");

            addAccountDetails(contact, TPF_PERSONA(persona));
        }
        g_object_unref(persona);
    }
    g_object_unref (iter);

#undef CONNECT_PERSONA_NOTIFY

    iter = gee_iterable_iterator(GEE_ITERABLE(removed));
    while(gee_iterator_next(iter)) {
        FolksPersona *persona = FOLKS_PERSONA(gee_iterator_get(iter));
        if(TPF_IS_PERSONA(persona)) {
            QContactLocalId localId = m_individualsToIds[individual];
            if(localId != 0) {
                ContactPair& contactPair = m_allContacts[localId];
                removeAccountDetails(contactPair.contact, TPF_PERSONA(persona));

                QPair<FolksIndividual *, FolksPersona *> pair(individual,
                        persona);
                gulong signalHandlerId = m_personasSignalHandlerIds[pair];
                if(signalHandlerId)
                    g_signal_handler_disconnect(persona, signalHandlerId);
                m_personasSignalHandlerIds.remove(pair);

                m_personasToIndividuals.remove(persona, individual);
            }
        }

        g_object_unref(persona);
    }
    g_object_unref (iter);
}

void ManagerEngine::addAccountDetails(
        QContact& contact,
        TpfPersona *persona)
{
    TpfPersonaStore *store = TPF_PERSONA_STORE(
            folks_persona_get_store(FOLKS_PERSONA(persona)));
    TpAccount *tpAccount = tpf_persona_store_get_account(store);
    if(!tpAccount)
        return;

    QContactOnlineAccount account;
    account.setAccountUri(
            QString::fromUtf8(folks_persona_get_display_id(FOLKS_PERSONA(persona))));
    account.setValue("TelepathyAccountPath",
            QString::fromUtf8(tp_proxy_get_object_path(TP_PROXY(tpAccount))));

    QRegExp lettersRegexp(QLatin1String("^[a-z]+$"));
    QString protocol = QString::fromUtf8(tp_account_get_protocol(tpAccount));
    QStringList subTypes;
    if(protocol == QLatin1String("sip"))
        subTypes << QLatin1String(QContactOnlineAccount::SubTypeSip)
                 << QLatin1String(QContactOnlineAccount::SubTypeSipVoip);
    else if(protocol == QLatin1String("jabber"))
        subTypes << QLatin1String("Xmpp");
    else if(protocol.contains(lettersRegexp))
        // Just make the protocol name look like the default ones
        subTypes << protocol[0].toUpper() + protocol.mid(1);
    account.setSubTypes(subTypes);

    account.setDetailUri(getPersonaAccountUri(FOLKS_PERSONA(persona)));

    // FIXME: We should set the capabilities here, but from what I understood
    // the current capabilities functions could be deprecated and replaced by
    // QContactAction and related classes (that don't exist yet in the most
    // recently released QtContacts).

    QContactPresence presence;
    if(setPresenceDetail(presence, persona)) {
        QString presenceUri = getPersonaPresenceUri(FOLKS_PERSONA(persona));
        presence.setDetailUri(presenceUri);
        contact.saveDetail(&presence);
        account.setLinkedDetailUris(presenceUri);
    }

    contact.saveDetail(&account);
}

void ManagerEngine::removeAccountDetails(
        QContact& contact,
        TpfPersona *persona)
{
    QList<QContactOnlineAccount> accounts =
        contact.details<QContactOnlineAccount>();
    QString accountUri = getPersonaAccountUri(FOLKS_PERSONA(persona));
    foreach(QContactOnlineAccount account, accounts) {
        if(account.detailUri() == accountUri) {
            contact.removeDetail(&account);
            break;
        }
    }

    QContactPresence presence = getPresenceForPersona(contact,
            FOLKS_PERSONA(persona));
    if(!presence.isEmpty())
        contact.removeDetail(&presence);
}

QString ManagerEngine::getPersonaPresenceUri(
        FolksPersona *persona)
{
    return QLatin1String("presence:") +
        QString::fromUtf8(folks_persona_get_uid(persona));
}

QString ManagerEngine::getPersonaAccountUri(
        FolksPersona *persona)
{
    return QLatin1String("account:") +
        QString::fromUtf8(folks_persona_get_uid(persona));
}

void ManagerEngine::individualsChangedCb(
        FolksIndividualAggregator *aggregator,
        GeeSet *added,
        GeeSet *removed,
        gchar *message,
        FolksPersona *actor,
        FolksGroupDetailsChangeReason reason)
{
    QList<QContactLocalId> removedIds;
    QList<QContactLocalId> addedIds;

    /* this will be used throughout this function */
    GeeIterator *iter;

    iter = gee_iterable_iterator(GEE_ITERABLE(removed));
    while(gee_iterator_next(iter)) {
        FolksIndividual *individual = FOLKS_INDIVIDUAL(gee_iterator_get(iter));
        QContactLocalId lid = removeIndividual(individual);
        if(lid)
            removedIds << lid;

        g_object_unref(individual);
    }
    g_object_unref (iter);

    iter = gee_iterable_iterator(GEE_ITERABLE(added));
    while(gee_iterator_next(iter)) {
        FolksIndividual *individual = FOLKS_INDIVIDUAL(gee_iterator_get(iter));
        QContactLocalId lid = addIndividual(individual);
        if(lid)
            addedIds << lid;

        g_object_unref(individual);
    }
    g_object_unref (iter);

    if(!removedIds.isEmpty())
        emit contactsRemoved(removedIds);
    if(!addedIds.isEmpty())
        emit contactsAdded(addedIds);
}

QContactId ManagerEngine::newContactId()
{
    QContactId id;
    id.setManagerUri(managerUri());
    id.setLocalId(m_currentLocalId++);

    return id;
}

QContactPresence::PresenceState ManagerEngine::folksToQtPresence(
        FolksPresenceType fp)
{
    switch(fp) {
    case FOLKS_PRESENCE_TYPE_OFFLINE:
        return QContactPresence::PresenceOffline;
    case FOLKS_PRESENCE_TYPE_AVAILABLE:
        return QContactPresence::PresenceAvailable;
    case FOLKS_PRESENCE_TYPE_AWAY:
        return QContactPresence::PresenceAway;
    case FOLKS_PRESENCE_TYPE_EXTENDED_AWAY:
        return QContactPresence::PresenceExtendedAway;
    case FOLKS_PRESENCE_TYPE_HIDDEN:
        return QContactPresence::PresenceHidden;
    case FOLKS_PRESENCE_TYPE_BUSY:
        return QContactPresence::PresenceBusy;
    case FOLKS_PRESENCE_TYPE_UNSET:
        // This should not happen as we don't set any presence in this
        // case
    case FOLKS_PRESENCE_TYPE_UNKNOWN:
    case FOLKS_PRESENCE_TYPE_ERROR:
    default:
        return QContactPresence::PresenceUnknown;
    }
}

QContactLocalId ManagerEngine::addIndividual(
        FolksIndividual *individual)
{
    QContact contact;
    contact.setId(newContactId());

    if(folks_individual_get_is_user(individual)) {
        debug() << "Skipping self contact:"
            << folks_alias_details_get_alias(FOLKS_ALIAS_DETAILS(individual))
            << individual;
        return 0;
    }

    debug() << "Added:"
        << folks_alias_details_get_alias(FOLKS_ALIAS_DETAILS(individual))
        << individual;

    updateGuidFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "local-ids", localIdsChangedCb);
    updateLocalIdsFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "alias", aliasChangedCb);
    updateAliasFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "structured-name", structuredNameChangedCb);
    updateStructuredNameFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "full-name", fullNameChangedCb);
    updateFullNameFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "nickname", nicknameChangedCb);
    updateNicknameFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "presence-type", presenceChangedCb);
    C_NOTIFY_CONNECT(individual, "presence-message", presenceChangedCb);
    updatePresenceFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "birthday", birthdayChangedCb);
    updateBirthdayFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "email-addresses", emailAddressesChangedCb);
    updateEmailAddressesFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "im-addresses", imAddressesChangedCb);
    updateImAddressesFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "favourite", favouriteChangedCb);
    updateFavoriteFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "gender", genderChangedCb);
    updateGenderFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "notes", notesChangedCb);
    updateNotesFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "roles", rolesChangedCb);
    updateOrganizationFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "phone-numbers", phoneNumbersChangedCb);
    updatePhoneNumbersFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "postal-addresses", postalAddressesChangedCb);
    updateAddressesFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "urls", urlsChangedCb);
    updateUrlsFromIndividual(contact, individual);

    C_NOTIFY_CONNECT(individual, "avatar", avatarChangedCb);
    updateAvatarFromIndividual(contact, individual);

    GeeSet *empty_set = gee_set_empty(G_TYPE_NONE, NULL, NULL);
    C_CONNECT(individual, "personas-changed", personasChangedCb);
    updatePersonas(contact, individual,
            folks_individual_get_personas(individual), empty_set);
    g_object_unref(empty_set);

    // Store the contact
    ContactPair pair(contact, individual);
    m_allContacts.insert(contact.localId(), pair);

    m_individualsToIds.insert(individual, contact.localId());

    return contact.localId();
}

QContactLocalId ManagerEngine::removeIndividual(
        FolksIndividual *individual)
{
    debug() << "Removed:"
        << folks_alias_details_get_alias(FOLKS_ALIAS_DETAILS(individual))
        << individual;

    QContactLocalId lid = m_individualsToIds[individual];
    if(lid) {
        m_individualsToIds.remove(individual);
        m_allContacts.remove(lid);
    }

    return lid;
}

// DetailType can be a QContactGlobalPresence or a QContactPresence
// FolkType can be a FolksIndividual or a FolksPersona
template<typename DetailType, typename FolkType>
bool ManagerEngine::setPresenceDetail(
        DetailType& detail,
        FolkType *folk)
{
    Q_ASSERT(FOLKS_IS_PRESENCE_DETAILS(folk) && FOLKS_IS_ALIAS_DETAILS(folk));

    if(folks_presence_details_get_presence_type(FOLKS_PRESENCE_DETAILS(folk))
            != FOLKS_PRESENCE_TYPE_UNSET) {
        detail.setCustomMessage(QString::fromUtf8(
            folks_presence_details_get_presence_message(
                FOLKS_PRESENCE_DETAILS(folk))));
        detail.setPresenceState(folksToQtPresence(
            folks_presence_details_get_presence_type(
                FOLKS_PRESENCE_DETAILS(folk))));
        detail.setNickname(QString::fromUtf8(
            folks_alias_details_get_alias(FOLKS_ALIAS_DETAILS(folk))));
        return true;
    }
    else
        return false;
}

QMap<QString, QContactDetailDefinition> ManagerEngine::detailDefinitions(
        const QString& contactType,
        QContactManager::Error *error) const
{
    if(m_detailDefinitions.isEmpty()) {
        QMap<QString, QContactDetailDefinition> full_schema =
            QContactManagerEngine::schemaDefinitions(2)[contactType];
        // Unimplemented:
        //   QContactSyncTarget
        //   QContactTimestamp
        //   QContactAnniversary
        //   QContactRingtone
        //   QContactThumbnail
        //   QContactGeoLocation
        //   QContactTag
        //   QContactFamily
        //   QContactHobby
        QList<QString> def_names = QList<QString>()
            << QContactType::DefinitionName
            << QContactDisplayLabel::DefinitionName
            << QContactEmailAddress::DefinitionName
            << QContactOrganization::DefinitionName
            << QContactPhoneNumber::DefinitionName
            << QContactBirthday::DefinitionName
            << QContactNickname::DefinitionName
            << QContactNote::DefinitionName
            << QContactUrl::DefinitionName
            << QContactGender::DefinitionName
            << QContactOnlineAccount::DefinitionName
            << QContactPresence::DefinitionName
            << QContactGlobalPresence::DefinitionName
            << QContactAvatar::DefinitionName
            << QContactAddress::DefinitionName
            << QContactName::DefinitionName
            << QContactFavorite::DefinitionName
            << QContactGuid::DefinitionName
            << ContactCustomId::DefinitionName
            ;
        foreach(const QString& def_name, def_names) {
            m_detailDefinitions.insert(def_name, full_schema[def_name]);
        }
    }

    return m_detailDefinitions;
}

QList<QContactLocalId> ManagerEngine::contactIds(
        const QContactFilter& filter,
        const QList<QContactSortOrder>& sortOrders,
        QContactManager::Error *error) const
{
    QList<QContactLocalId> ids;
    QContactManager::Error tmpError = QContactManager::NoError;
    QList<QContact> cnts = contacts(filter, sortOrders,
            QContactFetchHint(), &tmpError);
    if(tmpError == QContactManager::NoError) {
        foreach(const QContact& contact, cnts)
            ids << contact.localId();
    }
    else
        *error = tmpError;

    return ids;
}

QContact ManagerEngine::compatibleContact (const QContact & original,
        QContactManager::Error * error ) const
{
    // Let's keep all information about contact we will need that on EDS
    return original;
}

QContact ManagerEngine::contact(
        const QContactLocalId& contactId,
        const QContactFetchHint& fetchHint,
        QContactManager::Error* error) const
{
    Q_UNUSED(fetchHint);

    QContact contact = QContact();
    if(m_allContacts.contains(contactId)) {
        ContactPair pair = m_allContacts[contactId];
        contact = pair.contact;
        *error = QContactManager::NoError;
    }

    return contact;
}

QList<QContact> ManagerEngine::contacts(
        const QContactFilter& filter,
        const QList<QContactSortOrder>& sortOrders,
        const QContactFetchHint& fetchHint,
        QContactManager::Error *error) const
{
    QList<QContact> cnts;

    foreach(const ContactPair& pair, m_allContacts) {
        if(QContactManagerEngine::testFilter(filter, pair.contact))
            QContactManagerEngine::addSorted(&cnts, pair.contact, sortOrders);
    }

    return cnts;
}

bool ManagerEngine::hasFeature(
        QContactManager::ManagerFeature feature,
        const QString &contactType) const
{
    switch(feature) {
    case QContactManager::Anonymous:
        return true;
    default:
        return false;
    }
}

void ManagerEngine::personasChangedCb(
        FolksIndividual *individual,
        GeeSet *added,
        GeeSet *removed)
{
    if(!added && !removed)
        return;

    QContactLocalId localId = m_individualsToIds[individual];
    if(localId == 0)
        return;

    ContactPair& pair = m_allContacts[localId];
    updatePersonas(pair.contact, individual, added, removed);

    emit contactsChanged(QList<QContactLocalId>() << localId);
}

#define IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(cb, updateFunction) \
    void ManagerEngine::cb( \
            FolksIndividual *individual) \
    { \
        QContactLocalId localId = m_individualsToIds[individual]; \
        if(localId == 0) \
            return; \
        \
        ContactPair& pair = m_allContacts[localId]; \
        updateFunction(pair.contact, individual); \
        \
        emit contactsChanged(QList<QContactLocalId>() << localId); \
    }

IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        localIdsChangedCb,
        updateLocalIdsFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        aliasChangedCb,
        updateAliasFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        structuredNameChangedCb,
        updateStructuredNameFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        fullNameChangedCb,
        updateFullNameFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        nicknameChangedCb,
        updateNicknameFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        presenceChangedCb,
        updatePresenceFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        birthdayChangedCb,
        updateBirthdayFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        avatarChangedCb,
        updateAvatarFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        emailAddressesChangedCb,
        updateEmailAddressesFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        imAddressesChangedCb,
        updateImAddressesFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        favouriteChangedCb,
        updateFavoriteFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        genderChangedCb,
        updateGenderFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        notesChangedCb,
        updateNotesFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        rolesChangedCb,
        updateOrganizationFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        phoneNumbersChangedCb,
        updatePhoneNumbersFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        postalAddressesChangedCb,
        updateAddressesFromIndividual)
IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK(
        urlsChangedCb,
        updateUrlsFromIndividual)

#undef IMPLEMENT_INDIVIDUAL_NOTIFY_CALLBACK

#define IMPLEMENT_PERSONA_NOTIFY_CALLBACK(cb, updateFunction) \
    void ManagerEngine::cb( \
            FolksPersona *persona) \
    { \
        QList<FolksIndividual *> individuals = \
            m_personasToIndividuals.values(persona); \
        QList<QContactLocalId> changedIds; \
        foreach(FolksIndividual * individual, individuals) { \
            QContactLocalId localId = m_individualsToIds[individual]; \
            if(localId == 0) \
                continue; \
            \
            ContactPair& pair = m_allContacts[localId]; \
            updateFunction(pair.contact, individual, persona); \
            \
            changedIds << localId; \
        } \
        \
        if(changedIds.size() > 0) \
            emit contactsChanged(changedIds); \
    }

IMPLEMENT_PERSONA_NOTIFY_CALLBACK(
        personaPresenceChangedCb,
        updatePresenceFromPersona)

#undef IMPLEMENT_PERSONA_NOTIFY_CALLBACK

static GValue* asvSetStrNew(QMultiMap<QString, QString> providerUidMap)
{
    GeeMultiMap *hashSet = GEE_MULTI_MAP(
            gee_hash_multi_map_new(G_TYPE_STRING, (GBoxedCopyFunc) g_strdup,
                    g_free,
                    FOLKS_TYPE_IM_FIELD_DETAILS,
                    g_object_ref, g_object_unref,
                    g_str_hash, g_str_equal,
                    (GHashFunc) folks_abstract_field_details_hash,
                    (GEqualFunc) folks_abstract_field_details_equal));

    GValue *retval = gValueSliceNew (G_TYPE_OBJECT);
    g_value_take_object (retval, hashSet);

    QList<QString> keys = providerUidMap.keys();
    foreach(const QString& key, keys) {
        QList<QString> values = providerUidMap.values(key);
        foreach(const QString& value, values) {
            FolksImFieldDetails *imfd;

            imfd = folks_im_field_details_new (value.toUtf8().data(), NULL);

            gee_multi_map_set(hashSet,
                              key.toUtf8().data(),
                              imfd);
            g_object_unref(imfd);
        }
    }

    return retval;
}

static void gValueGeeSetAddStringFieldDetails(GValue *value,
        GType g_type,
        const char* v_string,
        QStringList contexts)
{
    GeeCollection *collection = (GeeCollection*) g_value_get_object(value);

    if(collection == NULL) {
        collection = GEE_COLLECTION(SET_AFD_NEW());
        g_value_take_object(value, collection);
    }

    FolksAbstractFieldDetails *fieldDetails = NULL;

    if(FALSE) {
    } else if(g_type == FOLKS_TYPE_EMAIL_FIELD_DETAILS) {
        fieldDetails = FOLKS_ABSTRACT_FIELD_DETAILS (
                folks_email_field_details_new(v_string, NULL));
    } else if(g_type == FOLKS_TYPE_IM_FIELD_DETAILS) {
        fieldDetails = FOLKS_ABSTRACT_FIELD_DETAILS (
                folks_im_field_details_new(v_string, NULL));
    } else if(g_type == FOLKS_TYPE_NOTE_FIELD_DETAILS) {
        fieldDetails = FOLKS_ABSTRACT_FIELD_DETAILS (
                folks_note_field_details_new(v_string, NULL, NULL));
    } else if(g_type == FOLKS_TYPE_PHONE_FIELD_DETAILS) {
        fieldDetails = FOLKS_ABSTRACT_FIELD_DETAILS (
                folks_phone_field_details_new(v_string, NULL));
    } else if(g_type == FOLKS_TYPE_URL_FIELD_DETAILS) {
        fieldDetails = FOLKS_ABSTRACT_FIELD_DETAILS (
                folks_url_field_details_new(v_string, NULL));
    } else if(g_type == FOLKS_TYPE_WEB_SERVICE_FIELD_DETAILS) {
        fieldDetails = FOLKS_ABSTRACT_FIELD_DETAILS (
                folks_web_service_field_details_new(v_string, NULL));
    }

    if (fieldDetails == NULL) {
        qWarning() << "Invalid fieldDetails type" << g_type;
    } else {
        setFieldDetailsFromContexts (fieldDetails, contexts);
        gee_collection_add(collection, fieldDetails);

        g_object_unref(fieldDetails);
    }
}

static QContactManager::Error managerErrorFromIndividualAggregatorError(
        FolksIndividualAggregatorError errornum)
{
    switch(errornum) {
        // XXX: Add Failed -> Bad Argument isn't a perfect mapping
        case FOLKS_INDIVIDUAL_AGGREGATOR_ERROR_ADD_FAILED:
        {
            return QContactManager::BadArgumentError;
        }
        break;
        break;
        case FOLKS_INDIVIDUAL_AGGREGATOR_ERROR_STORE_OFFLINE:
        {
            return QContactManager::LockedError;
        }
        break;
        case FOLKS_INDIVIDUAL_AGGREGATOR_ERROR_NO_WRITEABLE_STORE:
        {
            return QContactManager::NotSupportedError;
        }
        break;
        default:
        {
            return QContactManager::UnspecifiedError;
        }
        break;
    }
}

void
ManagerEngine::aggregatorAddPersonaFromDetailsCb(GObject *source,
    GAsyncResult *result,
    QContactSaveRequest *request,
    QContact &contact)
{
    FolksIndividualAggregator *aggregator = FOLKS_INDIVIDUAL_AGGREGATOR(source);
    FolksIndividual *individual;
    FolksPersona *persona;
    GError *error = NULL;

    QContactManager::Error opError = QContactManager::NoError;
    QMap<int, QContactManager::Error> contactErrors;

    persona = folks_individual_aggregator_add_persona_from_details_finish(
        aggregator, result, &error);
    if(error != NULL) {
        qWarning() << "Failed to add individual from contact:"
            << error->message;

        opError = managerErrorFromIndividualAggregatorError(
                (FolksIndividualAggregatorError) error->code);

        g_clear_error(&error);
    } else if(persona == NULL) {
        opError = QContactManager::AlreadyExistsError;
        contactErrors.insert(0, opError);
    } else {
        contact.setId(newContactId());
        individual = folks_persona_get_individual(persona);
        if (individual) {
            updateGuidFromIndividual(contact, individual);
            updateLocalIdsFromIndividual(contact, individual);
        }
    }

    contactErrors.insert(0, opError);
    QList<QContact> contacts;
    contacts << contact;
    updateContactSaveRequest(request, contacts, opError, contactErrors,
            QContactAbstractRequest::FinishedState);
}

#define PERSONA_DETAILS_INSERT(details, key, value) \
    g_hash_table_insert((details), \
            (gpointer) folks_persona_store_detail_key((key)), (value));

#define PERSONA_DETAILS_INSERT_STRING_FIELD_DETAILS(\
        details, key, value, q_type, g_type, member) \
{ \
    QList<q_type> contactDetails = contact.details<q_type>(); \
    if(contactDetails.size() > 0) { \
        value = gValueSliceNew(G_TYPE_OBJECT); \
        foreach(const q_type& detail, contact.details<q_type>()) { \
            if(!detail.isEmpty()) { \
                gValueGeeSetAddStringFieldDetails(value, (g_type), \
                        detail.member().toUtf8().data(), \
                        detail.contexts()); \
            } \
        } \
        PERSONA_DETAILS_INSERT((details), (key), (value)); \
    } \
}

/* free the returned GHashTable* with g_hash_table_destroy() */
static GHashTable* personaDetailsHashFromQContact(
        const QContact &contact)
{
    GHashTable *details = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
            (GDestroyNotify) gValueSliceFree);
    GValue *value;

    /*
     * Addresses
     */
    QList<QContactAddress> addresses = contact.details<QContactAddress>();
    if(addresses.size() > 0) {
        value = gValueSliceNew(G_TYPE_OBJECT);
        foreach(const QContactAddress& address, addresses) {
            if(!address.isEmpty()) {
                QStringList contexts = address.contexts();
                QStringList subTypes = address.subTypes();

                if (subTypes.size() > 0) {
                    contexts.append("type=" + subTypes.join(","));
                }

                FolksPostalAddress *postalAddress = folks_postal_address_new(
                        address.postOfficeBox().toUtf8().data(),
                        NULL,
                        address.street().toUtf8().data(),
                        address.locality().toUtf8().data(),
                        address.region().toUtf8().data(),
                        address.postcode().toUtf8().data(),
                        address.country().toUtf8().data(),
                        NULL,
                        NULL);

                GeeCollection *collection =
                    (GeeCollection*) g_value_get_object(value);
                if(collection == NULL) {
                    collection = GEE_COLLECTION(SET_AFD_NEW());
                    g_value_take_object(value, collection);
                }

                FolksPostalAddressFieldDetails *pafd =
                    folks_postal_address_field_details_new (postalAddress,
                            NULL);

                setFieldDetailsFromContexts ((FolksAbstractFieldDetails*)pafd,
                                             contexts);

                gee_collection_add(collection, pafd);

                g_object_unref(pafd);
                g_object_unref(postalAddress);
            }
        }

        PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_POSTAL_ADDRESSES,
                value);
    }

    /*
     * Avatar
     */
    QContactAvatar avatar = contact.detail<QContactAvatar>();
    if(!avatar.isEmpty()) {
        value = gValueSliceNew(G_TYPE_FILE_ICON);
        QUrl avatarUri = avatar.imageUrl();
        if(!avatarUri.isEmpty()) {
            QString formattedUri = avatarUri.toString(QUrl::RemoveUserInfo);
            if(!formattedUri.isEmpty()) {
                GFile *avatarFile =
                    g_file_new_for_uri(formattedUri.toUtf8().data());
                GFileIcon *avatarFileIcon = G_FILE_ICON(
                        g_file_icon_new(avatarFile));
                g_value_take_object(value, avatarFileIcon);

                PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_AVATAR,
                        value);

                gObjectClear((GObject**) &avatarFile);
            }
        }
    }

    /*
     * Birthday
     */
    QContactBirthday birthday = contact.detail<QContactBirthday>();
    if(!birthday.isEmpty()) {
        value = gValueSliceNew(G_TYPE_DATE_TIME);
        GDateTime *dateTime = g_date_time_new_from_unix_utc(
                birthday.dateTime().toMSecsSinceEpoch() / 1000);
        g_value_set_boxed(value, dateTime);

        PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_BIRTHDAY, value);

        g_date_time_unref(dateTime);
    }

    /*
     * Email addresses
     */
    PERSONA_DETAILS_INSERT_STRING_FIELD_DETAILS(details,
        FOLKS_PERSONA_DETAIL_EMAIL_ADDRESSES, value, QContactEmailAddress,
        FOLKS_TYPE_EMAIL_FIELD_DETAILS, emailAddress);


    /*
     * Favorite
     */
    QContactFavorite favorite = contact.detail<QContactFavorite>();
    if(!favorite.isEmpty()) {
        value = gValueSliceNew(G_TYPE_BOOLEAN);
        g_value_set_boolean(value, favorite.isFavorite());

        PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_IS_FAVOURITE,
                value);
    }

    /*
     * Gender
     */
    QContactGender gender = contact.detail<QContactGender>();
    if(!gender.isEmpty()) {
        value = gValueSliceNew(FOLKS_TYPE_GENDER);
        FolksGender genderEnum = FOLKS_GENDER_UNSPECIFIED;
        if(gender.gender().compare(
                    QLatin1String(QContactGender::GenderMale)) == 0) {
            genderEnum = FOLKS_GENDER_MALE;
        } else if(gender.gender().compare(
                    QLatin1String(QContactGender::GenderFemale)) == 0) {
            genderEnum = FOLKS_GENDER_FEMALE;
        }
        g_value_set_enum(value, genderEnum);

        PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_GENDER, value);
    }

    /*
     * Names
     */
    QContactName name = contact.detail<QContactName>();
    if(!name.isEmpty()) {
        value = gValueSliceNew(FOLKS_TYPE_STRUCTURED_NAME);
        FolksStructuredName *sn = folks_structured_name_new(
                name.lastName().toUtf8().data(),
                name.firstName().toUtf8().data(),
                name.middleName().toUtf8().data(),
                name.prefix().toUtf8().data(),
                name.suffix().toUtf8().data());
        g_value_take_object(value, sn);

        PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_STRUCTURED_NAME,
                value);

        QString fullName = name.customLabel();
        if(fullName.isEmpty())
            fullName = contact.displayLabel();
        if(!fullName.isEmpty()) {
            value = gValueSliceNew(G_TYPE_STRING);
            g_value_set_string(value, fullName.toUtf8().data());
            PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_FULL_NAME,
                    value);
        }
    }

    /* XXX: this assumes that the QContactNickname is user-set (what Folks calls
     * an "alias"), not contact-set (what Folks calls a "nickname") */
    QContactNickname nickname = contact.detail<QContactNickname>();
    if(!nickname.isEmpty()) {
        value = gValueSliceNew(G_TYPE_STRING);
        g_value_set_string(value, nickname.nickname().toUtf8().data());
        PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_ALIAS, value);
    }

    /*
     * Notes
     */
    PERSONA_DETAILS_INSERT_STRING_FIELD_DETAILS(details,
        FOLKS_PERSONA_DETAIL_NOTES, value, QContactNote,
        FOLKS_TYPE_NOTE_FIELD_DETAILS, note);

    /*
     * OnlineAccounts
     */
    QList<QContactOnlineAccount> accounts =
        contact.details<QContactOnlineAccount>();
    if(accounts.size() > 0) {
        QMultiMap<QString, QString> providerUidMap;
        foreach(const QContactOnlineAccount& account, accounts) {
            if (!account.isEmpty()) {
                providerUidMap.insert(account.protocol(), account.accountUri());
            }
        }

        if(!providerUidMap.isEmpty()) {
            value = asvSetStrNew(providerUidMap);
            PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_IM_ADDRESSES,
                    value);
        }
    }

    /*
     * Organization
     */
    QList<QContactOrganization> orgs = contact.details<QContactOrganization>();
    if(orgs.size() > 0) {
        value = gValueSliceNew(G_TYPE_OBJECT);
        GeeHashSet *hashSet = gee_hash_set_new(FOLKS_TYPE_ROLE,
            (GBoxedCopyFunc) g_object_ref, g_object_unref,
            (GHashFunc) folks_role_hash, (GEqualFunc) folks_role_equal);
        g_value_take_object(value, hashSet);
        foreach(const QContactOrganization& org, orgs) {
            if(!org.isEmpty()) {
                FolksRole *role = folks_role_new(org.title().toUtf8().data(),
                        org.name().toUtf8().data(), NULL);

                gee_collection_add(GEE_COLLECTION(hashSet), role);
            }
        }

        PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_ROLES, value);
    }

    /*
     * Phone Numbers
     */
    QList<QContactPhoneNumber> phones = contact.details<QContactPhoneNumber>();
    if (phones.size() > 0) {
        value = gValueSliceNew(G_TYPE_OBJECT);
        foreach(QContactPhoneNumber detail, phones) {
            if(!detail.isEmpty()) {
                QStringList contexts = detail.contexts();
                QStringList subTypes = detail.subTypes();

                if (subTypes.size() > 0) {
                    contexts.append("type=" + subTypes.join(","));
                }

                gValueGeeSetAddStringFieldDetails(value,
                                                  FOLKS_TYPE_PHONE_FIELD_DETAILS,
                                                  detail.number().toUtf8().data(),
                                                  contexts);
            }
        }
        PERSONA_DETAILS_INSERT(details, FOLKS_PERSONA_DETAIL_PHONE_NUMBERS, value);
    }

    /*
     * URLs
     */
    PERSONA_DETAILS_INSERT_STRING_FIELD_DETAILS(details,
        FOLKS_PERSONA_DETAIL_URLS, value, QContactUrl,
        FOLKS_TYPE_URL_FIELD_DETAILS, url);

    return details;
}

void
ManagerEngine::aggregatorRemoveIndividualCb(GObject *source,
    GAsyncResult *result,
    QContactRemoveRequest *request)
{
    FolksIndividualAggregator *aggregator = FOLKS_INDIVIDUAL_AGGREGATOR(source);
    GError *error = NULL;

    QContactManager::Error opError = QContactManager::NoError;
    QMap<int, QContactManager::Error> contactErrors;

    folks_individual_aggregator_remove_individual_finish(aggregator, result,
            &error);
    if(error != NULL) {
        qWarning() << "Failed to remove an individual from contact:"
            << error->message;

        opError = managerErrorFromIndividualAggregatorError(
                (FolksIndividualAggregatorError) error->code);
        contactErrors.insert(0, opError);

        g_clear_error(&error);
    }

    updateContactRemoveRequest(request, opError, contactErrors,
            QContactAbstractRequest::FinishedState);
}

FolksPersona* ManagerEngine::getPrimaryPersona(FolksIndividual *individual)
{
    if(individual == NULL) {
        return NULL;
    }

    FolksPersona *retval = NULL;

    GeeSet *personas = folks_individual_get_personas(individual);
    GeeIterator *iter = gee_iterable_iterator(GEE_ITERABLE(personas));

    while(retval == NULL && gee_iterator_next(iter)) {
        FolksPersona *persona = FOLKS_PERSONA(gee_iterator_get(iter));
        FolksPersonaStore *primaryStore =
            folks_individual_aggregator_get_primary_store(m_aggregator);
        if(folks_persona_get_store(persona) == primaryStore) {
            retval = persona;
            g_object_ref(retval);
        }

        g_object_unref(persona);
    }
    g_object_unref (iter);

    return retval;
}

bool ManagerEngine::contactSaveChangesToFolks(const QContact& contact)
{
    bool retval = true;
    ContactPair pair = m_allContacts[contact.localId()];
    FolksIndividual *ind = pair.individual;

    if(ind == NULL) {
        qWarning("Failed to save changes to contact %u: no known corresponding "
                "FolksIndividual", contact.localId());
        return false;
    }

    FolksPersona *persona = getPrimaryPersona(ind);

    /*
     * Addresses
     */
    if(FOLKS_IS_POSTAL_ADDRESS_DETAILS(persona)) {
        FolksPostalAddressDetails *postalAddressDetails =
            FOLKS_POSTAL_ADDRESS_DETAILS(persona);

        QList<QContactAddress> addresses = contact.details<QContactAddress>();
        GeeSet *addressSet = SET_AFD_NEW();
        foreach(const QContactAddress& address, addresses) {
            if(!address.isEmpty()) {
                FolksPostalAddress *postalAddress = folks_postal_address_new(
                        address.postOfficeBox().toUtf8().data(),
                        NULL,
                        address.street().toUtf8().data(),
                        address.locality().toUtf8().data(),
                        address.region().toUtf8().data(),
                        address.postcode().toUtf8().data(),
                        address.country().toUtf8().data(),
                        NULL,
                        NULL);
                FolksPostalAddressFieldDetails *pafd =
                    folks_postal_address_field_details_new (postalAddress,
                            NULL);

                foreach (QString type, address.subTypes()) {
                    folks_abstract_field_details_add_parameter ((FolksAbstractFieldDetails*)pafd,
                                                                "type",
                                                                type.toUtf8().data());
                }

                setFieldDetailsFromContexts ((FolksAbstractFieldDetails*)pafd, address);

                gee_collection_add(GEE_COLLECTION(addressSet), pafd);

                g_object_unref(pafd);
                g_object_unref(postalAddress);
            }
        }

        folks_postal_address_details_set_postal_addresses(postalAddressDetails,
                addressSet);

        g_object_unref(addressSet);
    }

    /*
     * Avatar
     */
    if(FOLKS_IS_AVATAR_DETAILS(persona)) {
        FolksAvatarDetails *avatarDetails = FOLKS_AVATAR_DETAILS(persona);

        QContactAvatar avatar = contact.detail<QContactAvatar>();
        QUrl avatarUri = avatar.imageUrl();
        GFileIcon *avatarFileIcon = NULL;
        if(!avatarUri.isEmpty()) {
            QString formattedUri = avatarUri.toString(QUrl::RemoveUserInfo);
            if(!formattedUri.isEmpty()) {
                GFile *avatarFile =
                    g_file_new_for_uri(formattedUri.toUtf8().data());
                avatarFileIcon = G_FILE_ICON(g_file_icon_new(avatarFile));

                gObjectClear((GObject**) &avatarFile);
            }
        }

        folks_avatar_details_set_avatar(avatarDetails,
                G_LOADABLE_ICON(avatarFileIcon));
        gObjectClear((GObject**) &avatarFileIcon);
    }

    /*
     * Birthday
     */
    if(FOLKS_IS_BIRTHDAY_DETAILS(persona)) {
        FolksBirthdayDetails *birthdayDetails = FOLKS_BIRTHDAY_DETAILS(persona);

        QContactBirthday birthday = contact.detail<QContactBirthday>();
        if(!birthday.isEmpty()) {
            GDateTime *dateTime = g_date_time_new_from_unix_utc(
                    birthday.dateTime().toMSecsSinceEpoch() / 1000);
            folks_birthday_details_set_birthday(birthdayDetails, dateTime);
            folks_birthday_details_set_calendar_event_id(birthdayDetails,
                    birthday.calendarId().toUtf8().data());
            g_date_time_unref(dateTime);
        }
    }

    /*
     * Favorite
     */
    if(FOLKS_IS_FAVOURITE_DETAILS(persona)) {
        FolksFavouriteDetails *favouriteDetails =
            FOLKS_FAVOURITE_DETAILS(persona);

        QContactFavorite favorite = contact.detail<QContactFavorite>();

        folks_favourite_details_set_is_favourite(favouriteDetails,
                favorite.isFavorite());
    }

    /*
     * Names
     */
    if(FOLKS_IS_NAME_DETAILS(persona)) {
        FolksNameDetails *nameDetails = FOLKS_NAME_DETAILS(persona);
        FolksStructuredName *sn = NULL;

        QContactName name = contact.detail<QContactName>();
        folks_name_details_set_full_name(nameDetails,
                name.customLabel().toUtf8().data());

        /* XXX: this is a slight conflation, since Folks treats Nickname as
         * contact-set and alias as user-set (but QtContacts doesn't treat them
         * differently) */
        if(FOLKS_IS_ALIAS_DETAILS(persona)) {
            QContactNickname nickname = contact.detail<QContactNickname>();
            FolksAliasDetails *alias_details = FOLKS_ALIAS_DETAILS(persona);
            folks_alias_details_set_alias(alias_details,
                    nickname.nickname().toUtf8().data());
        }

        if(!name.isEmpty()) {
            sn = folks_structured_name_new(
                    name.lastName().toUtf8().data(),
                    name.firstName().toUtf8().data(),
                    name.middleName().toUtf8().data(),
                    name.prefix().toUtf8().data(),
                    name.suffix().toUtf8().data());
        }

        folks_name_details_set_structured_name(nameDetails, sn);
        gObjectClear((GObject**) &sn);
    }

    /*
     * Notes
     */
    if(FOLKS_IS_NOTE_DETAILS(persona)) {
        FolksNoteDetails *noteDetails = FOLKS_NOTE_DETAILS(persona);

        QList<QContactNote> notes = contact.details<QContactNote>();

        GeeSet *noteSet = SET_AFD_NEW();
        foreach(const QContactNote& note, notes) {
            if(!note.isEmpty()) {
                FolksAbstractFieldDetails *fieldDetails =
                    FOLKS_ABSTRACT_FIELD_DETAILS (folks_note_field_details_new(
                        note.note().toUtf8().data(), NULL, NULL));
                setFieldDetailsFromContexts (fieldDetails, note);
                gee_collection_add(GEE_COLLECTION(noteSet), fieldDetails);
                g_object_unref(fieldDetails);
            }
        }

        folks_note_details_set_notes(noteDetails, noteSet);

        g_object_unref(noteSet);
    }

    /*
     * Phone numbers
     */
    if(FOLKS_IS_PHONE_DETAILS(persona)) {
        FolksPhoneDetails *phoneDetails = FOLKS_PHONE_DETAILS(persona);

        QList<QContactPhoneNumber> numbers =
            contact.details<QContactPhoneNumber>();
        GeeSet *numberSet = SET_AFD_NEW();

        foreach(const QContactPhoneNumber& number, numbers) {
            if(!number.isEmpty()) {
                FolksAbstractFieldDetails *fieldDetails =
                    FOLKS_ABSTRACT_FIELD_DETAILS (folks_phone_field_details_new(
                        number.number().toUtf8().data(), NULL));

                setFieldDetailsFromContexts (fieldDetails, number);

                foreach (QString type, number.subTypes()) {
                    folks_abstract_field_details_add_parameter (fieldDetails,
                        "type",
                        type.toUtf8().data());
                }

                gee_collection_add(GEE_COLLECTION(numberSet), fieldDetails);
                g_object_unref(fieldDetails);
            }
        }

        folks_phone_details_set_phone_numbers(phoneDetails, numberSet);

        g_object_unref(numberSet);
    }

    /*
     * OnlineAccount
     */
    if(FOLKS_IS_IM_DETAILS(persona)) {
        FolksImDetails *imDetails = FOLKS_IM_DETAILS(persona);

        QList<QContactOnlineAccount> accounts =
            contact.details<QContactOnlineAccount>();

        GeeMultiMap *imAddressHash = GEE_MULTI_MAP(
                gee_hash_multi_map_new(G_TYPE_STRING, (GBoxedCopyFunc) g_strdup,
                        g_free,
                        FOLKS_TYPE_IM_FIELD_DETAILS,
                        g_object_ref, g_object_unref,
                        g_str_hash, g_str_equal,
                        (GHashFunc) folks_abstract_field_details_hash,
                        (GEqualFunc) folks_abstract_field_details_equal));

        foreach(const QContactOnlineAccount& account, accounts) {
            if(!account.isEmpty()) {
                QString accountUri = account.accountUri();
                if(!account.protocol().isEmpty()) {
                    if(!accountUri.isEmpty()) {
                        FolksImFieldDetails *imfd =
                            folks_im_field_details_new (
                                    accountUri.toUtf8().data(), NULL);
                        setFieldDetailsFromContexts (FOLKS_ABSTRACT_FIELD_DETAILS (imfd),
                                account);

                        foreach (QString type, account.subTypes()) {
                            folks_abstract_field_details_add_parameter (FOLKS_ABSTRACT_FIELD_DETAILS (imfd),
                                "type",
                                type.toUtf8().data());
                        }

                        gee_multi_map_set(imAddressHash,
                                account.protocol().toUtf8().data(),
                                imfd);

                        g_object_unref(imfd);
                    }
                }
            }
        }

        folks_im_details_set_im_addresses(imDetails, imAddressHash);

        g_object_unref(imAddressHash);
    }

    /*
     * Organization
     */
    if(FOLKS_IS_ROLE_DETAILS(persona)) {
        FolksRoleDetails *roleDetails = FOLKS_ROLE_DETAILS(persona);

        QList<QContactOrganization> orgs =
            contact.details<QContactOrganization>();

        GeeSet *roleSet = SET_AFD_NEW();
        foreach(const QContactOrganization& org, orgs) {
            if(!org.isEmpty()) {


                // The role values can not be NULL
                const gchar* title = org.title().toUtf8().data();
                const gchar* name = org.name().toUtf8().data();
                const gchar* roleName = org.role().toUtf8().data();

                FolksRole *role = folks_role_new(title ? title : "",
                        name ? name : "", "");
                folks_role_set_role (role, roleName ? roleName : "");

                FolksRoleFieldDetails *fieldDetails = folks_role_field_details_new(
                        role, NULL);

                setFieldDetailsFromContexts (FOLKS_ABSTRACT_FIELD_DETAILS (fieldDetails),
                        org);
                gee_collection_add(GEE_COLLECTION(roleSet), fieldDetails);

                g_object_unref(role);
                g_object_unref(fieldDetails);
            }
        }

        folks_role_details_set_roles(roleDetails, roleSet);
        g_object_unref(roleSet);
    }

    /*
     * URLs
     */
    if(FOLKS_IS_URL_DETAILS(persona)) {
        FolksUrlDetails *urlDetails = FOLKS_URL_DETAILS(persona);

        QList<QContactUrl> urls = contact.details<QContactUrl>();
        GeeSet *urlSet = SET_AFD_NEW();
        foreach(const QContactUrl& url, urls) {
            if(!url.isEmpty()) {
                FolksAbstractFieldDetails *fieldDetails =
                    FOLKS_ABSTRACT_FIELD_DETAILS (folks_url_field_details_new(
                        url.url().toUtf8().data(), NULL));
                gee_collection_add(GEE_COLLECTION(urlSet), fieldDetails);
                setFieldDetailsFromContexts (fieldDetails, url);
                g_object_unref(fieldDetails);
            }
        }

        folks_url_details_set_urls(urlDetails, urlSet);

        g_object_unref(urlSet);
    }

    /*
     * Email addresses
     */
    if(FOLKS_IS_EMAIL_DETAILS(persona)) {
        FolksEmailDetails *emailDetails = FOLKS_EMAIL_DETAILS(persona);

        QList<QContactEmailAddress> addresses =
            contact.details<QContactEmailAddress>();
        GeeSet *addressSet = SET_AFD_NEW();
        foreach(const QContactEmailAddress& address, addresses) {
            if(!address.isEmpty()) {
                FolksAbstractFieldDetails *fieldDetails =
                    FOLKS_ABSTRACT_FIELD_DETAILS (folks_email_field_details_new(
                        address.emailAddress().toUtf8().data(), NULL));
                setFieldDetailsFromContexts (fieldDetails, address);
                gee_collection_add(GEE_COLLECTION(addressSet), fieldDetails);
            }
        }

        folks_email_details_set_email_addresses(emailDetails, addressSet);

        g_object_unref(addressSet);
    }

    /*
     * Gender
     */
    if(FOLKS_IS_GENDER_DETAILS(persona)) {
        FolksGenderDetails *genderDetails = FOLKS_GENDER_DETAILS(persona);

        QContactGender gender = contact.detail<QContactGender>();
        FolksGender genderEnum = FOLKS_GENDER_UNSPECIFIED;
        if(gender.gender().compare(
                    QLatin1String(QContactGender::GenderMale)) == 0) {
            genderEnum = FOLKS_GENDER_MALE;
        } else if(gender.gender().compare(
                    QLatin1String(QContactGender::GenderFemale)) == 0) {
            genderEnum = FOLKS_GENDER_FEMALE;
        }

        folks_gender_details_set_gender(genderDetails, genderEnum);
    }

    FolksPersonaStore *primaryStore =
        folks_individual_aggregator_get_primary_store(m_aggregator);
    if(primaryStore == NULL) {
        qWarning() << "Cannot save Contact changes: Failed determine primary "
            "data store";
        retval = false;
    } else {
        folks_persona_store_flush(
                primaryStore,
                0, 0);
    }

    gObjectClear((GObject**) &persona);

    return retval;
}

bool ManagerEngine::startRequest(QContactAbstractRequest* request)
{
    if(request == NULL)
        return false;

    qDebug() << "START REQUEST" << request->type();
    updateRequestState(request, QContactAbstractRequest::ActiveState);
    switch(request->type()) {
        case QContactAbstractRequest::ContactSaveRequest:
        {
            QContactSaveRequest *save_request =
                    qobject_cast<QContactSaveRequest*>(request);

            if((save_request == 0) || (save_request->contacts().size() < 1)) {
                qWarning() << "Contact save request NULL or has zero contacts";
                break;
            }

            FolksPersonaStore *primaryStore =
                folks_individual_aggregator_get_primary_store(m_aggregator);
            if(primaryStore == NULL) {
                qWarning() << "Failed to add individual from contact: "
                        "couldn't get the only persona store";
            } else {
                qDebug () << "Save contact" << save_request->contacts();
                /* guaranteed to have >= 1 contacts above */
                foreach(const QContact& contact, save_request->contacts()) {
                    if(contact.localId() == QContactLocalId(0)) {
                        AddPersonaFromDetailsClosure *closure =
                            new AddPersonaFromDetailsClosure;
                        closure->this_ = this;
                        closure->request = save_request;
                        closure->contact = contact;

                        GHashTable *details = personaDetailsHashFromQContact(
                                contact);

                        folks_individual_aggregator_add_persona_from_details(
                                m_aggregator, NULL, primaryStore, details,
                                (GAsyncReadyCallback)
                                    STATIC_C_HANDLER_NAME(
                                        aggregatorAddPersonaFromDetailsCb),
                                closure);

                        g_hash_table_destroy(details);
                    } else {
                        contactSaveChangesToFolks(contact);
                    }
                }
            }
        }
        break;

        case QContactAbstractRequest::ContactRemoveRequest:
        {
            QContactRemoveRequest *remove_request =
                    qobject_cast<QContactRemoveRequest*>(request);

            if((remove_request == 0) ||
                    (remove_request->contactIds().size() < 1)) {
                qWarning() << "Contact remove request NULL or has zero "
                    "contacts";
                break;
            }

            foreach(const QContactLocalId& contactId,
                    remove_request->contactIds()) {
                RemoveIndividualClosure *closure = new RemoveIndividualClosure;
                closure->this_ = this;
                closure->request = remove_request;

                if(!m_allContacts.contains(contactId)) {
                    qWarning() << "Attempted to remove unknown Contact";
                    continue;
                }

                ContactPair pair = m_allContacts[contactId];
                FolksIndividual *individual = pair.individual;

                folks_individual_aggregator_remove_individual(
                        m_aggregator, individual,
                        (GAsyncReadyCallback)
                            STATIC_C_HANDLER_NAME(
                                aggregatorRemoveIndividualCb),
                        closure);
            }
        }
        break;
        case QContactAbstractRequest::ContactFetchRequest:
        {
            QContactFetchRequest* fetchRequest = qobject_cast<QContactFetchRequest*>(request);
            QContactFilter fetchFilter = fetchRequest->filter();
            QList<QContactSortOrder> fetchSorting = fetchRequest->sorting();
            QContactFetchHint fetchHint = fetchRequest->fetchHint();

            QContactManager::Error error = QContactManager::NoError;
            QList<QContact> allContacts = contacts(fetchFilter, fetchSorting, fetchHint, &error);

            if (!allContacts.isEmpty() || error != QContactManager::NoError)
                updateContactFetchRequest(fetchRequest, allContacts, error, QContactAbstractRequest::FinishedState);
            else
                updateRequestState(request, QContactAbstractRequest::FinishedState);
        }
        break;

        default:
        {
            updateRequestState(request, QContactAbstractRequest::CanceledState);
            return false;
        }
        break;
    }

    return true;
}

// Here is the factory used to allocate new manager engines.

QContactManagerEngine* ManagerEngineFactory::engine(
        const QMap<QString, QString>& parameters,
        QContactManager::Error *error)
{
    if(debugEnabled()) {
        debug() << "Creating a new folks engine";
        foreach(const QString& key, parameters)
            debug() << "    " << key << ": " << parameters[key];
    }

    return new ManagerEngine(parameters, error);
}

QString ManagerEngineFactory::managerName() const
{
    return QLatin1String(FOLKS_MANAGER_NAME);
}

Q_EXPORT_PLUGIN2(qtcontacts_folks, ManagerEngineFactory);

} // namespace Folks
