/* 
 * Copyright 2007 Matthieu CASTET <castet.matthieu@free.fr>
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * PRECISION INSIGHT AND/OR ITS SUPPLIERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "drmP.h"
#include "drm.h"
#include "nouveau_drv.h"
#include "nouveau_drm.h"

#define NV20_GRCTX_SIZE (3529*4)

int nv20_graph_create_context(struct nouveau_channel *chan) {
	struct drm_device *dev = chan->dev;
	struct drm_nouveau_private *dev_priv = dev->dev_private;
	unsigned int ctx_size = NV20_GRCTX_SIZE;
	int ret;

	if ((ret = nouveau_gpuobj_new_ref(dev, chan, NULL, 0, ctx_size, 16,
					  NVOBJ_FLAG_ZERO_ALLOC,
					  &chan->ramin_grctx)))
		return ret;

	/* Initialise default context values */
	INSTANCE_WR(chan->ramin_grctx->gpuobj, 10, chan->id<<24); /* CTX_USER */

	INSTANCE_WR(dev_priv->ctx_table->gpuobj, chan->id,
		    chan->ramin_grctx->instance >> 4);
	return 0;
}

void nv20_graph_destroy_context(struct nouveau_channel *chan) {
	struct drm_device *dev = chan->dev;
	struct drm_nouveau_private *dev_priv = dev->dev_private;

	nouveau_gpuobj_ref_del(dev, &chan->ramin_grctx);

	INSTANCE_WR(dev_priv->ctx_table->gpuobj, chan->id, 0);
}

static void nv20_graph_rdi(struct drm_device *dev) {
	struct drm_nouveau_private *dev_priv = dev->dev_private;
	int i;

	NV_WRITE(NV10_PGRAPH_RDI_INDEX, 0x2c80000);
	for (i = 0; i < 32; i++)
		NV_WRITE(NV10_PGRAPH_RDI_DATA, 0);

	nouveau_wait_for_idle(dev);
}

/* Save current context (from PGRAPH) into the channel's context
 */
int nv20_graph_save_context(struct nouveau_channel *chan) {
	struct drm_device *dev = chan->dev;
	struct drm_nouveau_private *dev_priv = dev->dev_private;
	uint32_t instance;

	instance = INSTANCE_RD(dev_priv->ctx_table->gpuobj, chan->id);
	if (!instance) {
		return -EINVAL;
	}
	if (instance != (chan->ramin_grctx->instance >> 4))
		DRM_ERROR("nv20_graph_save_context : bad instance\n");

	NV_WRITE(NV10_PGRAPH_CHANNEL_CTX_SIZE, instance);
	NV_WRITE(NV10_PGRAPH_CHANNEL_CTX_POINTER, 2 /* save ctx */);
	return 0;
}


/* Restore the context for a specific channel into PGRAPH
 */
int nv20_graph_load_context(struct nouveau_channel *chan) {
	struct drm_device *dev = chan->dev;
	struct drm_nouveau_private *dev_priv = dev->dev_private;
	uint32_t instance;

	instance = INSTANCE_RD(dev_priv->ctx_table->gpuobj, chan->id);
	if (!instance) {
		return -EINVAL;
	}
	if (instance != (chan->ramin_grctx->instance >> 4))
		DRM_ERROR("nv20_graph_load_context_current : bad instance\n");

	NV_WRITE(NV10_PGRAPH_CTX_USER, chan->id << 24);
	NV_WRITE(NV10_PGRAPH_CHANNEL_CTX_SIZE, instance);
	NV_WRITE(NV10_PGRAPH_CHANNEL_CTX_POINTER, 1 /* restore ctx */);
	return 0;
}

void nouveau_nv20_context_switch(struct drm_device *dev)
{
	struct drm_nouveau_private *dev_priv = dev->dev_private;
	struct nouveau_channel *next, *last;
	int chid;

	chid = NV_READ(NV03_PFIFO_CACHE1_PUSH1)&(nouveau_fifo_number(dev)-1);
	next = dev_priv->fifos[chid];

	chid = (NV_READ(NV10_PGRAPH_CTX_USER) >> 24) & (nouveau_fifo_number(dev)-1);
	last = dev_priv->fifos[chid];

	DRM_DEBUG("NV: PGRAPH context switch interrupt channel %x -> %x\n",
		  last->id, next->id);

	NV_WRITE(NV04_PGRAPH_FIFO,0x0);

	nv20_graph_save_context(last);
	
	nouveau_wait_for_idle(dev);

	NV_WRITE(NV10_PGRAPH_CTX_CONTROL, 0x10000000);

	nv20_graph_load_context(next);

	nouveau_wait_for_idle(dev);
	
	if ((NV_READ(NV10_PGRAPH_CTX_USER) >> 24) != next->id)
		DRM_ERROR("nouveau_nv20_context_switch : wrong channel restored %x %x!!!\n", next->id, NV_READ(NV10_PGRAPH_CTX_USER) >> 24);

	NV_WRITE(NV10_PGRAPH_CTX_CONTROL, 0x10010100);
	NV_WRITE(NV10_PGRAPH_FFINTFC_ST2, NV_READ(NV10_PGRAPH_FFINTFC_ST2)&0xCFFFFFFF);

	NV_WRITE(NV04_PGRAPH_FIFO,0x1);
}

int nv20_graph_init(struct drm_device *dev) {
	struct drm_nouveau_private *dev_priv =
		(struct drm_nouveau_private *)dev->dev_private;
	uint32_t tmp, vramsz;
	int ret, i;

	NV_WRITE(NV03_PMC_ENABLE, NV_READ(NV03_PMC_ENABLE) &
			~NV_PMC_ENABLE_PGRAPH);
	NV_WRITE(NV03_PMC_ENABLE, NV_READ(NV03_PMC_ENABLE) |
			 NV_PMC_ENABLE_PGRAPH);

	/* Create Context Pointer Table */
	dev_priv->ctx_table_size = 32 * 4;
	if ((ret = nouveau_gpuobj_new_ref(dev, NULL, NULL, 0,
					  dev_priv->ctx_table_size, 16,
					  NVOBJ_FLAG_ZERO_ALLOC,
					  &dev_priv->ctx_table)))
		return ret;

	NV_WRITE(NV10_PGRAPH_CHANNEL_CTX_TABLE,
		 dev_priv->ctx_table->instance >> 4);

	//XXX need to be done and save/restore for each fifo ???
	nv20_graph_rdi(dev);

	NV_WRITE(NV03_PGRAPH_INTR   , 0xFFFFFFFF);
	NV_WRITE(NV03_PGRAPH_INTR_EN, 0xFFFFFFFF);

	NV_WRITE(NV04_PGRAPH_DEBUG_0, 0xFFFFFFFF);
	NV_WRITE(NV04_PGRAPH_DEBUG_0, 0x00000000);
	NV_WRITE(NV04_PGRAPH_DEBUG_1, 0x00118700);
	NV_WRITE(NV04_PGRAPH_DEBUG_3, 0xF20E0431);
	NV_WRITE(NV10_PGRAPH_DEBUG_4, 0x00000000);
	NV_WRITE(0x40009C           , 0x00000040);

	if (dev_priv->chipset >= 0x25) {
		NV_WRITE(0x400890, 0x00080000);
		NV_WRITE(0x400610, 0x304B1FB6);
		NV_WRITE(0x400B80, 0x18B82880);
		NV_WRITE(0x400B84, 0x44000000);
		NV_WRITE(0x400098, 0x40000080);
		NV_WRITE(0x400B88, 0x000000ff);
	} else {
		NV_WRITE(0x400880, 0x00080000);
		NV_WRITE(0x400094, 0x00000005);
		NV_WRITE(0x400B80, 0x45CAA208);
		NV_WRITE(0x400B84, 0x24000000);
		NV_WRITE(0x400098, 0x00000040);
		NV_WRITE(NV10_PGRAPH_RDI_INDEX, 0x00E00038);
		NV_WRITE(NV10_PGRAPH_RDI_DATA , 0x00000030);
		NV_WRITE(NV10_PGRAPH_RDI_INDEX, 0x00E10038);
		NV_WRITE(NV10_PGRAPH_RDI_DATA , 0x00000030);
	}

	/* copy tile info from PFB */
	for (i=0; i<NV10_PFB_TILE__SIZE; i++) {
		NV_WRITE(NV10_PGRAPH_TILE(i), NV_READ(NV10_PFB_TILE(i)));
		NV_WRITE(NV10_PGRAPH_TLIMIT(i), NV_READ(NV10_PFB_TLIMIT(i)));
		NV_WRITE(NV10_PGRAPH_TSIZE(i), NV_READ(NV10_PFB_TSIZE(i)));
		NV_WRITE(NV10_PGRAPH_TSTATUS(i), NV_READ(NV10_PFB_TSTATUS(i)));
	}

	NV_WRITE(NV10_PGRAPH_CTX_CONTROL, 0x10010100);
	NV_WRITE(NV10_PGRAPH_STATE      , 0xFFFFFFFF);
	NV_WRITE(NV04_PGRAPH_FIFO       , 0x00000001);

	tmp = NV_READ(NV10_PGRAPH_SURFACE) & 0x0007ff00;
	NV_WRITE(NV10_PGRAPH_SURFACE, tmp);
	tmp = NV_READ(NV10_PGRAPH_SURFACE) | 0x00020100;
	NV_WRITE(NV10_PGRAPH_SURFACE, tmp);

	/* begin RAM config */
	vramsz = drm_get_resource_len(dev, 0) - 1;
	NV_WRITE(0x4009A4, NV_READ(NV04_PFB_CFG0));
	NV_WRITE(0x4009A8, NV_READ(NV04_PFB_CFG1));
	NV_WRITE(NV10_PGRAPH_RDI_INDEX, 0x00EA0000);
	NV_WRITE(NV10_PGRAPH_RDI_DATA , NV_READ(NV04_PFB_CFG0));
	NV_WRITE(NV10_PGRAPH_RDI_INDEX, 0x00EA0004);
	NV_WRITE(NV10_PGRAPH_RDI_DATA , NV_READ(NV04_PFB_CFG1));
	NV_WRITE(0x400820, 0);
	NV_WRITE(0x400824, 0);
	NV_WRITE(0x400864, vramsz-1);
	NV_WRITE(0x400868, vramsz-1);

	/* interesting.. the below overwrites some of the tile setup above.. */
	NV_WRITE(0x400B20, 0x00000000);
	NV_WRITE(0x400B04, 0xFFFFFFFF);

	NV_WRITE(NV03_PGRAPH_ABS_UCLIP_XMIN, 0);
	NV_WRITE(NV03_PGRAPH_ABS_UCLIP_YMIN, 0);
	NV_WRITE(NV03_PGRAPH_ABS_UCLIP_XMAX, 0x7fff);
	NV_WRITE(NV03_PGRAPH_ABS_UCLIP_YMAX, 0x7fff);

	return 0;
}

void nv20_graph_takedown(struct drm_device *dev)
{
	struct drm_nouveau_private *dev_priv = dev->dev_private;

	nouveau_gpuobj_ref_del(dev, &dev_priv->ctx_table);
}

