/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/


/*
 * iocontrols.h
 *
 *        The driver io controls.
 */

#ifndef _IOCONTROLS_H_
#define _IOCONTROLS_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_VMMEXT
#define INCLUDE_ALLOW_VMMON
#define INCLUDE_ALLOW_VMCORE
#include "includeCheck.h"

#include "pshare_ext.h"

/*
 * Driver version.
 *
 * Increment major version when you make an incompatible change.
 * Compatibility goes both ways (old driver with new executable
 * as well as new driver with old executable).
 *
 * Note: Vmcore compatibility is different from driver versioning.
 * For vmcore puposes, the bora tree is conceptually split in two:
 * vmcore, and rest-of-bora. The vmmon driver is largely outside
 * vmcore and vmcore imports functionality from vmmon. Addition,
 * deletion or modification of an iocontrol used only by rest-of-bora
 * does not break vmcore compatibility. 
 *
 * See bora/doc/vmcore details.
 *
 */

#define VMMON_VERSION           (138 << 16 | 0)
#define VMMON_VERSION_MAJOR(v)  ((uint32) (v) >> 16)
#define VMMON_VERSION_MINOR(v)  ((uint16) (v))


/*
 * ENOMEM returned after MAX_VMS virtual machines created
 */

#ifdef VMX86_SERVER
#define MAX_VMS 128
#else
#define MAX_VMS 24
#endif


/*
 * We need to specify a special base for ioctl command.
 *
 * This is because in x86_64 Linux, we need to explicitly register
 * handlers for ioctl command numbers with the host kernel to handle
 * 32 bit ioctl system calls, which is a stupid thing done by the
 * kernel folks. In i386 Linux, such a requirement is not necessary.
 * 
 * A cleaner interface would be to define only one ioctl cmd to be
 * used by all calls into vmmon and pass in our customized commands
 * through a parameter. The advantage is that it's more flexible
 * for use to work around Linux kernel hack like in x86_64. The
 * disadvantage is that we need to translate user address to kernel
 * address for all ioctl calls.
 */

typedef enum IOCTLCmd {
   IOCTLCMD_NONE = 2000,
   IOCTLCMD_VERSION,
   IOCTLCMD_CREATE_VM,
   IOCTLCMD_BIND_VM,
   IOCTLCMD_RELEASE_VM,
   IOCTLCMD_GET_NUM_VMS,
   IOCTLCMD_INIT_VM,
   IOCTLCMD_LATE_INIT_VM,
   IOCTLCMD_RUN_VM,
   IOCTLCMD_LOOK_UP_MPN,
   IOCTLCMD_LOCK_PAGE,
   IOCTLCMD_UNLOCK_PAGE,
   IOCTLCMD_APIC_BASE,
   IOCTLCMD_GET_HARD_LIMIT,
   IOCTLCMD_SET_HARD_LIMIT,
   IOCTLCMD_GET_MEM_INFO,
   IOCTLCMD_ADMIT,
   IOCTLCMD_SET_MEM_USAGE,
   IOCTLCMD_READMIT,
   IOCTLCMD_PAE_ENABLED,
   IOCTLCMD_HOST_X86_64,
   IOCTLCMD_GET_TOTAL_MEM_USAGE,
   IOCTLCMD_ACK_USER_CALL,
   IOCTLCMD_COMPLETE_USER_CALL,
   IOCTLCMD_GET_KHZ_ESTIMATE,
   IOCTLCMD_SET_HOST_CLOCK_RATE,
   IOCTLCMD_READ_PAGE,
   IOCTLCMD_WRITE_PAGE,
   IOCTLCMD_LOCK_PAGE_NEW,
   IOCTLCMD_UNLOCK_PAGE_BY_MPN,
   IOCTLCMD_MARK_LOCKEDVARANGE_CLEAN,
   IOCTLCMD_COW_SHARE,
   IOCTLCMD_COW_CHECK,
   IOCTLCMD_COW_UPDATE_HINT,
   IOCTLCMD_COW_COPY_PAGE,
   IOCTLCMD_COW_REMOVE_HINT,
   IOCTLCMD_COW_GET_ZERO_MPN,
    /* AWE calls */
   IOCTLCMD_ALLOC_LOCKED_PAGES,
   IOCTLCMD_FREE_LOCKED_PAGES,
   IOCTLCMD_GET_LOCKED_PAGES_LIST,
   IOCTLCMD_MAP_LOCKED_PAGES,
   IOCTLCMD_UNMAP_LOCKED_PAGES,

   IOCTLCMD_APIC_ID,
   IOCTLCMD_VT_CAPABLE_CPU,
   IOCTLCMD_VT_SUPPORTED_CPU,
   
   /*
    * Keep host-specific calls at the end so they can be undefined
    * without renumbering the common calls.
    */

#ifdef linux
   IOCTLCMD_SET_UID,		// VMX86_DEVEL only
   IOCTLCMD_IS_MP_SAFE,
   IOCTLCMD_ALLOW_CORE_DUMP,

   IOCTLCMD_BROADCAST_IPI,	// SMP 2.2.8+ only

   IOCTLCMD_REGISTER_PASSTHROUGH_IO,
   IOCTLCMD_REGISTER_PASSTHROUGH_IRQ,
   IOCTLCMD_FREE_PASSTHROUGH_IO,
   IOCTLCMD_FREE_PASSTHROUGH_IRQ,
   IOCTLCMD_START_PASSTHROUGH,
   IOCTLCMD_STOP_PASSTHROUGH,
   IOCTLCMD_QUERY_PASSTHROUGH,

   IOCTLCMD_REGISTER_PERFCTR,
   IOCTLCMD_START_PERFCTR,
   IOCTLCMD_STOP_PERFCTR,
   IOCTLCMD_RELEASE_PERFCTR,

   IOCTLCMD_ALLOC_LOW_PAGES,
   IOCTLCMD_FREE_LOW_PAGES,

   IOCTLCMD_GET_ALL_CPUID,

   IOCTLCMD_SET_THREAD_AFFINITY,
   IOCTLCMD_GET_THREAD_AFFINITY,

   IOCTLCMD_SET_POLL_TIMEOUT_PTR,
   IOCTLCMD_GET_KERNEL_CLOCK_RATE,
#endif

#ifdef _WIN32
   IOCTLCMD_BEEP,
   IOCTLCMD_HARD_LIMIT_MONITOR_STATUS,	// Windows 2000 only
   IOCTLCMD_BLUE_SCREEN,	// USE_BLUE_SCREEN only
   IOCTLCMD_CHANGE_HARD_LIMIT,
   IOCTLCMD_ALLOC_LOW_PAGES,
   IOCTLCMD_FREE_LOW_PAGES,
   IOCTLCMD_SYNC_GET_TSCS,
   IOCTLCMD_SYNC_SET_TSCS,
   IOCTLCMD_SEND_IPI,
   IOCTLCMD_GET_KERNEL_PROC_ADDRESS,
   IOCTLCMD_READ_VA64,
   IOCTLCMD_SET_MEMORY_PARAMS,
#endif

   IOCTLCMD_LAST
} IOCTLCmd;


/*
 * Windows ioctl definitions.
 *
 * We use the IRP Information field for the return value
 * of IOCTLCMD_RUN, to be faster since it is use a lot.
 */

#ifdef _WIN32

#define FILE_DEVICE_VMX86        0x8101
#define VMX86_IOCTL_BASE_INDEX   0x801
#define VMIOCTL_BUFFERED(name) \
      CTL_CODE(FILE_DEVICE_VMX86, \
	       VMX86_IOCTL_BASE_INDEX+IOCTLCMD_##name, \
	       METHOD_BUFFERED, \
	       FILE_ANY_ACCESS)
#define VMIOCTL_NEITHER(name) \
      CTL_CODE(FILE_DEVICE_VMX86, \
	       VMX86_IOCTL_BASE_INDEX+IOCTLCMD_##name, \
	       METHOD_NEITHER, \
	       FILE_ANY_ACCESS)

#define IOCTL_VMX86_VERSION		VMIOCTL_BUFFERED(VERSION)
#define IOCTL_VMX86_CREATE_VM		VMIOCTL_BUFFERED(CREATE_VM)
#define IOCTL_VMX86_BIND_VM		VMIOCTL_BUFFERED(BIND_VM)
#define IOCTL_VMX86_RELEASE_VM		VMIOCTL_BUFFERED(RELEASE_VM)
#define IOCTL_VMX86_GET_NUM_VMS		VMIOCTL_BUFFERED(GET_NUM_VMS)
#define IOCTL_VMX86_INIT_VM		VMIOCTL_BUFFERED(INIT_VM)
#define IOCTL_VMX86_LATE_INIT_VM	VMIOCTL_BUFFERED(LATE_INIT_VM)
#define IOCTL_VMX86_RUN_VM		VMIOCTL_NEITHER(RUN_VM)
#define IOCTL_VMX86_SEND_IPI		VMIOCTL_NEITHER(SEND_IPI)
#define IOCTL_VMX86_LOOK_UP_MPN		VMIOCTL_BUFFERED(LOOK_UP_MPN)
#define IOCTL_VMX86_LOCK_PAGE		VMIOCTL_BUFFERED(LOCK_PAGE)
#define IOCTL_VMX86_UNLOCK_PAGE		VMIOCTL_BUFFERED(UNLOCK_PAGE)
#define IOCTL_VMX86_APIC_BASE		VMIOCTL_BUFFERED(APIC_BASE)
#define IOCTL_VMX86_GET_HARD_LIMIT	VMIOCTL_BUFFERED(GET_HARD_LIMIT)
#define IOCTL_VMX86_SET_HARD_LIMIT	VMIOCTL_BUFFERED(SET_HARD_LIMIT)
#define IOCTL_VMX86_GET_MEM_INFO	VMIOCTL_BUFFERED(GET_MEM_INFO)
#define IOCTL_VMX86_ADMIT  	        VMIOCTL_BUFFERED(ADMIT)
#define IOCTL_VMX86_READMIT             VMIOCTL_BUFFERED(READMIT)
#define IOCTL_VMX86_SET_MEM_USAGE       VMIOCTL_BUFFERED(SET_MEM_USAGE)
#define IOCTL_VMX86_PAE_ENABLED		VMIOCTL_BUFFERED(PAE_ENABLED)
#define IOCTL_VMX86_HOST_X86_64         VMIOCTL_BUFFERED(HOST_X86_64)
#define IOCTL_VMX86_COW_SHARE           VMIOCTL_BUFFERED(COW_SHARE)
#define IOCTL_VMX86_COW_CHECK           VMIOCTL_BUFFERED(COW_CHECK)
#define IOCTL_VMX86_COW_UPDATE_HINT     VMIOCTL_BUFFERED(COW_UPDATE_HINT)
#define IOCTL_VMX86_COW_COPY_PAGE       VMIOCTL_BUFFERED(COW_COPY_PAGE)
#define IOCTL_VMX86_COW_REMOVE_HINT     VMIOCTL_BUFFERED(COW_REMOVE_HINT)
#define IOCTL_VMX86_COW_GET_ZERO_MPN    VMIOCTL_BUFFERED(COW_GET_ZERO_MPN)
#define IOCTL_VMX86_BEEP		VMIOCTL_BUFFERED(BEEP)
#define IOCTL_VMX86_HARD_LIMIT_MONITOR_STATUS	VMIOCTL_BUFFERED(HARD_LIMIT_MONITOR_STATUS)
#define IOCTL_VMX86_CHANGE_HARD_LIMIT   VMIOCTL_BUFFERED(CHANGE_HARD_LIMIT)
#define IOCTL_VMX86_ALLOC_LOW_PAGES     VMIOCTL_BUFFERED(ALLOC_LOW_PAGES)
#define IOCTL_VMX86_FREE_LOW_PAGES      VMIOCTL_BUFFERED(FREE_LOW_PAGES)
#ifdef USE_BLUE_SCREEN
#define IOCTL_VMX86_BLUE_SCREEN		VMIOCTL_BUFFERED(BLUE_SCREEN)
#endif

#define IOCTL_VMX86_GET_VM_LIST		VMIOCTL_BUFFERED(GET_VM_LIST)
#define IOCTL_VMX86_GET_VM_INFO		VMIOCTL_BUFFERED(GET_VM_INFO)
#define IOCTL_VMX86_SET_VM_INFO		VMIOCTL_BUFFERED(SET_VM_INFO)
#define IOCTL_VMX86_GET_TOTAL_MEM_USAGE	VMIOCTL_BUFFERED(GET_TOTAL_MEM_USAGE)
#define IOCTL_VMX86_ACK_USER_CALL       VMIOCTL_BUFFERED(ACK_USER_CALL)
#define IOCTL_VMX86_COMPLETE_USER_CALL  VMIOCTL_BUFFERED(COMPLETE_USER_CALL)
#define IOCTL_VMX86_GET_KHZ_ESTIMATE    VMIOCTL_BUFFERED(GET_KHZ_ESTIMATE)
#define IOCTL_VMX86_SET_HOST_CLOCK_RATE VMIOCTL_BUFFERED(SET_HOST_CLOCK_RATE)
#define IOCTL_VMX86_SYNC_GET_TSCS       VMIOCTL_BUFFERED(SYNC_GET_TSCS)
#define IOCTL_VMX86_SYNC_SET_TSCS       VMIOCTL_BUFFERED(SYNC_SET_TSCS)
#define IOCTL_VMX86_READ_PAGE           VMIOCTL_BUFFERED(READ_PAGE)
#define IOCTL_VMX86_WRITE_PAGE          VMIOCTL_BUFFERED(WRITE_PAGE)
#define IOCTL_VMX86_LOCK_PAGE_NEW	VMIOCTL_BUFFERED(LOCK_PAGE_NEW)
#define IOCTL_VMX86_UNLOCK_PAGE_BY_MPN  VMIOCTL_BUFFERED(UNLOCK_PAGE_BY_MPN)
#define IOCTL_VMX86_ALLOC_LOCKED_PAGES	VMIOCTL_BUFFERED(ALLOC_LOCKED_PAGES)
#define IOCTL_VMX86_FREE_LOCKED_PAGES   VMIOCTL_BUFFERED(FREE_LOCKED_PAGES)
#define IOCTL_VMX86_GET_LOCKED_PAGES_LIST VMIOCTL_BUFFERED(GET_LOCKED_PAGES_LIST)
#define IOCTL_VMX86_MAP_LOCKED_PAGES    VMIOCTL_BUFFERED(MAP_LOCKED_PAGES)
#define IOCTL_VMX86_UNMAP_LOCKED_PAGES  VMIOCTL_BUFFERED(UNMAP_LOCKED_PAGES)

#define IOCTL_VMX86_GET_KERNEL_PROC_ADDRESS  VMIOCTL_BUFFERED(GET_KERNEL_PROC_ADDRESS)
#define IOCTL_VMX86_READ_VA64		VMIOCTL_BUFFERED(READ_VA64)
#define IOCTL_VMX86_SET_MEMORY_PARAMS   VMIOCTL_BUFFERED(SET_MEMORY_PARAMS)

#define IOCTL_VMX86_APIC_ID	        VMIOCTL_BUFFERED(APIC_ID)
#define IOCTL_VMX86_VT_CAPABLE_CPU VMIOCTL_BUFFERED(VT_CAPABLE_CPU)
#define IOCTL_VMX86_VT_SUPPORTED_CPU	VMIOCTL_BUFFERED(VT_SUPPORTED_CPU)

#endif


/*
 * Return codes from page locking, unlocking, and MPN lookup.
 * They share an error code space because they call one another
 * internally.
 *
 *    PAGE_LOCK_FAILED              The host refused to lock a page.
 *    PAGE_LOCK_LIMIT_EXCEEDED      We have reached the limit of locked
 *                                  pages for all VMs
 *    PAGE_LOCK_TOUCH_FAILED        Failed to touch page after lock.
 *    PAGE_LOCK_IN_TRANSITION       The page is locked but marked by Windows
 *                                  as nonpresent in CPU PTE and in transition 
 *                                  in Windows PFN.  
 *
 *    PAGE_LOCK_SYS_ERROR           System call error.
 *    PAGE_LOCK_ALREADY_LOCKED      Page already locked.
 *    PAGE_LOCK_MEMTRACKER_ERROR    MemTracker fails.
 *    PAGE_LOCK_PHYSTRACKER_ERROR   PhysTracker fails.
 *    PAGE_LOCK_MDL_ERROR           Mdl error on Windows.
 *
 *    PAGE_UNLOCK_NO_ERROR          Unlock successful (must be 0).
 *    PAGE_UNLOCK_NOT_TRACKED       Not in memtracker.
 *    PAGE_UNLOCK_NO_MPN            Tracked but no MPN.
 *    PAGE_UNLOCK_NOT_LOCKED        Not locked.
 *    PAGE_UNLOCK_TOUCH_FAILED      Failed to touch page.
 *    PAGE_UNLOCK_MISMATCHED_TYPE   Tracked but was locked by different API 
 *
 *    PAGE_LOOKUP_INVALID_ADDR      Consistency checking on Windows.
 *    PAGE_LOOKUP_BAD_HIGH_ADDR     Consistency checking on Windows.
 *    PAGE_LOOKUP_ZERO_ADDR         Consistency checking on Windows.
 *    PAGE_LOOKUP_SMALL_ADDR        Consistency checking on Windows.
 *
 * PAGE_LOCK_ERROR_BIT is used to encode the error value as a positive
 * number to get through ioctl().
 *
 * -- edward
 */

#define PAGE_LOCK_FAILED                  -1
#define PAGE_LOCK_LIMIT_EXCEEDED          -2
#define PAGE_LOCK_TOUCH_FAILED            -3
#define PAGE_LOCK_IN_TRANSITION           -4

#define PAGE_LOCK_SYS_ERROR              -10
#define PAGE_LOCK_ALREADY_LOCKED         -11
#define PAGE_LOCK_MEMTRACKER_ERROR       -12
#define PAGE_LOCK_PHYSTRACKER_ERROR      -13
#define PAGE_LOCK_MDL_ERROR              -14

#define PAGE_UNLOCK_NO_ERROR		   0
#define PAGE_UNLOCK_NOT_TRACKED		-100
#define PAGE_UNLOCK_NO_MPN		-101
#define PAGE_UNLOCK_NOT_LOCKED		-102
#define PAGE_UNLOCK_TOUCH_FAILED	-103
#define PAGE_UNLOCK_MISMATCHED_TYPE	-104

#define PAGE_LOOKUP_INVALID_ADDR        -200
#define PAGE_LOOKUP_BAD_HIGH_ADDR       -201
#define PAGE_LOOKUP_ZERO_ADDR           -202
#define PAGE_LOOKUP_SMALL_ADDR          -203
#define PAGE_LOOKUP_NOT_TRACKED          -10	// added to another code
#define PAGE_LOOKUP_NO_MPN               -20	// added to another code
#define PAGE_LOOKUP_NOT_LOCKED           -30	// added to another code
#define PAGE_LOOKUP_NO_VM                -40	// added to another code

#define PAGE_LOCK_SUCCESS(mpn) ((int) (mpn) >= 0)
#define PAGE_LOCK_SOFT_FAILURE(mpn) ((int) (mpn) < 0 && (int) (mpn) > -10)

#define PAGE_LOCK_ERROR_BIT 0x4000000


/*
 * Flags sent into APICBASE ioctl
 */

#define APIC_FLAG_DISABLE_NMI       0x00000001
#define APIC_FLAG_PROBE             0x00000002
#define APIC_FLAG_FORCE_ENABLE      0x00000004


/*
 * Structure sent into REGISTER_PERFCOUNTERS ioctl
 */

#define PERFCTR_MAX_COUNTERS        2
#define PERFCTR_INVALID_EVENT       0
#define PERFCTR_INVALID_IRQ         -1

typedef struct PerfCtrInfo {
   uint32 eventNum;
   uint32 samplingRate;
} PerfCtrInfo;

typedef struct PerfCtrRegisterArgs {
   PerfCtrInfo counterInfo[PERFCTR_MAX_COUNTERS];
   int irq;
   Bool totalOnly;
} PerfCtrRegisterArgs;

typedef struct VMAPICInfo {
   uint32 flags;
   MA32   apicBase;
} VMAPICInfo; 

#ifndef MAX_PROCESSORS
#define MAX_PROCESSORS 32
#endif

typedef struct VMSyncWriteTSCs {
   uint32 cpuSet;
   uint32 TSCs[MAX_PROCESSORS];
} VMSyncWriteTSCs;


#define VMX86_DRIVER_VCPUID_OFFSET	1000


/*
 * We keep track of 3 different limits on the number of pages we can lock.
 * The host limit is determined at driver load time (in windows only) to
 * make sure we do not starve the host by locking too many pages.
 * The static limit is user defined in the UI and the dynamic limit is 
 * set by authd's hardLimitMonitor code (windows only), which queries
 * host load and adjusts the limit accordingly.  We lock the minimum of 
 * all these values at any given time.
 */
typedef struct LockedPageLimit {
   uint32 host;        // driver calculated maximum for this host
   uint32 configured;  // user defined maximum pages to lock
   uint32 dynamic;     // authd hardLimitMonitor pages to lock
} LockedPageLimit;

/*
 * Data structures for the GET_MEM_INFO and ADMIT ioctls.
 */

typedef struct VMMemMgmtInfo {
   uint32          minAllocation;   // minimum pages for vm
   uint32          maxAllocation;   // maximum pages the vm could lock
   uint32          shares;          // proportional sharing weight
   uint32          nonpaged;        // overhead memory
   uint32          paged;           // guest memory(mainmem + other regions)
   uint32          mainMemSize;     // guest main memory size
   uint32          locked;          // number of pages locked by this vm
   uint32          shared;          // number of guest pages shared
   uint32          usedPct;         // % of guest memory being touched
   uint32          admitted;        // admission control
} VMMemMgmtInfo;

#define VMMEM_COW_HOT_PAGES 5

typedef struct VMMemCOWHotPage {
   uint32          mpn;
   uint32          ref;
   uint64          key;
} VMMemCOWHotPage;

typedef struct VMMemCOWInfo {
   uint32          numHints;
   uint32          uniqueMPNs;       // current MPNs used for COW
   uint32          totalUniqueMPNs;  // total MPNs ever used as COW
   uint32          numBreaks;
   uint32          numRef;
   uint32          _pad[1];
   VMMemCOWHotPage hot[VMMEM_COW_HOT_PAGES];
} VMMemCOWInfo;

typedef struct VMMemInfoArgs {
   VMMemCOWInfo    cowInfo;            // global page sharing state
   uint32          numVMs;             
   uint32          minVmMemPct;        // % of vm that must fit in maxLockedPages
   uint32          globalMinAllocation;// pages that must fit in maxLockedPages
   uint32          numLockedPages;     // total locked pages by all vms
   LockedPageLimit lockedPageLimit;    // set of locked page limits
   uint32          maxLockedPages;     // effective limit on locked pages
   uint32          callerIndex;        // this vm's index memInfo array
   VMMemMgmtInfo   memInfo[1];
   uint8           _pad[4];
} VMMemInfoArgs;

#define VM_GET_MEM_INFO_SIZE(numVMs) \
   ((numVMs - 1) * sizeof (VMMemMgmtInfo) + sizeof (VMMemInfoArgs))

typedef struct VMMPNList {
   uint32       mpnCount;
   VA32         mpn32List; /* list of 32 bit mpns */
} VMMPNList;

typedef struct VARange {
   VA32     addr;
   unsigned len;
   VA32     bv;
} VARange;

typedef struct VMMPNListVA {
   uint32       mpnCount;
   VA32         mpn32List;  /* list of 32 bit mpns */
   VA32		va;	    /* userspace VA. */
} VMMPNListVA;

typedef struct VMMUnlockPageByMPN {
   MPN32	  mpn;
   VA32 	  va;	    /* optional userspace VA */
} VMMUnlockPageByMPN;

typedef struct VMMReadWritePage {
   MPN32        mpn;
   VA32         buf;
} VMMReadWritePage;


struct passthrough_iorange {
  unsigned short ioBase;	/* Base of range to pass through */
  unsigned short numPorts;	/* Length of range */
};

typedef struct COWShareInfo {
   uint32       numPages;
   MPN          pshareMPN;
   Bool         shareFailure;
   Bool         updateHints;
} COWShareInfo;

typedef struct COWHintInfo {
   uint32            nUpdates;
   PShare_HintUpdate updates[PSHARE_HINT_BATCH_PAGES_MAX];
} COWHintInfo;

typedef struct COWCheckInfo {
   uint32              numPages;
   PShare_COWCheckInfo check[PSHARE_MAX_COW_CHECK_PAGES];
} COWCheckInfo;

#ifdef linux

typedef struct {
   uint64              ioarg;	// 64bit address
   uint64              iocmd;	// 64bit command
} VMIoctl64;

/*
 * We can move it at the beginning of our ioctl list (just below
 * IOCTLCMD_VERSION). It just should not change very often...
 */
#define IOCTLCMD_IOCTL64  _IOW(0x99, 0x80, VMIoctl64)


/*
 * Structures for passing process affinity between vmmon and userspace.
 */
struct VMMonAffinity {
   uint32   pid;      // IN
   uint32   affinity; // IN for set, OUT for get, bit 0 == processor 0...
};

#endif

#endif // ifndef _IOCONTROLS_H_
