/*
 * Copyright 2009 Freescale Semiconductor, Inc. All Rights Reserved.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */
/*---------------------------------------------------------------------------
 * Pegatron Modify History
 *---------------------------------------------------------------------------
 * 2009/11/18  Huimin Lin - 1. add debug message
 2. enable battery platform
 3. change charge to AUTO
 *---------------------------------------------------------------------------
 * 2009/11/20  Eric Chou  - 1. modify
 *                          2. fine-tune
 *---------------------------------------------------------------------------
 * 2009/11/24  Kevin Wei  - 1. use EVENT_CHGDETI = charger attach , to detect PC VBUS
 *                          2. modify mc13892_charger_update_status() to send KOBJ event for USB
 *---------------------------------------------------------------------------
 * 2009/11/27  Eric Chou  - 1. modify
 *                          2. fine-tune
 *---------------------------------------------------------------------------
 * 2009/12/01  Kevin Wei  - 1. add callback EVENT_VBUSVI = VBUS valid detect ( option )
 *                          2. use work queue to read CHRGSE1BI = wall charger detect ( option )
 *                          3. use work queue to send KOBJ event for USB ( option )
 *---------------------------------------------------------------------------
 * 2009/12/02  Eric Chou  - 1. modify
 *                          2. fine-tune
 *                          3. use EVENT_VBUSVI callback , need enable mc13892.c line 209
 *---------------------------------------------------------------------------
 * 2009/12/08  Huimin Lin  - 1 modify - to sync_BATTERY_LOW_INTERRUPT mechanism from Spider1 codebase
 *---------------------------------------------------------------------------
 * 2009/12/11  Huimin Lin  - 1.modify - to use Battery charging S/W monitor 2-hr timeout alogrithm
 *---------------------------------------------------------------------------
 * 2010/01/04  Huimin Lin  - 1. modify - to sync_BATTERY_LOW_INTERRUPT mechanism from Spider1 codebase
 *---------------------------------------------------------------------------
 * 2010/01/29  Huimin Lin  - 1. add - to fix the charger LED issue
 *---------------------------------------------------------------------------
 * 2010/04/16                1. add - batt_cap.
 *                           2. add - mc13892_battery_sample() for BAT voltage average sample.
 *                           3. modify - battery_low_resume_check().
 *                                        don't use BAT_VOLTAGE_LOW_SHUTDOWN.
 *                                        use mc13892_battery_sample(), batt_cap.
 *---------------------------------------------------------------------------
 * 2010/04/20                1. modify - batt_cap[] table.
 *---------------------------------------------------------------------------
 * 2010/04/24                1. modify - batt_cap[] table.
 *                           2. modify - check ACadapter at online/offline.
 *                           3. modify - compiler warning fix in battery_chgfaulti_work().
 *                           4. modify - marge Spider1 patch, check di->charger.dev
 *                                        in mc13892_charger_update_status().
 *                           5. modify - adjust A/D <-> mV scale.
 *                                        change BAT_VOLTAGE_UNIT_UV -> BATT_RAW_TO_uV().
 *---------------------------------------------------------------------------
 * 2010/05/14                1. modify - USE_SW_MONITOR_TIME_OUT 1->0
 *                           2. modify - USE_CHARGER_LED_FIX 1->0
 *                                        charge LED auto(don't force ON)
 *                           3. add    - add CHARGE_CURRENT,
 *                                           pmic_battery_start/stop_charge().
 *                           4. modify - don't check online BATT_CAP_SHUTDOWN_LEVEL at battery_low_resume_check().
 *                           5. add    - add USE_BATTERY_THERMISTOR_CHECK,
 *---------------------------------------------------------------------------
 * 2010/05/28                1. modify - undef USE_BATTERY_THERMISTOR_CHECK.
 *                                        power-off when BATT eject.
 *                           2. add    - add USE_BATTERY_TRICKLE_CHARGE.
 *---------------------------------------------------------------------------
 * 2010/09/30                1. add    - USE_SOFT WITH AUTO
 *---------------------------------------------------------------------------
 */

/*
 * Includes
 */
#include <linux/platform_device.h>
#include <linux/power_supply.h>

#include <linux/delay.h>
#include <asm/mach-types.h>
#include <linux/pmic_battery.h>
#include <linux/pmic_adc.h>
#include <linux/pmic_status.h>
#include <mach/arc_otg.h>

#define USE_EVENT_CHGDETI	0	
#define USE_EVENT_VBUSVI	0	

#define DEBUG_TRACE        0
#if ( DEBUG_TRACE )
#define DEBUG_PRINTK(x)    printk x
#else
#define DEBUG_PRINTK(x)
#endif

#define BIT_CHG_VOL_LSH		0
#define BIT_CHG_VOL_WID		3

#define BIT_CHG_CURR_LSH		3
#define BIT_CHG_CURR_WID		4

#define BIT_CHG_PLIM_LSH		15
#define BIT_CHG_PLIM_WID		2

#define BIT_CHG_DETS_LSH 6
#define BIT_CHG_DETS_WID 1
#define BIT_CHG_CURRS_LSH 11
#define BIT_CHG_CURRS_WID 1

#define TRICKLE_CHG_EN_LSH	7
#define	LOW_POWER_BOOT_ACK_LSH	8
#define BAT_TH_CHECK_DIS_LSH	9
#define	BATTFET_CTL_EN_LSH	10
#define BATTFET_CTL_LSH		11
#define	REV_MOD_EN_LSH		13
#define PLIM_DIS_LSH		17
#define	CHG_LED_EN_LSH		18
#define RESTART_CHG_TIMER_LSH	19
#define RESTART_CHG_STAT_LSH	20
#define	AUTO_CHG_DIS_LSH	21
#define CYCLING_DIS_LSH		22
#define	VI_PROGRAM_EN_LSH	23

#define TRICKLE_CHG_EN_WID	1
#define	LOW_POWER_BOOT_ACK_WID	1
#define BAT_TH_CHECK_DIS_WID	1
#define	BATTFET_CTL_EN_WID	1
#define BATTFET_CTL_WID		1
#define	REV_MOD_EN_WID		1
#define PLIM_DIS_WID		1
#define	CHG_LED_EN_WID		1
#define RESTART_CHG_TIMER_WID	1
#define RESTART_CHG_STAT_WID	1
#define	AUTO_CHG_DIS_WID	1
#define CYCLING_DIS_WID		1
#define	VI_PROGRAM_EN_WID	1

#define ACC_STARTCC_LSH		0
#define ACC_STARTCC_WID		1
#define ACC_RSTCC_LSH		1
#define ACC_RSTCC_WID		1
#define ACC_CCFAULT_LSH		7
#define ACC_CCFAULT_WID		7
#define ACC_CCOUT_LSH		8
#define ACC_CCOUT_WID		16
#define ACC1_ONEC_LSH		0
#define ACC1_ONEC_WID		15

#define ACC_CALIBRATION 0x17
#define ACC_START_COUNTER 0x07
#define ACC_STOP_COUNTER 0x2
#define ACC_CONTROL_BIT_MASK 0x1f
#define ACC_ONEC_VALUE 2621
#define ACC_COULOMB_PER_LSB 1
#define ACC_CALIBRATION_DURATION_MSECS 20

#define BAT_CURRENT_UNIT_UA 5870	/* AD1  3,000,000(uA) / 511  (+/-) */
#define CHG_VOLTAGE_UINT_UV 23474	/* AD3 20,000,000(uV) / 852  (+)   */
#define CHG_MIN_CURRENT_UA 3500

#define COULOMB_TO_UAH(c) (10000 * c / 36)

enum chg_setting {
  TRICKLE_CHG_EN,
  LOW_POWER_BOOT_ACK,
  BAT_TH_CHECK_DIS,
  BATTFET_CTL_EN,
  BATTFET_CTL,
  REV_MOD_EN,
  PLIM_DIS,
  CHG_LED_EN,
  RESTART_CHG_TIMER,
  RESTART_CHG_STAT,
  AUTO_CHG_DIS,
  CYCLING_DIS,
  VI_PROGRAM_EN
};

//Modify by pega Huimin 2010/1/4, to sync_BATTERY_LOW_INTERRUPT mechanism from Spider1 codebase
#define USE_BATTERY_LOW_POLLING  1
//Modify by pega Huimin 2009/12/8, to sync_BATTERY_LOW_INTERRUPT mechanism from Spider1 codebase
#define USE_BATTERY_LOW_INTERRUPT	0	
//Modify by pega Huimin 2009/12/11, to use Battery charging S/W monitor 2-hr timeout alogrithm
#define USE_SW_MONITOR_TIME_OUT	1
//add by pega Huimin 2010.01.29, to fix the charger LED issue
#define USE_CHARGER_LED_FIX	0

#define USE_BATTERY_TRICKLE_CHARGE
#undef  USE_BATTERY_THERMISTOR_CHECK
#define CHARGE_CURRENT		6		/* ICHRG 6:560mA */

#define USE_RESTART_TIME_OUT	0
#define USE_SOFT_WITH_AUTO	1

//Modify by pega Huimin 2009/12/8 USE_BATTERY_LOW_INTERRUPT mechanism from spider 1
#if ( USE_BATTERY_LOW_INTERRUPT )
#include <linux/input.h>
/*
 * battery low event key
 */
#define BATTERY_INPUT_NAME  ("battery_stat")
struct _batt_input_stat {
  int  event_type;		/* EV_KEY/... */
  int  event_code;		/* KEY_xxx/... */
}; 
static struct _batt_input_stat batt_input_stat [2] = {
  { EV_KEY, KEY_POWER },	/* LOBATLI */
  { EV_KEY, KEY_BATTERY },	/* LOBATHI */
};
#endif /* USE_BATTERY_LOW_INTERRUPT */


#if ( USE_SW_MONITOR_TIME_OUT )
#define BAT_VOLTAGE_UPPER	4180000
#define BAT_VOLTAGE_LOWER	4120000
#define BAT_VOLTAGE_FAULTI_UV	4120000
static int pmic_stop_charging(void);
static int pmic_get_batt_voltage(unsigned short *voltage);
static int pmic_battery_boot_flag = 0;
static int charging_flag = 0;
#endif
static int pmic_restart_charging(void);

#if ( USE_BATTERY_LOW_POLLING )
/*
 * BATTERY POLLING interval (sec)
 */
static int batt_poll_interval = (30 * 60);	//modify by Huimin from 30*60 to 5*60 secs
static struct mc13892_dev_info *batt_poll_di;

#endif /* USE_BATTERY_LOW_POLLING */

#ifdef CONFIG_MACH_MX51_SENDAI

#define IGNORE_SAMPLE_VOLT	2000	/* sample dara ignore limit */

/*
 * Battery 4.8V - 1023
 *  Battery(3.95V) -> A/Dinput(3.88V) -> A/D(827)
 *  Battery(3.40V) -> A/Dinput(3.33V) -> A/D(710)
 */
#define BATT_RAW_TO_uV(raw)	(((raw) * (4800000 / 1023)) + 70000)	// mV
#define BATT_RAW_TO_VOLT(raw)	(((raw) * 4800 / 1023) + 70)		// mV
#define BATT_VOLT_TO_RAW(mV)	(((mV) - 70) * 1023 / 4800)

/*
 * BATT capacity level
 */
#define BATT_CAP_CAUTION_LEVEL    2
#define BATT_CAP_SHUTDOWN_LEVEL   4
struct _batt_cap {
    int  raw;		/* BATT raw value */
    int  cap;		/* BATT capacity level (%) */
};
static struct _batt_cap batt_cap [] = {
  { 806, 90 },		/* [0] */
  { 753, 50 },		/* [1] */
  { 731, 20 },		/* [2] caution judge level */
  { 710,  8 },		/* [3] */
  { 709,  2 },		/* [4] shutdown judge level */
  {   0,  2 } };	/* [5] */

/*
 * Battery/DCinput voltage input data
 *  [0] : BATTERY(ADIN0)
 *  [1] : BPSNS  (ADIN2)
 */
#define GEN_PURPOSE_AD2 APPLICATION_SUPPLY	/* AD2 */
#define VOLT_MEASURE_INTERVAL (2 * HZ)
struct _volt_input_value {
    unsigned long jif;		/* jiffies at measure */
    int           volt;		/* voltage(mV) */
    int           raw;		/* A/D raw value */
};
static struct _volt_input_value volt_input_value [2];

/*
 * PMIC access lock flag
 */
extern int g_apm_spi_disable;

#endif /* CONFIG_MACH_MX51_SENDAI */

#if ( USE_SW_MONITOR_TIME_OUT )
//modify by pega Huimin 600 sec
static int batt_on_poll_interval = (10 * 60);
//Modify by pega Huimin by secs 60
#define	BATTERY_SYSTEM_ON_POLL_INTERVAL		10 

//Modify by pega Huimin based on batt_poll_interval/60
#define	BATTERY_SYSTEM_SUSPEND_POLL_INTERVAL	30

//Modify by pega Huimin by miniutes 30 
#define	BATTERY_TIMEOUT_LIMIT	240
#endif /* USE_SW_MONITOR_TIME_OUT */

#if ( 1 )
u32 start_time_charging = 0;
u32 cur_time_charging = 0;
int timeout_flag_charging = 0;
u32 timeout_limitation = 60 * 1000; 

int timeout_counter_charging = 0;
//int timeout_counter_limit_charging = 30*1; //modify by Huimin from 60*4 to 30*1 30 mins 
#endif

extern int isACadapter (void);


static int pmic_set_chg_current(unsigned short curr)
{
  unsigned int mask;
  unsigned int value;

  DEBUG_PRINTK(("[%s] \n",  __func__));

  value = BITFVAL(BIT_CHG_CURR, curr);
  mask = BITFMASK(BIT_CHG_CURR);
  CHECK_ERROR(pmic_write_reg(REG_CHARGE, value, mask));

  return 0;
}

static int pmic_set_chg_misc(enum chg_setting type, unsigned short flag)
{
  unsigned int reg_value = 0;
  unsigned int mask = 0;

  DEBUG_PRINTK(("[%s] \n",  __func__));

  switch (type) {
    case TRICKLE_CHG_EN:
//      printk("PMIC:TRICKLE_CHG_EN %08x\n", flag);
      reg_value = BITFVAL(TRICKLE_CHG_EN, flag);
      mask = BITFMASK(TRICKLE_CHG_EN);
      break;
    case LOW_POWER_BOOT_ACK:
//      printk("PMIC:LOW_POWER_BOOT_ACK %08x\n", flag);
      reg_value = BITFVAL(LOW_POWER_BOOT_ACK, flag);
      mask = BITFMASK(LOW_POWER_BOOT_ACK);
      break;
    case BAT_TH_CHECK_DIS:
//      printk("PMIC:BAT_TH_CHECK_DIS %08x\n", flag);
      reg_value = BITFVAL(BAT_TH_CHECK_DIS, flag);
      mask = BITFMASK(BAT_TH_CHECK_DIS);
      break;
    case BATTFET_CTL_EN:
//      printk("PMIC:BATTFET_CTL_EN %08x\n", flag);
      reg_value = BITFVAL(BATTFET_CTL_EN, flag);
      mask = BITFMASK(BATTFET_CTL_EN);
      break;
    case BATTFET_CTL:
//      printk("PMIC:BATTFET_CTL %08x\n", flag);
      reg_value = BITFVAL(BATTFET_CTL, flag);
      mask = BITFMASK(BATTFET_CTL);
      break;
    case REV_MOD_EN:
//      printk("PMIC:REV_MOD_EN %08x\n", flag);
      reg_value = BITFVAL(REV_MOD_EN, flag);
      mask = BITFMASK(REV_MOD_EN);
      break;
    case PLIM_DIS:
//      printk("PMIC:LIM_DIS %08x\n", flag);
      reg_value = BITFVAL(PLIM_DIS, flag);
      mask = BITFMASK(PLIM_DIS);
      break;
    case CHG_LED_EN:
//      printk("PMIC:CHG_LED_EN %08x\n", flag);
      reg_value = BITFVAL(CHG_LED_EN, flag);
      mask = BITFMASK(CHG_LED_EN);
      break;
    case RESTART_CHG_TIMER:
//      printk("PMIC:RESTART_CHG_TIMER %08x\n", flag);
      reg_value = BITFVAL(RESTART_CHG_TIMER, flag);
      mask = BITFMASK(RESTART_CHG_TIMER);
      break;
    case RESTART_CHG_STAT:
//      printk("PMIC:RESTART_CHG_STAT %08x\n", flag);
      reg_value = BITFVAL(RESTART_CHG_STAT, flag);
      mask = BITFMASK(RESTART_CHG_STAT);
      break;
    case AUTO_CHG_DIS:
//      printk("PMIC:AUTO_CHG_DIS %08x\n", flag);
      reg_value = BITFVAL(AUTO_CHG_DIS, flag);
      mask = BITFMASK(AUTO_CHG_DIS);
      break;
    case CYCLING_DIS:
//      printk("PMIC:CYCLING_DIS %08x\n", flag);
      reg_value = BITFVAL(CYCLING_DIS, flag);
      mask = BITFMASK(CYCLING_DIS);
      break;
    case VI_PROGRAM_EN:
//      printk("PMIC:VI_PROGRAM_EN %08x\n", flag);
      reg_value = BITFVAL(VI_PROGRAM_EN, flag);
      mask = BITFMASK(VI_PROGRAM_EN);
      break;
    default:
      return PMIC_PARAMETER_ERROR;
  }

  CHECK_ERROR(pmic_write_reg(REG_CHARGE, reg_value, mask));

  return 0;
}

#ifdef CONFIG_MACH_MX51_SENDAI

/*
 * Battery charge start [auto]
 */
int pmic_battery_start_charge (void)
{
	int  ret;
	ret = pmic_restart_charging ();
	return ret;
}

/*
 * Battery charge stop
 */
int pmic_battery_stop_charge (void)
{
	int ret;

	/*
	 * software control, current=0mA
	 */
	ret = pmic_set_chg_misc (AUTO_CHG_DIS, 1);
	if (ret != 0) {
		return ret;
	}
	ret = pmic_set_chg_current (0);
	if (ret != 0) {
		return ret;
	}
	return 0;
}

/*
 * get Battery/DCinput voltage from PMIC
 *  voltage [mV]
 */
static int pmic_get_voltage(t_channel channel, unsigned short *voltage, int *rawval,
			    int force)
{
	unsigned short            result[8];
	struct _volt_input_value *info;
	unsigned long             dif, jif = jiffies;
	int                       rc, volt;

	if (channel == BATTERY_VOLTAGE) {
		/*
		 * Battery Voltage
		 */
		info = &(volt_input_value [0]);
	} else if (channel == GEN_PURPOSE_AD2) {
		/*
		 * BPSNS Voltage
		 */
		info = &(volt_input_value [1]);
	} else {
		return -1;
	}

	if (g_apm_spi_disable) {
		/*
		 * can't access PMIC, set GOOD condition.
		 */
		if (voltage != 0) {
			*voltage = 3700;
		}
		if (rawval != 0) {
			*rawval = 3700 * 1023 / 4800;
		}
		return 0;
	}

	/*
	 * check measure timing
	 */
	if (force == 0) {
		if (info->jif <= jif) {
			dif = jif - info->jif;
		} else {
			dif = 0xffffffff - info->jif + jif;
		}
		if (dif < VOLT_MEASURE_INTERVAL) {
			/*
			 * using already measure value
			 */
			if (voltage != 0) {
				*voltage = info->volt;
			}
			if (rawval != 0) {
				*rawval = info->raw;
			}
			return 0;
		}
	}

	rc = pmic_adc_convert(channel, result);
	if (rc == 0) {
		/*
		 * scale convert(mV)
		 */
		volt       = (int)BATT_RAW_TO_VOLT(result[0]);
		info->volt = volt;
		info->raw  = result[0];
		info->jif  = jif;
		if (voltage != 0) {
			*voltage = volt;
		}
		if (rawval != 0) {
			*rawval = result[0];
		}
	}

	return rc;
}

/*
 * mc13892_battery_sample (void)
 *  battery volatge sample for average
 *   return  N - sample count
 *   return -1 - error
 */
struct _batt_sample {
    int     mode_caution;		/* mode mormal(0)/caution(1) */
    int     interval;			/* sample interval */
    int     raw;			/* sample raw A/D */
    int     volt;			/* sample volt */
    int     prev_use;			/* previous sample use count */
    int     error;			/* sample error */
    int     update;			/* sample update */
    int     ignore;			/* sample ignore */
};
static struct _batt_sample batt_sample;
static int                 batt_sample_initialized;
static int                 batt_sample_rate = 60;	/* sample rate (sec) */
static int                 batt_sample_flag;
static int mc13892_battery_sample (int online, int wait, int force)
{
	int   i, raw, rc;
	int   rdata [5], rcnt, rmin, rmax, rttl, fmin, fmax;

	/*
	 * check sample interval
	 *  mormal - 60sec, caution - 5sec
	 */
	rc = 0;
	if (batt_sample_initialized != 0) {
		if (batt_sample.mode_caution == 0) {
			batt_sample.interval++;
			if ((force == 0) &&
			    (batt_sample.interval < (batt_sample_rate / 5))) {
				/*
				 * not sample timing
				 */
				return rc;
			}
			batt_sample.interval = 0;
		}
	}

	/*
	 * BATT voltage
         *  lowest  (1)   (2)   (3)  highest
         *    *      +- average -+      *    (* ignore)
	 */
	rmin = 0x7fffffff;
	rmax = 0;
	rcnt = 0;
	for ( i = 0 ; i < 5 ; i++ ) {
		if (pmic_get_voltage (BATTERY_VOLTAGE, 0, &raw, 1) == 0) {
			/*
			 * check ignore value
			 */
			if (BATT_VOLT_TO_RAW(IGNORE_SAMPLE_VOLT) <= raw) {
				rdata [rcnt++] = raw;
				if (raw < rmin) {
					rmin = raw;
				} else if (rmax < raw) {
					rmax = raw;
				}
			} else {
				batt_sample.ignore++;
			}
		} else {
			batt_sample.error++;
		}
		if (wait != 0) {
			mdelay ( 10 );	/* 10ms measure interval */
		}
	}
	rttl = 0;
	fmin = 0;
	fmax = 0;
	for ( i = 0 ; i < rcnt ; i++ ) {
		if ((fmin == 0) && (rmin == rdata [i])) {
			fmin++;		/* lowest value ignore */
		} else if ((fmax == 0) && (rmax == rdata [i])) {
			fmax++;		/* highest value ignore */
		} else {
			rttl += rdata [i];
		}
	}
	if (0 < (rcnt - fmin - fmax)) {
		raw = rttl / (rcnt - fmin - fmax);
		batt_sample.update++;
		rc = rcnt;
	} else {
		raw = batt_sample.raw;
		batt_sample.prev_use++;
		rc = -1;
	}
	if ((batt_sample_flag == 1) && (rcnt != 5)) {
		printk ("BATT: (%d) %d(%d) %d(%d) %d(%d) %d(%d) %d(%d)\n", rc,
			BATT_RAW_TO_VOLT(rdata [0]), rdata [0],
			BATT_RAW_TO_VOLT(rdata [1]), rdata [1],
			BATT_RAW_TO_VOLT(rdata [2]), rdata [2],
			BATT_RAW_TO_VOLT(rdata [3]), rdata [3],
			BATT_RAW_TO_VOLT(rdata [4]), rdata [4]);
	}

	/*
	 * check BATT use(AC adapter disconnect)
	 *  voltage of BATT use is decrease only.
	 */
	if ((online == 0) && (batt_sample_initialized != 0) && (batt_sample.raw < raw)) {
		if (batt_sample_flag == 1) {
			printk ("BATT: (%d) %d(%d) / %d(%d)[prev value]\n", rc,
				BATT_RAW_TO_VOLT(raw), raw,
				BATT_RAW_TO_VOLT(batt_sample.raw), batt_sample.raw);
		}
		raw = batt_sample.raw;	/* prev sample use */
	} else {
		if (batt_sample_flag == 1) {
			printk ("BATT: (%d) %d(%d)\n", rc, BATT_RAW_TO_VOLT(raw), raw);
		}
	}

	/*
	 * check normal/caution band
	 */
	if (raw <= batt_cap [BATT_CAP_CAUTION_LEVEL].raw) {
		batt_sample.mode_caution = 1;
	} else {
		batt_sample.mode_caution = 0;
	}

	batt_sample.raw = raw;
	/*
	 * Convert voltage
	 */
	batt_sample.volt = BATT_RAW_TO_VOLT(raw);

	batt_sample_initialized = 1;

	return rc;
}

/*
 * pmic_get_battery_raw (unsigned short *voltage, int *raw)
 *  sampled battery voltage & raw input
 */
static int pmic_get_battery_raw (unsigned short *voltage, int *raw)
{
	if (voltage != 0) {
		*voltage = batt_sample.volt;
	}
	if (raw != 0) {
		*raw = batt_sample.raw;
	}
	return 0;
}

/*
 * measure BATT voltage(A/D raw data)
 */
static int pmic_get_batt_voltage(unsigned short *voltage)
{
	int rc, raw;
	rc = pmic_get_battery_raw (0, &raw);
	if ((rc == 0) && ( voltage )) {
		*voltage = (unsigned short)raw;
	}
	return rc;
}

#else /* CONFIG_MACH_MX51_SENDAI */

static int pmic_get_batt_voltage(unsigned short *voltage)
{
  t_channel channel;
  unsigned short result[8];

  channel = BATTERY_VOLTAGE;
  CHECK_ERROR(pmic_adc_convert(channel, result));
  *voltage = result[0];

  DEBUG_PRINTK(("[%s] voltage = %d \n",  __func__, *voltage));
  return 0;
}

#endif /* CONFIG_MACH_MX51_SENDAI */

static int pmic_get_batt_current(unsigned short *curr)
{
  t_channel channel;
  unsigned short result[8];

  channel = BATTERY_CURRENT;
  CHECK_ERROR(pmic_adc_convert(channel, result));
  *curr = result[0];

  DEBUG_PRINTK(("[%s] current = %d \n",  __func__, *curr));
  return 0;
}

static int coulomb_counter_calibration;
static unsigned int coulomb_counter_start_time_msecs;

static int pmic_start_coulomb_counter(void)
{
  DEBUG_PRINTK(("[%s] \n",  __func__));

  /* set scaler */
  CHECK_ERROR(pmic_write_reg(REG_ACC1,
        ACC_COULOMB_PER_LSB * ACC_ONEC_VALUE, BITFMASK(ACC1_ONEC)));

  CHECK_ERROR(pmic_write_reg(
        REG_ACC0, ACC_START_COUNTER, ACC_CONTROL_BIT_MASK));
  coulomb_counter_start_time_msecs = jiffies_to_msecs(jiffies);
  pr_debug("coulomb counter start time %u\n",
      coulomb_counter_start_time_msecs);
  return 0;
}

static int pmic_stop_coulomb_counter(void)
{
  DEBUG_PRINTK(("[%s] \n",  __func__));

  CHECK_ERROR(pmic_write_reg(
        REG_ACC0, ACC_STOP_COUNTER, ACC_CONTROL_BIT_MASK));
  return 0;
}

static int pmic_calibrate_coulomb_counter(void)
{
  int ret;
  unsigned int value;

  /* set scaler */
  CHECK_ERROR(pmic_write_reg(REG_ACC1,
        0x1, BITFMASK(ACC1_ONEC)));

  CHECK_ERROR(pmic_write_reg(
        REG_ACC0, ACC_CALIBRATION, ACC_CONTROL_BIT_MASK));
  msleep(ACC_CALIBRATION_DURATION_MSECS);

  ret = pmic_read_reg(REG_ACC0, &value, BITFMASK(ACC_CCOUT));
  if (ret != 0)
    return -1;
  value = BITFEXT(value, ACC_CCOUT);
  pr_debug("calibrate value = %x\n", value);
  coulomb_counter_calibration = (int)((s16)((u16) value));
  pr_debug("coulomb_counter_calibration = %d\n",
      coulomb_counter_calibration);

  return 0;

}

static int pmic_get_charger_coulomb(int *coulomb)
{
  int ret;
  unsigned int value;
  int calibration;
  unsigned int time_diff_msec;

  ret = pmic_read_reg(REG_ACC0, &value, BITFMASK(ACC_CCOUT));
  if (ret != 0)
    return -1;
  value = BITFEXT(value, ACC_CCOUT);
  pr_debug("counter value = %x\n", value);
  *coulomb = ((s16)((u16)value)) * ACC_COULOMB_PER_LSB;

  if (abs(*coulomb) >= ACC_COULOMB_PER_LSB) {
    /* calibrate */
    time_diff_msec = jiffies_to_msecs(jiffies);
    time_diff_msec =
      (time_diff_msec > coulomb_counter_start_time_msecs) ?
      (time_diff_msec - coulomb_counter_start_time_msecs) :
      (0xffffffff - coulomb_counter_start_time_msecs
       + time_diff_msec);
    calibration = coulomb_counter_calibration * (int)time_diff_msec
      / (ACC_ONEC_VALUE * ACC_CALIBRATION_DURATION_MSECS);
    *coulomb -= calibration;
  }

  DEBUG_PRINTK(("[%s] value = %d , coulomb = %d \n",  __func__, value, *coulomb));
  return 0;
}

static int pmic_restart_charging(void)
{
  printk("%s\n",__func__);

  //add by pega Huimin 2010.01.29, to fix the charger LED issue
#if ( USE_CHARGER_LED_FIX )
  pmic_set_chg_misc(CHG_LED_EN, 1);
#endif

#ifdef USE_BATTERY_TRICKLE_CHARGE
  pmic_set_chg_misc(TRICKLE_CHG_EN, 1);
#endif /* USE_BATTERY_TRICKLE_CHARGE */

#ifdef USE_BATTERY_THERMISTOR_CHECK
  pmic_set_chg_misc(BAT_TH_CHECK_DIS, 0);
  pmic_set_chg_misc(AUTO_CHG_DIS, 0);
  pmic_set_chg_misc(VI_PROGRAM_EN, 1); // set not auto charge
  pmic_set_chg_current(CHARGE_CURRENT); //ICHRG[3:0] change from 1000 (720mA) to 0110(520mA)
#else /* USE_BATTERY_THERMISTOR_CHECK */
  pmic_set_chg_misc(BAT_TH_CHECK_DIS, 1);
  pmic_set_chg_misc(AUTO_CHG_DIS, 0);
#if ( 1 ) // modify by pega Huimin 2009.11.18
  pmic_set_chg_misc(VI_PROGRAM_EN, 0); // set auto charge
  pmic_set_chg_current(CHARGE_CURRENT); //ICHRG[3:0] change from 1000 (720mA) to 0110(520mA)
#else
  pmic_set_chg_misc(VI_PROGRAM_EN, 1); // set not auto charge
  pmic_set_chg_current(0x8);           // set charge current 720 mA = ICHRG[3:0] = 8
#endif
#endif /* USE_BATTERY_THERMISTOR_CHECK */

  pmic_set_chg_misc(RESTART_CHG_STAT, 1);
  return 0;
}
//Modify by pega Huimin
#if ( USE_SW_MONITOR_TIME_OUT )
static int pmic_charger_removal(void)
{
  printk("%s\n",__func__);

  //add by pega Huimin 2010.01.29, to fix the charger LED issue
#if ( USE_CHARGER_LED_FIX )
  pmic_set_chg_misc(CHG_LED_EN, 0);
#endif

  pmic_set_chg_misc(AUTO_CHG_DIS, 0);	
  pmic_set_chg_misc(BATTFET_CTL_EN, 0);
  pmic_set_chg_misc(BATTFET_CTL, 0);
  return 0;
}
#endif

struct mc13892_dev_info {
  struct device *dev;

  unsigned short voltage_raw;
  int voltage_uV;
  unsigned short current_raw;
  int current_uA;
  int battery_status;
  int full_counter;
  int charger_online;
  int charger_voltage_uV;
  int accum_current_uAh;

  struct power_supply bat;
  struct power_supply charger;

  struct workqueue_struct *monitor_wqueue;
  struct delayed_work monitor_work;

#if ( USE_EVENT_VBUSVI ) // add by pega Kevin 2009.12.01 - use EVENT_VBUSVI = VBUS valid detect
  struct workqueue_struct *vbus_wqueue;
  struct delayed_work vbus_work;
#endif
#if ( USE_SW_MONITOR_TIME_OUT )
  struct workqueue_struct *sw_monitor_wqueue;
  struct delayed_work sw_monitor_work;
#else
  struct workqueue_struct *sw_monitor_wqueue;
  struct delayed_work sw_monitor_work;
#endif
};

#define mc13892_SENSER	25
#define to_mc13892_dev_info(x) container_of((x), struct mc13892_dev_info, \
    bat);

static enum power_supply_property mc13892_battery_props[] = {
  POWER_SUPPLY_PROP_VOLTAGE_NOW,
  POWER_SUPPLY_PROP_CURRENT_NOW,
  POWER_SUPPLY_PROP_CHARGE_NOW,
  POWER_SUPPLY_PROP_STATUS,
#ifdef CONFIG_MACH_MX51_SENDAI
  POWER_SUPPLY_PROP_CAPACITY,
#endif /* CONFIG_MACH_MX51_SENDAI */
};

static enum power_supply_property mc13892_charger_props[] = {
  POWER_SUPPLY_PROP_ONLINE,
#ifdef CONFIG_MACH_MX51_SENDAI
  POWER_SUPPLY_PROP_VOLTAGE_NOW,
  POWER_SUPPLY_PROP_CAPACITY,
  POWER_SUPPLY_PROP_STATUS,
#endif /* CONFIG_MACH_MX51_SENDAI */
};

static int mc13892_charger_update_status(struct mc13892_dev_info *di)
{
  int ret;
  unsigned int value;
  int online;
  int ac = isACadapter ();

  DEBUG_PRINTK(("[%s] \n",  __func__));

  ret = pmic_read_reg(REG_INT_SENSE0, &value, BITFMASK(BIT_CHG_DETS));

  if (ret == 0) {
    online = BITFEXT(value, BIT_CHG_DETS);
    /*
     * check AC adapter use
     */
    if (ac == 0) {
      online = 0;
    }
    if (online != di->charger_online) {
      di->charger_online = online;
      if (di->charger.dev == 0) {
        printk ("mc13892_charger_update_status: charger status: %s\n",
                online ? "online" : "offline");
      } else {
      dev_info(di->charger.dev, "charger status: %s\n",
          online ? "online" : "offline");

      // add by pega Kevin 2009.11.24 - for USB connect PC
#if ( USE_EVENT_CHGDETI )
      if ( online )
        kobject_uevent(&di->charger.dev->kobj, KOBJ_ONLINE); // include/linux/kobject.h
      else
        kobject_uevent(&di->charger.dev->kobj, KOBJ_OFFLINE);
#endif

      power_supply_changed(&di->charger);
      }
      cancel_delayed_work(&di->monitor_work);
      queue_delayed_work(di->monitor_wqueue,
          &di->monitor_work, HZ / 10);
      if (online) {
        pmic_start_coulomb_counter();
        pmic_restart_charging();
      } else
        pmic_stop_coulomb_counter();
      //Modify by pega Huimin
#if ( USE_SW_MONITOR_TIME_OUT )
      if (!online)
        pmic_charger_removal();

#if ( 1 )
      if (online) {
        printk("%s:sw_monitor_work start\n",__func__);
        cancel_delayed_work(&di->sw_monitor_work);
        if (pmic_battery_boot_flag)
        {
          pmic_battery_boot_flag = 0;
          queue_delayed_work(di->sw_monitor_wqueue, &di->sw_monitor_work, 20*HZ);
        }
        else
          queue_delayed_work(di->sw_monitor_wqueue, &di->sw_monitor_work, 2*HZ);
      }
#endif
#endif

#if ( 1 )
      if ( !online ) 
      {
        start_time_charging = 0;
        timeout_flag_charging = 0;
        timeout_counter_charging = 0;
      }
      else
      {
        if (!start_time_charging)
          start_time_charging = jiffies_to_msecs(jiffies);
      }
#endif

    }
  }

  return ret;
}

static int mc13892_charger_get_property(struct power_supply *psy,
    enum power_supply_property psp,
    union power_supply_propval *val)
{
  struct mc13892_dev_info *di =
    container_of((psy), struct mc13892_dev_info, charger);

  DEBUG_PRINTK(("[%s] \n",  __func__));

  switch (psp) {
    case POWER_SUPPLY_PROP_ONLINE:
      val->intval = di->charger_online;
      return 0;
#ifdef CONFIG_MACH_MX51_SENDAI
    case POWER_SUPPLY_PROP_STATUS:
      if ( di->charger_online == 0 ) {
        val->intval = POWER_SUPPLY_STATUS_DISCHARGING;
      } else {
        unsigned int value;
        int retval = pmic_read_reg(REG_INT_SENSE0, &value, BITFMASK(BIT_CHG_CURRS));
        if (retval == 0) {
          if ( BITFEXT(value, BIT_CHG_CURRS) ) {
            val->intval = POWER_SUPPLY_STATUS_CHARGING;
          } else {
            val->intval = POWER_SUPPLY_STATUS_FULL;
          }
        } else {
          val->intval = POWER_SUPPLY_STATUS_NOT_CHARGING;
        }
      }
      return 0;
      break;
    case POWER_SUPPLY_PROP_VOLTAGE_NOW:
      {
        unsigned short mV = 0;
        pmic_get_battery_raw (&mV, 0);
        val->intval = mV;
      }
      return 0;
      break;
    case POWER_SUPPLY_PROP_CAPACITY:
      {
        /*
         * BATT life temp
         */
        unsigned short    mV = 0;
        int               raw = batt_cap [0].raw;
        struct _batt_cap *tbl = &(batt_cap [0]);
        int               i, mx;
        pmic_get_battery_raw (&mV, &raw);
        mx = sizeof(batt_cap) / sizeof(struct _batt_cap);
        for ( i = 0 ; i < mx ; i++ , tbl++ ) {
          if (tbl->raw <= raw) {
            val->intval = tbl->cap;
            break;
          }
        }
      }
      return 0;
      break;
#endif /* CONFIG_MACH_MX51_SENDAI */
    default:
      break;
  }
  return -EINVAL;
}

#ifdef CONFIG_MACH_MX51_SENDAI
static volatile int mc13892_battery_read_status_flag = 0;
#endif /* CONFIG_MACH_MX51_SENDAI */
static int mc13892_battery_read_status(struct mc13892_dev_info *di)
{
  int retval;
  int coulomb;

  DEBUG_PRINTK(("[%s] \n",  __func__));

  retval = pmic_get_batt_voltage(&(di->voltage_raw));
  if (retval == 0)
#ifdef CONFIG_MACH_MX51_SENDAI
    di->voltage_uV = BATT_RAW_TO_uV(di->voltage_raw);
#else
    di->voltage_uV = di->voltage_raw * BAT_VOLTAGE_UNIT_UV;
#endif /* CONFIG_MACH_MX51_SENDAI */

  retval = pmic_get_batt_current(&(di->current_raw));
  if (retval == 0) {
    if (di->current_raw & 0x200)
      di->current_uA =
        (0x1FF - (di->current_raw & 0x1FF)) *
        BAT_CURRENT_UNIT_UA * (-1);
    else
      di->current_uA =
        (di->current_raw & 0x1FF) * BAT_CURRENT_UNIT_UA;
  }
  retval = pmic_get_charger_coulomb(&coulomb);
  if (retval == 0)
    di->accum_current_uAh = COULOMB_TO_UAH(coulomb);

#ifdef CONFIG_MACH_MX51_SENDAI
  if (mc13892_battery_read_status_flag == 1) {
    printk ("battery_read_status[%s]: %duV %duA %duAh\n",
            current->comm, di->voltage_uV, di->current_uA, di->accum_current_uAh);
    }
#endif /* CONFIG_MACH_MX51_SENDAI */
  return retval;
}



static void mc13892_battery_update_status(struct mc13892_dev_info *di)
{
  unsigned int value;
  int retval;
  int old_battery_status = di->battery_status;

  DEBUG_PRINTK(("[%s] \n",  __func__));


  if (di->battery_status == POWER_SUPPLY_STATUS_UNKNOWN)
    di->full_counter = 0;

  if (di->charger_online) {
    retval = pmic_read_reg(REG_INT_SENSE0,
        &value, BITFMASK(BIT_CHG_CURRS));

    if (retval == 0) {
      value = BITFEXT(value, BIT_CHG_CURRS);
      if (value)
        di->battery_status =
          POWER_SUPPLY_STATUS_CHARGING;
      else
#ifdef CONFIG_MACH_MX51_SENDAI
        di->battery_status =
          POWER_SUPPLY_STATUS_FULL;
#else
        di->battery_status =
          POWER_SUPPLY_STATUS_NOT_CHARGING;
#endif /* CONFIG_MACH_MX51_SENDAI */
    }

    if (di->battery_status == POWER_SUPPLY_STATUS_NOT_CHARGING)
      di->full_counter++;
    else
      di->full_counter = 0;
  } else {
    di->battery_status = POWER_SUPPLY_STATUS_DISCHARGING;
    di->full_counter = 0;
  }

  dev_dbg(di->bat.dev, "bat status: %d\n",
      di->battery_status);

  if (di->battery_status != old_battery_status)
    power_supply_changed(&di->bat);
}

static void mc13892_battery_work(struct work_struct *work)
{
  struct mc13892_dev_info *di = container_of(work,
      struct mc13892_dev_info,
      monitor_work.work);
#ifdef CONFIG_MACH_MX51_SENDAI
  const int interval = HZ * 5;

  mc13892_charger_update_status(di);
  mc13892_battery_sample(di->charger_online, 1, 0);
#else
  const int interval = HZ * 60;
#endif /* CONFIG_MACH_MX51_SENDAI */

  DEBUG_PRINTK(("[%s] \n",  __func__));

  dev_dbg(di->dev, "%s\n", __func__);

  mc13892_battery_update_status(di);
  queue_delayed_work(di->monitor_wqueue, &di->monitor_work, interval);
}

#if ( USE_EVENT_VBUSVI ) // add by pega Kevin 2009.12.01 - use EVENT_VBUSVI = VBUS valid detect
static void vbus_detect_work(struct work_struct *work)
{
  struct mc13892_dev_info *di = container_of(work,
      struct mc13892_dev_info,
      monitor_work.work);

  unsigned int wall_charger_detect;
  unsigned int vbus_valid_sense;

  pmic_read_reg(REG_INT_SENSE0, &vbus_valid_sense, ( 1 << 3 ));      // REG_INT_SENSE0 = 2[3] = VBUSVALIDI sense bit
  pmic_read_reg(REG_INT_STATUS0, &wall_charger_detect, ( 1 << 21 )); // REG_INT_STATUS0 = 0[21] = CHRGSE1BI = wall charger detect
  pmic_write_reg(REG_INT_STATUS0, wall_charger_detect, ( 1 << 21 )); // clean CHRGSE1BI

  printk("REG_INT_SENSE0[3] = vbus_valid_sense = %08x \n", vbus_valid_sense);
  printk("REG_INT_STATUS0[21] = wall_charger_detect = %08x \n", wall_charger_detect);

  if ( !wall_charger_detect ) // connect to PC
  {
    if ( vbus_valid_sense )
      kobject_uevent(&di->charger.dev->kobj, KOBJ_ONLINE); // include/linux/kobject.h
    else
      kobject_uevent(&di->charger.dev->kobj, KOBJ_OFFLINE);
  }
  cancel_delayed_work(&di->vbus_work);
}

static void vbus_detect_callback(void *para)
{
  extern int usb_otg_init_flag;

  struct mc13892_dev_info *di = (struct mc13892_dev_info *) para;

  if ( usb_otg_init_flag == 1 ) // usb otg driver ready , can read USB ID pin
  {
    if ( USBOTG_REG32(0x1A4) & OTGSC_STS_USB_ID ) // read USB ID pin
      queue_delayed_work(di->vbus_wqueue, &di->vbus_work, HZ);
  }
}
#endif

static void charger_online_event_callback(void *para)
{
#if ( 1 ) // modify by pega Kevin 2009.11.27 - for USB OTG
  extern int usb_otg_init_flag;
#endif

  struct mc13892_dev_info *di = (struct mc13892_dev_info *) para;
  pr_info("\n\n DETECTED charger plug/unplug event\n");

  DEBUG_PRINTK(("[%s] \n",  __func__));

#if ( 1 ) // modify by pega Kevin 2009.11.27 - for USB OTG
  if ( usb_otg_init_flag == 0 )	// usb otg driver not ready , can't read USB ID pin
  {
    mc13892_charger_update_status(di);
  }
  else if ( USBOTG_REG32(0x1A4) & OTGSC_STS_USB_ID ) // read USB ID pin
  {
    mc13892_charger_update_status(di);
  }
  //add by pega Huimin 2009/12/11 to switch off the charging LED when mini-USB device in	
#if ( 1 )	
  else
  {
    pmic_set_chg_misc(CHG_LED_EN, 0);
  }
#endif	
#else
  mc13892_charger_update_status(di);
#endif
}

static int mc13892_battery_get_property(struct power_supply *psy,
    enum power_supply_property psp,
    union power_supply_propval *val)
{
  struct mc13892_dev_info *di = to_mc13892_dev_info(psy);

  DEBUG_PRINTK(("[%s] \n",  __func__));

  switch (psp) {
    case POWER_SUPPLY_PROP_STATUS:
      if (di->battery_status == POWER_SUPPLY_STATUS_UNKNOWN) {
        mc13892_charger_update_status(di);
        mc13892_battery_update_status(di);
      }
      val->intval = di->battery_status;
      return 0;
    default:
      break;
  }

  mc13892_battery_read_status(di);

  switch (psp) {
    case POWER_SUPPLY_PROP_VOLTAGE_NOW:
#ifdef CONFIG_MACH_MX51_SENDAI
      val->intval = di->voltage_uV / 1000;	/* (mV) */
#else
      val->intval = di->voltage_uV;
#endif /* CONFIG_MACH_MX51_SENDAI */
      break;
#ifdef CONFIG_MACH_MX51_SENDAI
    case POWER_SUPPLY_PROP_CAPACITY:
      {
        /*
         * BATT life temp
         */
        unsigned short    mV = 0;
        int               raw = batt_cap [0].raw;
        struct _batt_cap *tbl = &(batt_cap [0]);
        int               i, mx;
        pmic_get_battery_raw (&mV, &raw);
        mx = sizeof(batt_cap) / sizeof(struct _batt_cap);
        for ( i = 0 ; i < mx ; i++ , tbl++ ) {
          if (tbl->raw <= raw) {
            val->intval = tbl->cap;
            break;
          }
        }
      }
      break;
    case POWER_SUPPLY_PROP_CURRENT_NOW:
      val->intval = di->current_uA;
      break;
    case POWER_SUPPLY_PROP_CHARGE_NOW:
      val->intval = di->accum_current_uAh;
      break;
#else /* CONFIG_MACH_MX51_SENDAI */
    case POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN:
      val->intval = 3800000;
      break;
    case POWER_SUPPLY_PROP_VOLTAGE_MIN_DESIGN:
      val->intval = 3300000;
      break;
#endif /* CONFIG_MACH_MX51_SENDAI */
    default:
      return -EINVAL;
  }

  return 0;
}

//Modify by pega Huimin 2009/12/8 USE_BATTERY_LOW_INTERRUPT mechanism from spider 1
#if ( USE_BATTERY_LOW_INTERRUPT )
/*
 * battery_create_inputdevice : create BATTERY key
 */
static struct input_dev *battery_input_dev;
static void battery_create_inputdevice (void)
{
  int                      i, ret, inited;
  struct input_dev        *input = 0;
  struct _batt_input_stat *tbl;

  inited = 0;
  tbl = &(batt_input_stat [0]);
  for ( i = 0 ; i < 2 ; i++ , tbl++ ) {
    if (0 < tbl->event_type) {
      if (inited == 0) {
        input = input_allocate_device ();
        if (input == 0) {
          printk ("battery_create_inputdevice: can't create input device\n");
          return;
        }
        input->name = BATTERY_INPUT_NAME;
        inited = 1;
      }
      input_set_capability (input, tbl->event_type, tbl->event_code);
    }
  }
  if (inited == 1) {
    ret = input_register_device (input);
    if (ret != 0) {
      printk ("battery_create_inputdevice: can't register input device\n");
      input_free_device (input);
    } else {
      battery_input_dev = input;
    }
  }
  return;
}

/*
 * battery_remove_inputdevice : delete BATTERY key
 */
static void battery_remove_inputdevice (void)
{
  if (battery_input_dev != 0) {
    input_unregister_device (battery_input_dev);
    input_free_device (battery_input_dev);
  }
}

/*
 * battery_input_event : event report
 */
static void battery_input_event (struct _batt_input_stat *ev)
{
  if ((battery_input_dev != 0) && (0 < ev->event_type)) {
    input_event (battery_input_dev, ev->event_type, ev->event_code, 1);
    input_sync (battery_input_dev);
    input_event (battery_input_dev, ev->event_type, ev->event_code, 0);
    input_sync (battery_input_dev);
  }
}

/*
 * LOBATLI interrupt
 */
static void battery_lobatli_callback (void *para)
{
  unsigned int value;
  int          ret;

  /*
   * check BATLI level lower
   */
  ret = pmic_read_reg (REG_INT_SENSE0, &value, 0x00ffffff);
  printk ("battery_lobatli_callback: reg2 (%d %x)\n", ret, value);
  if ((ret == PMIC_SUCCESS) && ((value & (0x1 << SENSE_LOBATLS)) != 0x00000000)) {
    battery_input_event (&(batt_input_stat [0]));
  }
}

/*
 * LOBATHI interrupt
 */
static void battery_lobathi_callback (void *para)
{
  unsigned int value;
  int          ret;

  /*
   * check BATHI level lower
   */
  ret = pmic_read_reg (REG_INT_SENSE0, &value, 0x00ffffff);
  printk ("battery_lobathi_callback: reg2 (%d %x)\n", ret, value);
  if ((ret == PMIC_SUCCESS) && ((value & (0x1 << SENSE_LOBATHS)) == 0x00000000)) {

    /*
     * caution mode
     */
    batt_sample.interval = 999;
    battery_input_event (&(batt_input_stat [1]));
  }
}

/*
 * Battery low suspend
 */
static void battery_low_suspend (void)
{
  /*
   * interrupt enable LOBATLI only
   */
  extern int pmic_event_for_suspend (type_event event);
  pmic_event_for_suspend (EVENT_LOBATLI);
#if ( USE_SOFT_WITH_AUTO )
  pmic_set_chg_misc(AUTO_CHG_DIS, 0);
#endif

  //Modify by pega Huimin no need in spider 2
#if ( 0 )
  /*
   * DCinput Power mode
   */
  gpio_battery_enable ( 0 );
#endif
}

/*
 * Battery low resume
 */
static void battery_low_resume (void)
{
  /*
   * interrupt restore
   */
  extern int pmic_event_for_resume (void);
  pmic_event_for_resume ();
}

/*
 * Battery low check initial
 */
static void battery_low_init (struct mc13892_dev_info *di)
{
  pmic_event_callback_t bat_event_callback; //Modify by pega Huimin add the definition in spider 2
  int retval = 0;


  //Modify by pega Huimin 2009/12/11
#if ( 0 )
  /*
   * LOBATLI/HI level setting
   *  BPSNS1/0 0:0 -> 2.8V 3.0V
   */
  retval = pmic_write_reg (REG_POWER_CTL0, 0x00000000, 0x00030000);
#else
  /*
   * Change the LOBATLI/HI level setting demanded by EE Charles
   *  BPSNS1/0 1:1 -> 3.1V 3.4V
   */
  retval = pmic_write_reg (REG_POWER_CTL0, 0x00030000, 0x00030000);

#endif

  if (retval != PMIC_SUCCESS) {
    printk ("pmic_battery_probe: can't set BPSNS1/0 %d\n", retval);
  }
  battery_create_inputdevice ();
  bat_event_callback.func  = battery_lobatli_callback;
  bat_event_callback.param = (void *) di;
  pmic_event_subscribe (EVENT_LOBATLI, bat_event_callback);

  bat_event_callback.func  = battery_lobathi_callback;
  bat_event_callback.param = (void *) di;
  pmic_event_subscribe (EVENT_LOBATHI, bat_event_callback);
}

/*
 * Battery low check remove
 */
static void battery_low_remove (struct mc13892_dev_info *di)
{
  pmic_event_callback_t bat_event_callback;

  bat_event_callback.func  = battery_lobatli_callback;
  bat_event_callback.param = (void *) di;
  pmic_event_unsubscribe(EVENT_LOBATLI, bat_event_callback);

  bat_event_callback.func  = battery_lobathi_callback;
  bat_event_callback.param = (void *) di;
  pmic_event_unsubscribe(EVENT_LOBATHI, bat_event_callback);
  battery_remove_inputdevice ();
}
#endif /* USE_BATTERY_LOW_INTERRUPT */





#if ( USE_BATTERY_LOW_POLLING)

extern void resume_power_sw_clean (void);

static volatile int battery_poll_statistics [16];
static volatile int battpollC;
static volatile int battpoll [260][4];
static void LogIn (int a, int b, int c)
{
  int *p;
  if (256 <= battpollC)
    battpollC = 0;
  p = (int *)&(battpoll [battpollC++][0]);
  p [0] = jiffies;
  p [1] = a;
  p [2] = b;
  p [3] = c;
}

/*
 * PMIC alarm set
 */
static int battery_set_alarm (void)
{
  int          rc;
  unsigned int toda, daya;


  /*
   * wakeup after check interval
   */
  rc = pmic_read_reg (REG_RTC_TIME, &toda, 0x01FFFF);
  if (rc == 0) {
    rc = pmic_read_reg (REG_RTC_DAY, &daya, 0x007FFF);
  }
  if (rc != 0) {
    printk ("battery_set_alarm: get TODA/DAYA error %d\n", rc);
    return -1;
  }
  toda += batt_poll_interval;
  if ((24 * 60 * 60) <= toda) {
    toda -= (24 * 60 * 60);
    daya++;
  }
  rc = pmic_write_reg (REG_RTC_ALARM, toda, 0x01FFFF);
  if (rc == 0) {
    rc = pmic_write_reg (REG_RTC_DAY_ALARM, daya, 0x007FFF);
  }
  if (rc != 0) {
    printk ("battery_set_alarm: set TODA/DAYA error %d\n", rc);
    return -1;
  }
  return 0;
}

#if ( USE_SOFT_WITH_AUTO )
/*
 * PMIC battery check
 */
static int battery_poll_voltage_check (void)
{
//  printk("PMIC:%s:BATTCHECK\n",__func__);
    if (pmic_get_batt_voltage(&(batt_poll_di->voltage_raw)))
      printk("PMIC:%s:get_batt_voltage\n",__func__);

    batt_poll_di->voltage_uV = BATT_RAW_TO_uV(batt_poll_di->voltage_raw);
    if (batt_poll_di->voltage_uV <= BAT_VOLTAGE_LOWER) {
      pmic_set_chg_misc(RESTART_CHG_TIMER, 1);
      pmic_set_chg_misc(RESTART_CHG_STAT, 1);
//      printk ("PMIC:BAT_LOW %s: %duV\n", __func__, batt_poll_di->voltage_uV);
    } else {
//      printk ("PMIC:BAT_HIGH %s: %duV\n", __func__, batt_poll_di->voltage_uV);
    }
}
#endif

/*
 * poll wake interrupt
 */
static void battery_poll_callback (void *para)
{

  battery_poll_statistics [0]++;

  /* check battery voltage */
#if ( USE_SOFT_WITH_AUTO )
  battery_poll_voltage_check();
#endif

  /*
   * wakeup after check interval
   */
  battery_set_alarm ();
}

/*
 * Battery low suspend
 *  called by PM(suspend)
 */
extern int pmic_event_for_suspend (type_event event);
extern int pmic_event_for_resume (void);
static unsigned int pre_alarm [3];
static void battery_low_suspend (void)
{
  int  rc;
  pmic_event_callback_t bat_event_callback;


  battery_poll_statistics [1]++;
  battery_poll_statistics [6] = 0;
  LogIn (0x00010000, battery_poll_statistics [1], battery_poll_statistics [0]);
  /*
   * previous alarm setting
   */
  pre_alarm [0] = 1;
  pre_alarm [1] = 0x01FFFF;
  pre_alarm [2] = 0x007FFF;
  pmic_read_reg (REG_RTC_ALARM,     &(pre_alarm [1]), 0x01FFFF);
  pmic_read_reg (REG_RTC_DAY_ALARM, &(pre_alarm [2]), 0x007FFF);

  /*
   * wakeup after check interval
   */
  rc = battery_set_alarm ();
  if (rc == 0) {
    /*
     * interrupt enable TODAI only
     */
    bat_event_callback.func  = battery_poll_callback;
    bat_event_callback.param = (void *)0;
    pmic_event_subscribe (EVENT_TODAI, bat_event_callback);
    pmic_event_for_suspend (EVENT_TODAI);
  }

#if ( 0 )
  /*
   * DCinput Power mode
   */
  gpio_battery_enable ( 0 );
#endif

  /*
   * cleanup Power-SW flag
   */
  resume_power_sw_clean ();


}

/*
 * Battery low cancel_suspend
 *  cancel suspend mode
 */
static void battery_low_suspend_cancel (void)
{
  int        rc;
  pmic_event_callback_t bat_event_callback;

  /*
   * interrupt restore
   */
  pmic_event_for_resume ();

  /*
   * restore alarm
   */
  if (pre_alarm [0] == 1) {
    pre_alarm [0] = 0;
    rc = pmic_write_reg (REG_RTC_ALARM, pre_alarm [1], 0x01FFFF);
    if (rc == 0) {
      rc = pmic_write_reg (REG_RTC_DAY_ALARM, pre_alarm [2], 0x007FFF);
    }
    if (rc != 0) {
      printk ("battery_low_resume: restore TODA/DAYA error %d\n", rc);
    }

    bat_event_callback.func  = battery_poll_callback;
    bat_event_callback.param = (void *)0;
    pmic_event_unsubscribe (EVENT_TODAI, bat_event_callback);
  }
}

/*
 * Battery low resume
 *  called by PM(resume)
 */
static void battery_low_resume (void)
{
  battery_poll_statistics [2]++;
  LogIn (0x00020000, battery_poll_statistics [2], battery_poll_statistics [0]);
  battery_low_suspend_cancel ();
  resume_power_sw_clean ();

}

/*
 * check resume condition
 *  called by exit suspend
 *   return 0 - re-suspend
 *   return 1 - wakeup
 */
extern struct platform_driver pmic_adc_driver_ldm;
static int battery_low_resume_check (void)
{

#if ( USE_SW_MONITOR_TIME_OUT )
#if ( USE_SOFT_WITH_AUTO )
  // no definition
#else
  //add by pega Huimin 2010,0316
  int time_diff;

  //add by pega Huimin 2010.1.4
  int ret;
  int value;
#endif /* USE_SOFT_WITH_AUTO */
#endif /* USE_SW_MONITOR_TIME_OUT */




  extern int is_resume_power_sw (void);
  int rc, raw;
  int resume_by_sw = is_resume_power_sw ();


  battery_poll_statistics [3]++;
  battery_poll_statistics [6]++;
  /*
   * check resume by Power-SW
   */
  if (resume_by_sw != 0) {
    battery_low_suspend_cancel ();
    battery_poll_statistics [4]++;
    LogIn (0x00070099, (battery_poll_statistics [4]<<16)|(battery_poll_statistics [6]&0xFFFF), resume_by_sw);
    return 1;		/* wakeup by Power-SW */
  }

  /*
   * check BATTERY shutdown voltage
   */
  if (pmic_adc_driver_ldm.resume != 0) {
    (*pmic_adc_driver_ldm.resume)((struct platform_device *)0);
  }
  pmic_event_for_suspend (EVENT_ADCBISDONEI);	/* ONLY enable ADC interrupt */

#if ( 1 )
  rc = mc13892_battery_sample (1, 0, 1);		/* fake ONLINE/noWAIT/FORCE */
  raw = batt_sample.raw;
#endif

#if ( USE_SW_MONITOR_TIME_OUT )
#if ( USE_SOFT_WITH_AUTO )
  // no action
#else
  //add by pega Huimin 2010.1.4
#if ( 1 )
  ret = pmic_read_reg(REG_INT_SENSE0, &value, BITFMASK(BIT_CHG_DETS));
  if (ret == 0) {
    int ac = isACadapter ();
    /*
     * check AC adapter use
     */
    if (( ac ) && ( BITFEXT(value, BIT_CHG_DETS) )) {
      batt_poll_di->charger_online = 1;
    } else {
      batt_poll_di->charger_online = 0;
    }
  } else
    printk("%s:detect charger online error\n",__func__);


#if ( 1 )
  if ( !batt_poll_di->charger_online ) 
  {
    start_time_charging = 0;
    timeout_flag_charging = 0;
    timeout_counter_charging = 0;
  }
  else
  {
    if (!start_time_charging)
      start_time_charging = jiffies_to_msecs(jiffies);
  }

#endif

#if ( 1 )
  //struct timespec *tv1; 	//for testing
  //getnstimeofday(tv1); //for testing

  if (start_time_charging) {
    cur_time_charging = jiffies_to_msecs(jiffies);
    if ((time_diff = (cur_time_charging - start_time_charging)) >= timeout_limitation) {
      //timeout_flag_charging = 1; //do nothing
    }

    printk("%s:cur_time_charging:%x msec, start_time_charging:%x msec, timeout_limitation:%x msec, timeout_flag:%d, time_diff:%d\n",
        __func__, cur_time_charging, start_time_charging, timeout_limitation, timeout_flag_charging, time_diff);

  }

#endif


#if ( 1 )
  if (start_time_charging) {
    timeout_counter_charging += BATTERY_SYSTEM_SUSPEND_POLL_INTERVAL;
    if (timeout_counter_charging >= BATTERY_TIMEOUT_LIMIT )	  
      timeout_flag_charging = 1;

    printk("%s:timeout_counter_charging:%d, timeout_flag:%d\n",
        __func__, timeout_counter_charging, timeout_flag_charging);
  }
#endif


  if (batt_poll_di->charger_online && pmic_stop_charging() ) {

    mdelay(1000);
    if (pmic_get_batt_voltage(&(batt_poll_di->voltage_raw)))
      printk("%s:online get_batt_voltage\n",__func__);

#ifdef CONFIG_MACH_MX51_SENDAI
    batt_poll_di->voltage_uV = BATT_RAW_TO_uV(batt_poll_di->voltage_raw);
#else
    batt_poll_di->voltage_uV = batt_poll_di->voltage_raw * BAT_VOLTAGE_UNIT_UV;
#endif /* CONFIG_MACH_MX51_SENDAI */

    if (batt_poll_di->voltage_uV >= BAT_VOLTAGE_UPPER)
      charging_flag = 1;

    if (batt_poll_di->voltage_uV <= BAT_VOLTAGE_LOWER)
      charging_flag = 0;

    printk ("%s: %d,%d\n", __func__, batt_poll_di->voltage_uV, charging_flag);

#if ( 1 )
    if (charging_flag == 0 && !timeout_flag_charging)
      pmic_restart_charging();
    else
      pmic_stop_charging();
#else
    if (charging_flag == 0)
      pmic_restart_charging();
    else
      pmic_stop_charging();
#endif

  }
  else
  {
    if (pmic_get_batt_voltage(&(batt_poll_di->voltage_raw)))
      printk("%s:offline get_batt_voltage error\n",__func__);

#ifdef CONFIG_MACH_MX51_SENDAI
    batt_poll_di->voltage_uV = BATT_RAW_TO_uV(batt_poll_di->voltage_raw);
#else
    batt_poll_di->voltage_uV = batt_poll_di->voltage_raw * BAT_VOLTAGE_UNIT_UV;
#endif /* CONFIG_MACH_MX51_SENDAI */
    printk("%s:%d\n",__func__,batt_poll_di->voltage_uV);
  }

#endif
#endif /* USE_SOFT_WITH_AUTO */
#endif /* USE_SW_MONITOR_TIME_OUT */

  pmic_event_for_suspend (EVENT_TODAI);		/* ONLY enable ALARM interrupt */
  if (pmic_adc_driver_ldm.suspend != 0) {
    (*pmic_adc_driver_ldm.suspend)((struct platform_device *)0, PMSG_SUSPEND);
  }

#if ( 1 )	
  if (0 < rc) {
    if (raw <= batt_cap[BATT_CAP_SHUTDOWN_LEVEL].raw) {
      battery_low_suspend_cancel ();
      battery_poll_statistics [5]++;
      LogIn (0x00070098, (battery_poll_statistics [5]<<16)|(battery_poll_statistics [6]&0xFFFF), (raw<<16)|batt_cap[BATT_CAP_SHUTDOWN_LEVEL].raw);
      return 1;	/* shutdown voltage */
    }
  }
#endif

  //add by pega Huimin 2010.1.4
#if ( 0 )
  if (!batt_poll_di->charger_online && batt_poll_di->voltage_uV <= BAT_VOLTAGE_LOW_SHUTDOWN) {
    battery_low_suspend_cancel ();
    battery_poll_statistics [5]++;
    return 1;	/* shutdown voltage */			
  }
#endif


  return 0;
}

/*
 * Battery low check initial
 */
static void battery_low_init (struct mc13892_dev_info *di)
{
  extern int resume_check_register_ops (int (*func)(void));
  int rc;

  batt_poll_di = di;

  rc = resume_check_register_ops (battery_low_resume_check);
  if (rc != 0) {
    printk ("battery_low_init: resume_check_register_ops error %d\n", rc);
  }
}

/*
 * Battery low check remove
 */
static void battery_low_remove (struct mc13892_dev_info *di)
{
  extern int resume_check_unregister_ops (int (*func)(void));
  int rc;

  rc = resume_check_unregister_ops (battery_low_resume_check);
  if (rc != 0) {
    printk ("battery_low_remove: resume_check_unregister_ops error %d\n", rc);
  }
}

#endif /* USE_BATTERY_LOW_POLLING */




//Modify by pega Huimin
#ifdef CONFIG_PM
static int pmic_battery_suspend (struct platform_device *pdev, pm_message_t state)
{


#if( USE_BATTERY_LOW_INTERRUPT )
  battery_low_suspend ();
#endif /* USE_BATTERY_LOW_INTERRUPT */


#if( USE_BATTERY_LOW_POLLING )
  battery_low_suspend ();
#endif /* USE_BATTERY_LOW_POLLING */

#if ( USE_SOFT_WITH_AUTO )
  pmic_battery_start_charge ();
#endif

  return 0;
}

static int pmic_battery_resume (struct platform_device *pdev)
{


#if( USE_BATTERY_LOW_INTERRUPT )
  battery_low_resume ();
#endif /* USE_BATTERY_LOW_INTERRUPT */

#if( USE_BATTERY_LOW_POLLING )
  battery_low_resume ();
#endif /* USE_BATTERY_LOW_POLLING */

#if ( USE_SOFT_WITH_AUTO )
  pmic_restart_charging();
#endif

  return 0;
}
#else
#define pmic_battery_suspend 0
#define pmic_battery_resume  0
#endif /* CONFIG_PM */


//Modify by pega Huimin
#if ( USE_SW_MONITOR_TIME_OUT )

static int pmic_stop_charging(void)
{

  int ret;
  unsigned int reg_value = 0;

  pmic_set_chg_misc(AUTO_CHG_DIS, 1);	
  pmic_set_chg_misc(BATTFET_CTL_EN, 1);
  pmic_set_chg_misc(BATTFET_CTL, 0);
  pmic_set_chg_current(CHARGE_CURRENT); //ICHRG[3:0] change from 1000 (720mA) to 0110(520mA)

#if ( USE_CHARGER_LED_FIX )
  pmic_set_chg_misc(CHG_LED_EN, !charging_flag);
#endif

  ret = pmic_read_reg (REG_CHARGE, &reg_value, 0x00ffffff);
  if(ret == 0)
    printk("--%s:Read REG(%d,0x%x)\n",__func__,REG_CHARGE, reg_value);

  return 1;

}


static void battery_chgfaulti_work(struct work_struct *work)
{
  struct mc13892_dev_info *di = container_of(work,
      struct mc13892_dev_info,
      sw_monitor_work.work);
  //add by pega Huimin 2010,0316
  int time_diff = 0;

#if ( 1 )
  if (start_time_charging) {
    cur_time_charging = jiffies_to_msecs(jiffies);
    if ((time_diff = (cur_time_charging - start_time_charging)) >= timeout_limitation) {
      //timeout_flag_charging = 1; //do nothing
    }
  }
  printk("%s:cur_time_charging:%x msec, start_time_charging:%x msec, timeout_limitation:%x msec, timeout_flag:%d, time_diff:%d\n",
      __func__, cur_time_charging, start_time_charging, timeout_limitation, timeout_flag_charging, time_diff);
#endif


#if ( 1 )
  if ( start_time_charging ) {
    timeout_counter_charging += BATTERY_SYSTEM_ON_POLL_INTERVAL;
    if (timeout_counter_charging >= BATTERY_TIMEOUT_LIMIT )	  
      timeout_flag_charging = 1;

    printk("%s:timeout_counter_charging:%d, timeout_flag:%d\n",
        __func__, timeout_counter_charging, timeout_flag_charging);		
  }
#endif



  if (di->charger_online && pmic_stop_charging() ) {

    msleep(1000);
    if (pmic_get_batt_voltage(&(di->voltage_raw)))
      return;

#ifdef CONFIG_MACH_MX51_SENDAI
    di->voltage_uV = BATT_RAW_TO_uV(di->voltage_raw);
#else
    di->voltage_uV = di->voltage_raw * BAT_VOLTAGE_UNIT_UV;
#endif /* CONFIG_MACH_MX51_SENDAI */

    if (di->voltage_uV >= BAT_VOLTAGE_UPPER)
      charging_flag = 1;

    if (di->voltage_uV <= BAT_VOLTAGE_LOWER)
      charging_flag = 0;

    printk ("%s: %d,%d\n", __func__, di->voltage_uV, charging_flag);

#if ( 1 )		
    if (charging_flag == 0 && !timeout_flag_charging)
      pmic_restart_charging();
#else		
    if (charging_flag == 0)
      pmic_restart_charging();
#endif


  }

  queue_delayed_work(di->sw_monitor_wqueue, &di->sw_monitor_work, HZ * batt_on_poll_interval); //modify by Huimin from 60*HZ

  return;
}



static void battery_chgfaulti_callback (void *para)
{
  struct mc13892_dev_info *di= para;
  const int interval = HZ * 1;

  int ret;
  unsigned int value;

  DEBUG_PRINTK(("[%s] \n",  __func__));

  ret = pmic_read_reg (REG_INT_SENSE0, &value, 0x00ffffff);
  printk ("battery_chgfaulti_callback: reg2 (%d %x)\n", ret, value);

  if ( ret == PMIC_SUCCESS && ((value & (0x1 << SENSE_CHGRSHORTS | 0x1 << SENSE_CHGREVS)) >> SENSE_CHGREVS) == 0x2  ) 
    queue_delayed_work(di->sw_monitor_wqueue, &di->sw_monitor_work, interval);

}

#endif

#if ( USE_RESTART_TIME_OUT )
static void battery_chgfaulti_callback2 (void *para)
{
  int ret;
  unsigned int value;

  DEBUG_PRINTK(("[%s] \n",  __func__));

  ret = pmic_read_reg (REG_INT_SENSE0, &value, 0x00ffffff);
  printk ("battery_chgfaulti_callback2: reg2 (%d %x)\n", ret, value);

  if ( ret == PMIC_SUCCESS && ((value & (0x1 << SENSE_CHGRSHORTS | 0x1 << SENSE_CHGREVS)) >> SENSE_CHGREVS) == 0x2  ) {
    printk ("battery_chgfaulti_callback2: Restart Timer");
    pmic_set_chg_misc(RESTART_CHG_TIMER, 1);
    pmic_set_chg_misc(RESTART_CHG_STAT, 1);
  }
}
#endif



static int pmic_battery_remove(struct platform_device *pdev)
{
  pmic_event_callback_t bat_event_callback;
  struct mc13892_dev_info *di = platform_get_drvdata(pdev);

  DEBUG_PRINTK(("[%s] \n",  __func__));

  //Modify by pega Huimin
#if ( USE_BATTERY_LOW_INTERRUPT )
  battery_low_remove (di);
#endif
#if (USE_BATTERY_LOW_POLLING)
  battery_low_remove (di);
#endif
  bat_event_callback.func = charger_online_event_callback;
  bat_event_callback.param = (void *) di;
  pmic_event_unsubscribe(EVENT_CHGDETI, bat_event_callback);

  cancel_rearming_delayed_workqueue(di->monitor_wqueue,
      &di->monitor_work);
  destroy_workqueue(di->monitor_wqueue);
  power_supply_unregister(&di->bat);
  power_supply_unregister(&di->charger);

  kfree(di);

  return 0;
}

#ifdef CONFIG_MACH_MX51_SENDAI
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
static int proc_battery_show(struct seq_file *m, void *v)
{
	unsigned short    mV;
	int               raw;
	int               i, mx;
	struct _batt_cap *tbl = &(batt_cap [0]);

	if (pmic_get_voltage(BATTERY_VOLTAGE, &mV, &raw, 1) == 0) {
		seq_printf (m, "%dmV (%d)  ", mV, raw);
	} else {
		seq_printf (m, "---mV (---)  ");
	}
	mx = sizeof(batt_cap) / sizeof(struct _batt_cap);
	for ( i = 0 ; i < mx ; i++ , tbl++ ) {
		seq_printf (m, " %d:%d", tbl->raw, tbl->cap);
	}

	seq_printf (m, " caution(%d:%d - %d)", batt_cap[BATT_CAP_CAUTION_LEVEL].raw,
		    batt_cap[BATT_CAP_CAUTION_LEVEL].cap, batt_sample.mode_caution);

#if USE_BATTERY_LOW_INTERRUPT
	if (pmic_get_voltage (GEN_PURPOSE_AD2, &mV, &raw, 0) == 0) {
		seq_printf (m, " bpsns:%dmV(%d)", mV, raw);
	} else {
		seq_printf (m, " bpsns:---mV(---)");
	}
#endif /* USE_BATTERY_LOW_INTERRUPT */
#if USE_BATTERY_LOW_POLLING
	seq_printf (m, " shutdown(%d:%d) poll:%dsec",
		    batt_cap[BATT_CAP_SHUTDOWN_LEVEL].raw,
		    batt_cap[BATT_CAP_SHUTDOWN_LEVEL].cap, batt_poll_interval);
#endif /* USE_BATTERY_LOW_POLLING */

	seq_printf (m, "\n");
	return 0;
}

static int proc_battery_open(struct inode *inode, struct file *file)
{
	return single_open(file, proc_battery_show, NULL);
}

/*
 * update batt_cap[] table
 *  capacity rate
 *  echo "raw:cap raw:cap ..." >/proc/battery
 */
static ssize_t proc_battery_write(struct file *file, const char __user *user,
                                  size_t count, loff_t *data)
{
	int   ret, offset, id, mx, raw, cap, mode;
	char  c, str [64];

	if ((count == 0) || (sizeof(str) <= count)) {
		return -EINVAL;
	}
	ret = copy_from_user(str, user, count);
	if (ret != 0) {
		return -EFAULT;
	}

	/*
	 * front space cut
	 */
	for ( offset = 0 ; offset < count ; offset++ ) {
		c = str [offset];
		if (c != ' ') {
			break;
		}
	}
#if USE_BATTERY_LOW_POLLING
	if (strncmp (&(str [offset]), "poll", 4) == 0) {
		int  val;
		char mm [16];
		if (sscanf (&(str [offset]), "%s %d", mm, &val) == 2) {
			if (0 < val) {
				batt_poll_interval = val;
			}
		}
		return count;
	}
#endif /* USE_BATTERY_LOW_POLLING */

	/*
	 * check sample rate
	 */
	if (strncmp (&(str [offset]), "rate", 4) == 0) {
		int  val;
		char mm [16];
		if (sscanf (&(str [offset]), "%s %d", mm, &val) == 2) {
			if (0 < val) {
				batt_sample_rate = val;
			}
		}
		return count;
	}

	raw  = 0;
	cap  = 0;
	mode = 0;
	id   = 0;
	mx   = sizeof(batt_cap) / sizeof(struct _batt_cap);
	for ( offset = 0 ; offset < count ; offset++ ) {
		c = str [offset];
		if (('0' <= c) && (c <= '9')) {
			if (mode == 0) {
				raw = raw * 10 + (c - '0');
			} else {
				cap = cap * 10 + (c - '0');
			}
		} else if (c == ':') {
			mode = 1;
		}
		if ((c == ' ') || ((offset + 1) == count)) {
			if (mx <= id) {
				break;
			}
			batt_cap [id].raw = raw;
			batt_cap [id].cap = cap;
			id++;
			raw  = 0;
			cap  = 0;
			mode = 0;
		}
	}
	return count;
}

static const struct file_operations battery_proc_fops = {
	.owner   = THIS_MODULE,
	.open    = proc_battery_open,
	.read    = seq_read,
	.write   = proc_battery_write,
	.llseek  = seq_lseek,
	.release = single_release,
};
#endif /* CONFIG_PROC_FS */
#endif /* CONFIG_MACH_MX51_SENDAI */

static int pmic_battery_probe(struct platform_device *pdev)
{
  int retval = 0;
  struct mc13892_dev_info *di;
  pmic_event_callback_t bat_event_callback;
  pmic_version_t pmic_version;

#if ( USE_EVENT_VBUSVI ) // add by pega Kevin 2009.12.01 - use EVENT_VBUSVI = VBUS valid detect
  pmic_event_callback_t vbus_event_callback;
#endif

  /* Only apply battery driver for MC13892 V2.0 due to ENGR108085 */
  pmic_version = pmic_get_version();
  if (pmic_version.revision < 20) {
    pr_debug("Battery driver is only applied for MC13892 V2.0\n");
    return -1;
  }

#if ( 0 ) // modify by pega Huimin 2009.11.18 - enable battery platform
  if (machine_is_mx51_babbage()) {
    pr_debug("mc13892 charger is not used for this platform\n");
    return -1;
  }
#endif

  di = kzalloc(sizeof(*di), GFP_KERNEL);
  if (!di) {
    retval = -ENOMEM;
    goto di_alloc_failed;
  }

  platform_set_drvdata(pdev, di);

  di->dev	= &pdev->dev;
  di->bat.name	= "mc13892_bat";
  di->bat.type = POWER_SUPPLY_TYPE_BATTERY;
  di->bat.properties = mc13892_battery_props;
  di->bat.num_properties = ARRAY_SIZE(mc13892_battery_props);
  di->bat.get_property = mc13892_battery_get_property;
  di->bat.use_for_apm = 1;

  di->battery_status = POWER_SUPPLY_STATUS_UNKNOWN;

  retval = power_supply_register(&pdev->dev, &di->bat);
  if (retval) {
    dev_err(di->dev, "failed to register battery\n");
    goto batt_failed;
  }
  di->charger.name	= "mc13892_charger";
  di->charger.type = POWER_SUPPLY_TYPE_MAINS;
  di->charger.properties = mc13892_charger_props;
  di->charger.num_properties = ARRAY_SIZE(mc13892_charger_props);
  di->charger.get_property = mc13892_charger_get_property;
  retval = power_supply_register(&pdev->dev, &di->charger);
  if (retval) {
    dev_err(di->dev, "failed to register charger\n");
    goto charger_failed;
  }
  INIT_DELAYED_WORK(&di->monitor_work, mc13892_battery_work);
  di->monitor_wqueue = create_singlethread_workqueue(pdev->dev.bus_id);
  if (!di->monitor_wqueue) {
    retval = -ESRCH;
    goto workqueue_failed;
  }
  queue_delayed_work(di->monitor_wqueue, &di->monitor_work, HZ * 10);


  //Modify by pega Huimin 2009/12/11
#if ( USE_SW_MONITOR_TIME_OUT )
  bat_event_callback.func = battery_chgfaulti_callback;
  bat_event_callback.param = (void *) di;
  pmic_event_subscribe(EVENT_CHGFAULTI, bat_event_callback);
  INIT_DELAYED_WORK(&di->sw_monitor_work, battery_chgfaulti_work);
  di->sw_monitor_wqueue = create_singlethread_workqueue("battery_chgfaulti");
  pmic_battery_boot_flag = 1;
  if (!di->sw_monitor_wqueue) {
    retval = -ESRCH;
    goto workqueue_failed;
  }
#endif

#if ( USE_RESTART_TIME_OUT )
  bat_event_callback.func = battery_chgfaulti_callback2;
  bat_event_callback.param = (void *) di;
  pmic_event_subscribe(EVENT_CHGFAULTI, bat_event_callback);
#endif

  bat_event_callback.func = charger_online_event_callback;
  bat_event_callback.param = (void *) di;
  pmic_event_subscribe(EVENT_CHGDETI, bat_event_callback);

#if ( USE_EVENT_VBUSVI ) // add by pega Kevin 2009.12.01 - use EVENT_VBUSVI = VBUS valid detect
  INIT_DELAYED_WORK(&di->vbus_work, vbus_detect_work);
  di->vbus_wqueue = create_singlethread_workqueue("vbus_detect");
  vbus_event_callback.func = vbus_detect_callback;
  vbus_event_callback.param = (void *) di;
  pmic_event_subscribe(EVENT_VBUSVI, vbus_event_callback); //use vbus to detect
#endif

#ifdef CONFIG_MACH_MX51_SENDAI
#ifdef CONFIG_PROC_FS
  proc_create ("battery", 0, NULL, &battery_proc_fops);
#endif /* CONFIG_PROC_FS */
#endif /* CONFIG_MACH_MX51_SENDAI */

  //Modify by pega Huimin 2009/12/8 USE_BATTERY_LOW_INTERRUPT mechanism from spider 1
#if ( USE_BATTERY_LOW_INTERRUPT )
  battery_low_init (di);
#endif


#if (USE_BATTERY_LOW_POLLING)
  battery_low_init (di);
#endif


  pmic_stop_coulomb_counter();
  pmic_calibrate_coulomb_counter();
  goto success;

workqueue_failed:
  power_supply_unregister(&di->charger);
charger_failed:
  power_supply_unregister(&di->bat);
batt_failed:
  kfree(di);
di_alloc_failed:
success:
  dev_dbg(di->dev, "%s battery probed!\n", __func__);
  return retval;


  return 0;
}

static struct platform_driver pmic_battery_driver_ldm = {
  .driver = {
    .name = "pmic_battery",
    .bus = &platform_bus_type,
  },
  .probe = pmic_battery_probe,
  .remove = pmic_battery_remove,
#ifdef CONFIG_PM	
  .suspend = pmic_battery_suspend,
  .resume = pmic_battery_resume,
#endif
};

static int __init pmic_battery_init(void)
{
  pr_debug("PMIC Battery driver loading...\n");
  return platform_driver_register(&pmic_battery_driver_ldm);
}

static void __exit pmic_battery_exit(void)
{
  platform_driver_unregister(&pmic_battery_driver_ldm);
  pr_debug("PMIC Battery driver successfully unloaded\n");
}

module_init(pmic_battery_init);
module_exit(pmic_battery_exit);

MODULE_DESCRIPTION("pmic_battery driver");
MODULE_AUTHOR("Freescale Semiconductor, Inc.");
MODULE_LICENSE("GPL");
