/*
 *  Copyright (C) 2008-2010 Freescale Semiconductor, Inc. All Rights Reserved.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/clk.h>
#include <linux/platform_device.h>
#include <linux/regulator/consumer.h>
#include <linux/suspend.h>
#include <linux/proc_fs.h>
#include <linux/cpufreq.h>
#include <linux/iram_alloc.h>
#include <linux/fsl_devices.h>
#include <linux/gpio.h>
#include <linux/pmic_light.h>
#include <linux/pmic_status.h>
#include <linux/delay.h>
#include <asm/cacheflush.h>
#include <asm/tlb.h>
#include <asm/mach/map.h>
#include <mach/hardware.h>
#include <mach/gpio.h>
#include <mach/iomux-v3.h>
#if defined(CONFIG_ARCH_MX50) || defined (CONFIG_ARCH_MX51)
#include <mach/iomux-mx51.h>
#endif

#define MXC_SRPG_EMPGC0_SRPGCR	(IO_ADDRESS(GPC_BASE_ADDR) + 0x2C0)
#define MXC_SRPG_EMPGC1_SRPGCR	(IO_ADDRESS(GPC_BASE_ADDR) + 0x2D0)
#define DATABAHN_CTL_REG0	0
#define DATABAHN_CTL_REG19	0x4c
#define DATABAHN_CTL_REG79	0x13c
#define DATABAHN_PHY_REG25	0x264
/* ON9 SYSTEM POWER GPIO */
#define ON9_SYSTEM_3V3_ENABLE   	(3*32 + 10) /* GPIO_4_10 */
/* ON9 LCD GPIO */
#define ON9_LCD_BACKLIGHT_ENABLE	(0*32 + 9)  /* GPIO_1_9 */
#define ON9_LCD_3V3_ENABLE		(2*32 + 3)  /* GPIO_3_3 */
#define ON9_LCD_POWER_ENABLE		(3*32 + 9)  /* GPIO_4_9 */
/* ON9 CODEC EN */
#define ON9_CODEC_POWER_EN		(0*32 + 0)  /* GPIO_1_0 */
/* ON9 AMP STBY */
#define ON9_AUDAMP_STBY			(0*32 + 1)  /* GPIO_1_1 */
/* ON9 Audio RT5632 reset */
#define ON9_AUDIO_RST_N			(3*32 + 11) /* GPIO_4_11 */
/* ON9 clock enable */
#define ON9_12M_OSC_EN			(2*32 + 1)  /* GPIO_3_1 */
/* ON9 USB POWER */
#define ON9_USB_POWER_ENABLE		(0*32 + 6)  /* GPIO_1_6 */
/* ON9 INTERNAL KEYBOARD */
#define ON9_KEYBOARD_ENABLE		(2*32 + 4)  /* GPIO_3_4 */
#define ON9_KEYBOARD_RESET		(2*23 + 5 ) /* GPIO_3_5 */
/* ON9 ETHERNET GPIO */
#define ON9_ETHERNET_RESET		(1*32 + 20) /* GPIO_2_20 */
/* ON9 RTL8712 WIFI device */
#define ON9_WIFI_3V3_EN			(0*32 + 3)  /* GPIO_1_3 */
#define ON9_WIFI_PWRDN			(3*32 + 26) /* GPIO_4_26 */
/* ON9 USB HUB RST GPIO */
#define ON9_USB_HUB_RESET		(0*32 + 7)  /* GPIO_1_7 */

static struct pad_desc mx51on9_rtl8712_pads[] = {
	MX51_PAD_SD2_CMD__SD2_CMD,
	MX51_PAD_SD2_CLK__SD2_CLK,
	MX51_PAD_SD2_DATA0__SD2_DATA0,
	MX51_PAD_SD2_DATA1__SD2_DATA1,
	MX51_PAD_SD2_DATA2__SD2_DATA2,
	MX51_PAD_SD2_DATA3__SD2_DATA3,
};

static struct pad_desc mx51on9_rtl8712_pull_down_pads[] = {
	MX51_PAD_SD2_CMD__SD2_CMD_PDOWN,
	MX51_PAD_SD2_CLK__SD2_CLK_PDOWN,
	MX51_PAD_SD2_DATA0__SD2_DATA0_PDOWN,
	MX51_PAD_SD2_DATA1__SD2_DATA1_PDOWN,
	MX51_PAD_SD2_DATA2__SD2_DATA2_PDOWN,
	MX51_PAD_SD2_DATA3__SD2_DATA3_PDOWN,
};

static struct cpu_wp *cpu_wp_tbl;
static struct clk *cpu_clk;
static struct mxc_pm_platform_data *pm_data;

#if defined(CONFIG_CPU_FREQ)
static int org_freq;
extern int cpufreq_suspended;
extern int set_cpu_freq(int wp);
#endif


static struct device *pm_dev;
struct clk *gpc_dvfs_clk;
extern void cpu_do_suspend_workaround(u32 sdclk_iomux_addr);
extern void mx50_suspend(u32 databahn_addr);
extern struct cpu_wp *(*get_cpu_wp)(int *wp);
extern void __iomem *databahn_base;

extern int iram_ready;
void *suspend_iram_base;
void (*suspend_in_iram)(void *sdclk_iomux_addr) = NULL;
void __iomem *suspend_param1;

#define FEC_EN (5*32 + 23) /*GPIO_6_23*/
static void on9_device_power_down(void)
{
	pr_info("%s:Turn off KB related voltage\n", __func__);
	gpio_set_value(ON9_KEYBOARD_ENABLE, 0);
	pr_info("%s:Turn off WIFI related voltage\n", __func__);
	gpio_set_value(ON9_WIFI_PWRDN, 0);
	gpio_set_value(ON9_WIFI_3V3_EN, 0);
	pr_info("%s:Turn off AUDIO related voltage\n", __func__);
	gpio_set_value(ON9_AUDAMP_STBY, 0);
	gpio_set_value(ON9_AUDIO_RST_N, 0);
	gpio_set_value(ON9_12M_OSC_EN, 0);
	gpio_set_value(ON9_CODEC_POWER_EN, 0);
	pr_info("%s:Turn off USB related voltage\n", __func__);
	gpio_set_value(ON9_USB_HUB_RESET, 0);
	gpio_set_value(ON9_USB_POWER_ENABLE, 0);
	pr_info("%s:Turn off LCD related voltage\n", __func__);
	gpio_set_value(ON9_LCD_BACKLIGHT_ENABLE, 0);
	gpio_set_value(ON9_LCD_3V3_ENABLE, 0);
	gpio_set_value(ON9_LCD_POWER_ENABLE, 0);
	pr_info("%s:Turn off SYS_3V SYS_5V voltage\n", __func__);
	gpio_set_value(ON9_SYSTEM_3V3_ENABLE, 0);
}
static int mx5_suspend_enter(suspend_state_t state)
{
	if (gpc_dvfs_clk == NULL)
		gpc_dvfs_clk = clk_get(NULL, "gpc_dvfs_clk");

	on9_device_power_down();

	/* gpc clock is needed for SRPG */
	clk_enable(gpc_dvfs_clk);
	switch (state) {
	case PM_SUSPEND_MEM:
		mxc_cpu_lp_set(STOP_POWER_OFF);
		break;
	case PM_SUSPEND_STANDBY:
		mxc_cpu_lp_set(WAIT_UNCLOCKED_POWER_OFF);
		break;
	default:
		return -EINVAL;
	}

	if (tzic_enable_wake(0) != 0)
		return -EAGAIN;

	if (state == PM_SUSPEND_MEM) {
		local_flush_tlb_all();
		flush_cache_all();

		if (cpu_is_mx51() || cpu_is_mx53()) {
			/* Run the suspend code from iRAM. */
			suspend_in_iram(suspend_param1);

			/*clear the EMPGC0/1 bits */
			__raw_writel(0, MXC_SRPG_EMPGC0_SRPGCR);
			__raw_writel(0, MXC_SRPG_EMPGC1_SRPGCR);
		} else {
			/* Setup GPIO/IOMUX settings to lower power. */
			if (pm_data->suspend_enter)
				pm_data->suspend_enter();
			/* Suspend now. */
			suspend_in_iram(databahn_base);

			if (pm_data->suspend_exit)
				pm_data->suspend_exit();
		}
	} else {
			cpu_do_idle();
	}
	clk_disable(gpc_dvfs_clk);

	return 0;
}

/*
 * Called after processes are frozen, but before we shut down devices.
 */
static int mx5_suspend_prepare(void)
{
#if defined(CONFIG_CPU_FREQ)
	struct cpufreq_freqs freqs;

	pr_info("%s:Turn off ETHERNET related voltage\n", __func__);
	gpio_set_value(ON9_ETHERNET_RESET, 0);
	pmic_write_reg(REG_MODE_1, 0, 1);

	org_freq = clk_get_rate(cpu_clk);
	freqs.old = org_freq / 1000;
	freqs.new = cpu_wp_tbl[0].cpu_rate / 1000;
	freqs.cpu = 0;
	freqs.flags = 0;

	cpufreq_suspended = 1;
	if (clk_get_rate(cpu_clk) != cpu_wp_tbl[0].cpu_rate) {
		set_cpu_freq(cpu_wp_tbl[0].cpu_rate);
		cpufreq_notify_transition(&freqs, CPUFREQ_PRECHANGE);
		cpufreq_notify_transition(&freqs, CPUFREQ_POSTCHANGE);
	}
#endif
	return 0;
}

static void on9_device_power_up(void)
{
	pr_info("%s:Turn on SYS_3V SYS_5V voltage\n", __func__);
	gpio_set_value(ON9_SYSTEM_3V3_ENABLE, 1);
	pr_info("%s:Turn on LCD related voltage\n", __func__);
	gpio_set_value(ON9_LCD_POWER_ENABLE, 1);
	msleep(10);
	gpio_set_value(ON9_LCD_3V3_ENABLE, 1);
	msleep(200);
	gpio_set_value(ON9_LCD_BACKLIGHT_ENABLE, 1);
	pr_info("%s:Turn on KB related voltage\n", __func__);
	gpio_set_value(ON9_KEYBOARD_ENABLE, 1);
	pr_info("%s:Turn on WIFI related voltage\n", __func__);
	mxc_iomux_v3_setup_multiple_pads(mx51on9_rtl8712_pull_down_pads,
					ARRAY_SIZE(mx51on9_rtl8712_pull_down_pads));
	gpio_set_value(ON9_WIFI_3V3_EN, 1);
	msleep(500);
	gpio_set_value(ON9_WIFI_3V3_EN, 0);
	msleep(500);
	gpio_set_value(ON9_WIFI_3V3_EN, 1);
	gpio_set_value(ON9_WIFI_PWRDN, 1);
	mxc_iomux_v3_setup_multiple_pads(mx51on9_rtl8712_pads,
					ARRAY_SIZE(mx51on9_rtl8712_pads));
	pr_info("%s:Turn on AUDIO related voltage\n", __func__);
	gpio_set_value(ON9_CODEC_POWER_EN, 1);
	gpio_set_value(ON9_12M_OSC_EN, 1);
	msleep(5);
	pr_info("%s:Turn on USB related voltage\n", __func__);
	gpio_set_value(ON9_USB_POWER_ENABLE, 1);
	gpio_set_value(ON9_USB_HUB_RESET, 0);
	gpio_set_value(ON9_USB_HUB_RESET, 1);
	gpio_set_value(ON9_AUDIO_RST_N, 0);
	msleep(5);
	gpio_set_value(ON9_AUDIO_RST_N, 1);
	gpio_set_value(ON9_AUDAMP_STBY, 1);
	pr_info("%s:Turn on KB related voltage\n", __func__);
	gpio_set_value(ON9_KEYBOARD_ENABLE, 1);
	gpio_set_value(ON9_KEYBOARD_RESET, 0);
	msleep(5);
	gpio_set_value(ON9_KEYBOARD_RESET, 1);
	pr_info("%s:Turn on ETHERNET related voltage\n", __func__);
	pmic_write_reg(REG_MODE_1, 1, 1);
	gpio_set_value(ON9_ETHERNET_RESET, 0);
	msleep(10);
	gpio_set_value(ON9_ETHERNET_RESET, 1);
}

static void mx5_suspend_wake(void)
{
}

/*
 * Called before devices are re-setup.
 */
static void mx5_suspend_finish(void)
{
#if defined(CONFIG_CPU_FREQ)
	struct cpufreq_freqs freqs;

	freqs.old = clk_get_rate(cpu_clk) / 1000;
	freqs.new = org_freq / 1000;
	freqs.cpu = 0;
	freqs.flags = 0;

	cpufreq_suspended = 0;

	if (org_freq != clk_get_rate(cpu_clk)) {
		set_cpu_freq(org_freq);
		cpufreq_notify_transition(&freqs, CPUFREQ_PRECHANGE);
		cpufreq_notify_transition(&freqs, CPUFREQ_POSTCHANGE);
	}
#endif
	on9_device_power_up();
}

/*
 * Called after devices are re-setup, but before processes are thawed.
 */
static void mx5_suspend_end(void)
{
	pr_info("%s: Turn on the Power LED\n", __func__);
	pmic_write_reg(REG_LED_CTL3, 0xffff, PMIC_ALL_BITS);
}

static int mx5_pm_valid(suspend_state_t state)
{
	return (state > PM_SUSPEND_ON && state <= PM_SUSPEND_MAX);
}

struct platform_suspend_ops mx5_suspend_ops = {
	.valid = mx5_pm_valid,
	.prepare = mx5_suspend_prepare,
	.enter = mx5_suspend_enter,
	.wake = mx5_suspend_wake,
	.finish = mx5_suspend_finish,
	.end = mx5_suspend_end,
};

static int __devinit mx5_pm_probe(struct platform_device *pdev)
{
	pm_dev = &pdev->dev;
	pm_data = pdev->dev.platform_data;

	return 0;
}

static struct platform_driver mx5_pm_driver = {
	.driver = {
		   .name = "mx5_pm",
		   },
	.probe = mx5_pm_probe,
};

static int __init pm_init(void)
{
	int cpu_wp_nr;
	unsigned long iram_paddr;

	pr_info("Static Power Management for Freescale i.MX5\n");
	if (platform_driver_register(&mx5_pm_driver) != 0) {
		printk(KERN_ERR "mx5_pm_driver register failed\n");
		return -ENODEV;
	}
	suspend_set_ops(&mx5_suspend_ops);
	/* Move suspend routine into iRAM */
	iram_alloc(SZ_4K, &iram_paddr);
	/* Need to remap the area here since we want the memory region
		 to be executable. */
	suspend_iram_base = __arm_ioremap(iram_paddr, SZ_4K,
					  MT_HIGH_VECTORS);

	if (cpu_is_mx51() || cpu_is_mx53()) {
		suspend_param1 = IO_ADDRESS(IOMUXC_BASE_ADDR + 0x4b8);
		memcpy(suspend_iram_base, cpu_do_suspend_workaround,
				SZ_4K);
	} else if (cpu_is_mx50()) {
		/*
		 * Need to run the suspend code from IRAM as the DDR needs
		 * to be put into self refresh mode manually.
		 */
		memcpy(suspend_iram_base, mx50_suspend, SZ_4K);

		suspend_param1 = databahn_base;
	}
	suspend_in_iram = (void *)suspend_iram_base;

	cpu_wp_tbl = get_cpu_wp(&cpu_wp_nr);

	cpu_clk = clk_get(NULL, "cpu_clk");
	if (IS_ERR(cpu_clk)) {
		printk(KERN_DEBUG "%s: failed to get cpu_clk\n", __func__);
		return PTR_ERR(cpu_clk);
	}
	printk(KERN_INFO "PM driver module loaded\n");

	return 0;
}


static void __exit pm_cleanup(void)
{
	/* Unregister the device structure */
	platform_driver_unregister(&mx5_pm_driver);
}

module_init(pm_init);
module_exit(pm_cleanup);

MODULE_AUTHOR("Freescale Semiconductor, Inc.");
MODULE_DESCRIPTION("PM driver");
MODULE_LICENSE("GPL");
